/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.CastExpression;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.NullType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.UndefinedType;
import org.eclipse.n4js.ts.types.util.AllSuperTypesCollector;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.validators.N4JSDependencyInjectionValidator;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Validations related to callsites targeting N4Injector methods.
 * <p>
 * For other DI-related validations see {@link N4JSDependencyInjectionValidator}
 */
@SuppressWarnings("all")
public class N4JSInjectorCallsitesValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Detects callsites targeting either N4Injector.of() or N4Injector.create()
   * and forwards to appropriate validator.
   */
  @Check
  public void checkCallExpression(final ParameterizedCallExpression callExpression) {
    if ((null == callExpression)) {
      return;
    }
    Expression _target = callExpression.getTarget();
    boolean _not = (!(_target instanceof ParameterizedPropertyAccessExpression));
    if (_not) {
      return;
    }
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(callExpression.getArguments());
    if (_isNullOrEmpty) {
      return;
    }
    Expression _target_1 = callExpression.getTarget();
    final ParameterizedPropertyAccessExpression propAccess = ((ParameterizedPropertyAccessExpression) _target_1);
    if ((null == propAccess)) {
      return;
    }
    IdentifiableElement _property = propAccess.getProperty();
    boolean _not_1 = (!(_property instanceof TMethod));
    if (_not_1) {
      return;
    }
    IdentifiableElement _property_1 = propAccess.getProperty();
    final TMethod property = ((TMethod) _property_1);
    EObject _eContainer = property.eContainer();
    boolean _not_2 = (!(_eContainer instanceof TClass));
    if (_not_2) {
      return;
    }
    EObject _eContainer_1 = property.eContainer();
    final TClass tclassOfReceiver = ((TClass) _eContainer_1);
    boolean _isInjectorType = this.isInjectorType(tclassOfReceiver);
    if (_isInjectorType) {
      String _name = property.getName();
      boolean _equals = Objects.equal(_name, "of");
      if (_equals) {
        this.internalCheckInjectorOfCallsite(callExpression);
      } else {
        String _name_1 = property.getName();
        boolean _equals_1 = Objects.equal(_name_1, "create");
        if (_equals_1) {
          this.internalCheckInjectorCreateCallsite(callExpression);
        }
      }
    }
  }
  
  /**
   * This method validates callsites of the form:
   * <p/>
   * <code>
   * public static N4Injector of(constructor{N4Object} ctorOfDIC, N4Injector? parentDIC, N4Object... providedBinders)
   * </code>
   * <p/>
   * At runtime:
   * <p/>
   * <ul>
   * <li>(1) the first argument denotes a DIC constructor</li>
   * <li>(2) the second (optional) argument is an injector</li>
   * <li>(3) lastly, the purpose of providedBinders is as follows:
   * <ul>
   * <li>the DIC above in (1) is marked with one or more (at)UseBinder.</li>
   * <li>Some of those binders may require injection (to recap, a classifier declaring injected members is said to require injection).</li>
   * <li>Some of those binders may have constructor(s) taking params.</li>
   * <li>The set of binders described above should match the providedBinders.</li>
   * </ul>
   * </li>
   * </ul>
   */
  private void internalCheckInjectorOfCallsite(final ParameterizedCallExpression callExpression) {
    final Function1<Argument, Expression> _function = (Argument it) -> {
      return it.getExpression();
    };
    final List<Expression> args = ListExtensions.<Argument, Expression>map(callExpression.getArguments(), _function);
    Expression ctorOfDICArg = null;
    Expression _get = args.get(0);
    if ((_get instanceof CastExpression)) {
      Expression _get_1 = args.get(0);
      ctorOfDICArg = ((CastExpression) _get_1).getExpression();
    } else {
      ctorOfDICArg = args.get(0);
    }
    final TClass dicTClass = this.holdsDenotesDICConstructor(ctorOfDICArg);
    if ((null == dicTClass)) {
      return;
    }
    final Iterable<TClass> usedBinders = N4JSDependencyInjectionValidator.usedBindersOf(dicTClass);
    final Function1<TClass, Boolean> _function_1 = (TClass binder) -> {
      return Boolean.valueOf((N4JSDependencyInjectionValidator.requiresInjection(binder) || N4JSInjectorCallsitesValidator.hasNonEmptyCtor(binder)));
    };
    final Iterable<TClass> bindersForWhichInstancesAreNeeded = IterableExtensions.<TClass>filter(usedBinders, _function_1);
    int _size = args.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      this.holdsNoProvidedBindersNeeded(callExpression, bindersForWhichInstancesAreNeeded);
      return;
    }
    Expression parentDIC = args.get(1);
    this.holdsDenotesInjector(parentDIC);
    int _size_1 = args.size();
    boolean _equals_1 = (_size_1 == 2);
    if (_equals_1) {
      this.holdsNoProvidedBindersNeeded(callExpression, bindersForWhichInstancesAreNeeded);
      return;
    }
    List<Expression> providedBinders = args.subList(2, args.size());
    this.holdsAllNeededBinderInstancesAreProvided(callExpression, providedBinders, bindersForWhichInstancesAreNeeded);
  }
  
  /**
   * Does the argument declare (or inherit) a ctor requiring parameters?
   */
  private static boolean hasNonEmptyCtor(final TClass tClass) {
    final Function1<TClassifier, Boolean> _function = (TClassifier t) -> {
      final Function1<TMember, Boolean> _function_1 = (TMember ctor) -> {
        return Boolean.valueOf((ctor.isConstructor() && N4JSInjectorCallsitesValidator.lacksParams(((TMethod) ctor))));
      };
      return Boolean.valueOf(IterableExtensions.<TMember>exists(t.getOwnedMembers(), _function_1));
    };
    return IterableExtensions.<TClassifier>exists(AllSuperTypesCollector.collect(tClass), _function);
  }
  
  private static boolean lacksParams(final TMethod m) {
    return m.getFpars().isEmpty();
  }
  
  /**
   * A callsite target N4Injector.of(ctorOfDIC, ...) but lacks providedBinders.
   * If one or more are required (as dictated by "ctorOfDIC") this method raises an issue positioned at the callsite.
   * 
   * TODO is it possible for a parent injector to also require provided-binders? should we also issue an error for them?
   */
  private void holdsNoProvidedBindersNeeded(final ParameterizedCallExpression callExpression, final Iterable<TClass> bindersForWhichInstancesAreNeeded) {
    boolean _isEmpty = IterableExtensions.isEmpty(bindersForWhichInstancesAreNeeded);
    if (_isEmpty) {
      return;
    }
    final Function1<TClass, String> _function = (TClass binder) -> {
      return binder.getTypeAsString();
    };
    final String missing = IterableExtensions.join(IterableExtensions.<TClass, String>map(bindersForWhichInstancesAreNeeded, _function), ", ");
    final String msg = IssueCodes.getMessageForDI_ANN_MISSING_PROVIDED_BINDERS(missing);
    this.addIssue(msg, callExpression, IssueCodes.DI_ANN_MISSING_PROVIDED_BINDERS);
  }
  
  /**
   * Expression must denote a DIC constructor. If so, return the TClass of the DIC. Otherwise return null.
   */
  private TClass holdsDenotesDICConstructor(final Expression ctorOfDICArg) {
    final TypeRef ctorOfDICTypeRef = this.ts.tau(ctorOfDICArg);
    if ((ctorOfDICTypeRef instanceof TypeTypeRef)) {
      TypeArgument _typeArg = ((TypeTypeRef)ctorOfDICTypeRef).getTypeArg();
      if ((_typeArg instanceof TypeRef)) {
        TypeArgument _typeArg_1 = ((TypeTypeRef)ctorOfDICTypeRef).getTypeArg();
        final TClass dicTClass = N4JSInjectorCallsitesValidator.dicTClassOf(((TypeRef) _typeArg_1));
        if ((null != dicTClass)) {
          return dicTClass;
        }
      }
    }
    this.addIssue(IssueCodes.getMessageForDI_ANN_INJECTOR_REQUIRED(), ctorOfDICArg, IssueCodes.DI_ANN_INJECTOR_REQUIRED);
    return null;
  }
  
  /**
   * In case the argument refers to class marked (at)GenerateInjector, returns its TClass.
   * Otherwise null.
   */
  private static TClass dicTClassOf(final TypeRef ref) {
    final TClass injtorClassDecl = N4JSDependencyInjectionValidator.tClassOf(ref);
    if (((null != injtorClassDecl) && AnnotationDefinition.GENERATE_INJECTOR.hasAnnotation(injtorClassDecl))) {
      return injtorClassDecl;
    }
    return null;
  }
  
  /**
   * This method validates callsites of the form:
   * <p/>
   * <code>
   * public static N4Injector create(type{T} ctor)
   * </code>
   * <p/>
   * Validation: <code>type{T}</code> should be injectable (in particular, it may be an N4Provider).
   */
  private void internalCheckInjectorCreateCallsite(final ParameterizedCallExpression callExpression) {
    Argument _head = IterableExtensions.<Argument>head(callExpression.getArguments());
    Expression _expression = null;
    if (_head!=null) {
      _expression=_head.getExpression();
    }
    final Expression ctorArg = _expression;
    final TypeRef ctorArgTypeRef = this.ts.tau(ctorArg);
    if ((ctorArgTypeRef instanceof TypeTypeRef)) {
      boolean _isInjectableType = N4JSDependencyInjectionValidator.isInjectableType(((TypeTypeRef)ctorArgTypeRef).getTypeArg());
      if (_isInjectableType) {
        return;
      }
    }
    this.addIssue(IssueCodes.getMessageForDI_NOT_INJECTABLE(ctorArgTypeRef.getTypeRefAsString(), ""), ctorArg, IssueCodes.DI_NOT_INJECTABLE);
  }
  
  /**
   * Is type one of: N4Injector, NullType, or UndefinedType?
   */
  private boolean isAllowedAsInjectorInstance(final Type type) {
    return (N4JSInjectorCallsitesValidator.isNullOrUndefinedType(type) || this.isInjectorType(type));
  }
  
  /**
   * Is the argument (a non-null, non-undefined) N4Injector?
   */
  private boolean isInjectorType(final Type type) {
    if ((!(type instanceof TClass))) {
      return false;
    }
    String _name = type.getName();
    boolean _equals = Objects.equal(_name, "N4Injector");
    boolean _not = (!_equals);
    if (_not) {
      return false;
    }
    final String fqn = this.resourceNameComputer.getFullyQualifiedTypeName(type);
    return Objects.equal(fqn, "runtime.n4.N4Injector.N4Injector");
  }
  
  private static boolean isNullOrUndefinedType(final Type type) {
    return ((type instanceof NullType) || (type instanceof UndefinedType));
  }
  
  /**
   * Expression must denote at runtime an N4Injector instance, null, or undefined.
   */
  private void holdsDenotesInjector(final Expression expr) {
    final TypeRef tr = this.ts.tau(expr);
    boolean _isAllowedAsInjectorInstance = this.isAllowedAsInjectorInstance(tr.getDeclaredType());
    if (_isAllowedAsInjectorInstance) {
      return;
    }
    this.addIssue(IssueCodes.getMessageForDI_ANN_INJECTOR_MISSING(), expr, IssueCodes.DI_ANN_INJECTOR_MISSING);
  }
  
  /**
   * This method checks that instances are provided for the binders that are in use and that moreover:
   * <ul>
   * <li>require injection (to recap, a classifier declaring injected members is said to require injection)</li>
   * <li>or, have constructor(s) taking params.</li>
   * </ul>
   */
  private void holdsAllNeededBinderInstancesAreProvided(final ParameterizedCallExpression callExpression, final List<Expression> providedBinders, final Iterable<TClass> bindersForWhichInstancesAreNeeded) {
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(providedBinders.get(0));
    final Function1<Expression, TypeRef> _function = (Expression expr) -> {
      return this.ts.tau(expr);
    };
    final Function1<TypeRef, Boolean> _function_1 = (TypeRef tr) -> {
      boolean _isNullOrUndefinedType = N4JSInjectorCallsitesValidator.isNullOrUndefinedType(tr.getDeclaredType());
      return Boolean.valueOf((!_isNullOrUndefinedType));
    };
    final Iterable<TypeRef> availableTypeRefs = IterableExtensions.<TypeRef>filter(ListExtensions.<Expression, TypeRef>map(providedBinders, _function), _function_1);
    final Function1<TClass, Boolean> _function_2 = (TClass requirement) -> {
      boolean _someBinderSatisfies = this.someBinderSatisfies(G, availableTypeRefs, requirement);
      return Boolean.valueOf((!_someBinderSatisfies));
    };
    final Iterable<TClass> missingBinders = IterableExtensions.<TClass>filter(bindersForWhichInstancesAreNeeded, _function_2);
    boolean _isEmpty = IterableExtensions.isEmpty(missingBinders);
    if (_isEmpty) {
      return;
    }
    final Function1<TClass, String> _function_3 = (TClass binder) -> {
      return binder.getTypeAsString();
    };
    final String missing = IterableExtensions.join(IterableExtensions.<TClass, String>map(missingBinders, _function_3), ", ");
    final String msg = IssueCodes.getMessageForDI_ANN_MISSING_NEEDED_BINDERS(missing);
    this.addIssue(msg, callExpression, IssueCodes.DI_ANN_MISSING_NEEDED_BINDERS);
  }
  
  /**
   * Does one of the available instances serve as binder for the required binder TClass?
   */
  private boolean someBinderSatisfies(final RuleEnvironment G, final Iterable<? extends TypeRef> availableTypeRefs, final TClass requirement) {
    final Function1<TypeRef, Boolean> _function = (TypeRef availableTR) -> {
      return Boolean.valueOf(this.ts.subtypeSucceeded(G, availableTR, TypeUtils.createTypeRef(requirement)));
    };
    return IterableExtensions.exists(availableTypeRefs, _function);
  }
}
