/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.ResourceType;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.parser.IParseResult;
import org.eclipse.xtext.resource.IContainer;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Contains module-level validations, i.e. validations that need to be checked once per module / file.
 * For example: unique module names.
 * <p>
 * It contains a helper method to conveniently add issues that do not relate to a particular statement, expression, etc.
 * but instead relate to the entire file. See {@link N4JSModuleValidator#addIssue(String,Script,String)}
 */
@SuppressWarnings("all")
public class N4JSModuleValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private ResourceDescriptionsProvider resourceDescriptionsProvider;
  
  @Inject
  private IResourceDescription.Manager resourceDescriptionManager;
  
  @Inject
  private IContainer.Manager containerManager;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  @Inject
  private IN4JSCore n4jscore;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkModuleSpecifier(final Script script) {
    final ResourceType resType = ResourceType.getResourceType(script.getModule().eResource());
    final boolean correctType = (((resType == ResourceType.N4JS) || (resType == ResourceType.N4JSD)) || (resType == ResourceType.N4JSX));
    final String moduleQN = script.getModule().getQualifiedName();
    if ((moduleQN.equals("js.cookie") || moduleQN.endsWith("/js.cookie"))) {
      return;
    }
    if ((correctType && moduleQN.contains("."))) {
      final QualifiedName qualifiedName = this.qualifiedNameConverter.toQualifiedName(moduleQN);
      final boolean filenameContainsDot = qualifiedName.getLastSegment().contains(".");
      final Function1<String, Boolean> _function = (String s) -> {
        return Boolean.valueOf(s.contains("."));
      };
      String _findFirst = IterableExtensions.<String>findFirst(qualifiedName.skipLast(1).getSegments(), _function);
      final boolean moduleFoldersContainDot = (_findFirst != null);
      ArrayList<Object> locationSubject = CollectionLiterals.<Object>newArrayList();
      if (filenameContainsDot) {
        locationSubject.add("filename");
      }
      if (moduleFoldersContainDot) {
        locationSubject.add("containing folders");
      }
      final String locationSubjectDescription = StringExtensions.toFirstUpper(IterableExtensions.join(locationSubject, " and "));
      this.addIssue(IssueCodes.getMessageForMOD_NAME_MUST_NOT_CONTAIN_DOTS(locationSubjectDescription, script.getModule().getQualifiedName()), script, 0, 0, IssueCodes.MOD_NAME_MUST_NOT_CONTAIN_DOTS);
    }
  }
  
  /**
   * Make sure .n4js files are not located in runtime environments or runtime libraries.
   */
  @Check
  public void checkNoN4jsInRuntimeEnvOrLib(final Script script) {
    boolean _requirecheckNoN4jsInRuntimeEnvOrLib = this.jsVariantHelper.requirecheckNoN4jsInRuntimeEnvOrLib(script);
    if (_requirecheckNoN4jsInRuntimeEnvOrLib) {
      final IN4JSProject n4jsProject = this.n4jscore.findProject(script.eResource().getURI()).orNull();
      if ((n4jsProject != null)) {
        final ProjectType projectType = n4jsProject.getProjectType();
        if (((projectType == ProjectType.RUNTIME_ENVIRONMENT) || (projectType == ProjectType.RUNTIME_LIBRARY))) {
          this.addIssue(IssueCodes.getMessageForNO_N4JS_IN_RUNTIME_COMPONENT(), script, IssueCodes.NO_N4JS_IN_RUNTIME_COMPONENT);
        }
      }
    }
  }
  
  /**
   * Validates the module name that is derived from the given script.
   * 
   * This validation is defined for the script itself since we need it for the error marker anyway
   */
  @Check
  public void checkUniqueName(final Script script) {
    String _fileExtension = script.eResource().getURI().fileExtension();
    boolean _notEquals = (!Objects.equal(_fileExtension, N4JSGlobals.JS_FILE_EXTENSION));
    if (_notEquals) {
      this.checkUniqueName(script, script.getModule());
    }
  }
  
  /**
   * If the module exists, obtain its name and validate it for uniqueness.
   */
  private void checkUniqueName(final Script script, final TModule module) {
    if ((module == null)) {
      return;
    }
    final QualifiedName name = this.qualifiedNameConverter.toQualifiedName(module.getQualifiedName());
    if ((name != null)) {
      this.doCheckUniqueName(script, module, name);
    }
  }
  
  /**
   * Find all other modules in the index with the same name and check all the obtained
   * candidates for reachable duplicates.
   */
  private void doCheckUniqueName(final Script script, final TModule module, final QualifiedName name) {
    Resource _eResource = module.eResource();
    final XtextResource resource = ((XtextResource) _eResource);
    final IResourceDescriptions index = this.resourceDescriptionsProvider.getResourceDescriptions(resource);
    final Iterable<IEObjectDescription> others = index.getExportedObjects(TypesPackage.Literals.TMODULE, name, false);
    final Provider<List<IContainer>> _function = () -> {
      IResourceDescription _elvis = null;
      IResourceDescription _resourceDescription = index.getResourceDescription(resource.getURI());
      if (_resourceDescription != null) {
        _elvis = _resourceDescription;
      } else {
        IResourceDescription _resourceDescription_1 = this.resourceDescriptionManager.getResourceDescription(resource);
        _elvis = _resourceDescription_1;
      }
      final IResourceDescription description = _elvis;
      if ((description == null)) {
        return CollectionLiterals.<IContainer>emptyList();
      }
      return this.containerManager.getVisibleContainers(description, index);
    };
    this.checkUniqueInIndex(script, module, others, _function);
  }
  
  /**
   * Check the found candidates for reach	ability via the visible containers.
   */
  private void checkUniqueInIndex(final Script script, final TModule module, final Iterable<IEObjectDescription> descriptions, final Provider<List<IContainer>> lazyContainersList) {
    final Function1<IEObjectDescription, URI> _function = (IEObjectDescription it) -> {
      return it.getEObjectURI().trimFragment();
    };
    final Function1<URI, Boolean> _function_1 = (URI it) -> {
      return Boolean.valueOf(((!Objects.equal(it, EcoreUtil2.getPlatformResourceOrNormalizedURI(module.eResource()))) && (!Objects.equal(it.fileExtension(), N4JSGlobals.JS_FILE_EXTENSION))));
    };
    final Set<URI> resourceURIs = IterableExtensions.<URI>toSet(IterableExtensions.<URI>filter(IterableExtensions.<IEObjectDescription, URI>map(descriptions, _function), _function_1));
    int _size = resourceURIs.size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      final HashSet<URI> visibleResourceURIs = CollectionLiterals.<URI>newHashSet();
      final Consumer<IContainer> _function_2 = (IContainer container) -> {
        final Function1<URI, Boolean> _function_3 = (URI uri) -> {
          return Boolean.valueOf(container.hasResourceDescription(uri));
        };
        Iterable<URI> _filter = IterableExtensions.<URI>filter(resourceURIs, _function_3);
        Iterables.<URI>addAll(visibleResourceURIs, _filter);
      };
      lazyContainersList.get().forEach(_function_2);
      int _size_1 = visibleResourceURIs.size();
      boolean _greaterThan_1 = (_size_1 > 0);
      if (_greaterThan_1) {
        boolean _isStaticPolyfillAware = module.isStaticPolyfillAware();
        if (_isStaticPolyfillAware) {
          int _size_2 = visibleResourceURIs.size();
          boolean _equals = (_size_2 == 1);
          if (_equals) {
            return;
          }
        } else {
          boolean _isStaticPolyfillModule = module.isStaticPolyfillModule();
          if (_isStaticPolyfillModule) {
            int _size_3 = visibleResourceURIs.size();
            boolean _equals_1 = (_size_3 == 1);
            if (_equals_1) {
              return;
            }
          }
        }
        if ((this.hasPolyfill(script) && (!module.isStaticPolyfillModule()))) {
          return;
        }
        Set<URI> filteredMutVisibleResourceURIs = visibleResourceURIs;
        boolean _isMainModule = module.isMainModule();
        if (_isMainModule) {
          final IN4JSProject pr = this.n4jscore.findProject(module.eResource().getURI()).get();
          final Function1<URI, Boolean> _function_3 = (URI u) -> {
            IN4JSProject _get = this.n4jscore.findProject(u).get();
            boolean _equals_2 = Objects.equal(pr, _get);
            if (_equals_2) {
              final String baseModuleSrcCon = this.n4jscore.findN4JSSourceContainer(module.eResource().getURI()).get().getLocation().toString();
              final String otherModuleSrcCon = this.n4jscore.findN4JSSourceContainer(u).get().getLocation().toString();
              final String baseModuleSrcContainerRelativePath = module.eResource().getURI().toString().substring(baseModuleSrcCon.length());
              final String otherModuleSrcContainerRelativePath = u.toString().substring(otherModuleSrcCon.length());
              return Boolean.valueOf(Objects.equal(baseModuleSrcContainerRelativePath, otherModuleSrcContainerRelativePath));
            } else {
              return Boolean.valueOf(false);
            }
          };
          filteredMutVisibleResourceURIs = IterableExtensions.<URI>toSet(IterableExtensions.<URI>filter(filteredMutVisibleResourceURIs, _function_3));
        }
        boolean _isEmpty = filteredMutVisibleResourceURIs.isEmpty();
        if (_isEmpty) {
          return;
        }
        final Function1<URI, String> _function_4 = (URI it) -> {
          return IterableExtensions.join(IterableExtensions.<String>drop(it.segmentsList(), 1), "/");
        };
        final String filePathStr = IterableExtensions.join(IterableExtensions.<URI, String>map(filteredMutVisibleResourceURIs, _function_4), "; ");
        final String message = IssueCodes.getMessageForCLF_DUP_MODULE(module.getQualifiedName(), filePathStr);
        this.addIssue(message, script, IssueCodes.CLF_DUP_MODULE);
      }
    }
  }
  
  /**
   * @returns true if the script contains a toplevel-type annotated with a {@code @Polyfill}.
   */
  private boolean hasPolyfill(final Script script) {
    EList<ScriptElement> _scriptElements = script.getScriptElements();
    for (final ScriptElement se : _scriptElements) {
      boolean _matched = false;
      if (se instanceof AnnotableElement) {
        _matched=true;
        boolean _isPolyfill = N4JSLanguageUtils.isPolyfill(((AnnotableElement)se));
        if (_isPolyfill) {
          return true;
        }
      }
    }
    return false;
  }
  
  private static final Pattern nonEmpty = Pattern.compile("^.+$", Pattern.MULTILINE);
  
  /**
   * Annotates the script with a an error marker on the first AST element or the first none-empty line.
   */
  private void addIssue(final String message, final Script script, final String issueCode) {
    final ScriptElement first = IterableExtensions.<ScriptElement>head(script.getScriptElements());
    if ((first != null)) {
      this.addIssue(message, first, issueCode);
      return;
    }
    Resource _eResource = script.eResource();
    final XtextResource resource = ((XtextResource) _eResource);
    final IParseResult parseResult = resource.getParseResult();
    final ICompositeNode rootNode = parseResult.getRootNode();
    final String text = rootNode.getText();
    final Matcher matcher = N4JSModuleValidator.nonEmpty.matcher(text);
    int start = 0;
    int end = text.length();
    boolean _find = matcher.find();
    if (_find) {
      start = matcher.start();
      end = matcher.end();
    }
    this.addIssue(message, script, start, (end - start), issueCode);
  }
}
