/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import javax.swing.text.html.HTML.Tag;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassExpression;
import org.eclipse.n4js.n4JS.N4FieldAccessor;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.es.assistants.BootstrapCallAssistant;
import org.eclipse.n4js.transpiler.es.assistants.ClassConstructorAssistant;
import org.eclipse.n4js.transpiler.es.transform.MemberPatchingTransformation;
import org.eclipse.n4js.transpiler.es.transform.SuperLiteralTransformation;
import org.eclipse.n4js.transpiler.im.DelegatingMember;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.utils.TranspilerUtils;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Transforms {@link N4ClassDeclaration}s into a constructor function and a <code>$makeClass</code> call.
 * <p>
 * Dependencies:
 * <ul>
 * <li>requiresBefore {@link MemberPatchingTransformation}:
 *     additional members must be in place before they are transformed within this transformation.
 * <li>requiresBefore {@link SuperLiteralTransformation}:
 *     requires the tag {@link Tag#explicitSuperCall}.
 * </ul>
 */
@TransformationDependency.RequiresBefore({ MemberPatchingTransformation.class, SuperLiteralTransformation.class })
@SuppressWarnings("all")
public class ClassDeclarationTransformation extends Transformation {
  @Inject
  private ClassConstructorAssistant classConstructorAssistant;
  
  @Inject
  private BootstrapCallAssistant bootstrapCallAssistant;
  
  @Inject
  private TypeAssistant typeAssistant;
  
  @Override
  public void assertPreConditions() {
    this.typeAssistant.assertClassifierPreConditions();
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      return Boolean.valueOf((it instanceof N4ClassExpression));
    };
    this.assertFalse("class expressions are not supported yet", 
      IteratorExtensions.<EObject>exists(this.getState().im.eAllContents(), _function));
    final Function1<N4ClassDeclaration, Boolean> _function_1 = (N4ClassDeclaration it) -> {
      boolean _isTopLevel = this.typeAssistant.isTopLevel(it);
      return Boolean.valueOf((!_isTopLevel));
    };
    this.assertFalse("only top-level classes are supported, for now", 
      IterableExtensions.<N4ClassDeclaration>exists(this.<N4ClassDeclaration>collectNodes(this.getState().im, N4ClassDeclaration.class, false), _function_1));
  }
  
  @Override
  public void assertPostConditions() {
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      return Boolean.valueOf((it instanceof N4ClassDeclaration));
    };
    this.assertFalse("there should not be any N4ClassDeclarations in the intermediate model", 
      IteratorExtensions.<EObject>exists(this.getState().im.eAllContents(), _function));
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<N4ClassDeclaration> _function = (N4ClassDeclaration it) -> {
      this.transformClassDecl(it);
    };
    this.<N4ClassDeclaration>collectNodes(this.getState().im, N4ClassDeclaration.class, false).forEach(_function);
  }
  
  private void transformClassDecl(final N4ClassDeclaration classDecl) {
    final SymbolTableEntry classSTE = this.findSymbolTableEntryForElement(classDecl, false);
    final SymbolTableEntryOriginal superClassSTE = this.typeAssistant.getSuperClassSTE(classDecl);
    final FunctionDeclaration ctorDecl = this.classConstructorAssistant.createCtorDecl(classDecl, superClassSTE);
    final ExpressionStatement makeClassCall = this.bootstrapCallAssistant.createMakeClassCall(classDecl, superClassSTE);
    final List<Statement> staticInits = this.createStaticFieldInitializations(classSTE, classDecl);
    this.getState().tracer.copyTrace(classDecl, ctorDecl);
    this.getState().tracer.copyTrace(classDecl, makeClassCall);
    this.replace(classDecl, ctorDecl);
    this.insertAfter(TranspilerUtils.orContainingExportDeclaration(ctorDecl), makeClassCall);
    this.insertAfter(makeClassCall, ((EObject[])Conversions.unwrapArray(staticInits, EObject.class)));
    this.getState().info.markAsToHoist(ctorDecl);
  }
  
  /**
   * Creates a new list of statements to initialize the static fields of the given {@code classDecl}.
   * 
   * Clients of this method may modify the returned list.
   */
  protected List<Statement> createStaticFieldInitializations(final SymbolTableEntry steClass, final N4ClassDeclaration classDecl) {
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf(it.isStatic());
    };
    final Function1<N4MemberDeclaration, Statement> _function_1 = (N4MemberDeclaration it) -> {
      return this.createStaticInitialiserCode(it, steClass);
    };
    return IterableExtensions.<Statement>toList(IterableExtensions.<Statement>filterNull(IterableExtensions.<N4MemberDeclaration, Statement>map(IterableExtensions.<N4MemberDeclaration>filter(classDecl.getOwnedMembers(), _function), _function_1)));
  }
  
  private Statement _createStaticInitialiserCode(final N4FieldDeclaration fieldDecl, final SymbolTableEntry steClass) {
    if (((!fieldDecl.isStatic()) || (fieldDecl.getExpression() == null))) {
      return null;
    }
    TMember _originalDefinedMember = this.getState().info.getOriginalDefinedMember(fieldDecl);
    final TField tField = ((TField) _originalDefinedMember);
    ParameterizedPropertyAccessExpression_IM __PropertyAccessExpr = TranspilerBuilderBlocks._PropertyAccessExpr();
    final Procedure1<ParameterizedPropertyAccessExpression_IM> _function = (ParameterizedPropertyAccessExpression_IM it) -> {
      it.setTarget(this.__NSSafe_IdentRef(steClass));
      it.setProperty_IM(this.getSymbolTableEntryOriginal(tField, 
        true));
    };
    ParameterizedPropertyAccessExpression_IM _doubleArrow = ObjectExtensions.<ParameterizedPropertyAccessExpression_IM>operator_doubleArrow(__PropertyAccessExpr, _function);
    final ExpressionStatement exprStmnt = TranspilerBuilderBlocks._ExprStmnt(TranspilerBuilderBlocks._AssignmentExpr(_doubleArrow, fieldDecl.getExpression()));
    this.getState().tracer.copyTrace(fieldDecl, exprStmnt);
    return exprStmnt;
  }
  
  private Statement _createStaticInitialiserCode(final N4FieldAccessor fieldDecl, final SymbolTableEntry steClass) {
    return null;
  }
  
  private Statement _createStaticInitialiserCode(final N4MethodDeclaration methodDecl, final SymbolTableEntry steClass) {
    return null;
  }
  
  private Statement _createStaticInitialiserCode(final DelegatingMember accDecl, final SymbolTableEntry steClass) {
    return null;
  }
  
  private Statement createStaticInitialiserCode(final N4MemberDeclaration methodDecl, final SymbolTableEntry steClass) {
    if (methodDecl instanceof N4MethodDeclaration) {
      return _createStaticInitialiserCode((N4MethodDeclaration)methodDecl, steClass);
    } else if (methodDecl instanceof N4FieldAccessor) {
      return _createStaticInitialiserCode((N4FieldAccessor)methodDecl, steClass);
    } else if (methodDecl instanceof N4FieldDeclaration) {
      return _createStaticInitialiserCode((N4FieldDeclaration)methodDecl, steClass);
    } else if (methodDecl instanceof DelegatingMember) {
      return _createStaticInitialiserCode((DelegatingMember)methodDecl, steClass);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(methodDecl, steClass).toString());
    }
  }
}
