/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.operations;

import com.google.common.collect.Iterables;
import java.util.Collection;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EObjectContainmentEList;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.TranspilerState;
import org.eclipse.n4js.transpiler.im.ImPackage;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.ReferencingElement_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.operations.SymbolTableManagement;
import org.eclipse.n4js.transpiler.utils.TranspilerUtils;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Methods of this class provide elementary operations on a transpiler state, mainly on the intermediate model. The
 * intermediate model should only be changed through the operations defined by this class.
 * <p>
 * Main clients are AST transformations, but they should not invoke these operations directly, but instead use the
 * delegation methods in {@link Transformation}.
 */
@SuppressWarnings("all")
public class TranspilerStateOperations {
  private static final class IM2IMCopier extends EcoreUtil.Copier {
    private static final EReference eRef__ReferencingElement_IM__rewiredTarget = ImPackage.eINSTANCE.getReferencingElement_IM_RewiredTarget();
    
    @Override
    protected void copyReference(final EReference eReference, final EObject eObject, final EObject copyEObject) {
      if ((eReference == TranspilerStateOperations.IM2IMCopier.eRef__ReferencingElement_IM__rewiredTarget)) {
        ((ReferencingElement_IM) copyEObject).setRewiredTarget(((ReferencingElement_IM) eObject).getRewiredTarget());
      } else {
        super.copyReference(eReference, eObject, copyEObject);
      }
    }
  }
  
  /**
   * Creates a new namespace import for the given module and adds it to the intermediate model of the given transpiler
   * state. The returned symbol table entry can be used to create references to the namespace, e.g. by passing it to
   * {@link TranspilerBuilderBlocks#_IdentRef(SymbolTableEntry)}. The newly created import can be obtained by calling
   * {@link SymbolTableEntryOriginal#getImportSpecifier()} on the returned symbol table entry.
   * <p>
   * IMPORTANT: this method does not check if an import for the given module exists already or if the given namespace
   * name is unique (i.e. does not avoid name clashes!).
   */
  public static SymbolTableEntryOriginal addNamespaceImport(final TranspilerState state, final TModule moduleToImport, final String namespaceName) {
    final NamespaceImportSpecifier importSpec = TranspilerBuilderBlocks._NamespaceImportSpecifier(namespaceName, true);
    final ImportDeclaration importDecl = TranspilerBuilderBlocks._ImportDecl(importSpec);
    final ModuleNamespaceVirtualType typeForNamespace = TypesFactory.eINSTANCE.createModuleNamespaceVirtualType();
    typeForNamespace.setName(namespaceName);
    state.resource.addTemporaryType(typeForNamespace);
    final SymbolTableEntryOriginal steForNamespace = SymbolTableManagement.getSymbolTableEntryOriginal(state, typeForNamespace, true);
    steForNamespace.setImportSpecifier(importSpec);
    final EList<ScriptElement> scriptElements = state.im.getScriptElements();
    boolean _isEmpty = scriptElements.isEmpty();
    if (_isEmpty) {
      scriptElements.add(importDecl);
    } else {
      TranspilerStateOperations.insertBefore(state, scriptElements.get(0), importDecl);
    }
    state.info.setImportedModule(importDecl, moduleToImport);
    return steForNamespace;
  }
  
  /**
   * Creates a new named import for the given element and adds it to the intermediate model of the given transpiler
   * state. The returned symbol table entry can be used to create references to the imported element, e.g. by passing
   * it to {@link TranspilerBuilderBlocks#_IdentRef(SymbolTableEntry)}. The newly created import can be obtained by
   * calling {@link SymbolTableEntryOriginal#getImportSpecifier()} on the returned symbol table entry.
   * <p>
   * If a named import already exists for the given element, nothing will be changed in the intermediate model and its
   * symbol table entry will be returned as described above. If the given element is of type
   * {@link ModuleNamespaceVirtualType} an exception will be thrown (because only namespace imports can be created for
   * those types).
   * <p>
   * IMPORTANT: this method does not check if the given namespace name is unique (i.e. does not avoid name clashes!).
   */
  public static SymbolTableEntryOriginal addNamedImport(final TranspilerState state, final IdentifiableElement elementToImport, final String aliasOrNull) {
    final SymbolTableEntryOriginal steOfElementToImport = SymbolTableManagement.getSymbolTableEntryOriginal(state, elementToImport, true);
    TranspilerStateOperations.addNamedImport(state, steOfElementToImport, aliasOrNull);
    return steOfElementToImport;
  }
  
  /**
   * Creates a new named import for the given STE and adds it to the intermediate model of the given transpiler
   * state. The passed-in symbol table entry can be used to create references to the imported element, e.g. by passing
   * it to {@link TranspilerBuilderBlocks#_IdentRef(SymbolTableEntry)}. The newly created import can be obtained by
   * calling {@link SymbolTableEntryOriginal#getImportSpecifier()} on the passed-in symbol table entry.
   * <p>
   * If a named import already exists for the given element, nothing will be changed in the intermediate model. If the
   * original target of the given symbol table entry is of type {@link ModuleNamespaceVirtualType} an exception will
   * be thrown (because only namespace imports can be created for those types).
   * <p>
   * IMPORTANT: this method does not check if the given namespace name is unique (i.e. does not avoid name clashes!).
   */
  public static void addNamedImport(final TranspilerState state, final SymbolTableEntryOriginal steOfElementToImport, final String aliasOrNull) {
    final IdentifiableElement originalTarget = steOfElementToImport.getOriginalTarget();
    if ((originalTarget instanceof ModuleNamespaceVirtualType)) {
      throw new IllegalArgumentException("cannot create named import for a ModuleNamespaceVirtualType");
    }
    final ImportSpecifier existingImportSpec = steOfElementToImport.getImportSpecifier();
    if ((existingImportSpec != null)) {
      return;
    }
    final NamedImportSpecifier importSpec = TranspilerBuilderBlocks._NamedImportSpecifier(steOfElementToImport.getExportedName(), aliasOrNull, true);
    final ImportDeclaration importDecl = TranspilerBuilderBlocks._ImportDecl(importSpec);
    final EList<ScriptElement> scriptElements = state.im.getScriptElements();
    boolean _isEmpty = scriptElements.isEmpty();
    if (_isEmpty) {
      scriptElements.add(importDecl);
    } else {
      TranspilerStateOperations.insertBefore(state, scriptElements.get(0), importDecl);
    }
    steOfElementToImport.setImportSpecifier(importSpec);
    final TModule moduleOfOriginalTarget = originalTarget.getContainingModule();
    state.info.setImportedModule(importDecl, moduleOfOriginalTarget);
  }
  
  /**
   * Adds an "empty" import to the intermediate model, i.e. an import of the form:
   * <pre>
   * import "&lt;moduleSpecifier>";
   * </pre>
   */
  public static void addEmptyImport(final TranspilerState state, final String moduleSpecifier) {
    ImportDeclaration __ImportDecl = TranspilerBuilderBlocks._ImportDecl();
    final Procedure1<ImportDeclaration> _function = (ImportDeclaration it) -> {
      it.setModuleSpecifierAsText(moduleSpecifier);
    };
    final ImportDeclaration importDecl = ObjectExtensions.<ImportDeclaration>operator_doubleArrow(__ImportDecl, _function);
    final EList<ScriptElement> scriptElements = state.im.getScriptElements();
    boolean _isEmpty = scriptElements.isEmpty();
    if (_isEmpty) {
      scriptElements.add(importDecl);
    } else {
      TranspilerStateOperations.insertBefore(state, scriptElements.get(0), importDecl);
    }
  }
  
  public static void setTarget(final TranspilerState state, final ParameterizedCallExpression callExpr, final Expression newTarget) {
    final Expression oldTarget = callExpr.getTarget();
    if ((oldTarget != null)) {
      TranspilerStateOperations.replaceWithoutRewire(state, oldTarget, newTarget);
    } else {
      callExpr.setTarget(newTarget);
    }
  }
  
  public static void setTarget(final TranspilerState state, final ParameterizedPropertyAccessExpression_IM accExpr, final Expression newTarget) {
    final Expression oldTarget = accExpr.getTarget();
    if ((oldTarget != null)) {
      TranspilerStateOperations.replaceWithoutRewire(state, oldTarget, newTarget);
    } else {
      accExpr.setTarget(newTarget);
    }
  }
  
  public static void addArgument(final TranspilerState state, final ParameterizedCallExpression callExpr, final int index, final Expression newArgument) {
    callExpr.getArguments().add(index, TranspilerBuilderBlocks._Argument(newArgument));
  }
  
  public static void remove(final TranspilerState state, final EObject elementInIM) {
    TranspilerStateOperations.replaceWithoutRewire(state, elementInIM);
    if ((elementInIM instanceof ReferencingElement_IM)) {
      ((ReferencingElement_IM)elementInIM).setRewiredTarget(null);
    }
  }
  
  /**
   * Removes the export-container (ExportDeclaration) by creating a new VariableStatement {@code varStmt}, moving all content from {@code exVarStmnt}
   * into it and replacing the ExportDeclaration with the newly created {@code varStmt}
   * @return newly created {@code varStmt} (already part of the intermediate model).
   */
  public static VariableStatement removeExport(final TranspilerState state, final ExportedVariableStatement exVarStmnt) {
    boolean _isIntermediateModelElement = TranspilerUtils.isIntermediateModelElement(exVarStmnt);
    boolean _not = (!_isIntermediateModelElement);
    if (_not) {
      throw new IllegalArgumentException(("not an element in the intermediate model: " + exVarStmnt));
    }
    EObject _eContainer = exVarStmnt.eContainer();
    final ExportDeclaration exportDecl = ((ExportDeclaration) _eContainer);
    VariableStatement __VariableStatement = TranspilerBuilderBlocks._VariableStatement();
    final Procedure1<VariableStatement> _function = (VariableStatement it) -> {
      EList<VariableDeclarationOrBinding> _varDeclsOrBindings = it.getVarDeclsOrBindings();
      EList<VariableDeclarationOrBinding> _varDeclsOrBindings_1 = exVarStmnt.getVarDeclsOrBindings();
      Iterables.<VariableDeclarationOrBinding>addAll(_varDeclsOrBindings, _varDeclsOrBindings_1);
      it.setVarStmtKeyword(exVarStmnt.getVarStmtKeyword());
    };
    final VariableStatement varStmnt = ObjectExtensions.<VariableStatement>operator_doubleArrow(__VariableStatement, _function);
    TranspilerStateOperations.replaceWithoutRewire(state, exportDecl, varStmnt);
    return varStmnt;
  }
  
  public static void replace(final TranspilerState state, final Statement stmnt, final ReturnStatement returnStmnt) {
    TranspilerStateOperations.replaceWithoutRewire(state, stmnt, returnStmnt);
  }
  
  public static void replace(final TranspilerState state, final N4ClassDeclaration classDecl, final FunctionDeclaration funDecl) {
    TranspilerStateOperations.replaceWithoutRewire(state, classDecl, funDecl);
    SymbolTableManagement.rewireSymbolTable(state, classDecl, funDecl);
  }
  
  /**
   * Replace an interface declaration by a variable declaration. The variable declaration will be wrapped in a
   * newly created [Exported]VariableStatement.
   */
  public static void replace(final TranspilerState state, final N4InterfaceDeclaration ifcDecl, final VariableDeclaration varDecl) {
    EObject _eContainer = ifcDecl.eContainer();
    final boolean isExported = (_eContainer instanceof ExportDeclaration);
    final VariableStatement varStmnt = TranspilerBuilderBlocks._VariableStatement(isExported, varDecl);
    TranspilerStateOperations.replaceWithoutRewire(state, ifcDecl, varStmnt);
    SymbolTableManagement.rewireSymbolTable(state, ifcDecl, varDecl);
  }
  
  /**
   * Replace an enum declaration by a variable declaration. The variable declaration will be wrapped in a
   * newly created [Exported]VariableStatement.
   */
  public static void replace(final TranspilerState state, final N4EnumDeclaration enumDecl, final VariableDeclaration varDecl) {
    EObject _eContainer = enumDecl.eContainer();
    final boolean isExported = (_eContainer instanceof ExportDeclaration);
    final VariableStatement varStmnt = TranspilerBuilderBlocks._VariableStatement(isExported, varDecl);
    TranspilerStateOperations.replaceWithoutRewire(state, enumDecl, varStmnt);
    SymbolTableManagement.rewireSymbolTable(state, enumDecl, varDecl);
  }
  
  /**
   * Replace an enum declaration by a function declaration.
   */
  public static void replace(final TranspilerState state, final N4EnumDeclaration enumDecl, final FunctionDeclaration funDecl) {
    TranspilerStateOperations.replaceWithoutRewire(state, enumDecl, funDecl);
    SymbolTableManagement.rewireSymbolTable(state, enumDecl, funDecl);
  }
  
  public static void replace(final TranspilerState state, final FunctionDeclaration funDecl, final VariableDeclaration varDecl) {
    EObject _eContainer = funDecl.eContainer();
    final boolean isExported = (_eContainer instanceof ExportDeclaration);
    final VariableStatement varStmnt = TranspilerBuilderBlocks._VariableStatement(isExported, varDecl);
    TranspilerStateOperations.replaceWithoutRewire(state, funDecl, varStmnt);
    SymbolTableManagement.rewireSymbolTable(state, funDecl, varDecl);
    final Expression varValue = varDecl.getExpression();
    if ((varValue instanceof FunctionExpression)) {
      SymbolTableManagement.rewireSymbolTable(state, funDecl.getLocalArgumentsVariable(), ((FunctionExpression)varValue).getLocalArgumentsVariable());
    } else {
      throw new IllegalArgumentException(
        ("when replacing a function declaration by a variable declaration, " + 
          "we expect the variable to be initialized with a function expression"));
    }
  }
  
  public static void replace(final TranspilerState state, final FunctionDeclaration functionDecl, final ExpressionStatement stmt) {
    TranspilerStateOperations.replaceWithoutRewire(state, functionDecl, stmt);
  }
  
  public static void replace(final TranspilerState state, final N4MemberDeclaration memberDecl, final N4MemberDeclaration replacement) {
    TranspilerStateOperations.replaceWithoutRewire(state, memberDecl, replacement);
    SymbolTableManagement.rewireSymbolTable(state, memberDecl, replacement);
  }
  
  public static void replace(final TranspilerState state, final VariableStatement varStmnt, final Statement... newStmnts) {
    TranspilerStateOperations.replaceWithoutRewire(state, varStmnt, newStmnts);
  }
  
  public static void replace(final TranspilerState state, final VariableBinding varBinding, final VariableDeclaration... varDecls) {
    TranspilerStateOperations.replaceWithoutRewire(state, varBinding, varDecls);
  }
  
  public static void replace(final TranspilerState state, final Expression exprOld, final Expression exprNew) {
    TranspilerStateOperations.replaceWithoutRewire(state, exprOld, exprNew);
  }
  
  public static void replace(final TranspilerState state, final ArrowFunction exprOld, final ParameterizedCallExpression exprNew, final FunctionExpression rewireTarget) {
    TranspilerStateOperations.replaceWithoutRewire(state, exprOld, exprNew);
    SymbolTableManagement.rewireSymbolTable(state, exprOld, rewireTarget);
  }
  
  /**
   * Replace formal parameter with a variableStmt. Rewire the fpar to the VariableDeclaration. Relocate the Stmt
   */
  public static void replaceAndRelocate(final TranspilerState state, final FormalParameter fPar_to_remove, final VariableStatement varStmnt, final VariableDeclaration varDecl_wireTo, final Block newContainer) {
    EObject _eContainer = varDecl_wireTo.eContainer();
    boolean _tripleNotEquals = (_eContainer != varStmnt);
    if (_tripleNotEquals) {
      throw new IllegalArgumentException("varDecl must be contained in varStmnt");
    }
    TranspilerStateOperations.replaceAndRelocateWithoutRewire_internal(state, fPar_to_remove, varStmnt, newContainer.getStatements(), 0);
    SymbolTableManagement.rewireSymbolTable(state, fPar_to_remove, varDecl_wireTo);
  }
  
  public static <T extends Expression> void wrapExistingExpression(final TranspilerState state, final T exprToWrap, final Expression outerExpr_without_exprToWrap, final Procedure1<? super T> inserterFunction) {
    TranspilerStateOperations.insertOrReplace_internal(state, exprToWrap, new EObject[] { outerExpr_without_exprToWrap }, true, false);
    inserterFunction.apply(exprToWrap);
  }
  
  /**
   * append( pos < 0 or > current size ), prepend(pos==0) or insert at {@code pos} the object {@code insertThis}
   * to {@code newContainer}. Also delete {@code removeThis} from the IM.  Does not rewire. But keeps trace.
   */
  private static void replaceAndRelocateWithoutRewire_internal(final TranspilerState state, final EObject removeThis, final EObject insertThis, final EList<?> newContainer, final int pos) {
    final EReference eRefRemove = TranspilerStateOperations.checkedContainmentFeature(removeThis);
    EObject _eContainer = insertThis.eContainer();
    boolean _tripleNotEquals = (_eContainer != null);
    if (_tripleNotEquals) {
      EObject _eContainer_1 = insertThis.eContainer();
      String _plus = (((("The new element must not be contained anywhere." + 
        " insertThis=") + insertThis) + " is currently contained in ") + _eContainer_1);
      throw new IllegalArgumentException(_plus);
    }
    if ((newContainer instanceof EObjectContainmentEList<?>)) {
      final EObjectContainmentEList<EObject> newContainerCasted = ((EObjectContainmentEList<EObject>) newContainer);
      state.tracer.copyTrace(removeThis, insertThis);
      state.tracer.discardIntermediateModelNode(removeThis);
      int _upperBound = eRefRemove.getUpperBound();
      boolean _equals = (_upperBound == 1);
      if (_equals) {
        boolean _isUnsettable = eRefRemove.isUnsettable();
        if (_isUnsettable) {
          removeThis.eContainer().eUnset(eRefRemove);
        } else {
          removeThis.eContainer().eSet(eRefRemove, null);
        }
      } else {
        Object _eGet = removeThis.eContainer().eGet(eRefRemove);
        final List<EObject> l = ((List<EObject>) _eGet);
        int idx = l.indexOf(removeThis);
        l.remove(idx);
      }
      int _xifexpression = (int) 0;
      if (((pos < 0) || (pos >= ((EObjectContainmentEList<?>)newContainer).size()))) {
        _xifexpression = ((EObjectContainmentEList<?>)newContainer).size();
      } else {
        _xifexpression = pos;
      }
      final int idx_1 = _xifexpression;
      newContainerCasted.add(idx_1, insertThis);
    } else {
      throw new IllegalArgumentException("designated new container-list must be a subtype of type EObjectContainmentList");
    }
  }
  
  /**
   * {@code elementInIntermediateModel} is going away (ie, should be garbage-collected) and therefore we clear all
   * references to it from tracing. The {@code replacements} IM nodes take over whatever AST node was previously
   * traced-back-to via the element going away.
   */
  private static void replaceWithoutRewire(final TranspilerState state, final EObject elementInIntermediateModel, final EObject... replacements) {
    state.tracer.copyTrace(elementInIntermediateModel, replacements);
    state.tracer.discardIntermediateModelNode(elementInIntermediateModel);
    TranspilerStateOperations.insertOrReplace_internal(state, elementInIntermediateModel, replacements, true, false);
  }
  
  public static void insertBefore(final TranspilerState state, final EObject elementInIntermediateModel, final EObject... newElements) {
    TranspilerStateOperations.insertOrReplace_internal(state, elementInIntermediateModel, newElements, false, false);
  }
  
  public static void insertAfter(final TranspilerState state, final EObject elementInIntermediateModel, final EObject... newElements) {
    TranspilerStateOperations.insertOrReplace_internal(state, elementInIntermediateModel, newElements, false, true);
  }
  
  private static void insertOrReplace_internal(final TranspilerState state, final EObject elementInIntermediateModel, final EObject[] newElements, final boolean replace, final boolean after) {
    if ((((List<EObject>)Conversions.doWrapArray(newElements)).isEmpty() && (!replace))) {
      return;
    }
    final EReference eRef = TranspilerStateOperations.checkedContainmentFeature(elementInIntermediateModel);
    final EClass eRefType = eRef.getEReferenceType();
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      boolean _isSuperTypeOf = eRefType.isSuperTypeOf(it.eClass());
      return Boolean.valueOf((!_isSuperTypeOf));
    };
    final Iterable<EObject> replElemsOfWrongType = IterableExtensions.<EObject>filter(((Iterable<EObject>)Conversions.doWrapArray(newElements)), _function);
    boolean _isEmpty = IterableExtensions.isEmpty(replElemsOfWrongType);
    boolean _not = (!_isEmpty);
    if (_not) {
      String _name = eRef.getEReferenceType().getName();
      String _plus = ("one or more elements are of wrong type, expected: " + _name);
      String _plus_1 = (_plus + ", actual: ");
      final Function1<EObject, String> _function_1 = (EObject it) -> {
        return it.eClass().getName();
      };
      String _join = IterableExtensions.join(IterableExtensions.<EObject, String>map(replElemsOfWrongType, _function_1), ", ");
      String _plus_2 = (_plus_1 + _join);
      throw new IllegalArgumentException(_plus_2);
    }
    int _upperBound = eRef.getUpperBound();
    boolean _equals = (_upperBound == 1);
    if (_equals) {
      int _length = newElements.length;
      boolean _greaterThan = (_length > 1);
      if (_greaterThan) {
        String _name_1 = eRef.getName();
        String _plus_3 = ("the single-valued reference " + _name_1);
        String _plus_4 = (_plus_3 + " in class ");
        EClass _eContainingClass = eRef.getEContainingClass();
        String _plus_5 = (_plus_4 + _eContainingClass);
        String _plus_6 = (_plus_5 + " is not able to hold ");
        int _length_1 = newElements.length;
        String _plus_7 = (_plus_6 + Integer.valueOf(_length_1));
        String _plus_8 = (_plus_7 + " elements.");
        throw new IllegalArgumentException(_plus_8);
      }
      int _length_2 = newElements.length;
      boolean _equals_1 = (_length_2 == 1);
      if (_equals_1) {
        if ((!replace)) {
          String _name_2 = eRef.getName();
          String _plus_9 = ("Cannot insert another element into a single-valued containment reference " + _name_2);
          String _plus_10 = (_plus_9 + " in class ");
          EClass _eContainingClass_1 = eRef.getEContainingClass();
          String _plus_11 = (_plus_10 + _eContainingClass_1);
          throw new IllegalArgumentException(_plus_11);
        }
        elementInIntermediateModel.eContainer().eSet(eRef, newElements[0]);
      } else {
        if ((!replace)) {
          throw new IllegalArgumentException("Inserting zero elements with replace==false is pointless.");
        }
        boolean _isUnsettable = eRef.isUnsettable();
        if (_isUnsettable) {
          elementInIntermediateModel.eContainer().eUnset(eRef);
        } else {
          elementInIntermediateModel.eContainer().eSet(eRef, null);
        }
      }
    } else {
      Object _eGet = elementInIntermediateModel.eContainer().eGet(eRef);
      final List<EObject> l = ((List<EObject>) _eGet);
      int idx = l.indexOf(elementInIntermediateModel);
      if (replace) {
        l.remove(idx);
      } else {
        if (after) {
          idx++;
        }
      }
      l.addAll(idx, ((Collection<? extends EObject>)Conversions.doWrapArray(newElements)));
    }
  }
  
  /**
   * Retrieves the EReference of the Container. Throws Exceptions if a) not part of the IM or b) not contained anywhere
   */
  private static EReference checkedContainmentFeature(final EObject elementInIntermediateModel) {
    boolean _isIntermediateModelElement = TranspilerUtils.isIntermediateModelElement(elementInIntermediateModel);
    boolean _not = (!_isIntermediateModelElement);
    if (_not) {
      throw new IllegalArgumentException(("not an element in the intermediate model: " + elementInIntermediateModel));
    }
    final EReference eRef = elementInIntermediateModel.eContainmentFeature();
    if ((eRef == null)) {
      throw new IllegalArgumentException("element is not contained anywhere");
    }
    return eRef;
  }
  
  /**
   * Rename the given symbol table entry and all named elements in the intermediate model that are using this name.
   * During AST transformations in the transpiler, the 'name' property of a symbol table entry should never be changed
   * directly, but this operation should be used instead.
   * <p>
   * <b>WARNING:</b> renaming is currently only implemented partially and used only in a single, very specific use
   * case; if renaming is required in the future, then the implementation of this method has to be complemented!
   */
  public static void rename(final TranspilerState state, final SymbolTableEntry entry, final String newName) {
    SymbolTableManagement.rename(state, entry, newName);
  }
  
  /**
   * Copy a subtree of the intermediate model.
   */
  public static <T extends EObject> T copy(final TranspilerState state, final T elementInIM) {
    final TranspilerStateOperations.IM2IMCopier copier = new TranspilerStateOperations.IM2IMCopier();
    final EObject result = copier.copy(elementInIM);
    copier.copyReferences();
    state.tracer.copyTrace(elementInIM, result);
    return ((T) result);
  }
}
