/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ts.scoping;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.xtext.resource.IEObjectDescription;

/**
 * Poor mans filter to reduce number of elements, filtering out elements which are not valid in a certain context, although
 * more elaborated filtering may be needed for content assist.
 */
@SuppressWarnings("all")
public class TypesScopeFilter {
  public Predicate<? super IEObjectDescription> getTypesFilterCriteria(final EObject context, final EReference reference) {
    Predicate<? super IEObjectDescription> _switchResult = null;
    boolean _matched = false;
    if (context instanceof TypeRef) {
      _matched=true;
      _switchResult = this.getTypesFilterCriteria(((TypeRef)context), reference);
    }
    if (!_matched) {
      _switchResult = this.getTypeReferenceFilterCriteria(context, reference);
    }
    return _switchResult;
  }
  
  protected Predicate<? super IEObjectDescription> getTypesFilterCriteria(final TypeRef context, final EReference reference) {
    EReference _parameterizedTypeRef_AstNamespace = TypeRefsPackage.eINSTANCE.getParameterizedTypeRef_AstNamespace();
    boolean _tripleEquals = (reference == _parameterizedTypeRef_AstNamespace);
    if (_tripleEquals) {
      return this.getNamespaceCriterion();
    }
    EObject container = context;
    EReference containmentFeature = null;
    while ((container instanceof TypeRef)) {
      {
        EStructuralFeature _eContainingFeature = ((TypeRef)container).eContainingFeature();
        containmentFeature = ((EReference) _eContainingFeature);
        container = ((TypeRef)container).eContainer();
      }
    }
    return this.getTypeReferenceFilterCriteria(container, containmentFeature);
  }
  
  protected Predicate<? super IEObjectDescription> getTypeReferenceFilterCriteria(final EObject container, final EReference containmentFeature) {
    Predicate<? super IEObjectDescription> _switchResult = null;
    boolean _matched = false;
    if (container instanceof TClass) {
      _matched=true;
      _switchResult = this.getInheritanceFilterCriteria(containmentFeature);
    }
    if (!_matched) {
      if (container instanceof TypeVariable) {
        _matched=true;
        _switchResult = this.getTypeVariableBoundCriteria();
      }
    }
    if (!_matched) {
      if (container instanceof TField) {
        _matched=true;
        _switchResult = this.getFieldTypeCriteria();
      }
    }
    if (!_matched) {
      if (container instanceof TMethod) {
        _matched=true;
        _switchResult = this.getReturnTypeCriteria();
      }
    }
    if (!_matched) {
      if (container instanceof TFormalParameter) {
        _matched=true;
        _switchResult = this.getParameterTypeCriteria();
      }
    }
    if (!_matched) {
      if (container instanceof TFunction) {
        _matched=true;
        _switchResult = this.getReturnTypeCriteria();
      }
    }
    if (!_matched) {
      _switchResult = Predicates.alwaysTrue();
    }
    return _switchResult;
  }
  
  protected Predicate<? super IEObjectDescription> getNamespaceCriterion() {
    final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
      return TypesPackage.Literals.MODULE_NAMESPACE_VIRTUAL_TYPE.isSuperTypeOf(it.getEClass());
    };
    return _function;
  }
  
  protected Predicate<? super IEObjectDescription> getParameterTypeCriteria() {
    final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
      return ((((!TypesPackage.Literals.TFUNCTION.isSuperTypeOf(it.getEClass())) && (!Objects.equal(TypesPackage.Literals.UNDEFINED_TYPE, it.getEClass()))) && (!Objects.equal(TypesPackage.Literals.NULL_TYPE, it.getEClass()))) && (!Objects.equal(TypesPackage.Literals.VOID_TYPE, it.getEClass())));
    };
    return _function;
  }
  
  protected Predicate<? super IEObjectDescription> getReturnTypeCriteria() {
    final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
      return (((!TypesPackage.Literals.TFUNCTION.isSuperTypeOf(it.getEClass())) && (!Objects.equal(TypesPackage.Literals.UNDEFINED_TYPE, it.getEClass()))) && (!Objects.equal(TypesPackage.Literals.NULL_TYPE, it.getEClass())));
    };
    return _function;
  }
  
  protected Predicate<? super IEObjectDescription> getFieldTypeCriteria() {
    final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
      return ((((!TypesPackage.Literals.TFUNCTION.isSuperTypeOf(it.getEClass())) && (!Objects.equal(TypesPackage.Literals.UNDEFINED_TYPE, it.getEClass()))) && (!Objects.equal(TypesPackage.Literals.NULL_TYPE, it.getEClass()))) && (!Objects.equal(TypesPackage.Literals.VOID_TYPE, it.getEClass())));
    };
    return _function;
  }
  
  protected Predicate<? super IEObjectDescription> getTypeVariableBoundCriteria() {
    final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
      return TypesPackage.Literals.TCLASSIFIER.isSuperTypeOf(it.getEClass());
    };
    return _function;
  }
  
  protected Predicate<? super IEObjectDescription> getInheritanceFilterCriteria(final EReference reference) {
    Predicate<IEObjectDescription> _switchResult = null;
    boolean _matched = false;
    if (Objects.equal(reference, TypesPackage.Literals.TCLASS__SUPER_CLASS_REF)) {
      _matched=true;
      final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
        return TypesPackage.Literals.TCLASS.isSuperTypeOf(it.getEClass());
      };
      _switchResult = _function;
    }
    if (!_matched) {
      if (Objects.equal(reference, TypesPackage.Literals.TCLASS__IMPLEMENTED_INTERFACE_REFS)) {
        _matched=true;
        final Predicate<IEObjectDescription> _function_1 = (IEObjectDescription it) -> {
          return TypesPackage.Literals.TINTERFACE.isSuperTypeOf(it.getEClass());
        };
        _switchResult = _function_1;
      }
    }
    return _switchResult;
  }
}
