/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling.helper;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ui.labeling.EObjectWithContext;
import org.eclipse.n4js.ui.labeling.N4JSLabelProvider;
import org.eclipse.n4js.ui.labeling.N4JSStylers;
import org.eclipse.n4js.ui.labeling.helper.LabelCalculationHelper;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.ui.label.AbstractLabelProvider;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This helper class serves as replacement for the polymorphic dispatch done
 * by {@link AbstractLabelProvider} in favor of
 * Xtend dispatch methods. Here the dispatch of styled labels is done.
 * It is called in {@link N4JSLabelProvider#getStyledText}. getStyledText
 * calls {@link N4JSLabelProvider#doGetText} so in most cases its good
 * practice to call here getLabelProvider.getSuperStyledText to keep this
 * behavior. By this you can reuse the label calculated by
 * {@link LabelCalculationHelper} so far. Here you can then append the styled
 * parts of the label (so its still possible to append non styled strings as
 * well).
 * <br /><br />
 * General use case for styled string is the highlighting of inferred types
 * in the outline (to see the difference to declared types). So styled texts
 * are created for fields, methods, getters, setters and exported variables
 * as only those can have types.
 */
@SuppressWarnings("all")
public class StyledTextCalculationHelper {
  /**
   * Threshold length at which a type reference description is compressed
   */
  private static final int OUTLINE_TYPE_REF_LENGTH_THRESHOLD = 30;
  
  private N4JSLabelProvider labelProvider;
  
  @Inject
  private LabelCalculationHelper labelCalculationHelper;
  
  public N4JSLabelProvider setLabelProvider(final N4JSLabelProvider provider) {
    return this.labelProvider = provider;
  }
  
  public N4JSLabelProvider getLabelProvider() {
    return this.labelProvider;
  }
  
  /**
   * Should not happen as this will lead to consequential errors, still added here to make helper more robust.
   */
  protected StyledString _dispatchGetStyledText(final Void _null) {
    return new StyledString("*error*");
  }
  
  /**
   * fallback
   */
  protected StyledString _dispatchGetStyledText(final Object element) {
    return this.getLabelProvider().getSuperStyledText(element);
  }
  
  /**
   * Adds origin to inherited, consumed, or polyfilled members.
   * 
   * @see https://github.com/eclipse/n4js/issues/99
   */
  protected StyledString _dispatchGetStyledText(final EObjectWithContext objectWithContext) {
    StyledString styledText = this.dispatchGetStyledText(objectWithContext.obj);
    if ((styledText == null)) {
      return this.unknown();
    }
    TMember _xifexpression = null;
    if ((objectWithContext.obj instanceof N4MemberDeclaration)) {
      _xifexpression = ((N4MemberDeclaration) objectWithContext.obj).getDefinedTypeElement();
    } else {
      TMember _xifexpression_1 = null;
      if ((objectWithContext.obj instanceof TMember)) {
        _xifexpression_1 = ((TMember) objectWithContext.obj);
      } else {
        _xifexpression_1 = null;
      }
      _xifexpression = _xifexpression_1;
    }
    final TMember member = _xifexpression;
    if ((((member != null) && (member.getContainingType() != null)) && (!Objects.equal(member.getContainingType(), objectWithContext.context)))) {
      String _dispatchDoGetText = this.labelCalculationHelper.dispatchDoGetText(member.getContainingType());
      final String orgDest = (" from " + _dispatchDoGetText);
      boolean _isPolyfilled = member.isPolyfilled();
      if (_isPolyfilled) {
        styledText.append((" - polyfilled" + orgDest), N4JSStylers.POLYFILLED_MEMBERS_STYLER);
      } else {
        if (((member.getContainingType() instanceof TInterface) && (objectWithContext.context instanceof TClass))) {
          styledText.append((" - consumed" + orgDest), N4JSStylers.CONSUMED_MEMBERS_STYLER);
        } else {
          styledText.append((" - inherited" + orgDest), N4JSStylers.INHERITED_MEMBERS_STYLER);
        }
      }
    }
    return styledText;
  }
  
  /**
   * produces e.g. functionName(param1TypeName, param2TypeName): returnTypeName
   */
  protected StyledString _dispatchGetStyledText(final FunctionDefinition functionDefinition) {
    StyledString _xblockexpression = null;
    {
      final Type definedType = functionDefinition.getDefinedType();
      if ((definedType instanceof TFunction)) {
        return this.dispatchGetStyledText(definedType);
      }
      _xblockexpression = this.getLabelProvider().getSuperStyledText(functionDefinition);
    }
    return _xblockexpression;
  }
  
  /**
   * Does not access AST.
   */
  protected StyledString _dispatchGetStyledText(final TFunction tfunction) {
    StyledString styledText = this.getLabelProvider().getSuperStyledText(tfunction);
    if ((styledText == null)) {
      return this.unknown();
    }
    boolean _isConstructor = tfunction.isConstructor();
    if (_isConstructor) {
      styledText.setStyle(0, styledText.length(), N4JSStylers.CONSTRUCTOR_STYLER);
    }
    styledText = this.appendStyledTextForFormalParameters(styledText, tfunction);
    boolean _isEmpty = tfunction.getTypeVars().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Function1<TypeVariable, CharSequence> _function = (TypeVariable it) -> {
        return it.getName();
      };
      styledText.append(IterableExtensions.<TypeVariable>join(tfunction.getTypeVars(), "<", ", ", ">", _function));
    }
    if (((!tfunction.isConstructor()) && (tfunction.getReturnTypeRef() != null))) {
      styledText.append(": ");
      styledText.append(this.getTypeRefDescription(tfunction.getReturnTypeRef()));
    }
    return styledText;
  }
  
  /**
   * produces e.g. getterName(): returnTypeName
   */
  protected StyledString _dispatchGetStyledText(final N4GetterDeclaration n4GetterDeclaration) {
    TGetter _definedGetter = n4GetterDeclaration.getDefinedGetter();
    boolean _tripleNotEquals = (_definedGetter != null);
    if (_tripleNotEquals) {
      return this.dispatchGetStyledText(n4GetterDeclaration.getDefinedGetter());
    }
    return this.getLabelProvider().getSuperStyledText(n4GetterDeclaration);
  }
  
  /**
   * Returns the styled text for the given getter, does not access the AST.
   */
  protected StyledString _dispatchGetStyledText(final TGetter tgetter) {
    StyledString styledText = this.getLabelProvider().getSuperStyledText(tgetter);
    if ((styledText == null)) {
      return this.unknown();
    }
    styledText.setStyle(0, styledText.length(), N4JSStylers.FIELD_OR_VAR_STYLER);
    boolean _isOptional = tgetter.isOptional();
    if (_isOptional) {
      styledText.append("?");
    }
    TypeRef _declaredTypeRef = tgetter.getDeclaredTypeRef();
    boolean _tripleNotEquals = (_declaredTypeRef != null);
    if (_tripleNotEquals) {
      styledText.append(": ");
      styledText.append(this.getTypeRefDescription(tgetter.getDeclaredTypeRef()));
    }
    return styledText;
  }
  
  /**
   * produces e.g. setterName: paramTypeName
   */
  protected StyledString _dispatchGetStyledText(final N4SetterDeclaration n4SetterDeclaration) {
    TSetter _definedSetter = n4SetterDeclaration.getDefinedSetter();
    boolean _tripleNotEquals = (_definedSetter != null);
    if (_tripleNotEquals) {
      return this.dispatchGetStyledText(n4SetterDeclaration.getDefinedSetter());
    }
    return this.getLabelProvider().getSuperStyledText(n4SetterDeclaration);
  }
  
  /**
   * Returns the styled text for the given setter, does not access AST.
   */
  protected StyledString _dispatchGetStyledText(final TSetter tsetter) {
    StyledString _xblockexpression = null;
    {
      StyledString styledText = this.getLabelProvider().getSuperStyledText(tsetter);
      if ((styledText == null)) {
        return this.unknown();
      }
      styledText.setStyle(0, styledText.length(), N4JSStylers.FIELD_OR_VAR_STYLER);
      boolean _isOptional = tsetter.isOptional();
      if (_isOptional) {
        styledText.append("?");
      }
      StyledString _xifexpression = null;
      TFormalParameter _fpar = tsetter.getFpar();
      boolean _tripleNotEquals = (_fpar != null);
      if (_tripleNotEquals) {
        _xifexpression = styledText.append(": ").append(this.getStyledTextForFormalParameter(tsetter.getFpar()));
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * produces e.g. fieldName: fieldTypeName
   */
  protected StyledString _dispatchGetStyledText(final N4FieldDeclaration n4FieldDeclaration) {
    TField _definedField = n4FieldDeclaration.getDefinedField();
    boolean _tripleNotEquals = (_definedField != null);
    if (_tripleNotEquals) {
      return this.dispatchGetStyledText(n4FieldDeclaration.getDefinedField());
    }
    return this.getLabelProvider().getSuperStyledText(n4FieldDeclaration);
  }
  
  /**
   * Returns the styled text for the given field, does not accesses the AST.
   */
  protected StyledString _dispatchGetStyledText(final TField tfield) {
    StyledString styledText = this.getLabelProvider().getSuperStyledText(tfield);
    if ((styledText == null)) {
      return this.unknown();
    }
    styledText.setStyle(0, styledText.length(), N4JSStylers.FIELD_OR_VAR_STYLER);
    boolean _isOptional = tfield.isOptional();
    if (_isOptional) {
      styledText.append("?");
    }
    TypeRef _typeRef = tfield.getTypeRef();
    boolean _tripleNotEquals = (_typeRef != null);
    if (_tripleNotEquals) {
      styledText.append(": ");
      styledText.append(this.getTypeRefDescription(tfield.getTypeRef()));
    }
    return styledText;
  }
  
  /**
   * produces e.g. variableName: variableTypeName
   */
  protected StyledString _dispatchGetStyledText(final ExportedVariableDeclaration variableDeclaration) {
    StyledString styledText = this.getLabelProvider().getSuperStyledText(variableDeclaration);
    if ((styledText == null)) {
      return this.unknown();
    }
    styledText.setStyle(0, styledText.length(), N4JSStylers.FIELD_OR_VAR_STYLER);
    final TVariable definedVariable = variableDeclaration.getDefinedVariable();
    TypeRef _typeRef = null;
    if (definedVariable!=null) {
      _typeRef=definedVariable.getTypeRef();
    }
    boolean _tripleNotEquals = (_typeRef != null);
    if (_tripleNotEquals) {
      styledText.append(": ");
      styledText.append(this.getTypeRefDescription(definedVariable.getTypeRef()));
    }
    return styledText;
  }
  
  /**
   * produces e.g. (param1TypeName, param2TypeName)
   */
  private StyledString appendStyledTextForFormalParameters(final StyledString styledText, final TFunction tFunction) {
    StyledString _xblockexpression = null;
    {
      styledText.append("(");
      boolean _isEmpty = tFunction.getFpars().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final Function1<TFormalParameter, StyledString> _function = (TFormalParameter it) -> {
          return this.getStyledTextForFormalParameter(it);
        };
        final Function2<StyledString, StyledString, StyledString> _function_1 = (StyledString l, StyledString r) -> {
          StyledString _xifexpression = null;
          if ((l != null)) {
            _xifexpression = l.append(", ").append(r);
          } else {
            StyledString _xifexpression_1 = null;
            if ((r != null)) {
              _xifexpression_1 = r;
            } else {
              _xifexpression_1 = new StyledString();
            }
            _xifexpression = _xifexpression_1;
          }
          return _xifexpression;
        };
        styledText.append(IterableExtensions.<StyledString>reduce(ListExtensions.<TFormalParameter, StyledString>map(tFunction.getFpars(), _function), _function_1));
      }
      _xblockexpression = styledText.append(")");
    }
    return _xblockexpression;
  }
  
  private StyledString getStyledTextForFormalParameter(final TFormalParameter tFormalParameter) {
    StyledString styledText = this.getTypeRefDescription(tFormalParameter.getTypeRef());
    boolean _isVariadic = tFormalParameter.isVariadic();
    if (_isVariadic) {
      styledText = new StyledString("…").append(styledText);
    }
    boolean _isHasInitializerAssignment = tFormalParameter.isHasInitializerAssignment();
    if (_isHasInitializerAssignment) {
      styledText.append("=");
    }
    return styledText;
  }
  
  /**
   * Returns a type reference description which is compressed according to the {@link StyledTextCalculationHelper#OUTLINE_TYPE_REF_LENGTH_THRESHOLD} constant.
   */
  private StyledString getTypeRefDescription(final TypeRef ref) {
    String _compressedTypeRefDescription = this.getCompressedTypeRefDescription(ref, 0);
    final StyledString typeRefDescription = new StyledString(_compressedTypeRefDescription, N4JSStylers.TYPEREF_STYLER);
    return typeRefDescription;
  }
  
  /**
   * Appends the description of the given type reference to the styled string. If the {@link StyledTextCalculationHelper#OUTLINE_TYPE_REF_LENGTH_THRESHOLD} is exceeded
   * further nested type information is omitted and replaced by three dots (...).
   */
  private String getCompressedTypeRefDescription(final TypeRef typeRef, final int currentLength) {
    if ((currentLength > StyledTextCalculationHelper.OUTLINE_TYPE_REF_LENGTH_THRESHOLD)) {
      return "...";
    } else {
      if ((typeRef != null)) {
        return this.getTypeRefDescriptionString(typeRef);
      } else {
        return "<unknown>";
      }
    }
  }
  
  /**
   * appends the simple type name to the styled string
   */
  private String _getTypeRefDescriptionString(final TypeArgument ref) {
    final String name = ref.getTypeRefAsString();
    if ((name == null)) {
      return "<unknown>";
    } else {
      return name;
    }
  }
  
  /**
   * appends the simple type name to the styled string
   */
  private String _getTypeRefDescriptionString(final TypeRef ref) {
    Type _declaredType = ref.getDeclaredType();
    String _name = null;
    if (_declaredType!=null) {
      _name=_declaredType.getName();
    }
    final String name = _name;
    if ((name == null)) {
      return "<unknown>";
    } else {
      return name;
    }
  }
  
  /**
   * appends the simple type name to the styled string
   */
  private String _getTypeRefDescriptionString(final ParameterizedTypeRef ref) {
    Type _declaredType = ref.getDeclaredType();
    String _name = null;
    if (_declaredType!=null) {
      _name=_declaredType.getName();
    }
    String name = _name;
    if ((name == null)) {
      return "<unknown>";
    } else {
      if (((ref.getTypingStrategy() != null) && (!Objects.equal(ref.getTypingStrategy(), TypingStrategy.DEFAULT)))) {
        TypingStrategy _typingStrategy = ref.getTypingStrategy();
        String _plus = (_typingStrategy + name);
        name = _plus;
      }
      if ((name.equals("Array") && ref.getDeclaredType().isArrayLike())) {
        final Function1<TypeArgument, CharSequence> _function = (TypeArgument arg) -> {
          return this.getTypeRefDescriptionString(arg);
        };
        String _join = IterableExtensions.<TypeArgument>join(ref.getTypeArgs(), ",", _function);
        String _plus_1 = ("[" + _join);
        String _plus_2 = (_plus_1 + "]");
        name = _plus_2;
      } else {
        String _name_1 = name;
        final Function1<TypeArgument, CharSequence> _function_1 = (TypeArgument arg) -> {
          return this.getTypeRefDescriptionString(arg);
        };
        String _join_1 = IterableExtensions.<TypeArgument>join(ref.getTypeArgs(), "<", ",", ">", _function_1);
        name = (_name_1 + _join_1);
      }
      return name;
    }
  }
  
  /**
   * produces 'this' for ThisType references
   */
  private String _getTypeRefDescriptionString(final ThisTypeRef ref) {
    if (((ref.getTypingStrategy() != null) && (!Objects.equal(ref.getTypingStrategy(), TypingStrategy.DEFAULT)))) {
      TypingStrategy _typingStrategy = ref.getTypingStrategy();
      return (_typingStrategy + "this");
    }
    return "this";
  }
  
  /**
   * produces (param1, param2,...)=>returnType
   */
  private String _getTypeRefDescriptionString(final FunctionTypeExpression ref) {
    final StringBuilder result = new StringBuilder();
    result.append("(");
    final StyledString parameterString = new StyledString();
    final Function1<TFormalParameter, TypeRef> _function = (TFormalParameter it) -> {
      return it.getTypeRef();
    };
    this.appendCommaSeparatedTypeRefList(ListExtensions.<TFormalParameter, TypeRef>map(ref.getFpars(), _function), parameterString, true, ", ");
    result.append(parameterString).toString();
    result.append(")=>");
    TypeRef _returnTypeRef = ref.getReturnTypeRef();
    boolean _tripleNotEquals = (_returnTypeRef != null);
    if (_tripleNotEquals) {
      result.append(this.getCompressedTypeRefDescription(ref.getReturnTypeRef(), 0));
    } else {
      result.append("void");
    }
    return result.toString();
  }
  
  /**
   * produces type{typeName} or constructor{typeName}
   */
  private String _getTypeRefDescriptionString(final TypeTypeRef ref) {
    String _switchResult = null;
    TypeArgument _typeArg = ref.getTypeArg();
    boolean _matched = false;
    if (_typeArg instanceof ThisTypeRef) {
      _matched=true;
      _switchResult = "this";
    }
    if (!_matched) {
      _switchResult = this.nominalTypeNameOrWildCard(ref);
    }
    final String typeName = _switchResult;
    boolean _isConstructorRef = ref.isConstructorRef();
    if (_isConstructorRef) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("constructor{");
      _builder.append(typeName);
      _builder.append("}");
      return _builder.toString();
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("type{");
      _builder_1.append(typeName);
      _builder_1.append("}");
      return _builder_1.toString();
    }
  }
  
  /**
   * produces type1 | type2 or type1 & type2, adding parenthesis if needed
   */
  private String _getTypeRefDescriptionString(final ComposedTypeRef ref) {
    final StyledString result = new StyledString();
    final EStructuralFeature cf = ref.eContainingFeature();
    final boolean needsParens = ((ref.eContainer() instanceof ComposedTypeRef) || (((Objects.equal(cf, TypesPackage.Literals.TFUNCTION__RETURN_TYPE_REF) || Objects.equal(cf, TypeRefsPackage.Literals.FUNCTION_TYPE_EXPR_OR_REF___GET_RETURN_TYPE_REF)) || Objects.equal(cf, TypeRefsPackage.Literals.FUNCTION_TYPE_EXPRESSION__RETURN_TYPE_REF)) && 
      (ref.eContainer().eContainer() instanceof ComposedTypeRef)));
    if (needsParens) {
      result.append("(");
    }
    String _xifexpression = null;
    if ((ref instanceof UnionTypeExpression)) {
      _xifexpression = " | ";
    } else {
      _xifexpression = " & ";
    }
    final String separator = _xifexpression;
    this.appendCommaSeparatedTypeRefList(ref.getTypeRefs(), result, true, separator);
    if (needsParens) {
      result.append(")");
    }
    return result.toString();
  }
  
  /**
   * produces a comma separated TypeRef description list element by element
   */
  private void appendCommaSeparatedTypeRefList(final Iterable<TypeRef> refs, final StyledString styledString, final boolean first, final String separator) {
    boolean _hasNext = refs.iterator().hasNext();
    boolean _not = (!_hasNext);
    if (_not) {
      return;
    }
    int _length = styledString.length();
    boolean _greaterThan = (StyledTextCalculationHelper.OUTLINE_TYPE_REF_LENGTH_THRESHOLD > _length);
    if (_greaterThan) {
      if ((!first)) {
        styledString.append(separator);
      }
      styledString.append(this.getCompressedTypeRefDescription(refs.iterator().next(), styledString.length()));
      this.appendCommaSeparatedTypeRefList(IterableExtensions.<TypeRef>drop(refs, 1), styledString, false, separator);
    } else {
      if ((!first)) {
        styledString.append((separator + "…"));
      } else {
        styledString.append("…");
      }
    }
  }
  
  private String getWildcardDescription(final Wildcard wildcard) {
    final StyledString string = new StyledString();
    string.append("?");
    TypeRef _declaredLowerBound = wildcard.getDeclaredLowerBound();
    boolean _tripleNotEquals = (_declaredLowerBound != null);
    if (_tripleNotEquals) {
      string.append(" super ");
      string.append(this.getCompressedTypeRefDescription(wildcard.getDeclaredLowerBound(), string.length()));
    } else {
      string.append(" extends ");
      string.append(this.getCompressedTypeRefDescription(wildcard.getDeclaredUpperBound(), string.length()));
    }
    return string.getString();
  }
  
  private String nominalTypeNameOrWildCard(final TypeTypeRef ref) {
    String _switchResult = null;
    TypeArgument _typeArg = ref.getTypeArg();
    boolean _matched = false;
    if (_typeArg instanceof TypeRef) {
      _matched=true;
      TypeArgument _typeArg_1 = ref.getTypeArg();
      Type _declaredType = ((TypeRef) _typeArg_1).getDeclaredType();
      String _name = null;
      if (_declaredType!=null) {
        _name=_declaredType.getName();
      }
      _switchResult = _name;
    }
    if (!_matched) {
      if (_typeArg instanceof Wildcard) {
        _matched=true;
        TypeArgument _typeArg_1 = ref.getTypeArg();
        _switchResult = this.getWildcardDescription(((Wildcard) _typeArg_1));
      }
    }
    if (!_matched) {
      _switchResult = "";
    }
    return _switchResult;
  }
  
  private StyledString unknown() {
    return new StyledString("<unknown>");
  }
  
  public StyledString dispatchGetStyledText(final Object tfunction) {
    if (tfunction instanceof TFunction) {
      return _dispatchGetStyledText((TFunction)tfunction);
    } else if (tfunction instanceof TGetter) {
      return _dispatchGetStyledText((TGetter)tfunction);
    } else if (tfunction instanceof TSetter) {
      return _dispatchGetStyledText((TSetter)tfunction);
    } else if (tfunction instanceof ExportedVariableDeclaration) {
      return _dispatchGetStyledText((ExportedVariableDeclaration)tfunction);
    } else if (tfunction instanceof N4GetterDeclaration) {
      return _dispatchGetStyledText((N4GetterDeclaration)tfunction);
    } else if (tfunction instanceof N4SetterDeclaration) {
      return _dispatchGetStyledText((N4SetterDeclaration)tfunction);
    } else if (tfunction instanceof TField) {
      return _dispatchGetStyledText((TField)tfunction);
    } else if (tfunction instanceof N4FieldDeclaration) {
      return _dispatchGetStyledText((N4FieldDeclaration)tfunction);
    } else if (tfunction instanceof FunctionDefinition) {
      return _dispatchGetStyledText((FunctionDefinition)tfunction);
    } else if (tfunction == null) {
      return _dispatchGetStyledText((Void)null);
    } else if (tfunction instanceof EObjectWithContext) {
      return _dispatchGetStyledText((EObjectWithContext)tfunction);
    } else if (tfunction != null) {
      return _dispatchGetStyledText(tfunction);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tfunction).toString());
    }
  }
  
  private String getTypeRefDescriptionString(final TypeArgument ref) {
    if (ref instanceof FunctionTypeExpression) {
      return _getTypeRefDescriptionString((FunctionTypeExpression)ref);
    } else if (ref instanceof ParameterizedTypeRef) {
      return _getTypeRefDescriptionString((ParameterizedTypeRef)ref);
    } else if (ref instanceof ThisTypeRef) {
      return _getTypeRefDescriptionString((ThisTypeRef)ref);
    } else if (ref instanceof TypeTypeRef) {
      return _getTypeRefDescriptionString((TypeTypeRef)ref);
    } else if (ref instanceof ComposedTypeRef) {
      return _getTypeRefDescriptionString((ComposedTypeRef)ref);
    } else if (ref instanceof TypeRef) {
      return _getTypeRefDescriptionString((TypeRef)ref);
    } else if (ref != null) {
      return _getTypeRefDescriptionString(ref);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ref).toString());
    }
  }
}
