/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.DefaultImportSpecifier;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.parser.InternalSemicolonInjectingParser;
import org.eclipse.n4js.ui.organize.imports.ImportsSorter;
import org.eclipse.n4js.ui.utils.ImportSpacerUserPreferenceHelper;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.ILeafNode;
import org.eclipse.xtext.nodemodel.SyntaxErrorMessage;
import org.eclipse.xtext.nodemodel.impl.LeafNodeWithSyntaxError;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper for creating representation of the {@link ImportDeclaration} as string with N4JS code.
 */
@SuppressWarnings("all")
public class ImportDeclarationTextHelper {
  @Inject
  private ImportSpacerUserPreferenceHelper spacerPreference;
  
  /**
   * Extracts the token text for existing import-declaration or creates new textual representation for
   * a new generated import declaration.
   */
  public String extractPureText(final ImportDeclaration declaration, final XtextResource resource, final Adapter nodelessMarker) {
    final String spacer = this.spacerPreference.getSpacingPreference(resource);
    boolean _contains = declaration.eAdapters().contains(nodelessMarker);
    if (_contains) {
      EList<ImportSpecifier> _importSpecifiers = declaration.getImportSpecifiers();
      final ArrayList<ImportSpecifier> impSpec = new ArrayList<ImportSpecifier>(_importSpecifiers);
      final String module = declaration.getModule().getModuleSpecifier();
      int _size = impSpec.size();
      boolean _tripleEquals = (_size == 1);
      if (_tripleEquals) {
        final ImportSpecifier onlyImpSpec = impSpec.get(0);
        if ((onlyImpSpec instanceof NamespaceImportSpecifier)) {
          return this.namespacePureText(((NamespaceImportSpecifier)onlyImpSpec), module);
        }
        final NamedImportSpecifier namedSpec = ((NamedImportSpecifier) onlyImpSpec);
        if ((namedSpec instanceof DefaultImportSpecifier)) {
          return this.defaultPureText(((DefaultImportSpecifier)namedSpec), module);
        } else {
          return this.namedPureText(namedSpec, module, spacer);
        }
      } else {
        return this.multiplePureText(impSpec, module, spacer);
      }
    } else {
      return ImportDeclarationTextHelper.rewriteTokenText(NodeModelUtils.findActualNodeFor(declaration), spacer, InternalSemicolonInjectingParser.SEMICOLON_INSERTED);
    }
  }
  
  private String namespacePureText(final NamespaceImportSpecifier is, final String module) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("import * as ");
    String _alias = is.getAlias();
    _builder.append(_alias);
    _builder.append(" from \"");
    _builder.append(module);
    _builder.append("\";");
    return _builder.toString();
  }
  
  private String defaultPureText(final DefaultImportSpecifier is, final String module) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("import ");
    String _name = is.getImportedElement().getName();
    _builder.append(_name);
    _builder.append(" from \"");
    _builder.append(module);
    _builder.append("\";");
    return _builder.toString();
  }
  
  private String namedPureText(final NamedImportSpecifier is, final String module, final String spacer) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("import {");
    _builder.append(spacer);
    String _name = is.getImportedElement().getName();
    _builder.append(_name);
    {
      String _alias = is.getAlias();
      boolean _tripleNotEquals = (_alias != null);
      if (_tripleNotEquals) {
        _builder.append(" as ");
        String _alias_1 = is.getAlias();
        _builder.append(_alias_1);
      }
    }
    _builder.append(spacer);
    _builder.append("} from \"");
    _builder.append(module);
    _builder.append("\";");
    return _builder.toString();
  }
  
  private String multiplePureText(final List<ImportSpecifier> impSpec, final String module, final String spacer) {
    ImportsSorter.sortByName(impSpec);
    final DefaultImportSpecifier defImp = IterableExtensions.<DefaultImportSpecifier>head(Iterables.<DefaultImportSpecifier>filter(impSpec, DefaultImportSpecifier.class));
    final NamespaceImportSpecifier nameImp = IterableExtensions.<NamespaceImportSpecifier>head(Iterables.<NamespaceImportSpecifier>filter(impSpec, NamespaceImportSpecifier.class));
    final Function1<ImportSpecifier, Boolean> _function = (ImportSpecifier specifier) -> {
      return Boolean.valueOf(((Boolean.valueOf((specifier instanceof DefaultImportSpecifier)) == Boolean.valueOf(false)) && 
        (Boolean.valueOf((specifier instanceof NamespaceImportSpecifier)) == Boolean.valueOf(false))));
    };
    final Iterable<ImportSpecifier> rest = IterableExtensions.<ImportSpecifier>filter(impSpec, _function);
    boolean _isEmpty = IterableExtensions.isEmpty(rest);
    final boolean normalImports = (!_isEmpty);
    String _xifexpression = null;
    if ((defImp == null)) {
      _xifexpression = "";
    } else {
      StringConcatenation _builder = new StringConcatenation();
      String _name = defImp.getImportedElement().getName();
      _builder.append(_name);
      _xifexpression = _builder.toString();
    }
    final String defaultImport = _xifexpression;
    StringConcatenation _builder_1 = new StringConcatenation();
    {
      if (((defImp != null) && (nameImp != null))) {
        _builder_1.append(", ");
      }
    }
    final String spacerDefName = _builder_1.toString();
    String _xifexpression_1 = null;
    if ((nameImp == null)) {
      _xifexpression_1 = "";
    } else {
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("* as ");
      String _alias = nameImp.getAlias();
      _builder_2.append(_alias);
      _xifexpression_1 = _builder_2.toString();
    }
    final String namespaceImport = _xifexpression_1;
    StringConcatenation _builder_3 = new StringConcatenation();
    _builder_3.append("import ");
    _builder_3.append(defaultImport);
    _builder_3.append(spacerDefName);
    _builder_3.append(namespaceImport);
    {
      if (normalImports) {
        _builder_3.append("{");
        _builder_3.append(spacer);
        {
          boolean _hasElements = false;
          for(final ImportSpecifier a : impSpec) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder_3.appendImmediate(", ", "");
            }
            String _name_1 = ((NamedImportSpecifier) a).getImportedElement().getName();
            _builder_3.append(_name_1);
            {
              String _alias_1 = ((NamedImportSpecifier) a).getAlias();
              boolean _tripleNotEquals = (_alias_1 != null);
              if (_tripleNotEquals) {
                _builder_3.append(" as ");
                String _alias_2 = ((NamedImportSpecifier) a).getAlias();
                _builder_3.append(_alias_2);
              }
            }
          }
        }
        _builder_3.append(spacer);
        _builder_3.append("}");
      }
    }
    _builder_3.append(" from \"");
    _builder_3.append(module);
    _builder_3.append("\";");
    return _builder_3.toString();
  }
  
  /**
   * Rewrites the node-content without comments.
   * 
   * Inspired by {@link NodeModelUtils#getTokenText(INode)} but can treat any {@link LeafNodeWithSyntaxError syntax error}
   * nodes as a hidden one and reformats White-space after opening and before closing curly brace according to spacer-policy
   * 
   * @param node the node to reformat.
   * @param spacer append after "{" and prepend before "}" can be empty string, then no white-space is inserted after
   * @param ignoredSyntaxErrorIssues - nodes of type LeafNodeWithSyntaxError having one of this issues are treated as ignored/hidden leafs
   * @returns modified textual form.
   */
  private static String rewriteTokenText(final ICompositeNode node, final String spacer, final String... ignoredSyntaxErrorIssues) {
    int _max = Math.max(node.getTotalLength(), 1);
    final StringBuilder builder = new StringBuilder(_max);
    boolean hiddenSeen = false;
    boolean openingCurlySeen = false;
    boolean fixedASI = false;
    Iterable<ILeafNode> _leafNodes = node.getLeafNodes();
    for (final ILeafNode leaf : _leafNodes) {
      {
        boolean _isIgnoredSyntaxErrorNode = UtilN4.isIgnoredSyntaxErrorNode(leaf, InternalSemicolonInjectingParser.SEMICOLON_INSERTED);
        if (_isIgnoredSyntaxErrorNode) {
          fixedASI = true;
        }
        boolean _isHiddenOrIgnoredSyntaxError = ImportDeclarationTextHelper.isHiddenOrIgnoredSyntaxError(leaf, ignoredSyntaxErrorIssues);
        boolean _not = (!_isHiddenOrIgnoredSyntaxError);
        if (_not) {
          final String text = leaf.getText();
          int _length = builder.length();
          boolean _greaterThan = (_length > 0);
          if (_greaterThan) {
            if (openingCurlySeen) {
              builder.append(spacer);
            } else {
              boolean _equals = Objects.equal("}", text);
              if (_equals) {
                builder.append(spacer);
              } else {
                boolean _equals_1 = Objects.equal(",", text);
                if (_equals_1) {
                } else {
                  if (hiddenSeen) {
                    builder.append(" ");
                  }
                }
              }
            }
          }
          boolean _equals_2 = Objects.equal("{", text);
          openingCurlySeen = _equals_2;
          builder.append(text);
          hiddenSeen = false;
        } else {
          hiddenSeen = true;
        }
      }
    }
    if (fixedASI) {
      builder.append(";");
    }
    return builder.toString();
  }
  
  /**
   * Returns with {@code true} if the leaf node argument is either hidden, or represents a
   * {@link LeafNodeWithSyntaxError syntax error} where the {@link SyntaxErrorMessage#getIssueCode() issue code} of
   * the syntax error matches with any of the given ignored syntax error issue codes. Otherwise returns with
   * {@code false}.
   */
  private static boolean isHiddenOrIgnoredSyntaxError(final ILeafNode leaf, final String... ignoredSyntaxErrorIssues) {
    return (leaf.isHidden() || UtilN4.isIgnoredSyntaxErrorNode(leaf, ignoredSyntaxErrorIssues));
  }
}
