/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.wizard.classifiers;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.packagejson.model.edit.IJSONDocumentModification;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.ui.wizard.classifiers.N4JSClassifierWizardModel;
import org.eclipse.n4js.ui.wizard.generator.ContentBlock;
import org.eclipse.n4js.ui.wizard.generator.ImportAnalysis;
import org.eclipse.n4js.ui.wizard.generator.ImportRequirement;
import org.eclipse.n4js.ui.wizard.generator.N4JSImportRequirementResolver;
import org.eclipse.n4js.ui.wizard.generator.WizardGeneratorHelper;
import org.eclipse.n4js.ui.wizard.generator.WorkspaceWizardGenerator;
import org.eclipse.n4js.ui.wizard.generator.WorkspaceWizardGeneratorException;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.util.StringInputStream;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;

/**
 * A file generator for a {@link N4JSClassifierWizardModel}
 */
@SuppressWarnings("all")
public abstract class N4JSNewClassifierWizardGenerator<M extends N4JSClassifierWizardModel> implements WorkspaceWizardGenerator<M> {
  private static final Logger LOGGER = Logger.getLogger(N4JSNewClassifierWizardGenerator.class);
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  @Extension
  private WizardGeneratorHelper generatorHelper;
  
  @Inject
  private N4JSImportRequirementResolver requirementResolver;
  
  @Override
  public ContentBlock[] generateContentPreview(final M model) {
    final IPath modulePath = model.computeFileLocation();
    boolean _exists = this.generatorHelper.exists(modulePath);
    if (_exists) {
      final XtextResource resource = this.generatorHelper.getResource(URI.createPlatformResourceURI(modulePath.toString(), true));
      final List<ImportRequirement> requirements = this.getImportRequirements(model);
      final ImportAnalysis importAnalysis = this.requirementResolver.analyzeImportRequirements(requirements, resource);
      final IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
      final IFile file = workspaceRoot.getFile(modulePath);
      String _xtrycatchfinallyexpression = null;
      try {
        _xtrycatchfinallyexpression = this.generatorHelper.readFileAsString(file);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          String _string = modulePath.toString();
          String _plus = ("Failed to create a content preview by overlaying existing module " + _string);
          N4JSNewClassifierWizardGenerator.LOGGER.error(_plus);
          return ((ContentBlock[])Conversions.unwrapArray(CollectionLiterals.<ContentBlock>emptyList(), ContentBlock.class));
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      final String fileContent = _xtrycatchfinallyexpression;
      final String importStatements = this.requirementResolver.generateImportStatements(importAnalysis.importRequirements);
      final String classCode = this.generateClassifierCode(model, importAnalysis.aliasBindings);
      final int importStatementOffset = this.requirementResolver.getImportStatementOffset(resource);
      final String importStatementFileContent = fileContent.substring(0, importStatementOffset);
      String _xifexpression = null;
      int _length = fileContent.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _xifexpression = this.generatorHelper.addLineBreak(fileContent.substring(importStatementOffset, fileContent.length()));
      } else {
        _xifexpression = "";
      }
      final String bodyFileContent = _xifexpression;
      ContentBlock _unhighlighted = ContentBlock.unhighlighted(importStatementFileContent);
      ContentBlock _highlighted = ContentBlock.highlighted(this.generatorHelper.addLineBreak(importStatements));
      ContentBlock _unhighlighted_1 = ContentBlock.unhighlighted(bodyFileContent);
      ContentBlock _highlighted_1 = ContentBlock.highlighted(classCode);
      return new ContentBlock[] { _unhighlighted, _highlighted, _unhighlighted_1, _highlighted_1 };
    } else {
      final List<ImportRequirement> importRequirements = this.getImportRequirements(model);
      final Map<URI, String> aliasBindings = this.requirementResolver.resolveImportNameConflicts(importRequirements, null);
      String importStatements_1 = this.requirementResolver.generateImportStatements(importRequirements);
      boolean _isEmpty = importStatements_1.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        String _importStatements = importStatements_1;
        importStatements_1 = (_importStatements + (WizardGeneratorHelper.LINEBREAK + WizardGeneratorHelper.LINEBREAK));
      }
      String _generateClassifierCode = this.generateClassifierCode(model, aliasBindings);
      String _plus_1 = (importStatements_1 + _generateClassifierCode);
      ContentBlock _highlighted_2 = ContentBlock.highlighted(_plus_1);
      return new ContentBlock[] { _highlighted_2 };
    }
  }
  
  /**
   * Writes the new classifier specified by the model to a file.
   * 
   * Depending on the model's module specifier the classifier will be written to a new file or inserted into an existing file.
   * 
   * <p>Note: Make sure to only a pass a valid model, no model validation is done. </p>
   * 
   * @param model The classifier wizard model to write to file
   */
  @Override
  public void writeToFile(final M model, final IProgressMonitor monitor) {
    final IPath modulePath = model.computeFileLocation();
    final IFile moduleFile = ResourcesPlugin.getWorkspace().getRoot().getFile(modulePath);
    try {
      boolean _exists = moduleFile.exists();
      if (_exists) {
        this.insertIntoFile(moduleFile, model);
      } else {
        final List<ImportRequirement> importRequirements = this.getImportRequirements(model);
        final Map<URI, String> aliasBindings = this.requirementResolver.resolveImportNameConflicts(importRequirements, null);
        String importStatements = this.requirementResolver.generateImportStatements(importRequirements);
        boolean _isEmpty = importRequirements.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          importStatements = ((importStatements + WizardGeneratorHelper.LINEBREAK) + WizardGeneratorHelper.LINEBREAK);
        }
        final String classifierCode = this.generateClassifierCode(model, aliasBindings);
        final StringInputStream classifierTextStream = new StringInputStream((importStatements + classifierCode));
        moduleFile.create(classifierTextStream, true, null);
      }
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        String _message = e.getMessage();
        throw new WorkspaceWizardGeneratorException(_message);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Performs the project description changes required by the classifier specified by the model.
   * 
   * This means for now the computation of necessary project dependencies and their addition to the project description file.
   * 
   * <p> IMPORTANT: This method should always be called before invoking {@link #writeToFile(N4JSClassWizardModel)} as
   * writeToFile may need project description changes to resolve all imports.</p>
   */
  @Override
  public void performProjectDescriptionChanges(final M model, final IProgressMonitor monitor) throws WorkspaceWizardGeneratorException {
    monitor.subTask("Performing project changes");
    final Optional<? extends IN4JSProject> project = this.n4jsCore.findProject(URI.createPlatformResourceURI(model.computeFileLocation().toString(), true));
    boolean _isPresent = project.isPresent();
    boolean _not = (!_isPresent);
    if (_not) {
      throw new WorkspaceWizardGeneratorException("The target project couldn\'t be found.");
    }
    final Optional<URI> projectDescriptionLocation = project.get().getProjectDescriptionLocation();
    final XtextResource projectDescriptionResource = this.generatorHelper.getResource(projectDescriptionLocation.get());
    final List<IN4JSProject> referencedProjects = this.getReferencedProjects(model);
    final URI moduleURI = URI.createPlatformResourceURI(model.computeFileLocation().toString(), true);
    final Collection<IJSONDocumentModification> projectDescriptionModifications = this.generatorHelper.projectDescriptionModifications(projectDescriptionResource, model, referencedProjects, moduleURI);
    int _length = ((Object[])Conversions.unwrapArray(projectDescriptionModifications, Object.class)).length;
    boolean _greaterThan = (_length > 0);
    if (_greaterThan) {
      boolean _applyJSONModifications = this.generatorHelper.applyJSONModifications(projectDescriptionResource, projectDescriptionModifications);
      boolean _not_1 = (!_applyJSONModifications);
      if (_not_1) {
        throw new WorkspaceWizardGeneratorException("Couldn\'t apply package.json changes.");
      }
    }
  }
  
  /**
   * Generates the classifier code with regard to the given alias bindings.
   */
  protected abstract String generateClassifierCode(final M model, final Map<URI, String> aliasBindings);
  
  /**
   * Returns all projects referenced by the given model.
   * 
   * This includes uses of interfaces or super classes from other projects.
   */
  protected abstract List<IN4JSProject> getReferencedProjects(final M model);
  
  /**
   * Returns the import requirements of the model.
   */
  protected abstract List<ImportRequirement> getImportRequirements(final M model);
  
  private void insertIntoFile(final IFile file, final M model) throws CoreException {
    final URI moduleURI = URI.createPlatformResourceURI(model.computeFileLocation().toString(), true);
    final XtextResource moduleResource = this.generatorHelper.getResource(moduleURI);
    final List<ImportRequirement> demandedImports = this.getImportRequirements(model);
    final ImportAnalysis importAnalysis = this.requirementResolver.analyzeImportRequirements(demandedImports, moduleResource);
    String classCode = this.generateClassifierCode(model, importAnalysis.aliasBindings);
    String _lastCharacterInFile = this.generatorHelper.lastCharacterInFile(file);
    boolean _notEquals = (!Objects.equal(_lastCharacterInFile, WizardGeneratorHelper.LINEBREAK));
    if (_notEquals) {
      classCode = (WizardGeneratorHelper.LINEBREAK + classCode);
    }
    final StringInputStream classifierCodeStream = new StringInputStream(classCode);
    this.generatorHelper.insertImportStatements(moduleResource, importAnalysis.importRequirements);
    file.appendContents(classifierCodeStream, true, true, null);
    this.generatorHelper.organizeImports(file, null);
  }
}
