/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.csv;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.eclipse.n4js.csv.CSVData;
import org.eclipse.n4js.csv.CSVParser;
import org.eclipse.n4js.csv.CSVRecord;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.junit.Assert;
import org.junit.Test;

/**
 * A few test cases for the CSV parser.
 */
@SuppressWarnings("all")
public class CSVParserTest {
  private static void assertResult(final List<List<String>> expected, final CSVData actual) {
    Iterator<List<String>> expectedIt = expected.iterator();
    Iterator<CSVRecord> actualIt = actual.iterator();
    while ((expectedIt.hasNext() || actualIt.hasNext())) {
      {
        Assert.assertEquals(Boolean.valueOf(expectedIt.hasNext()), Boolean.valueOf(actualIt.hasNext()));
        final List<String> expectedRecord = expectedIt.next();
        final CSVRecord actualRecord = actualIt.next();
        CSVParserTest.assertRecords(expectedRecord, actualRecord);
      }
    }
  }
  
  private static void assertRecords(final Iterable<String> expected, final Iterable<String> actual) {
    Iterator<String> expectedIt = expected.iterator();
    Iterator<String> actualIt = actual.iterator();
    while ((expectedIt.hasNext() && actualIt.hasNext())) {
      {
        Assert.assertEquals(Boolean.valueOf(expectedIt.hasNext()), Boolean.valueOf(actualIt.hasNext()));
        Assert.assertEquals(expectedIt.next(), actualIt.next());
      }
    }
  }
  
  @Test
  public void testWindowsLineEndings() {
    try {
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("1", "2", "3")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("1", "2", "3")))), 
        CSVParser.parse("1,2,3\r\n1,2,3"));
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("1", "2", "")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("1", "2", "3")))), 
        CSVParser.parse("1,2,\r\n1,2,3"));
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("1", "2", "")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "2", "3")))), 
        CSVParser.parse("1,2,\r\n,2,3"));
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "Supplier", "Class", "", "")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "Subject of Access", "Field, Accessor, Method", "", "Static Field, Getter, Setter, Method", "", "")))), 
        CSVParser.parse(",,Supplier,Class,,\r\n,,Subject of Access,\"Field, Accessor, Method\",,\"Static Field, Getter, Setter, Method\",,\r\n"));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testEmptyString() {
    try {
      CSVParserTest.assertResult(Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList()), CSVParser.parse(""));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testOneEmptyField() {
    try {
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")))), CSVParser.parse("\n"));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testTwoRowsWithOneFieldEach() {
    try {
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")))), CSVParser.parse("\n\n"));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testOneRowWithThreeFields() {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("One,Two,Three");
      _builder.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "Two", "Three")))), CSVParser.parse(_builder.toString()));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testOneRowWithThreeEmptyFields() {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(",,");
      _builder.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "")))), CSVParser.parse(_builder.toString()));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testOneRowWithSomeEmptyFields() {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("One,,,");
      _builder.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "", "", "")))), 
        CSVParser.parse(_builder.toString()));
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(",One,,");
      _builder_1.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "One", "", "")))), 
        CSVParser.parse(_builder_1.toString()));
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append(",,One,");
      _builder_2.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "One", "")))), 
        CSVParser.parse(_builder_2.toString()));
      StringConcatenation _builder_3 = new StringConcatenation();
      _builder_3.append(",,,One");
      _builder_3.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "", "One")))), 
        CSVParser.parse(_builder_3.toString()));
      StringConcatenation _builder_4 = new StringConcatenation();
      _builder_4.append("One,,Two,");
      _builder_4.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "", "Two", "")))), 
        CSVParser.parse(_builder_4.toString()));
      StringConcatenation _builder_5 = new StringConcatenation();
      _builder_5.append("One,,,Two");
      _builder_5.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "", "", "Two")))), 
        CSVParser.parse(_builder_5.toString()));
      StringConcatenation _builder_6 = new StringConcatenation();
      _builder_6.append(",One,Two,");
      _builder_6.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "One", "Two", "")))), 
        CSVParser.parse(_builder_6.toString()));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testSeveralRowsOfSameLength() {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("One,Two,Three,Four");
      _builder.newLine();
      _builder.append("Five,Six,Seven,Eight");
      _builder.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "Two", "Three", "Four")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder.toString()));
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(",Two,Three,");
      _builder_1.newLine();
      _builder_1.append(",Six,Seven,");
      _builder_1.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "Two", "Three", "")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "Six", "Seven", "")))), 
        CSVParser.parse(_builder_1.toString()));
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("One,Two,Three,Four");
      _builder_2.newLine();
      _builder_2.append(",,,");
      _builder_2.newLine();
      _builder_2.append("Five,Six,Seven,Eight");
      _builder_2.newLine();
      _builder_2.append(",,,");
      _builder_2.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "Two", "Three", "Four")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "", "")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "", "")))), 
        CSVParser.parse(_builder_2.toString()));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testSeveralRowsOfDifferingLengths() {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("One,Two,Three");
      _builder.newLine();
      _builder.append("Five,Six,Seven,Eight");
      _builder.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "Two", "Three")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder.toString()));
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.newLine();
      _builder_1.append("Five,Six,Seven,Eight");
      _builder_1.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder_1.toString()));
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("Five,Six,Seven,Eight");
      _builder_2.newLine();
      _builder_2.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")))), 
        CSVParser.parse(_builder_2.toString()));
      StringConcatenation _builder_3 = new StringConcatenation();
      _builder_3.append("Five,Six,Seven,Eight");
      _builder_3.newLine();
      _builder_3.newLine();
      _builder_3.append(",,");
      _builder_3.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "", "")))), 
        CSVParser.parse(_builder_3.toString()));
      StringConcatenation _builder_4 = new StringConcatenation();
      _builder_4.append("One,Two,Three");
      _builder_4.newLine();
      _builder_4.newLine();
      _builder_4.append(",");
      _builder_4.newLine();
      _builder_4.newLine();
      _builder_4.append("Five,Six,Seven,Eight");
      _builder_4.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One", "Two", "Three")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("", "")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder_4.toString()));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Test
  public void testFieldsWithControlChars() {
    try {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("\"One,Two, Three\"");
      _builder.newLine();
      _builder.append("Five,Six,Seven,Eight");
      _builder.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One,Two, Three")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder.toString()));
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("\"One,Two,");
      _builder_1.newLine();
      _builder_1.newLine();
      _builder_1.append("Three\"");
      _builder_1.newLine();
      _builder_1.append("\"");
      _builder_1.newLine();
      _builder_1.append("\"");
      _builder_1.newLine();
      _builder_1.append("Five,Six,Seven,Eight");
      _builder_1.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("One,Two, \n\nThree")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("\n")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder_1.toString()));
      StringConcatenation _builder_2 = new StringConcatenation();
      _builder_2.append("\"\"\"This is a quote!\"\"\",This isn\'t,\"This is not, either\"");
      _builder_2.newLine();
      _builder_2.append("Five,Six,Seven,Eight");
      _builder_2.newLine();
      CSVParserTest.assertResult(
        Collections.<List<String>>unmodifiableList(CollectionLiterals.<List<String>>newArrayList(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("\"This is a quote!\"", "This isn\'t", "This is not, either")), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("Five", "Six", "Seven", "Eight")))), 
        CSVParser.parse(_builder_2.toString()));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
