/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.tests.helper;

import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.IntFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import junit.framework.AssertionFailedError;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4idl.tests.helper.N4IDLParseHelper;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.testing.validation.ValidationTestHelper;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.junit.Assert;

/**
 * N4IDL test-helper to create valid {@link TypeRef} instances from a given
 * type expression string (e.g. "A#2") or expression strings (e.g. "new A#2()", "1", "true").
 */
@SuppressWarnings("all")
public class N4IDLTypeRefTestHelper {
  @Inject
  @Extension
  private N4IDLParseHelper _n4IDLParseHelper;
  
  @Inject
  @Extension
  private ValidationTestHelper _validationTestHelper;
  
  @Inject
  private N4JSTypeSystem typeSystem;
  
  /**
   * Creates an N4IDL-module containing the given type expression, extracts the corresponding {@link TypeRef}
   * element from the AST and returns it.
   * 
   * @param typeExpression A type expression as it may occur in a variable declaration
   * @param preamble A valid N4IDL preamble in which clients may declared additional testing types.
   */
  public TypeRef makeTypeRef(final String typeExpression, final String preamble) {
    return this.makeTypeRefs(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(typeExpression)), preamble).get(0);
  }
  
  /**
   * Creates {@link TypeRef}s for all given type expressions.
   * 
   * All returned {@link TypeRef}s are extracted from the same module, thus they
   * share the same type model instances.
   */
  public List<TypeRef> makeTypeRefs(final List<String> typeExpressions, final String preamble) {
    final List<Pair<String, String>> expressionVariables = this.<String>distinctNames(typeExpressions, "typeExpression");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(preamble);
    _builder.newLineIfNotEmpty();
    _builder.append("@VersionAware");
    _builder.newLine();
    _builder.append("function f() {");
    _builder.newLine();
    {
      for(final Pair<String, String> e : expressionVariables) {
        _builder.append("\t");
        _builder.append("let ");
        String _key = e.getKey();
        _builder.append(_key, "\t");
        _builder.append(" : ");
        String _value = e.getValue();
        _builder.append(_value, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    final String module = _builder.toString();
    final Function1<Pair<String, String>, String> _function = (Pair<String, String> it) -> {
      return it.getKey();
    };
    final Function1<VariableDeclaration, TypeRef> _function_1 = (VariableDeclaration decl) -> {
      return decl.getDeclaredTypeRef();
    };
    return IterableExtensions.<TypeRef>toList(ListExtensions.<VariableDeclaration, TypeRef>map(this.parseAndFindVariableDeclarations(module, IterableExtensions.<String>toList(ListExtensions.<Pair<String, String>, String>map(expressionVariables, _function))), _function_1));
  }
  
  /**
   * Parses the given module string as N4IDL and returns all {@link VariableDeclaration} whose names are
   * specified in {@code variableDeclarations} in that order.
   * 
   * Raises an {@link AssertionFailedError} if the module is invalid N4IDL or the specified {@code variableDeclarations}
   * cannot be found in the resulting AST.
   */
  private List<VariableDeclaration> parseAndFindVariableDeclarations(final String module, final List<String> variableDeclarations) {
    try {
      final Script script = this._n4IDLParseHelper.parseN4IDL(module);
      this._validationTestHelper.assertNoIssues(script);
      final Function1<VariableDeclaration, String> _function = (VariableDeclaration decl) -> {
        return decl.getName();
      };
      final Function1<VariableDeclaration, VariableDeclaration> _function_1 = (VariableDeclaration decl) -> {
        return decl;
      };
      final Map<String, VariableDeclaration> variableDeclarationsByName = IterableExtensions.<VariableDeclaration, String, VariableDeclaration>toMap(IteratorExtensions.<VariableDeclaration>toSet(Iterators.<VariableDeclaration>filter(script.eAllContents(), VariableDeclaration.class)), _function, _function_1);
      final Function1<String, VariableDeclaration> _function_2 = (String name) -> {
        return variableDeclarationsByName.get(name);
      };
      final List<VariableDeclaration> resultingVariableDeclaration = IterableExtensions.<VariableDeclaration>toList(IterableExtensions.<VariableDeclaration>filterNull(ListExtensions.<String, VariableDeclaration>map(variableDeclarations, _function_2)));
      int _size = resultingVariableDeclaration.size();
      int _size_1 = variableDeclarations.size();
      boolean _notEquals = (_size != _size_1);
      if (_notEquals) {
        Assert.fail(((("Failed to find all specified variable declarations " + variableDeclarations) + " in N4IDL module:\n ") + module));
      }
      return resultingVariableDeclaration;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Creates distinct names for each of the given elements and returns a list of
   * (name -> element) pairs.
   * 
   * Uses the given base name as a prefix for all names (e.g. base name "v" yield
   * distinct names "v0", "v1", "v2", etc.).
   * 
   * @param elements The elements to create distinct names for.
   * @param baseName The base name to use for the name generation.
   */
  public <T extends Object> List<Pair<String, T>> distinctNames(final List<T> elements, final String baseName) {
    final IntFunction<Pair<String, T>> _function = (int index) -> {
      T _get = elements.get(index);
      return Pair.<String, T>of((baseName + Integer.valueOf(index)), _get);
    };
    return IntStream.range(0, elements.size()).<Pair<String, T>>mapToObj(_function).collect(Collectors.<Pair<String, T>>toList());
  }
  
  /**
   * Creates an N4IDL-module containing the given type expression, extracts the corresponding {@link TypeRef}
   * element from the AST and returns it.
   * 
   * @param typeExpression A type expression as it may occur in a variable declaration
   * @param existingClasses A list of class names (with version, e.g. "A#1") which are assumed to exist in the namespace of the module.
   */
  public TypeRef makeTypeRef(final String typeExpression, final List<String> existingClasses) {
    return this.makeTypeRef(typeExpression, this.classes(((String[])Conversions.unwrapArray(existingClasses, String.class))));
  }
  
  /**
   * Creates {@link TypeRef}s for all value expressions (e.g. "new A#1()") and type expression (e.g. "A#1", "A#2")
   * and returns them in the given order.
   * 
   * All {@link TypeRef}s are extracted from the same module and thus share the same
   * type model instances.
   */
  public List<TypeRef> makeTypeRefs(final List<String> valueExpressions, final List<String> typeExpressions, final String preamble) {
    final List<Pair<String, String>> valueVariables = this.<String>distinctNames(valueExpressions, "value");
    final List<Pair<String, String>> typeExpressionVariables = this.<String>distinctNames(typeExpressions, "type");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(preamble);
    _builder.newLineIfNotEmpty();
    _builder.append("@VersionAware");
    _builder.newLine();
    _builder.append("function f() {");
    _builder.newLine();
    {
      for(final Pair<String, String> v : valueVariables) {
        _builder.append("\t");
        _builder.append("let ");
        String _key = v.getKey();
        _builder.append(_key, "\t");
        _builder.append(" = ");
        String _value = v.getValue();
        _builder.append(_value, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final Pair<String, String> t : typeExpressionVariables) {
        _builder.append("\t");
        _builder.append("let ");
        String _key_1 = t.getKey();
        _builder.append(_key_1, "\t");
        _builder.append(" : ");
        String _value_1 = t.getValue();
        _builder.append(_value_1, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    final String module = _builder.toString();
    final Function1<Pair<String, String>, String> _function = (Pair<String, String> v_1) -> {
      return v_1.getKey();
    };
    List<String> _map = ListExtensions.<Pair<String, String>, String>map(valueVariables, _function);
    final Function1<Pair<String, String>, String> _function_1 = (Pair<String, String> t_1) -> {
      return t_1.getKey();
    };
    List<String> _map_1 = ListExtensions.<Pair<String, String>, String>map(typeExpressionVariables, _function_1);
    final Iterable<String> allVariableNames = Iterables.<String>concat(_map, _map_1);
    final List<VariableDeclaration> declarations = this.parseAndFindVariableDeclarations(module, IterableExtensions.<String>toList(allVariableNames));
    final Function1<VariableDeclaration, TypeRef> _function_2 = (VariableDeclaration decl) -> {
      TypeRef _declaredTypeRef = decl.getDeclaredTypeRef();
      boolean _tripleNotEquals = (_declaredTypeRef != null);
      if (_tripleNotEquals) {
        return decl.getDeclaredTypeRef();
      } else {
        return this.typeSystem.tau(decl);
      }
    };
    return ListExtensions.<VariableDeclaration, TypeRef>map(declarations, _function_2);
  }
  
  /**
   * Returns N4IDL code which declares the given list of classes (incl. version).
   * 
   * Example: <code> classes(#["A#1"]) -> "class A#1 {}" </code>
   */
  public String classes(final String... classNames) {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final String e : classNames) {
        _builder.newLineIfNotEmpty();
        _builder.append("class ");
        _builder.append(e);
        _builder.append(" {}");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }
}
