/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import java.util.List;
import java.util.function.Function;
import org.eclipse.n4js.utils.Diff;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;

/**
 * Builder for creating a {@link Diff} via a fluent API. The fields are declared as protected only for debug purposes.
 */
@SuppressWarnings("all")
public abstract class DiffBuilder<F extends Object, T extends Object> {
  /**
   * The initial, ordered state of the relevant items.
   */
  @VisibleForTesting
  protected final List<T> oldItems;
  
  /**
   * The initial, ordered state of all items.
   */
  @VisibleForTesting
  protected final List<T> oldAllItems;
  
  /**
   * The items that have been added.
   */
  @VisibleForTesting
  protected final List<T> addedItems;
  
  /**
   * The removed items.
   */
  @VisibleForTesting
  protected final List<T> deletedItems;
  
  /**
   * Map of edited items. Keys are old state, values are the new state.
   */
  @VisibleForTesting
  protected final BiMap<T, T> editedItems;
  
  private final F input;
  
  public DiffBuilder(final F input) {
    this.input = input;
    this.oldItems = ((List<T>)Conversions.doWrapArray(this.getOldItemsFunction().apply(this.input)));
    this.oldAllItems = ((List<T>)Conversions.doWrapArray(this.getAllOldItemsFunction().apply(this.input)));
    boolean _containsAll = this.oldAllItems.containsAll(this.oldItems);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Not all old items is not a subset of all old items. Old items: ");
    _builder.append(this.oldItems);
    _builder.append(". All old items: ");
    _builder.append(this.oldAllItems);
    _builder.append(".");
    Preconditions.checkState(_containsAll, _builder);
    this.addedItems = CollectionLiterals.<T>newArrayList();
    this.deletedItems = CollectionLiterals.<T>newArrayList();
    this.editedItems = HashBiMap.<T, T>create();
  }
  
  /**
   * Adds a new item into the diff.
   */
  public DiffBuilder<F, T> add(final T item) {
    if (((!this.oldAllItems.contains(item)) && (!this.addedItems.contains(item)))) {
      this.addedItems.add(item);
    }
    this.deletedItems.remove(item);
    return this;
  }
  
  /**
   * Deletes an item from the diff.
   */
  public DiffBuilder<F, T> delete(final T item) {
    final int index = this.addedItems.indexOf(item);
    if ((index >= 0)) {
      this.addedItems.remove(index);
    } else {
      final T originalItem = this.editedItems.inverse().remove(item);
      this.deletedItems.add(originalItem);
    }
    return this;
  }
  
  /**
   * Updates an item in the diff.
   */
  public DiffBuilder<F, T> edit(final T oldState, final T newState) {
    final int index = this.addedItems.indexOf(oldState);
    if ((index >= 0)) {
      this.addedItems.remove(index);
      this.addedItems.add(index, newState);
    } else {
      final T originalState = this.editedItems.inverse().get(oldState);
      if ((null != originalState)) {
        this.editedItems.put(originalState, newState);
      } else {
        this.editedItems.put(oldState, newState);
      }
    }
    return this;
  }
  
  /**
   * Builds the diff instance using latest item states.
   */
  public Diff<T> build(final T[] newItems, final T[] newAllItems) {
    return new Diff<T>(((T[])Conversions.unwrapArray(this.oldItems, Object.class)), ((T[])Conversions.unwrapArray(this.oldAllItems, Object.class)), ((T[])Conversions.unwrapArray(this.addedItems, Object.class)), ((T[])Conversions.unwrapArray(this.deletedItems, Object.class)), this.editedItems, newItems, newAllItems);
  }
  
  /**
   * Returns with the input.
   */
  protected F getInput() {
    return this.input;
  }
  
  /**
   * Function from extracting the initial, ordered state of the relevant from the subject.
   */
  protected abstract Function<F, T[]> getOldItemsFunction();
  
  /**
   * Function from extracting the initial, ordered state of all items from the subject.
   */
  protected abstract Function<F, T[]> getAllOldItemsFunction();
}
