/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.runner.nodejs.ui.launch;

import com.google.common.base.Strings;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.n4js.runner.RunConfiguration;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Launch configuration tab for configuring Node.js NODE_PATH and other options.
 */
@SuppressWarnings("all")
public class NodejsLaunchConfigurationTab extends AbstractLaunchConfigurationTab {
  /**
   * Text field for storing any arbitrary options for execution engine.
   */
  private Text engineOptionsText;
  
  /**
   * Text field for storing any arbitrary options for the executed N4JS code.
   */
  private Text runOptionsText;
  
  /**
   * Text field storing env. variables (VAR=...)
   */
  private Text environmentVariablesText;
  
  /**
   * Converts a map to text, each entry is put on a line, key and value are separated by "=".
   * For convenience reasons, the map is sorted by key.
   * @param map may be empty or even null.
   */
  public static String mapToString(final Map<String, String> map) {
    String _xblockexpression = null;
    {
      if ((map == null)) {
        return "";
      }
      final Comparator<Map.Entry<String, String>> _function = (Map.Entry<String, String> o1, Map.Entry<String, String> o2) -> {
        return o1.getKey().compareTo(o2.getKey());
      };
      final Function1<Map.Entry<String, String>, CharSequence> _function_1 = (Map.Entry<String, String> it) -> {
        String _key = it.getKey();
        String _plus = (_key + "=");
        String _value = it.getValue();
        return (_plus + _value);
      };
      _xblockexpression = IterableExtensions.<Map.Entry<String, String>>join(IterableExtensions.<Map.Entry<String, String>>sortWith(map.entrySet(), _function), "\n", _function_1);
    }
    return _xblockexpression;
  }
  
  /**
   * Converts text to map, each entry is assumed on a line (separated by \n), key and value are
   * separated by "=". Empty lines are ignored.
   */
  public static Map<String, String> stringToMap(final String text) {
    final Map<String, String> map = new LinkedHashMap<String, String>();
    if ((text != null)) {
      final Consumer<String> _function = (String it) -> {
        final String line = it.trim();
        boolean _isEmpty = line.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final String[] keyVal = line.split("=");
          int _length = keyVal.length;
          boolean _notEquals = (_length != 2);
          if (_notEquals) {
            throw new IllegalArgumentException(
              ("Env. vars are expected to be saved in lines with key=val; found: " + line));
          }
          map.put(keyVal[0].trim(), keyVal[1].trim());
        }
      };
      ((List<String>)Conversions.doWrapArray(text.split("\n"))).forEach(_function);
    }
    return map;
  }
  
  @Override
  public void createControl(final Composite parent) {
    Composite _composite = new Composite(parent, SWT.NONE);
    final Procedure1<Composite> _function = (Composite it) -> {
      it.setLayout(GridLayoutFactory.swtDefaults().create());
      it.setLayoutData(GridDataFactory.swtDefaults().grab(true, true).align(SWT.FILL, SWT.FILL).create());
    };
    final Composite childControl = ObjectExtensions.<Composite>operator_doubleArrow(_composite, _function);
    this.engineOptionsText = this.createGroupWithMultiText(childControl, "Command line options passed to node.js:");
    this.runOptionsText = this.createGroupWithMultiText(childControl, "Command line options passed to executed N4JS code:");
    this.environmentVariablesText = this.createGroupWithMultiText(childControl, "Environment Variables (VAR=...):");
    this.setControl(childControl);
  }
  
  @Override
  public String getName() {
    return "Node.js settings";
  }
  
  @Override
  public void initializeFrom(final ILaunchConfiguration configuration) {
    try {
      this.engineOptionsText.setText(configuration.getAttribute(RunConfiguration.ENGINE_OPTIONS, ""));
      this.runOptionsText.setText(configuration.getAttribute(RunConfiguration.RUN_OPTIONS, ""));
      this.environmentVariablesText.setText(NodejsLaunchConfigurationTab.mapToString(configuration.getAttribute(RunConfiguration.ENV_VARS, Collections.<String, String>emptyMap())));
    } catch (final Throwable _t) {
      if (_t instanceof CoreException) {
        final CoreException e = (CoreException)_t;
        this.setErrorMessage(e.getMessage());
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Override
  public void performApply(final ILaunchConfigurationWorkingCopy configuration) {
    configuration.setAttribute(RunConfiguration.ENGINE_OPTIONS, Strings.nullToEmpty(this.engineOptionsText.getText()));
    configuration.setAttribute(RunConfiguration.RUN_OPTIONS, Strings.nullToEmpty(this.runOptionsText.getText()));
    configuration.setAttribute(RunConfiguration.ENV_VARS, NodejsLaunchConfigurationTab.stringToMap(this.environmentVariablesText.getText()));
  }
  
  @Override
  public void setDefaults(final ILaunchConfigurationWorkingCopy configuration) {
    configuration.setAttribute(RunConfiguration.ENGINE_OPTIONS, "");
    configuration.setAttribute(RunConfiguration.RUN_OPTIONS, "");
    configuration.setAttribute(RunConfiguration.ENV_VARS, "");
  }
  
  private Text createMultiText(final Composite parent) {
    Text _text = new Text(parent, (SWT.MULTI | SWT.BORDER));
    final Procedure1<Text> _function = (Text it) -> {
      it.setLayoutData(GridDataFactory.swtDefaults().grab(true, true).align(SWT.FILL, SWT.FILL).create());
      final ModifyListener _function_1 = (ModifyEvent it_1) -> {
        this.updateLaunchConfigurationDialog();
      };
      it.addModifyListener(_function_1);
    };
    return ObjectExtensions.<Text>operator_doubleArrow(_text, _function);
  }
  
  private Text createGroupWithMultiText(final Composite parent, final String groupText) {
    Group _group = new Group(parent, SWT.NONE);
    final Procedure1<Group> _function = (Group it) -> {
      it.setText(groupText);
      it.setLayout(GridLayoutFactory.swtDefaults().create());
      it.setLayoutData(GridDataFactory.swtDefaults().grab(true, true).align(SWT.FILL, SWT.FILL).create());
    };
    final Group group = ObjectExtensions.<Group>operator_doubleArrow(_group, _function);
    return this.createMultiText(group);
  }
}
