/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.packagejson;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.function.BiConsumer;
import org.eclipse.n4js.json.JSON.JSONArray;
import org.eclipse.n4js.json.JSON.JSONBooleanLiteral;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.JSON.JSONFactory;
import org.eclipse.n4js.json.JSON.JSONObject;
import org.eclipse.n4js.json.JSON.JSONStringLiteral;
import org.eclipse.n4js.json.JSON.NameValuePair;
import org.eclipse.n4js.json.model.utils.JSONModelUtils;
import org.eclipse.n4js.packagejson.PackageJsonProperties;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectDescription.SourceContainerType;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Class for providing the content of N4JS-specific package.json files.
 * 
 * Use {@link PackageJsonBuilder} for creating package.json model instances and file content.
 */
@SuppressWarnings("all")
class PackageJsonContentProvider {
  /**
   * Creates and returns with the N4JS package.json {@link JSONDocument} representation
   * based on the given arguments.
   * 
   * @param projectName the N4JS project name of the project (cf. name).
   * @param version The declared version of the project.
   * @param type The type of the N4JS project.
   * @param vendorId The vendorId to use.
   * @param vendorName The name of the vendor as string.
   * @param output The relative output folder location.
   * @param extendedRE The optional extended runtime environment.
   * @param dependencies A map of dependencies of the project (maps dependencies to their version constraints).
   * @param providedRL An iterable of provided runtime libraries.
   * @param requiredRL An iterable of required runtime libraries.
   * @param implementationId The implementationId of the project.
   * @param testedProject A list of all projects that are being tested.
   * @param sourceContainers A map of all source containers of the project.
   * 
   * @return the N4JS package.json content as a string.
   */
  static JSONDocument getModel(final Optional<String> projectName, final Optional<String> version, final Optional<Boolean> _private, final Iterable<String> workspaces, final Optional<ProjectType> type, final Optional<String> vendorId, final Optional<String> vendorName, final Optional<String> output, final Optional<String> extendedRE, final SortedMap<String, String> dependencies, final SortedMap<String, String> devDependencies, final Iterable<String> providedRL, final Iterable<String> requiredRL, final Optional<String> implementationId, final Iterable<String> implementedProjects, final Iterable<String> testedProjects, final Map<SourceContainerType, String> sourceContainers) {
    final JSONObject root = JSONFactory.eINSTANCE.createJSONObject();
    boolean _isPresent = projectName.isPresent();
    if (_isPresent) {
      JSONModelUtils.addProperty(root, PackageJsonProperties.NAME.name, projectName.get());
    }
    boolean _isPresent_1 = version.isPresent();
    if (_isPresent_1) {
      JSONModelUtils.addProperty(root, PackageJsonProperties.VERSION.name, version.get());
    }
    boolean _isPresent_2 = _private.isPresent();
    if (_isPresent_2) {
      JSONModelUtils.<JSONBooleanLiteral>addProperty(root, PackageJsonProperties.PRIVATE.name, 
        JSONModelUtils.createBooleanLiteral((_private.get()).booleanValue()));
    }
    boolean _isEmpty = IterableExtensions.isEmpty(workspaces);
    boolean _not = (!_isEmpty);
    if (_not) {
      JSONModelUtils.<JSONArray>addProperty(root, PackageJsonProperties.WORKSPACES.name, 
        JSONModelUtils.createStringArray(workspaces));
    }
    boolean _isEmpty_1 = dependencies.isEmpty();
    boolean _not_1 = (!_isEmpty_1);
    if (_not_1) {
      final JSONObject dependenciesValue = PackageJsonContentProvider.createDependenciesValue(dependencies);
      JSONModelUtils.<JSONObject>addProperty(root, PackageJsonProperties.DEPENDENCIES.name, dependenciesValue);
    }
    boolean _isEmpty_2 = devDependencies.isEmpty();
    boolean _not_2 = (!_isEmpty_2);
    if (_not_2) {
      final JSONObject devDependenciesValue = PackageJsonContentProvider.createDependenciesValue(devDependencies);
      JSONModelUtils.<JSONObject>addProperty(root, PackageJsonProperties.DEV_DEPENDENCIES.name, devDependenciesValue);
    }
    final JSONObject n4jsRoot = JSONFactory.eINSTANCE.createJSONObject();
    boolean _isPresent_3 = type.isPresent();
    if (_isPresent_3) {
      JSONModelUtils.addProperty(n4jsRoot, PackageJsonProperties.PROJECT_TYPE.name, PackageJsonContentProvider.getEnumAsString(type.get()));
    }
    boolean _isPresent_4 = vendorId.isPresent();
    if (_isPresent_4) {
      JSONModelUtils.addProperty(n4jsRoot, PackageJsonProperties.VENDOR_ID.name, vendorId.get());
    }
    boolean _isPresent_5 = vendorName.isPresent();
    if (_isPresent_5) {
      JSONModelUtils.addProperty(n4jsRoot, PackageJsonProperties.VENDOR_NAME.name, vendorName.get());
    }
    boolean _isEmpty_3 = sourceContainers.isEmpty();
    boolean _not_3 = (!_isEmpty_3);
    if (_not_3) {
      final JSONObject sourcesSection = JSONModelUtils.<JSONObject>addProperty(n4jsRoot, PackageJsonProperties.SOURCES.name, 
        JSONFactory.eINSTANCE.createJSONObject());
      final Function1<Map.Entry<SourceContainerType, String>, String> _function = (Map.Entry<SourceContainerType, String> e) -> {
        return e.getKey().getLiteral();
      };
      final Function1<Map.Entry<SourceContainerType, String>, SourceContainerType> _function_1 = (Map.Entry<SourceContainerType, String> e) -> {
        return e.getKey();
      };
      final BiConsumer<SourceContainerType, List<Map.Entry<SourceContainerType, String>>> _function_2 = (SourceContainerType containerType, List<Map.Entry<SourceContainerType, String>> paths) -> {
        final JSONArray typeSectionArray = JSONModelUtils.<JSONArray>addProperty(sourcesSection, 
          containerType.getLiteral().toLowerCase(), JSONFactory.eINSTANCE.createJSONArray());
        final Function1<Map.Entry<SourceContainerType, String>, JSONStringLiteral> _function_3 = (Map.Entry<SourceContainerType, String> pathEntry) -> {
          return JSONModelUtils.createStringLiteral(pathEntry.getValue());
        };
        final List<JSONStringLiteral> pathLiterals = ListExtensions.<Map.Entry<SourceContainerType, String>, JSONStringLiteral>map(paths, _function_3);
        typeSectionArray.getElements().addAll(pathLiterals);
      };
      IterableExtensions.<SourceContainerType, Map.Entry<SourceContainerType, String>>groupBy(IterableExtensions.<Map.Entry<SourceContainerType, String>, String>sortBy(sourceContainers.entrySet(), _function), _function_1).forEach(_function_2);
    }
    boolean _isPresent_6 = output.isPresent();
    if (_isPresent_6) {
      JSONModelUtils.addProperty(n4jsRoot, PackageJsonProperties.OUTPUT.name, output.get());
    }
    boolean _isEmpty_4 = IterableExtensions.isEmpty(providedRL);
    boolean _not_4 = (!_isEmpty_4);
    if (_not_4) {
      JSONModelUtils.<JSONArray>addProperty(n4jsRoot, PackageJsonProperties.PROVIDED_RUNTIME_LIBRARIES.name, 
        JSONModelUtils.createStringArray(providedRL));
    }
    boolean _isEmpty_5 = IterableExtensions.isEmpty(requiredRL);
    boolean _not_5 = (!_isEmpty_5);
    if (_not_5) {
      JSONModelUtils.<JSONArray>addProperty(n4jsRoot, PackageJsonProperties.REQUIRED_RUNTIME_LIBRARIES.name, 
        JSONModelUtils.createStringArray(requiredRL));
    }
    boolean _isPresent_7 = extendedRE.isPresent();
    if (_isPresent_7) {
      JSONModelUtils.addProperty(n4jsRoot, PackageJsonProperties.EXTENDED_RUNTIME_ENVIRONMENT.name, 
        extendedRE.get());
    }
    boolean _isPresent_8 = implementationId.isPresent();
    if (_isPresent_8) {
      JSONModelUtils.addProperty(n4jsRoot, PackageJsonProperties.IMPLEMENTATION_ID.name, 
        implementationId.get());
    }
    boolean _isEmpty_6 = IterableExtensions.isEmpty(implementedProjects);
    boolean _not_6 = (!_isEmpty_6);
    if (_not_6) {
      JSONModelUtils.<JSONArray>addProperty(n4jsRoot, PackageJsonProperties.IMPLEMENTED_PROJECTS.name, 
        JSONModelUtils.createStringArray(implementedProjects));
    }
    boolean _isEmpty_7 = IterableExtensions.isEmpty(testedProjects);
    boolean _not_7 = (!_isEmpty_7);
    if (_not_7) {
      JSONModelUtils.<JSONArray>addProperty(n4jsRoot, PackageJsonProperties.TESTED_PROJECTS.name, 
        JSONModelUtils.createStringArray(testedProjects));
    }
    boolean _isEmpty_8 = n4jsRoot.getNameValuePairs().isEmpty();
    boolean _not_8 = (!_isEmpty_8);
    if (_not_8) {
      JSONModelUtils.<JSONObject>addProperty(root, PackageJsonProperties.N4JS.name, n4jsRoot);
    }
    final JSONDocument document = JSONFactory.eINSTANCE.createJSONDocument();
    document.setContent(root);
    return document;
  }
  
  private static JSONObject createDependenciesValue(final Map<String, String> dependencies) {
    final JSONObject dependenciesValue = JSONFactory.eINSTANCE.createJSONObject();
    final Function1<Map.Entry<String, String>, NameValuePair> _function = (Map.Entry<String, String> e) -> {
      final NameValuePair pair = JSONFactory.eINSTANCE.createNameValuePair();
      pair.setName(e.getKey());
      pair.setValue(JSONModelUtils.createStringLiteral(e.getValue()));
      return pair;
    };
    Iterables.<NameValuePair>addAll(dependenciesValue.getNameValuePairs(), IterableExtensions.<Map.Entry<String, String>, NameValuePair>map(dependencies.entrySet(), _function));
    return dependenciesValue;
  }
  
  /**
   * Returns the string representation of the given {@link ProjectType}
   */
  private static String getEnumAsString(final ProjectType projectType) {
    boolean _equals = Objects.equal(projectType, ProjectType.RUNTIME_ENVIRONMENT);
    if (_equals) {
      return "runtimeEnvironment";
    }
    boolean _equals_1 = Objects.equal(projectType, ProjectType.RUNTIME_LIBRARY);
    if (_equals_1) {
      return "runtimeLibrary";
    }
    return projectType.getName().toLowerCase();
  }
}
