/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.scoping.members;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.MemberAccess;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.scoping.accessModifiers.MemberVisibilityChecker;
import org.eclipse.n4js.scoping.accessModifiers.StaticWriteAccessFilterScope;
import org.eclipse.n4js.scoping.accessModifiers.VisibilityAwareMemberScope;
import org.eclipse.n4js.scoping.members.IntersectionMemberScope;
import org.eclipse.n4js.scoping.members.MemberScope;
import org.eclipse.n4js.scoping.members.MemberScopeRequest;
import org.eclipse.n4js.scoping.members.TypingStrategyAwareMemberScope;
import org.eclipse.n4js.scoping.members.UnionMemberScope;
import org.eclipse.n4js.scoping.utils.CompositeScope;
import org.eclipse.n4js.scoping.utils.DynamicPseudoScope;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.types.AnyType;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TStructuralType;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.UndefinedType;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.xtext.scoping.FilterWithErrorMarkerScope;
import org.eclipse.n4js.xtext.scoping.IEObjectDescriptionWithError;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.Scopes;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class MemberScopingHelper {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private MemberScope.MemberScopeFactory memberScopeFactory;
  
  @Inject
  private MemberVisibilityChecker memberVisibilityChecker;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * Create a new member scope that filters using the given criteria (visibility, static access). Members retrieved
   * via the scope returned by this method are guaranteed to be contained in a resource.
   * <p>
   * When choosing static scope, the {@code context} is inspected to determine read/write access
   * but only if it's a {@link ParameterizedPropertyAccessExpression} or a {@code IndexedAccessExpression}.
   * 
   * @param receiverTypeRef
   *               TypeRef for the value whose scope is of interest.
   * @param context
   *               AST node used for (a) obtaining context resource, (b) visibility checking, and
   *               (c) caching composed members.
   * @param checkVisibility
   *               if true, the member scope will be wrapped in a {@link VisibilityAwareMemberScope}; if
   *               false, method {@link getPropertyTypeForNode(IScope,String)} will <b>never</b> return
   *               {@link #INVISIBLE_MEMBER}.
   * @param staticAccess
   *               true: only static members are relevant; false: only non-static ones.
   * @param structFieldInitMode
   *               see {@link AbstractMemberScope#structFieldInitMode}.
   */
  public IScope createMemberScope(final TypeRef receiverTypeRef, final MemberAccess context, final boolean checkVisibility, final boolean staticAccess, final boolean structFieldInitMode) {
    final boolean isDynamicType = receiverTypeRef.isDynamic();
    MemberScopeRequest _memberScopeRequest = new MemberScopeRequest(receiverTypeRef, context, true, checkVisibility, staticAccess, structFieldInitMode, isDynamicType);
    return this.decoratedMemberScopeFor(receiverTypeRef, _memberScopeRequest);
  }
  
  /**
   * Same as {@link #createMemberScope(TypeRef, MemberAccess, boolean, boolean)}, but the returned scope <b><u>DOES
   * NOT</u></b> guarantee that members will be contained in a resource (in particular, composed members will not be
   * contained in a resource). In turn, this method does not require a context of type {@link MemberAccess}.
   * <p>
   * This method can be used if members are only used temporarily for a purpose that does not require proper
   * containment of the member, e.g. retrieving the type of a field for validation purposes. There are two reasons
   * for using this method:
   * <ol>
   * <li>client code is unable to provide a context of type {@link MemberAccess},
   * <li>client code wants to invoke {@link IScope#getAllElements()} or similar methods on the returned scope and
   *     wants to avoid unnecessary caching of all those members.
   * </ol>
   */
  public IScope createMemberScopeAllowingNonContainedMembers(final TypeRef receiverTypeRef, final EObject context, final boolean checkVisibility, final boolean staticAccess, final boolean structFieldInitMode) {
    final boolean isDynamicType = receiverTypeRef.isDynamic();
    MemberScopeRequest _memberScopeRequest = new MemberScopeRequest(receiverTypeRef, context, false, checkVisibility, staticAccess, structFieldInitMode, isDynamicType);
    return this.decoratedMemberScopeFor(receiverTypeRef, _memberScopeRequest);
  }
  
  /**
   * Creates member scope via #members and decorates it via #decorate.
   */
  private IScope decoratedMemberScopeFor(final TypeRef typeRef, final MemberScopeRequest memberScopeRequest) {
    if ((typeRef == null)) {
      return IScope.NULLSCOPE;
    }
    IScope result = this.members(typeRef, memberScopeRequest);
    return result;
  }
  
  /**
   * Called only be members functions to decorate returned scope.
   */
  private IScope decorate(final IScope scope, final MemberScopeRequest memberScopeRequest, final TypeRef receiverTypeRef) {
    boolean _equals = Objects.equal(scope, IScope.NULLSCOPE);
    if (_equals) {
      return scope;
    }
    IScope decoratedScope = scope;
    if ((memberScopeRequest.checkVisibility && 
      (!FilterWithErrorMarkerScope.isDecoratedWithFilter(scope, VisibilityAwareMemberScope.class)))) {
      VisibilityAwareMemberScope _visibilityAwareMemberScope = new VisibilityAwareMemberScope(decoratedScope, this.memberVisibilityChecker, receiverTypeRef, 
        memberScopeRequest.context);
      decoratedScope = _visibilityAwareMemberScope;
    }
    if ((memberScopeRequest.staticAccess && 
      (!FilterWithErrorMarkerScope.isDecoratedWithFilter(scope, StaticWriteAccessFilterScope.class)))) {
      StaticWriteAccessFilterScope _staticWriteAccessFilterScope = new StaticWriteAccessFilterScope(decoratedScope, memberScopeRequest.context);
      decoratedScope = _staticWriteAccessFilterScope;
    }
    if ((memberScopeRequest.checkVisibility && 
      (!FilterWithErrorMarkerScope.isDecoratedWithFilter(scope, TypingStrategyAwareMemberScope.class)))) {
      TypingStrategyAwareMemberScope _typingStrategyAwareMemberScope = new TypingStrategyAwareMemberScope(decoratedScope, receiverTypeRef, 
        memberScopeRequest.context);
      decoratedScope = _typingStrategyAwareMemberScope;
    }
    return decoratedScope;
  }
  
  /**
   * For the member given by (name, staticAccess) return the erroneous descriptions from the given scope.
   * <p>
   * Precondition: {@link #isNonExistentMember} has negative answer.
   */
  public Iterable<IEObjectDescriptionWithError> getErrorsForMember(final IScope scope, final String memberName, final boolean staticAccess) {
    final Iterable<IEObjectDescription> descriptions = scope.getElements(QualifiedName.create(memberName));
    final Function1<IEObjectDescription, IEObjectDescriptionWithError> _function = (IEObjectDescription d) -> {
      return IEObjectDescriptionWithError.getDescriptionWithError(d);
    };
    final Iterable<IEObjectDescriptionWithError> errorsOrNulls = IterableExtensions.<IEObjectDescription, IEObjectDescriptionWithError>map(descriptions, _function);
    return IterableExtensions.<IEObjectDescriptionWithError>filterNull(errorsOrNulls);
  }
  
  private IScope _members(final TypeRef type, final MemberScopeRequest request) {
    return IScope.NULLSCOPE;
  }
  
  private IScope _members(final UnknownTypeRef type, final MemberScopeRequest request) {
    return new DynamicPseudoScope();
  }
  
  private IScope _members(final ParameterizedTypeRef ptr, final MemberScopeRequest request) {
    final IScope result = this.membersOfType(ptr.getDeclaredType(), request);
    if ((ptr.isDynamic() && (!(result instanceof DynamicPseudoScope)))) {
      IScope _decorate = this.decorate(result, request, ptr);
      return new DynamicPseudoScope(_decorate);
    }
    return this.decorate(result, request, ptr);
  }
  
  private IScope _members(final ParameterizedTypeRefStructural ptrs, final MemberScopeRequest request) {
    final IScope result = this.membersOfType(ptrs.getDeclaredType(), request);
    if ((ptrs.isDynamic() && (!(result instanceof DynamicPseudoScope)))) {
      IScope _decorate = this.decorate(result, request, ptrs);
      return new DynamicPseudoScope(_decorate);
    }
    boolean _isEmpty = ptrs.getStructuralMembers().isEmpty();
    if (_isEmpty) {
      return this.decorate(result, request, ptrs);
    }
    IScope _xifexpression = null;
    TStructuralType _structuralType = ptrs.getStructuralType();
    boolean _tripleNotEquals = (_structuralType != null);
    if (_tripleNotEquals) {
      _xifexpression = this.memberScopeFactory.create(result, ptrs.getStructuralType(), request.context, request.staticAccess, 
        request.structFieldInitMode, request.isDynamicType);
    } else {
      _xifexpression = this.memberScopeFactory.create(result, ptrs.getStructuralMembers(), request.context, request.staticAccess, 
        request.structFieldInitMode, request.isDynamicType);
    }
    final IScope memberScopeRaw = _xifexpression;
    return this.decorate(memberScopeRaw, request, ptrs);
  }
  
  /**
   * Note: N4JSScopeProvider already taking the upper bound before using this class (thus resolving ThisTypeRefs
   * beforehand), so we will never enter this method from there; still provided to support uses from other code.
   */
  private IScope _members(final ThisTypeRef thisTypeRef, final MemberScopeRequest request) {
    final TypeRef ub = this.ts.upperBound(RuleEnvironmentExtensions.newRuleEnvironment(request.context), thisTypeRef);
    if ((ub != null)) {
      return this.members(ub, request);
    }
    return IScope.NULLSCOPE;
  }
  
  private IScope _members(final TypeTypeRef ttr, final MemberScopeRequest request) {
    final MemberScopeRequest staticRequest = request.enforceStatic();
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(request.context);
    final Type ctrStaticType = this.tsh.getStaticType(G, ttr);
    IScope staticMembers = this.membersOfType(ctrStaticType, staticRequest);
    if ((ctrStaticType instanceof TEnum)) {
      staticMembers = this.decorate(Scopes.scopeFor(((TEnum)ctrStaticType).getLiterals(), staticMembers), request, ttr);
    }
    if ((ttr.isDynamic() && (!(staticMembers instanceof DynamicPseudoScope)))) {
      IScope _decorate = this.decorate(staticMembers, staticRequest, ttr);
      DynamicPseudoScope _dynamicPseudoScope = new DynamicPseudoScope(_decorate);
      staticMembers = _dynamicPseudoScope;
    }
    if (((ctrStaticType instanceof TEnum) && AnnotationDefinition.STRING_BASED.hasAnnotation(ctrStaticType))) {
      return this.decorate(staticMembers, staticRequest, ttr);
    }
    final MemberScopeRequest instanceRequest = request.enforceInstance();
    final BuiltInTypeScope builtInScope = BuiltInTypeScope.get(this.getResourceSet(ttr, request.context));
    TClassifier _xifexpression = null;
    boolean _isConstructorRef = ttr.isConstructorRef();
    if (_isConstructorRef) {
      _xifexpression = builtInScope.getFunctionType();
    } else {
      _xifexpression = builtInScope.getObjectType();
    }
    final TClassifier functionType = _xifexpression;
    final IScope ftypeScope = this.membersOfType(functionType, instanceRequest);
    final IScope result = CompositeScope.create(
      this.decorate(staticMembers, staticRequest, ttr), 
      this.decorate(ftypeScope, instanceRequest, ttr));
    return result;
  }
  
  private IScope _members(final UnionTypeExpression uniontypeexp, final MemberScopeRequest request) {
    boolean _activateDynamicPseudoScope = this.jsVariantHelper.activateDynamicPseudoScope(request.context);
    if (_activateDynamicPseudoScope) {
      return new DynamicPseudoScope();
    }
    final Function1<TypeRef, IScope> _function = (TypeRef elementTypeRef) -> {
      TypingStrategy _typingStrategy = elementTypeRef.getTypingStrategy();
      final boolean structFieldInitMode = Objects.equal(_typingStrategy, TypingStrategy.STRUCTURAL_FIELD_INITIALIZER);
      final IScope scope = this.members(elementTypeRef, request.setStructFieldInitMode(structFieldInitMode));
      return scope;
    };
    final List<IScope> subScopes = ListExtensions.<TypeRef, IScope>map(uniontypeexp.getTypeRefs(), _function);
    int _size = subScopes.size();
    switch (_size) {
      case 0:
        return IScope.NULLSCOPE;
      case 1:
        return subScopes.get(0);
      default:
        return new UnionMemberScope(uniontypeexp, request, subScopes, this.ts);
    }
  }
  
  private IScope _members(final IntersectionTypeExpression intersectiontypeexp, final MemberScopeRequest request) {
    boolean _isEmpty = intersectiontypeexp.getTypeRefs().isEmpty();
    if (_isEmpty) {
      return IScope.NULLSCOPE;
    }
    final Function1<TypeRef, IScope> _function = (TypeRef elementTypeRef) -> {
      TypingStrategy _typingStrategy = elementTypeRef.getTypingStrategy();
      final boolean structFieldInitMode = Objects.equal(_typingStrategy, TypingStrategy.STRUCTURAL_FIELD_INITIALIZER);
      final IScope scope = this.members(elementTypeRef, request.setStructFieldInitMode(structFieldInitMode));
      return scope;
    };
    final List<IScope> subScopes = ListExtensions.<TypeRef, IScope>map(intersectiontypeexp.getTypeRefs(), _function);
    return new IntersectionMemberScope(intersectiontypeexp, request, subScopes, this.ts);
  }
  
  private IScope _members(final FunctionTypeRef ftExpr, final MemberScopeRequest request) {
    return this.membersOfFunctionTypeRef(ftExpr, request);
  }
  
  private IScope _members(final FunctionTypeExpression ftExpr, final MemberScopeRequest request) {
    return this.membersOfFunctionTypeRef(ftExpr, request);
  }
  
  /**
   * delegated from two methods above, to avoid catch-all of ParameterizedTypeRef for FuntionTypeRefs while dispatching
   */
  private IScope membersOfFunctionTypeRef(final FunctionTypeExprOrRef ftExpr, final MemberScopeRequest request) {
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(this.getResourceSet(ftExpr, request.context));
    final TObjectPrototype fType = builtInTypeScope.getFunctionType();
    final IScope ret = this.membersOfType(fType, request);
    return this.decorate(ret, request, ftExpr);
  }
  
  private IScope _membersOfType(final Type type, final MemberScopeRequest request) {
    boolean _eIsProxy = type.eIsProxy();
    if (_eIsProxy) {
      return new DynamicPseudoScope();
    }
    boolean _activateDynamicPseudoScope = this.jsVariantHelper.activateDynamicPseudoScope(request.context);
    if (_activateDynamicPseudoScope) {
      return new DynamicPseudoScope();
    }
    return IScope.NULLSCOPE;
  }
  
  private IScope _membersOfType(final UndefinedType type, final MemberScopeRequest request) {
    boolean _activateDynamicPseudoScope = this.jsVariantHelper.activateDynamicPseudoScope(request.context);
    if (_activateDynamicPseudoScope) {
      return new DynamicPseudoScope();
    }
    return IScope.NULLSCOPE;
  }
  
  private IScope _membersOfType(final Void type, final MemberScopeRequest request) {
    return new DynamicPseudoScope();
  }
  
  /**
   * Primitive types have no members, but they can be auto-boxed to their
   * corresponding object type which then, transparently to the user, provide members.
   */
  private IScope _membersOfType(final PrimitiveType prim, final MemberScopeRequest request) {
    final TClassifier boxedType = prim.getAutoboxedType();
    IScope _xifexpression = null;
    if ((boxedType != null)) {
      _xifexpression = this.membersOfType(boxedType, request);
    } else {
      _xifexpression = IScope.NULLSCOPE;
    }
    return _xifexpression;
  }
  
  /**
   * Creates member scope with parent containing members of implicit super types.
   */
  private IScope _membersOfType(final ContainerType<?> type, final MemberScopeRequest request) {
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(this.getResourceSet(type, request.context));
    ContainerType<?> implicitSuperType = builtInTypeScope.getObjectType();
    if ((type instanceof TN4Classifier)) {
      TypingStrategy _typingStrategy = ((TN4Classifier)type).getTypingStrategy();
      boolean _tripleNotEquals = (_typingStrategy != TypingStrategy.DEFAULT);
      if (_tripleNotEquals) {
        implicitSuperType = builtInTypeScope.getObjectType();
      } else {
        if ((type instanceof TClass)) {
          if (((((TClass)type).getSuperClassRef() == null) || (!Objects.equal(((TClass)type).getSuperClassRef().getDeclaredType(), builtInTypeScope.getObjectType())))) {
            implicitSuperType = builtInTypeScope.getN4ObjectType();
          }
        } else {
          implicitSuperType = builtInTypeScope.getN4ObjectType();
        }
      }
    } else {
      if ((type instanceof TObjectPrototype)) {
        final Type rootSuperType = this.getRootSuperType(((TObjectPrototype)type));
        if ((rootSuperType instanceof PrimitiveType)) {
          final TClassifier boxedType = ((PrimitiveType)rootSuperType).getAutoboxedType();
          if (((boxedType != null) && (!request.staticAccess))) {
            implicitSuperType = boxedType;
          } else {
            implicitSuperType = ((ContainerType<?>)rootSuperType);
          }
        }
      }
    }
    IScope _xifexpression = null;
    boolean _activateDynamicPseudoScope = this.jsVariantHelper.activateDynamicPseudoScope(request.context);
    if (_activateDynamicPseudoScope) {
      DynamicPseudoScope _dynamicPseudoScope = new DynamicPseudoScope();
      _xifexpression = this.memberScopeFactory.create(_dynamicPseudoScope, implicitSuperType, request.context, 
        request.staticAccess, request.structFieldInitMode, request.isDynamicType);
    } else {
      _xifexpression = this.memberScopeFactory.create(implicitSuperType, request.context, request.staticAccess, 
        request.structFieldInitMode, request.isDynamicType);
    }
    final IScope implicitSuperTypeMemberScope = _xifexpression;
    return this.memberScopeFactory.create(implicitSuperTypeMemberScope, type, request.context, request.staticAccess, 
      request.structFieldInitMode, request.isDynamicType);
  }
  
  /**
   * Returns a scope of the literals, that is members such as name or value.
   * That is, the instance members of an enumeration. The static members are made available
   * in {@link #members(EnumTypeRef, EObject, boolean)}
   */
  private IScope _membersOfType(final TEnum enumeration, final MemberScopeRequest request) {
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(this.getResourceSet(enumeration, request.context));
    TObjectPrototype _xifexpression = null;
    boolean _isStringBasedEnumeration = TypeSystemHelper.isStringBasedEnumeration(enumeration);
    if (_isStringBasedEnumeration) {
      _xifexpression = builtInTypeScope.getN4StringBasedEnumType();
    } else {
      _xifexpression = builtInTypeScope.getN4EnumType();
    }
    final TObjectPrototype specificEnumType = _xifexpression;
    return this.membersOfType(specificEnumType, request);
  }
  
  private IScope _membersOfType(final TypeVariable typeVar, final MemberScopeRequest request) {
    final TypeRef declUB = typeVar.getDeclaredUpperBound();
    if ((declUB != null)) {
      return this.members(declUB, request);
    } else {
      final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(this.getResourceSet(typeVar, request.context));
      final AnyType anyType = builtInTypeScope.getAnyType();
      return this.membersOfType(anyType, request);
    }
  }
  
  private IScope _membersOfType(final TStructuralType structType, final MemberScopeRequest request) {
    boolean _isEmpty = structType.getOwnedMembers().isEmpty();
    if (_isEmpty) {
      return IScope.NULLSCOPE;
    }
    return this.memberScopeFactory.create(structType, request.context, request.staticAccess, request.structFieldInitMode, request.isDynamicType);
  }
  
  private ResourceSet getResourceSet(final EObject type, final EObject context) {
    ResourceSet result = EcoreUtilN4.getResourceSet(type, context);
    if ((result == null)) {
      throw new IllegalStateException("type or context must be contained in a ResourceSet");
    }
    return result;
  }
  
  private Type getRootSuperType(final TObjectPrototype type) {
    Type curr = type;
    Type next = null;
    do {
      {
        Type _xifexpression = null;
        if ((curr instanceof TObjectPrototype)) {
          ParameterizedTypeRef _superType = ((TObjectPrototype)curr).getSuperType();
          Type _declaredType = null;
          if (_superType!=null) {
            _declaredType=_superType.getDeclaredType();
          }
          _xifexpression = _declaredType;
        }
        next = _xifexpression;
        if ((next != null)) {
          curr = next;
        }
      }
    } while((next != null));
    return curr;
  }
  
  private IScope members(final TypeRef ftExpr, final MemberScopeRequest request) {
    if (ftExpr instanceof FunctionTypeRef) {
      return _members((FunctionTypeRef)ftExpr, request);
    } else if (ftExpr instanceof ParameterizedTypeRefStructural) {
      return _members((ParameterizedTypeRefStructural)ftExpr, request);
    } else if (ftExpr instanceof FunctionTypeExpression) {
      return _members((FunctionTypeExpression)ftExpr, request);
    } else if (ftExpr instanceof IntersectionTypeExpression) {
      return _members((IntersectionTypeExpression)ftExpr, request);
    } else if (ftExpr instanceof ParameterizedTypeRef) {
      return _members((ParameterizedTypeRef)ftExpr, request);
    } else if (ftExpr instanceof ThisTypeRef) {
      return _members((ThisTypeRef)ftExpr, request);
    } else if (ftExpr instanceof TypeTypeRef) {
      return _members((TypeTypeRef)ftExpr, request);
    } else if (ftExpr instanceof UnionTypeExpression) {
      return _members((UnionTypeExpression)ftExpr, request);
    } else if (ftExpr instanceof UnknownTypeRef) {
      return _members((UnknownTypeRef)ftExpr, request);
    } else if (ftExpr != null) {
      return _members(ftExpr, request);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ftExpr, request).toString());
    }
  }
  
  private IScope membersOfType(final Type prim, final MemberScopeRequest request) {
    if (prim instanceof PrimitiveType) {
      return _membersOfType((PrimitiveType)prim, request);
    } else if (prim instanceof TEnum) {
      return _membersOfType((TEnum)prim, request);
    } else if (prim instanceof TStructuralType) {
      return _membersOfType((TStructuralType)prim, request);
    } else if (prim instanceof UndefinedType) {
      return _membersOfType((UndefinedType)prim, request);
    } else if (prim instanceof ContainerType) {
      return _membersOfType((ContainerType<?>)prim, request);
    } else if (prim instanceof TypeVariable) {
      return _membersOfType((TypeVariable)prim, request);
    } else if (prim != null) {
      return _membersOfType(prim, request);
    } else if (prim == null) {
      return _membersOfType((Void)null, request);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(prim, request).toString());
    }
  }
}
