/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem.utils;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.YieldExpression;
import org.eclipse.n4js.n4idl.versioning.N4IDLVersionResolver;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ExistentialTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.StructuralTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeExtensions;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.BoundType;
import org.eclipse.n4js.typesystem.utils.DerivationComputer;
import org.eclipse.n4js.typesystem.utils.ExpectedTypeComputer;
import org.eclipse.n4js.typesystem.utils.GenericsComputer;
import org.eclipse.n4js.typesystem.utils.JoinComputer;
import org.eclipse.n4js.typesystem.utils.MeetComputer;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.SimplifyComputer;
import org.eclipse.n4js.typesystem.utils.StructuralTypingComputer;
import org.eclipse.n4js.typesystem.utils.StructuralTypingResult;
import org.eclipse.n4js.typesystem.utils.SubtypeComputer;
import org.eclipse.n4js.typesystem.utils.ThisTypeComputer;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.Log;
import org.eclipse.n4js.utils.StructuralTypesHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

/**
 * Utility methods used in the XSemantics type system. Must be injected.
 * 
 * <p>Simple implementations are directly contained here. Complex operations such as join or meet are to
 * be implemented in strategy classes. For those operations, this class acts as a facade.</p>
 * 
 * <p>EObject Reference Note: All methods prefer using {@link TypeUtils.copyIfContained(EObject)} instead
 * of {@link TypeUtils.copy(EObject)}. So, clients should follow this pattern as well.</p>
 * 
 * Bibliography:
 * <a name="Pierce02a">[Pierce02a]</a> B. C. Pierce: Types and Programming Languages. The MIT Press, 1, 2002.
 */
@Log
@Singleton
@SuppressWarnings("all")
public class TypeSystemHelper {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private N4IDLVersionResolver versionResolver;
  
  @Inject
  private DerivationComputer derivationComputer;
  
  @Inject
  private GenericsComputer genericsComputer;
  
  @Inject
  private SimplifyComputer simplifyComputer;
  
  @Inject
  private JoinComputer joinComputer;
  
  @Inject
  private MeetComputer meetComputer;
  
  @Inject
  private SubtypeComputer subtypeComputer;
  
  @Inject
  private ExpectedTypeComputer expectedTypeCompuer;
  
  @Inject
  private StructuralTypingComputer structuralTypingComputer;
  
  @Inject
  private ThisTypeComputer thisTypeComputer;
  
  @Inject
  private StructuralTypesHelper structuralTypesHelper;
  
  public StructuralTypesHelper getStructuralTypesHelper() {
    return this.structuralTypesHelper;
  }
  
  public StructuralTypingComputer getStructuralTypingComputer() {
    return this.structuralTypingComputer;
  }
  
  public FunctionTypeExpression createSubstitutionOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F) {
    return this.derivationComputer.createSubstitutionOfFunctionTypeExprOrRef(G, F);
  }
  
  public FunctionTypeExpression createUpperBoundOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F) {
    return this.derivationComputer.createUpperBoundOfFunctionTypeExprOrRef(G, F);
  }
  
  public FunctionTypeExpression createLowerBoundOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F) {
    return this.derivationComputer.createLowerBoundOfFunctionTypeExprOrRef(G, F);
  }
  
  public FunctionTypeExpression createBoundOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F, final BoundType boundType) {
    return this.derivationComputer.createBoundOfFunctionTypeExprOrRef(G, F, boundType);
  }
  
  public void addSubstitutions(final RuleEnvironment G, final TypeRef typeRef) {
    this.genericsComputer.addSubstitutions(G, typeRef);
  }
  
  public void addSubstitutions(final RuleEnvironment G, final ParameterizedCallExpression callExpr, final TypeRef targetTypeRef) {
    this.genericsComputer.addSubstitutions(G, callExpr, targetTypeRef);
  }
  
  public void addSubstitutions(final RuleEnvironment G, final ParameterizedPropertyAccessExpression accessExpr) {
    this.genericsComputer.addSubstitutions(G, accessExpr);
  }
  
  public TypeRef substTypeVariablesInStructuralMembers(final RuleEnvironment G, final StructuralTypeRef typeRef) {
    return this.genericsComputer.substTypeVariablesInStructuralMembers(G, typeRef);
  }
  
  public void storePostponedSubstitutionsIn(final RuleEnvironment G, final StructuralTypeRef typeRef) {
    this.genericsComputer.storePostponedSubstitutionsIn(G, typeRef);
  }
  
  public void restorePostponedSubstitutionsFrom(final RuleEnvironment G, final StructuralTypeRef typeRef) {
    this.genericsComputer.restorePostponedSubstitutionsFrom(G, typeRef);
  }
  
  public TypeRef createUnionType(final RuleEnvironment G, final TypeRef... elements) {
    return this.simplifyComputer.createUnionType(G, elements);
  }
  
  public TypeRef createIntersectionType(final RuleEnvironment G, final TypeRef... elements) {
    return this.simplifyComputer.createIntersectionType(G, elements);
  }
  
  public <T extends ComposedTypeRef> TypeRef simplify(final RuleEnvironment G, final T composedType) {
    return this.simplifyComputer.<T>simplify(G, composedType);
  }
  
  public List<TypeRef> getSimplifiedTypeRefs(final RuleEnvironment G, final ComposedTypeRef composedType) {
    return this.simplifyComputer.<ComposedTypeRef>getSimplifiedTypeRefs(G, composedType);
  }
  
  /**
   * Convenience method calling {@link join(RuleEnvironment, Iterable<TypeRef>)} with
   * type references inside an array.
   */
  public TypeRef join(final RuleEnvironment G, final TypeRef... typeRefs) {
    return this.joinComputer.join(G, Arrays.<TypeRef>asList(typeRefs));
  }
  
  /**
   * Returns the join, sometimes called least common super type (LCST),
   * of the given types.
   * @see JoinComputer#join(RuleEnvironment, Iterable<? extends TypeRef>)
   */
  public TypeRef join(final RuleEnvironment G, final Iterable<? extends TypeRef> typeRefsToJoin) {
    return this.joinComputer.join(G, typeRefsToJoin);
  }
  
  /**
   * Convenience method calling {@link meet(RuleEnvironment, Iterable<TypeRef>)} with
   * type references inside an array.
   */
  public TypeRef meet(final RuleEnvironment G, final TypeRef... typeRefs) {
    return this.meetComputer.meet(G, Arrays.<TypeRef>asList(typeRefs));
  }
  
  /**
   * Returns the meet (first common sub type) of the given types
   * @see  MeetComputer#meet(RuleEnvironment, Iterable<? extends TypeRef>)
   */
  public TypeRef meet(final RuleEnvironment G, final Iterable<? extends TypeRef> typeRefs) {
    return this.meetComputer.meet(G, typeRefs);
  }
  
  public boolean isSubtypeFunction(final RuleEnvironment G, final FunctionTypeExprOrRef left, final FunctionTypeExprOrRef right) {
    return this.subtypeComputer.isSubtypeFunction(G, left, right);
  }
  
  public StructuralTypingResult isStructuralSubtype(final RuleEnvironment G, final TypeRef left, final TypeRef right) {
    return this.structuralTypingComputer.isStructuralSubtype(G, left, right);
  }
  
  /**
   * @see ExpectedTypeComputer#getExpectedTypeOfReturnValueExpression(RuleEnvironment,Expression)
   */
  public TypeRef getExpectedTypeOfReturnValueExpression(final RuleEnvironment G, final Expression returnValueExpr) {
    return this.expectedTypeCompuer.getExpectedTypeOfReturnValueExpression(G, returnValueExpr);
  }
  
  /**
   * @see ExpectedTypeComputer#getExpectedTypeOfYieldValueExpression(RuleEnvironment,YieldExpression,Expression)
   */
  public TypeRef getExpectedTypeOfYieldValueExpression(final RuleEnvironment G, final YieldExpression yieldExpr, final TypeRef exprTypeRef) {
    return this.expectedTypeCompuer.getExpectedTypeOfYieldValueExpression(G, yieldExpr, exprTypeRef);
  }
  
  /**
   * @see ExpectedTypeComputer#getExpectedTypeOfYieldValueExpression(RuleEnvironment,YieldExpression,Expression)
   */
  public TypeRef getExpectedTypeOfFunctionOrFieldAccessor(final RuleEnvironment G, final FunctionOrFieldAccessor fofa) {
    return this.expectedTypeCompuer.getExpectedTypeOfFunctionOrFieldAccessor(G, fofa);
  }
  
  /**
   * @see ThisTypeComputer#getThisTypeAtLocation(RuleEnvironment,EObject)
   */
  public TypeRef getThisTypeAtLocation(final RuleEnvironment G, final EObject location) {
    return this.thisTypeComputer.getThisTypeAtLocation(G, location);
  }
  
  /**
   * see {@link N4JSTypeSystem#resolveType(RuleEnvironment,TypeArgument)}
   */
  public TypeRef resolveType(final RuleEnvironment G, final TypeArgument typeArg) {
    TypeRef _xifexpression = null;
    if ((typeArg != null)) {
      _xifexpression = this.ts.upperBound(G, typeArg);
    }
    TypeRef typeRef = _xifexpression;
    TypeRef _xifexpression_1 = null;
    if ((typeRef != null)) {
      _xifexpression_1 = TypeUtils.resolveTypeVariable(typeRef);
    }
    typeRef = _xifexpression_1;
    return typeRef;
  }
  
  public boolean allEqualType(final RuleEnvironment G, final TypeRef... typeRefs) {
    final int len = ((List<TypeRef>)Conversions.doWrapArray(typeRefs)).size();
    if ((len >= 2)) {
      final TypeRef firstRef = IterableExtensions.<TypeRef>head(((Iterable<TypeRef>)Conversions.doWrapArray(typeRefs)));
      for (int i = 1; (i < len); i++) {
        boolean _equaltypeSucceeded = this.ts.equaltypeSucceeded(G, firstRef, typeRefs[i]);
        boolean _not = (!_equaltypeSucceeded);
        if (_not) {
          return false;
        }
      }
    }
    return true;
  }
  
  public TypeRef sanitizeTypeOfVariableFieldProperty(final RuleEnvironment G, final TypeArgument typeRaw) {
    if (((typeRaw == null) || (typeRaw instanceof UnknownTypeRef))) {
      return RuleEnvironmentExtensions.anyTypeRef(G);
    }
    final TypeRef typeUB = this.ts.upperBound(G, typeRaw);
    final Type declType = typeUB.getDeclaredType();
    if ((((declType == RuleEnvironmentExtensions.undefinedType(G)) || (declType == RuleEnvironmentExtensions.nullType(G))) || (declType == RuleEnvironmentExtensions.voidType(G)))) {
      return RuleEnvironmentExtensions.anyTypeRef(G);
    }
    return typeUB;
  }
  
  public Iterator<ReturnStatement> returnStatements(final FunctionDefinition definition) {
    final Predicate<EObject> _function = (EObject it) -> {
      return (!((it instanceof Expression) || (it instanceof FunctionDefinition)));
    };
    final Function1<ReturnStatement, Boolean> _function_1 = (ReturnStatement it) -> {
      Expression _expression = it.getExpression();
      return Boolean.valueOf((_expression != null));
    };
    return IteratorExtensions.<ReturnStatement>filter(Iterators.<ReturnStatement>filter(EcoreUtilN4.getAllContentsFiltered(definition, _function), ReturnStatement.class), _function_1);
  }
  
  /**
   * If possible, a dynamic version of the given type ref is returned. If the type ref is already dynamic, it is returned.
   * This is used for making all type refs dynamic in JavaScript mode.
   */
  public TypeRef makeDynamic(final TypeRef typeRef) {
    boolean _isDynamic = typeRef.isDynamic();
    boolean _not = (!_isDynamic);
    if (_not) {
      if ((typeRef instanceof ParameterizedTypeRef)) {
        final ParameterizedTypeRef dyn = TypeUtils.<ParameterizedTypeRef>copyIfContained(((ParameterizedTypeRef)typeRef));
        dyn.setDynamic(true);
        return dyn;
      }
    }
    return typeRef;
  }
  
  /**
   * Returns the explicitly declared this type, or <code>null</code>.
   * @param type either subtype of TFunction, of FieldAccessor, or of FunctionTypeExprOrRef can have a declared this
   *             type ("@This")
   * @return declaredThisType if any, null in other cases.
   */
  public static TypeRef getDeclaredThisType(final IdentifiableElement type) {
    TypeRef _switchResult = null;
    boolean _matched = false;
    if (type instanceof TFunction) {
      _matched=true;
      _switchResult = ((TFunction)type).getDeclaredThisType();
    }
    if (!_matched) {
      if (type instanceof TGetter) {
        _matched=true;
        _switchResult = ((TGetter)type).getDeclaredThisType();
      }
    }
    if (!_matched) {
      if (type instanceof TSetter) {
        _matched=true;
        _switchResult = ((TSetter)type).getDeclaredThisType();
      }
    }
    if (!_matched) {
      if (type instanceof FunctionTypeExprOrRef) {
        _matched=true;
        _switchResult = ((FunctionTypeExprOrRef)type).getDeclaredThisType();
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  /**
   * Helper Method checking existence of "@StringBased" annotation.
   */
  public static boolean isStringBasedEnumeration(final TEnum tEnum) {
    final Function1<TAnnotation, Boolean> _function = (TAnnotation it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, AnnotationDefinition.STRING_BASED.name));
    };
    return IterableExtensions.<TAnnotation>exists(tEnum.getAnnotations(), _function);
  }
  
  /**
   * Binds and substitutes the given {@link ThisTypeRef this type reference} after wrapping the
   * given rule environment.
   * 
   * <p>
   * For instance after passing a {@code ~~this} type reference into a method in the context of container
   * {@code class A}, the type reference argument will be bound to {@code this[A]} and finally will be substituted
   * with {@code ~~this[A]} type reference. That will be the return value of the method.
   * 
   * @param G
   * 		the rule environment that will be wrapped for the operation.
   * @param location
   * 		location within the AST for which to create a BoundThisTypeRef. Same as the argument 'location' of
   * 		judgment 'thisTypeRef' in Xsemantics.
   * @param typeRef
   * 		type reference to substitute; this can either be an unbound ThisTypeRef or any other kind of TypeRef that
   * 		contains one or more unbound ThisTypeRefs. Need not be contained in the AST (as usual for type references).
   */
  public TypeRef bindAndSubstituteThisTypeRef(final RuleEnvironment G, final EObject location, final TypeRef typeRef) {
    final TypeRef boundThisTypeRef = this.getThisTypeAtLocation(G, location);
    final RuleEnvironment localG = RuleEnvironmentExtensions.wrap(G);
    RuleEnvironmentExtensions.addThisType(localG, boundThisTypeRef);
    return this.ts.substTypeVariables(localG, typeRef);
  }
  
  /**
   * Checks if a value of type <code>typeRef</code> is "callable", i.e. if it can be directly invoked using a call
   * expression.
   */
  public boolean isCallable(final RuleEnvironment G, final TypeRef typeRef) {
    boolean _isClassConstructorFunction = this.isClassConstructorFunction(G, typeRef);
    if (_isClassConstructorFunction) {
      TMethod _callableClassConstructorFunction = this.getCallableClassConstructorFunction(G, typeRef);
      boolean _tripleNotEquals = (_callableClassConstructorFunction != null);
      if (_tripleNotEquals) {
        return true;
      }
      return false;
    }
    Type _declaredType = typeRef.getDeclaredType();
    if ((_declaredType instanceof TFunction)) {
      return true;
    }
    if ((typeRef instanceof FunctionTypeExprOrRef)) {
      return true;
    }
    boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, typeRef, RuleEnvironmentExtensions.structuralFunctionTypeRef(G));
    if (_subtypeSucceeded) {
      return true;
    }
    boolean _subtypeSucceeded_1 = this.ts.subtypeSucceeded(G, typeRef, RuleEnvironmentExtensions.functionTypeRef(G));
    if (_subtypeSucceeded_1) {
      return true;
    }
    if ((typeRef.isDynamic() && this.ts.subtypeSucceeded(G, RuleEnvironmentExtensions.functionTypeRef(G), typeRef))) {
      return true;
    }
    return false;
  }
  
  /**
   * Checks if a value of type <code>typeRef</code> is a class constructor function.
   */
  public boolean isClassConstructorFunction(final RuleEnvironment G, final TypeRef typeRef) {
    final Type declaredType = typeRef.getDeclaredType();
    if ((declaredType instanceof TMethod)) {
      boolean _isConstructor = ((TMethod)declaredType).isConstructor();
      if (_isConstructor) {
        return true;
      }
    }
    if ((typeRef instanceof FunctionTypeExprOrRef)) {
      final TFunction ft = ((FunctionTypeExprOrRef)typeRef).getFunctionType();
      if ((ft instanceof TMethod)) {
        boolean _isConstructor_1 = ((TMethod)ft).isConstructor();
        if (_isConstructor_1) {
          return true;
        }
      }
    }
    if ((typeRef instanceof TypeTypeRef)) {
      final Type cls = this.getStaticType(G, ((TypeTypeRef)typeRef));
      if (((cls instanceof TClass) || (cls instanceof TObjectPrototype))) {
        return true;
      }
    }
    return false;
  }
  
  public TMethod getCallableClassConstructorFunction(final RuleEnvironment G, final TypeRef typeRef) {
    Type type = null;
    final Type declaredType = typeRef.getDeclaredType();
    if ((declaredType instanceof TMethod)) {
      boolean _isConstructor = ((TMethod)declaredType).isConstructor();
      if (_isConstructor) {
        type = ((TMethod)declaredType).getContainingType();
      }
    }
    if ((typeRef instanceof FunctionTypeExprOrRef)) {
      final TFunction ft = ((FunctionTypeExprOrRef)typeRef).getFunctionType();
      if ((ft instanceof TMethod)) {
        boolean _isConstructor_1 = ((TMethod)ft).isConstructor();
        if (_isConstructor_1) {
          type = ((TMethod)ft).getContainingType();
        }
      }
    }
    if ((typeRef instanceof TypeTypeRef)) {
      final Type cls = this.getStaticType(G, ((TypeTypeRef)typeRef));
      if (((cls instanceof TClass) || (cls instanceof TObjectPrototype))) {
        type = cls;
      }
    }
    if ((type instanceof ContainerType<?>)) {
      return ((ContainerType<?>)type).getCallableCtor();
    }
    return null;
  }
  
  /**
   * Returns the so-called "static type" of the given {@link TypeTypeRef} or <code>null</code> if not
   * available.
   * <p>
   * Formerly, this was a utility operation in {@code TypeRefs.xcore} but since the introduction of wildcards in
   * {@code TypeTypeRef}s the 'upperBound' judgment (and thus a RuleEnvironment) is required to compute this
   * and hence it was moved here.
   */
  public Type getStaticType(final RuleEnvironment G, final TypeTypeRef ctorTypeRef) {
    TypeRef _staticTypeRef = this.getStaticTypeRef(G, ctorTypeRef);
    Type _declaredType = null;
    if (_staticTypeRef!=null) {
      _declaredType=_staticTypeRef.getDeclaredType();
    }
    return _declaredType;
  }
  
  public TypeRef getStaticTypeRef(final RuleEnvironment G, final TypeTypeRef ctorTypeRef) {
    TypeArgument typeArg = ctorTypeRef.getTypeArg();
    while ((((typeArg instanceof Wildcard) || (typeArg instanceof ExistentialTypeRef)) || (typeArg instanceof BoundThisTypeRef))) {
      typeArg = this.ts.upperBound(G, typeArg);
    }
    return ((TypeRef) typeArg);
  }
  
  /**
   * Creates a parameterized type ref to the wrapped static type of a TypeTypeRef, configured with the given
   * TypeArguments. Returns UnknownTypeRef if the static type could not be retrieved (e.g. unbound This-Type).
   */
  public TypeRef createTypeRefFromStaticType(final RuleEnvironment G, final TypeTypeRef ctr, final TypeArgument... typeArgs) {
    final TypeRef typeRef = this.getStaticTypeRef(G, ctr);
    final Type type = typeRef.getDeclaredType();
    if ((type != null)) {
      TypeRef resultTypeRef = TypeExtensions.ref(type, typeArgs);
      return this.versionResolver.<TypeRef, TypeRef>resolveVersion(resultTypeRef, typeRef);
    }
    return TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
  }
  
  /**
   * This method computes the set of all subtypes in the set of TypeRefs.
   * It does not copy the TypeRefs!
   */
  public List<TypeRef> getSubtypesOnly(final RuleEnvironment G, final TypeRef... typeRefs) {
    final LinkedList<TypeRef> intersectTRs = new LinkedList<TypeRef>();
    for (final TypeRef s : typeRefs) {
      final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
        return Boolean.valueOf(this.ts.subtypeSucceeded(G, it, s));
      };
      boolean _exists = IterableExtensions.<TypeRef>exists(intersectTRs, _function);
      boolean _not = (!_exists);
      if (_not) {
        final Predicate<TypeRef> _function_1 = (TypeRef it) -> {
          return this.ts.subtypeSucceeded(G, s, it);
        };
        Iterables.<TypeRef>removeIf(intersectTRs, _function_1);
        intersectTRs.add(s);
      }
    }
    return intersectTRs;
  }
  
  /**
   * This method computes the set of all super types in the set of TypeRefs.
   * It does not copy the TypeRefs!
   */
  public List<TypeRef> getSuperTypesOnly(final RuleEnvironment G, final TypeRef... typeRefs) {
    final LinkedList<TypeRef> unionTRs = new LinkedList<TypeRef>();
    for (final TypeRef s : typeRefs) {
      final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
        return Boolean.valueOf(this.ts.subtypeSucceeded(G, s, it));
      };
      boolean _exists = IterableExtensions.<TypeRef>exists(unionTRs, _function);
      boolean _not = (!_exists);
      if (_not) {
        final Predicate<TypeRef> _function_1 = (TypeRef it) -> {
          return this.ts.subtypeSucceeded(G, s, it);
        };
        Iterables.<TypeRef>removeIf(unionTRs, _function_1);
        unionTRs.add(s);
      }
    }
    return unionTRs;
  }
  
  /**
   * From any expression within a generator function or method, the type TNext is returned (referring to the
   * actual (outer) return type, which is {@code Generator<TYield,TReturn,TNext>}).
   */
  public TypeRef getActualGeneratorReturnType(final RuleEnvironment G, final Expression expr) {
    EObject _eContainer = null;
    if (expr!=null) {
      _eContainer=expr.eContainer();
    }
    final FunctionDefinition funDef = EcoreUtil2.<FunctionDefinition>getContainerOfType(_eContainer, FunctionDefinition.class);
    final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
    final TypeRef myThisTypeRef = this.getThisTypeAtLocation(G, expr);
    RuleEnvironmentExtensions.addThisType(G2, myThisTypeRef);
    if (((funDef == null) || (!funDef.isGenerator()))) {
      return null;
    }
    final Type tFun = funDef.getDefinedType();
    if ((tFun instanceof TFunction)) {
      final TypeRef actualReturnTypeRef = ((TFunction)tFun).getReturnTypeRef();
      final BuiltInTypeScope scope = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope;
      boolean _isGenerator = TypeUtils.isGenerator(actualReturnTypeRef, scope);
      if (_isGenerator) {
        return actualReturnTypeRef;
      }
    }
    return TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
  }
  
  /**
   * Given a {@link TypeRef} to a {@code Generator<TYield,TReturn,TNext>} class, this method returns TYield, if existent.
   */
  public TypeRef getGeneratorTYield(final RuleEnvironment G, final TypeRef generatorTypeRef) {
    TypeRef yieldTypeRef = null;
    int _length = ((Object[])Conversions.unwrapArray(generatorTypeRef.getTypeArgs(), Object.class)).length;
    boolean _tripleEquals = (_length == 3);
    if (_tripleEquals) {
      final TypeArgument yieldTypeArg = generatorTypeRef.getTypeArgs().get(0);
      if ((yieldTypeArg != null)) {
        yieldTypeRef = this.ts.upperBound(G, yieldTypeArg);
      }
    }
    return yieldTypeRef;
  }
  
  /**
   * Given a {@link TypeRef} to a {@code Generator<TYield,TReturn,TNext>} class, this method returns TReturn, if existent.
   */
  public TypeRef getGeneratorTReturn(final RuleEnvironment G, final TypeRef generatorTypeRef) {
    TypeRef returnTypeRef = null;
    int _length = ((Object[])Conversions.unwrapArray(generatorTypeRef.getTypeArgs(), Object.class)).length;
    boolean _tripleEquals = (_length == 3);
    if (_tripleEquals) {
      final TypeArgument returnTypeArg = generatorTypeRef.getTypeArgs().get(1);
      if ((returnTypeArg != null)) {
        returnTypeRef = this.ts.upperBound(G, returnTypeArg);
      }
    }
    return returnTypeRef;
  }
  
  /**
   * Given a {@link TypeRef} to a {@code Generator<TYield,TReturn,TNext>} class, this method returns TNext, if existent.
   */
  public TypeRef getGeneratorTNext(final RuleEnvironment G, final TypeRef generatorTypeRef) {
    TypeRef nextTypeRef = null;
    int _length = ((Object[])Conversions.unwrapArray(generatorTypeRef.getTypeArgs(), Object.class)).length;
    boolean _tripleEquals = (_length == 3);
    if (_tripleEquals) {
      final TypeArgument nextTypeArg = generatorTypeRef.getTypeArgs().get(2);
      if ((nextTypeArg != null)) {
        nextTypeRef = this.ts.upperBound(G, nextTypeArg);
      }
    }
    return nextTypeRef;
  }
  
  /**
   * Given a {@link TypeRef} to an {@code Iterable<T>}, this method returns T, if existent.
   */
  public TypeRef getIterableTypeArg(final RuleEnvironment G, final TypeRef iterableTypeRef) {
    TypeRef typeRef = null;
    int _length = ((Object[])Conversions.unwrapArray(iterableTypeRef.getTypeArgs(), Object.class)).length;
    boolean _tripleEquals = (_length == 1);
    if (_tripleEquals) {
      final TypeArgument nextTypeArg = iterableTypeRef.getTypeArgs().get(0);
      if ((nextTypeArg != null)) {
        typeRef = this.ts.upperBound(G, nextTypeArg);
      }
    }
    return typeRef;
  }
  
  private static final Logger logger = Logger.getLogger(TypeSystemHelper.class);
}
