/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.utils.MembersByNameTypeAndAccessComparator;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.StructuralMembersPredicates;
import org.eclipse.n4js.utils.StructuralMembersTriple;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * An iterator over pairs of corresponding members from two structural types. Collecting the members is not
 * part of this class, see {@link StructuralTypesHelper} for this.
 */
@SuppressWarnings("all")
public class StructuralMembersTripleIterator implements Iterator<StructuralMembersTriple> {
  private static final MembersByNameTypeAndAccessComparator MEMBERS_BY_NameTypeAndAccess = new MembersByNameTypeAndAccessComparator(false);
  
  private static final MembersByNameTypeAndAccessComparator MEMBERS_BY_NameAndAccess = new MembersByNameTypeAndAccessComparator(true);
  
  private final TMember[] membersLeft;
  
  private final TMember[] membersRight;
  
  private final TypingStrategy leftStrategy;
  
  private final TypingStrategy rightStrategy;
  
  private int leftIndex = 0;
  
  private int rightIndex = 0;
  
  private Optional<StructuralMembersTriple> nextTriple = ((Optional<StructuralMembersTriple>) null);
  
  private StructuralMembersTripleIterator(final TMember[] membersLeft, final TMember[] membersRight, final TypingStrategy leftStrategy, final TypingStrategy rightStrategy) {
    this.membersLeft = membersLeft;
    this.membersRight = membersRight;
    this.leftStrategy = leftStrategy;
    this.rightStrategy = rightStrategy;
    this.computeNext();
  }
  
  /**
   * Creates a new iterator from two <em>unprepared</em> member lists, i.e. this method will remove duplicate
   * members and sort the members. If this has already been done, use method {@link #ofPrepared(TMember[],TMember[])}
   * instead.
   */
  public static StructuralMembersTripleIterator ofUnprepared(final Iterable<TMember> membersLeft, final Iterable<TMember> membersRight, final TypingStrategy leftStrategy, final TypingStrategy rightStrategy) {
    return StructuralMembersTripleIterator.ofPrepared(StructuralMembersTripleIterator.toSortedArray(membersLeft), StructuralMembersTripleIterator.toSortedArray(membersRight), leftStrategy, rightStrategy);
  }
  
  /**
   * Creates a new iterator from two <em>prepared</em> member lists, i.e. members must be sorted and must not
   * contain duplicates. <b>No copy of the given arrays will be created!</b>.
   */
  public static StructuralMembersTripleIterator ofPrepared(final TMember[] membersLeft, final TMember[] membersRight, final TypingStrategy leftStrategy, final TypingStrategy rightStrategy) {
    return new StructuralMembersTripleIterator(membersLeft, membersRight, leftStrategy, rightStrategy);
  }
  
  @Override
  public boolean hasNext() {
    return this.nextTriple.isPresent();
  }
  
  @Override
  public StructuralMembersTriple next() {
    boolean _isPresent = this.nextTriple.isPresent();
    boolean _not = (!_isPresent);
    if (_not) {
      throw new NoSuchElementException();
    }
    final StructuralMembersTriple currentTriple = this.nextTriple.get();
    this.computeNext();
    return currentTriple;
  }
  
  private void computeNext() {
    int _size = ((List<TMember>)Conversions.doWrapArray(this.membersRight)).size();
    boolean _lessThan = (this.rightIndex < _size);
    if (_lessThan) {
      TMember rightMember = this.membersRight[this.rightIndex];
      TMember leftMember = null;
      int compare = 0;
      do {
        {
          TMember _xifexpression = null;
          int _size_1 = ((List<TMember>)Conversions.doWrapArray(this.membersLeft)).size();
          boolean _lessThan_1 = (this.leftIndex < _size_1);
          if (_lessThan_1) {
            _xifexpression = this.membersLeft[this.leftIndex];
          } else {
            _xifexpression = null;
          }
          leftMember = _xifexpression;
          compare = StructuralMembersTripleIterator.MEMBERS_BY_NameAndAccess.compare(leftMember, rightMember);
          if (((compare == 0) && (!this.compatibleMemberTypes(leftMember, rightMember)))) {
            compare = StructuralMembersTripleIterator.MEMBERS_BY_NameTypeAndAccess.compare(leftMember, rightMember);
          }
          if ((compare < 0)) {
            this.leftIndex++;
          } else {
            if ((compare > 0)) {
              leftMember = null;
            } else {
            }
          }
        }
      } while((compare < 0));
      FieldAccessor leftOtherAccessor = ((FieldAccessor) null);
      if (((rightMember instanceof TField) && (leftMember instanceof FieldAccessor))) {
        TMember _xifexpression = null;
        int _size_1 = ((List<TMember>)Conversions.doWrapArray(this.membersLeft)).size();
        boolean _lessThan_1 = ((this.leftIndex + 1) < _size_1);
        if (_lessThan_1) {
          _xifexpression = this.membersLeft[(this.leftIndex + 1)];
        } else {
          _xifexpression = null;
        }
        final TMember leftMemberNext = _xifexpression;
        if (((leftMemberNext instanceof FieldAccessor) && Objects.equal(leftMemberNext.getName(), rightMember.getName()))) {
          leftOtherAccessor = ((FieldAccessor) leftMemberNext);
          this.leftIndex++;
        }
      }
      StructuralMembersTriple _structuralMembersTriple = new StructuralMembersTriple(leftMember, rightMember, leftOtherAccessor);
      this.nextTriple = Optional.<StructuralMembersTriple>of(_structuralMembersTriple);
      this.rightIndex++;
    } else {
      this.nextTriple = Optional.<StructuralMembersTriple>absent();
    }
  }
  
  /**
   * Returns true iff member 'left' can fulfill the structural requirement represented by member 'right'.
   * This method takes into account <b>only</b> the member kind (i.e. field, getter, etc.), not things like
   * return types, parameter types (in case of methods), etc.
   */
  private boolean compatibleMemberTypes(final TMember left, final TMember right) {
    if (((((TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == this.rightStrategy) && (TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS == this.leftStrategy)) && (left instanceof FieldAccessor)) && N4JSLanguageUtils.isWriteableField(right))) {
      return false;
    }
    return (((((((((((left instanceof TField) && (right instanceof TField)) || ((left instanceof TGetter) && (right instanceof TGetter))) || ((left instanceof TSetter) && (right instanceof TSetter))) || ((left instanceof TMethod) && (right instanceof TMethod))) || ((left instanceof FieldAccessor) && N4JSLanguageUtils.isWriteableField(right))) || (N4JSLanguageUtils.isWriteableField(left) && (right instanceof FieldAccessor))) || ((left instanceof TGetter) && N4JSLanguageUtils.isReadOnlyField(right))) || (N4JSLanguageUtils.isReadOnlyField(left) && (right instanceof TGetter))) || ((left instanceof TStructField) && (right instanceof TMethod))) || ((((TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == this.rightStrategy) && (TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS != this.leftStrategy)) && (StructuralMembersPredicates.WRITABLE_FIELDS_PREDICATE.apply(right)).booleanValue()) && (StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(left)).booleanValue()));
  }
  
  /**
   * Made public only to allow access from tests. Don't use from outside.
   */
  public static TMember[] toSortedArray(final Iterable<TMember> members) {
    return ((TMember[])Conversions.unwrapArray(IterableExtensions.<TMember>sortWith(IterableExtensions.<TMember>toSet(members), StructuralMembersTripleIterator.MEMBERS_BY_NameTypeAndAccess), TMember.class));
  }
}
