/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import javax.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.utils.Result;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@Singleton
@SuppressWarnings("all")
public class ValidatorMessageHelper {
  @Inject
  @Extension
  protected N4JSElementKeywordProvider _n4JSElementKeywordProvider;
  
  /**
   * Returns type of member and short qualified name, and, if this is similar to another member, including the line number.
   */
  public String descriptionDifferentFrom(final TMember member, final TMember differentMembers) {
    return this.descriptionDifferentFrom(member, Collections.<TMember>unmodifiableList(CollectionLiterals.<TMember>newArrayList(differentMembers)));
  }
  
  /**
   * Returns type of member and short qualified name, and, if this is similar one of the other members, including the line number.
   */
  public String descriptionDifferentFrom(final TMember member, final Iterable<TMember> otherMembers) {
    final Function1<TMember, String> _function = (TMember it) -> {
      return it.getContainingType().getName();
    };
    boolean _contains = IterableExtensions.<String>toList(IterableExtensions.<TMember, String>map(otherMembers, _function)).contains(member.getContainingType().getName());
    if (_contains) {
      Object _eGet = member.eGet(TypesPackage.Literals.SYNTAX_RELATED_TELEMENT__AST_ELEMENT, false);
      final EObject ast = ((EObject) _eGet);
      if (((null != ast) && (!ast.eIsProxy()))) {
        final ICompositeNode node = NodeModelUtils.getNode(ast);
        if ((node != null)) {
          final int memberLine = node.getStartLine();
          StringConcatenation _builder = new StringConcatenation();
          String _description = this.description(member);
          _builder.append(_description);
          _builder.append(" (line ");
          _builder.append(memberLine);
          _builder.append(")");
          return _builder.toString();
        }
      }
    }
    return this.description(member);
  }
  
  /**
   * Returns "static " if member is static, "const " for const fields and an empty strign otherwise;
   * used in description methods.
   */
  private String staticOrConstKeyword(final TMember member) {
    if (((member != null) && member.isStatic())) {
      if ((member instanceof TField)) {
        boolean _isConst = ((TField)member).isConst();
        if (_isConst) {
          return "const ";
        }
      }
      return "static ";
    } else {
      return "";
    }
  }
  
  /**
   * Returns "static " or "const "-prefixed keyword if member is static, used in description methods.
   */
  private String prefixedKeyword(final TMember member) {
    StringConcatenation _builder = new StringConcatenation();
    String _staticOrConstKeyword = this.staticOrConstKeyword(member);
    _builder.append(_staticOrConstKeyword);
    String _keyword = this._n4JSElementKeywordProvider.keyword(member);
    _builder.append(_keyword);
    return _builder.toString();
  }
  
  /**
   * Returns with the keyword and the name of the function definition.
   */
  public String _description(final FunctionDefinition definition) {
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(definition);
    _builder.append(_keyword);
    _builder.append(" ");
    String _name = definition.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  /**
   * Returns keyword and name of a named element; this method is seldom called, usually more concrete methods are used instead.
   */
  public String _description(final NamedElement namedElement) {
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(namedElement);
    _builder.append(_keyword);
    _builder.append(" ");
    String _name = namedElement.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  /**
   * Returns keyword and name of an identifiable element; this method is seldom called, usually more concrete methods are used instead.
   */
  public String _description(final IdentifiableElement identifiableElement) {
    if ((identifiableElement == null)) {
      return "unknown ";
    }
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(identifiableElement);
    _builder.append(_keyword);
    _builder.append(" ");
    String _name = identifiableElement.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  /**
   * Returns type of member and short qualified name.
   */
  public String _description(final TMember member) {
    boolean _isConstructor = member.isConstructor();
    if (_isConstructor) {
      final ContainerType<?> container = member.getContainingType();
      String _xifexpression = null;
      if ((container instanceof TInterface)) {
        _xifexpression = "in";
      } else {
        _xifexpression = "of";
      }
      final String preposition = _xifexpression;
      StringConcatenation _builder = new StringConcatenation();
      String _keyword = this._n4JSElementKeywordProvider.keyword(member);
      _builder.append(_keyword);
      _builder.append(" ");
      _builder.append(preposition);
      _builder.append(" ");
      String _description = null;
      if (container!=null) {
        _description=this.description(container);
      }
      _builder.append(_description);
      return _builder.toString();
    }
    StringConcatenation _builder_1 = new StringConcatenation();
    String _prefixedKeyword = this.prefixedKeyword(member);
    _builder_1.append(_prefixedKeyword);
    _builder_1.append(" ");
    String _shortQualifiedName = this.shortQualifiedName(member);
    _builder_1.append(_shortQualifiedName);
    return _builder_1.toString();
  }
  
  /**
   * Returns type of classifier and short qualified name.
   */
  public String _description(final TClassifier classifier) {
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(classifier);
    _builder.append(_keyword);
    _builder.append(" ");
    String _name = classifier.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  public String _description(final IdentifierRef identifierRef) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("identifier ");
    String _idAsText = identifierRef.getIdAsText();
    _builder.append(_idAsText);
    return _builder.toString();
  }
  
  public String _description(final EObject eObject) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<unnamed>");
    return _builder.toString();
  }
  
  /**
   * Returns type of member and simple name.
   */
  public String shortDescription(final TMember member) {
    StringConcatenation _builder = new StringConcatenation();
    String _prefixedKeyword = this.prefixedKeyword(member);
    _builder.append(_prefixedKeyword);
    _builder.append(" ");
    String _name = member.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  /**
   * Returns type of member and simple name.
   */
  public String shortDescription(final TFunction func) {
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(func);
    _builder.append(_keyword);
    _builder.append(" ");
    String _name = func.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  /**
   * Returns type of member and simple name.
   */
  public String shortDescription(final TVariable tvar) {
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(tvar);
    _builder.append(_keyword);
    _builder.append(" ");
    String _name = tvar.getName();
    _builder.append(_name);
    return _builder.toString();
  }
  
  /**
   * Returns descriptions, comma separated, last one added with "and".
   */
  public String descriptions(final Iterable<? extends TMember> members) {
    final TMember last = IterableExtensions.last(members);
    if ((last == null)) {
      return "";
    }
    TMember _head = IterableExtensions.head(members);
    boolean _tripleEquals = (last == _head);
    if (_tripleEquals) {
      return this.description(last);
    }
    final Function1<TMember, Boolean> _function = (TMember it) -> {
      return Boolean.valueOf((!Objects.equal(it, last)));
    };
    final Function1<TMember, CharSequence> _function_1 = (TMember it) -> {
      return this.description(it);
    };
    String _join = IterableExtensions.join(IterableExtensions.filter(members, _function), "", ", ", "", _function_1);
    String _plus = (_join + " and ");
    String _description = this.description(last);
    return (_plus + _description);
  }
  
  public String shortQualifiedName(final TMember member) {
    String _name = member.getContainingType().getName();
    String _plus = (_name + ".");
    String _name_1 = member.getName();
    return (_plus + _name_1);
  }
  
  public String shortQualifiedName(final TFunction func) {
    return func.getName();
  }
  
  public String shortQualifiedName(final TVariable tvar) {
    return tvar.getName();
  }
  
  /**
   * Returns description of "interface" and short qualified name for all interfaces.
   */
  public String names(final Iterable<? extends IdentifiableElement> namedElements) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<IdentifiableElement, Boolean> _function = (IdentifiableElement it) -> {
        IdentifiableElement _last = IterableExtensions.last(namedElements);
        return Boolean.valueOf((it != _last));
      };
      Iterable<? extends IdentifiableElement> _filter = IterableExtensions.filter(namedElements, _function);
      boolean _hasElements = false;
      for(final IdentifiableElement i : _filter) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(",", "");
        }
        String _name = i.getName();
        _builder.append(_name);
      }
      if (_hasElements) {
        _builder.append(" and ");
      }
    }
    String _name_1 = IterableExtensions.last(namedElements).getName();
    _builder.append(_name_1);
    return _builder.toString();
  }
  
  /**
   * Returns type of member and name of the member with line number in brackets.
   */
  public String descriptionWithLine(final TMember member) {
    String _xblockexpression = null;
    {
      final String memberLine = this.getMemberLine(member.getAstElement());
      StringConcatenation _builder = new StringConcatenation();
      String _prefixedKeyword = this.prefixedKeyword(member);
      _builder.append(_prefixedKeyword);
      _builder.append(" ");
      String _name = member.getName();
      _builder.append(_name);
      _builder.append(memberLine);
      _xblockexpression = _builder.toString();
    }
    return _xblockexpression;
  }
  
  private String getMemberLine(final EObject element) {
    if ((element != null)) {
      final ICompositeNode node = NodeModelUtils.getNode(element);
      if ((node != null)) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append(" ");
        _builder.append("(line ");
        int _startLine = node.getStartLine();
        _builder.append(_startLine, " ");
        _builder.append(")");
        return _builder.toString();
      }
    }
    return "";
  }
  
  /**
   * Returns type of ast element and line number in brackets.
   */
  public String descriptionWithLine(final EObject eAST) {
    final String memberLine = this.getMemberLine(eAST);
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(eAST);
    _builder.append(_keyword);
    _builder.append(memberLine);
    final String result = _builder.toString();
    return result.trim();
  }
  
  /**
   * Returns type and name of ast element and line number in brackets.
   */
  public String descriptionWithLine(final EObject eAST, final String name) {
    String _xblockexpression = null;
    {
      final String memberLine = this.getMemberLine(eAST);
      StringConcatenation _builder = new StringConcatenation();
      String _keyword = this._n4JSElementKeywordProvider.keyword(eAST);
      _builder.append(_keyword);
      _builder.append(" ");
      _builder.append(name);
      _builder.append(memberLine);
      _xblockexpression = _builder.toString();
    }
    return _xblockexpression;
  }
  
  /**
   * Returns type and name of the EObject.
   */
  public String description(final EObject eAST, final String name) {
    StringConcatenation _builder = new StringConcatenation();
    String _keyword = this._n4JSElementKeywordProvider.keyword(eAST);
    _builder.append(_keyword);
    _builder.append(" ");
    _builder.append(name);
    return _builder.toString();
  }
  
  /**
   * Returns verb of redefinition, that is either
   * "implemented", "consumed", or "overridden", or "inherited".
   */
  public String redefinitionString(final TClassifier currentClassifier, final TMember overriding, final TMember overridden) {
    final ContainerType<?> overriddenContainer = overridden.getContainingType();
    boolean _isAbstract = overridden.isAbstract();
    if (_isAbstract) {
      return "implemented";
    }
    if ((overriddenContainer instanceof TInterface)) {
      return "implemented";
    }
    ContainerType<?> _containingType = overriding.getContainingType();
    boolean _tripleEquals = (_containingType == currentClassifier);
    if (_tripleEquals) {
      return "overridden";
    }
    return "inherited";
  }
  
  /**
   * Returns verb of inheritance, that is either
   * "consumed", or "inherited".
   */
  public String inheritanceString(final TClassifier currentClassifier, final TMember inheritedMember) {
    final ContainerType<?> container = inheritedMember.getContainingType();
    if ((container instanceof TInterface)) {
      return "consumed";
    }
    return "inherited";
  }
  
  /**
   * Returns message provided by type system, or empty string if no message is provided.
   * The returned message is trimmed, that is, preceeding "failed: " is removed and
   * whitespaces are trimmed.
   */
  public String trimTypesystemMessage(final Result tsresult) {
    String _failureMessage = null;
    if (tsresult!=null) {
      _failureMessage=tsresult.getFailureMessage();
    }
    final String msg = _failureMessage;
    if ((msg == null)) {
      return "";
    }
    boolean _startsWith = msg.startsWith("failed:");
    if (_startsWith) {
      return msg.substring("failed:".length()).trim();
    }
    return msg.trim();
  }
  
  public String fullFunctionSignature(final TFunction tfunction) {
    final StringBuilder strb = new StringBuilder();
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<TAnnotation, Boolean> _function = (TAnnotation it) -> {
        String _name = it.getName();
        return Boolean.valueOf((!Objects.equal(_name, AnnotationDefinition.INTERNAL.name)));
      };
      Iterable<TAnnotation> _filter = IterableExtensions.<TAnnotation>filter(tfunction.getAnnotations(), _function);
      for(final TAnnotation a : _filter) {
        String _annotationAsString = a.getAnnotationAsString();
        _builder.append(_annotationAsString);
        _builder.append(" ");
      }
    }
    strb.append(_builder);
    if ((tfunction instanceof TMethod)) {
      MemberAccessModifier _memberAccessModifier = ((TMethod)tfunction).getMemberAccessModifier();
      boolean _tripleNotEquals = (_memberAccessModifier != null);
      if (_tripleNotEquals) {
        strb.append(this._n4JSElementKeywordProvider.keyword(((TMethod)tfunction).getMemberAccessModifier()));
        strb.append(" ");
      }
      boolean _isAbstract = ((TMethod)tfunction).isAbstract();
      if (_isAbstract) {
        strb.append("abstract ");
      }
    }
    if ((!(tfunction instanceof TMethod))) {
      boolean _isAsyncOrPromise = this.isAsyncOrPromise(tfunction);
      if (_isAsyncOrPromise) {
        strb.append("async ");
      }
      strb.append("function ");
    }
    boolean _isGenerator = this.isGenerator(tfunction);
    if (_isGenerator) {
      strb.append("* ");
    }
    boolean _isGeneric = tfunction.isGeneric();
    if (_isGeneric) {
      final Function1<TypeVariable, String> _function_1 = (TypeVariable it) -> {
        return it.getTypeAsString();
      };
      String _join = IterableExtensions.join(ListExtensions.<TypeVariable, String>map(tfunction.getTypeVars(), _function_1), ",");
      String _plus = ("<" + _join);
      String _plus_1 = (_plus + "> ");
      strb.append(_plus_1);
    }
    if ((tfunction instanceof TMethod)) {
      boolean _isAsyncOrPromise_1 = this.isAsyncOrPromise(tfunction);
      if (_isAsyncOrPromise_1) {
        strb.append("async ");
      }
    }
    String _name = tfunction.getName();
    String _plus_2 = (_name + "(");
    final Function1<TFormalParameter, String> _function_2 = (TFormalParameter it) -> {
      return it.getFormalParameterAsString();
    };
    String _join_1 = IterableExtensions.join(ListExtensions.<TFormalParameter, String>map(tfunction.getFpars(), _function_2), ", ");
    String _plus_3 = (_plus_2 + _join_1);
    String _plus_4 = (_plus_3 + ")");
    strb.append(_plus_4);
    strb.append(": ");
    this.appendPromisedReturnType(strb, tfunction);
    return strb.toString();
  }
  
  /**
   * Returns nicely worded enumeration of the given items.
   * 
   * Examples
   * <code>
   * "A" -> "A"
   * "A", "B" -> "A or B"
   * "A", "B", "C" -> "A, B or C"
   * </code>
   */
  public String orList(final List<String> items) {
    boolean _isEmpty = items.isEmpty();
    if (_isEmpty) {
      return "";
    }
    int _size = items.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      return IterableExtensions.<String>head(items);
    }
    int _size_1 = items.size();
    boolean _greaterEqualsThan = (_size_1 >= 2);
    if (_greaterEqualsThan) {
      Iterable<String> _tail = IterableExtensions.<String>tail(items);
      int _size_2 = items.size();
      int _minus = (_size_2 - 2);
      final Function1<String, CharSequence> _function = (String it) -> {
        return it;
      };
      StringBuilder _append = new StringBuilder().append(IterableExtensions.<String>head(items)).append(IterableExtensions.<String>join(IterableExtensions.<String>take(_tail, _minus), ", ", ", ", "", _function));
      String _last = IterableExtensions.<String>last(items);
      String _plus = (" or " + _last);
      return _append.append(_plus).toString();
    }
    return null;
  }
  
  /**
   * Returns true if a type ref has been appended.
   */
  public boolean appendPromisedReturnType(final StringBuilder strb, final TFunction tfunction) {
    final boolean async = this.isAsyncOrPromise(tfunction);
    final boolean generator = this.isGenerator(tfunction);
    if ((async || generator)) {
      final EObject astElem = tfunction.getAstElement();
      boolean _matched = false;
      if (astElem instanceof FunctionDeclaration) {
        _matched=true;
        final TypeRef retType = ((FunctionDeclaration)astElem).getReturnTypeRef();
        if ((retType != null)) {
          strb.append(retType.getTypeRefAsString());
          return true;
        }
      }
      if (!_matched) {
        if (astElem instanceof FunctionDefinition) {
          _matched=true;
          final TypeRef retType = ((FunctionDefinition)astElem).getReturnTypeRef();
          if ((retType != null)) {
            strb.append(retType.getTypeRefAsString());
            return true;
          }
        }
      }
      TypeRef _returnTypeRef = tfunction.getReturnTypeRef();
      final ParameterizedTypeRef ptr = ((ParameterizedTypeRef) _returnTypeRef);
      final TypeArgument asyncReturnType = ptr.getTypeArgs().get(0);
      if ((asyncReturnType != null)) {
        boolean _isUndefined = TypeUtils.isUndefined(asyncReturnType);
        if (_isUndefined) {
          strb.append("void");
        } else {
          strb.append(asyncReturnType.getTypeRefAsString());
        }
        return true;
      }
    } else {
      TypeRef _returnTypeRef_1 = tfunction.getReturnTypeRef();
      boolean _tripleNotEquals = (_returnTypeRef_1 != null);
      if (_tripleNotEquals) {
        strb.append(tfunction.getReturnTypeRef().getTypeRefAsString());
        return true;
      }
    }
    return false;
  }
  
  private boolean isAsyncOrPromise(final TFunction tfunction) {
    boolean _isDeclaredAsync = tfunction.isDeclaredAsync();
    boolean _not = (!_isDeclaredAsync);
    if (_not) {
      return false;
    }
    Resource _eResource = tfunction.eResource();
    ResourceSet _resourceSet = null;
    if (_eResource!=null) {
      _resourceSet=_eResource.getResourceSet();
    }
    final ResourceSet rs = _resourceSet;
    if ((rs == null)) {
      return false;
    }
    return TypeUtils.isPromise(tfunction.getReturnTypeRef(), BuiltInTypeScope.get(rs));
  }
  
  private boolean isGenerator(final TFunction tfunction) {
    boolean _isDeclaredGenerator = tfunction.isDeclaredGenerator();
    boolean _not = (!_isDeclaredGenerator);
    if (_not) {
      return false;
    }
    Resource _eResource = tfunction.eResource();
    ResourceSet _resourceSet = null;
    if (_eResource!=null) {
      _resourceSet=_eResource.getResourceSet();
    }
    final ResourceSet rs = _resourceSet;
    if ((rs == null)) {
      return false;
    }
    return TypeUtils.isGenerator(tfunction.getReturnTypeRef(), BuiltInTypeScope.get(rs));
  }
  
  public String description(final EObject classifier) {
    if (classifier instanceof TClassifier) {
      return _description((TClassifier)classifier);
    } else if (classifier instanceof IdentifierRef) {
      return _description((IdentifierRef)classifier);
    } else if (classifier instanceof FunctionDefinition) {
      return _description((FunctionDefinition)classifier);
    } else if (classifier instanceof TMember) {
      return _description((TMember)classifier);
    } else if (classifier instanceof IdentifiableElement) {
      return _description((IdentifiableElement)classifier);
    } else if (classifier instanceof NamedElement) {
      return _description((NamedElement)classifier);
    } else if (classifier != null) {
      return _description(classifier);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(classifier).toString());
    }
  }
}
