/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.validators.StaticPolyfillValidatorExtension;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class N4JSInterfaceValidator extends AbstractN4JSDeclarativeValidator {
  /**
   * NEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  public static final String STATIC_MEMBER_IN_ROLE = "STATIC_MEMBER_IN_ROLE";
  
  @Check
  public void checkN4InterfaceDeclaration(final N4InterfaceDeclaration n4Interface) {
    Type _definedType = n4Interface.getDefinedType();
    boolean _tripleEquals = (_definedType == null);
    if (_tripleEquals) {
      return;
    }
    this.holdsNoCyclicInheritance(n4Interface);
    this.internalCheckExtendedInterfaces(n4Interface);
    this.internalCheckNotFinal(n4Interface);
    this.getInternalCheckNoFieldInitizializer(n4Interface);
    StaticPolyfillValidatorExtension.internalCheckNotInStaticPolyfillModule(n4Interface, this);
  }
  
  @Override
  public void addIssue(final String message, final EObject source, final EStructuralFeature feature, final String issueCode, final String... issueData) {
    super.addIssue(message, source, feature, issueCode, issueData);
  }
  
  private void internalCheckNotFinal(final N4InterfaceDeclaration n4InterfaceDeclaration) {
    boolean _hasAnnotation = AnnotationDefinition.FINAL.hasAnnotation(n4InterfaceDeclaration);
    if (_hasAnnotation) {
      final String msg = IssueCodes.getMessageForITF_NO_FINAL();
      this.addIssue(msg, n4InterfaceDeclaration, N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, 
        IssueCodes.ITF_NO_FINAL);
    }
  }
  
  private void getInternalCheckNoFieldInitizializer(final N4InterfaceDeclaration n4Interface) {
    TypingStrategy _typingStrategy = n4Interface.getTypingStrategy();
    boolean _tripleEquals = (_typingStrategy == TypingStrategy.STRUCTURAL);
    if (_tripleEquals) {
      final Consumer<N4FieldDeclaration> _function = (N4FieldDeclaration f) -> {
        Expression _expression = f.getExpression();
        boolean _tripleNotEquals = (_expression != null);
        if (_tripleNotEquals) {
          this.addIssue(IssueCodes.getMessageForITF_NO_FIELD_INITIALIZER(f.getName(), n4Interface.getName()), f.getExpression(), 
            IssueCodes.ITF_NO_FIELD_INITIALIZER);
        }
      };
      n4Interface.getOwnedFields().forEach(_function);
    }
  }
  
  private void internalCheckExtendedInterfaces(final N4InterfaceDeclaration n4Interface) {
    final Consumer<ParameterizedTypeRef> _function = (ParameterizedTypeRef it) -> {
      final Type extendedType = it.getDeclaredType();
      if (((extendedType != null) && (extendedType.getName() != null))) {
        if ((!(extendedType instanceof TInterface))) {
          if ((extendedType instanceof PrimitiveType)) {
            final String message = IssueCodes.getMessageForCLF_EXTENDS_PRIMITIVE_GENERIC_TYPE(((PrimitiveType)extendedType).getName());
            this.addIssue(message, it, null, IssueCodes.CLF_EXTENDS_PRIMITIVE_GENERIC_TYPE);
          } else {
            final String message_1 = IssueCodes.getMessageForCLF_WRONG_META_TYPE(this.validatorMessageHelper.description(n4Interface), "extend", 
              this.validatorMessageHelper.description(extendedType));
            this.addIssue(message_1, it, null, IssueCodes.CLF_WRONG_META_TYPE);
          }
        }
      }
    };
    n4Interface.getSuperInterfaceRefs().forEach(_function);
  }
  
  private boolean holdsNoCyclicInheritance(final N4InterfaceDeclaration n4InterfaceDeclaration) {
    Type _definedType = n4InterfaceDeclaration.getDefinedType();
    final TInterface ifc = ((TInterface) _definedType);
    final List<TClassifier> cycle = this.findCyclicInheritance(ifc);
    if ((cycle != null)) {
      final Function1<ParameterizedTypeRef, Boolean> _function = (ParameterizedTypeRef it) -> {
        Type _declaredType = it.getDeclaredType();
        TClassifier _get = cycle.get(1);
        return Boolean.valueOf((_declaredType == _get));
      };
      final ParameterizedTypeRef miscreant = IterableExtensions.<ParameterizedTypeRef>findFirst(n4InterfaceDeclaration.getSuperInterfaceRefs(), _function);
      final Function1<TClassifier, String> _function_1 = (TClassifier it) -> {
        return it.getName();
      };
      final String message = IssueCodes.getMessageForCLF_INHERITANCE_CYCLE(IterableExtensions.join(ListExtensions.<TClassifier, String>map(cycle, _function_1), ", "));
      this.addIssue(message, miscreant, IssueCodes.CLF_INHERITANCE_CYCLE);
      return false;
    }
    return true;
  }
}
