/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.assistants;

import com.google.inject.Inject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ConditionalExpression;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4Modifier;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PropertyNameOwner;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.transpiler.TransformationAssistant;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.im.DelegatingMember;
import org.eclipse.n4js.transpiler.im.ImFactory;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.ReferencingElementExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.util.SuperInterfacesIterable;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.RecursionGuard;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;

/**
 * This assistant provides helper methods to create members that delegate to some other target member (see
 * {@link DelegationAssistant#createDelegatingMember(TClassifier, TMember) createDelegatingMember(TClassifier, TMember)})
 * and to create the Javascript output code to actually implement these delegating members in the transpiler output
 * (see {@link DelegationAssistant#createDelegation(DelegatingMember) createDelegation(DelegatingMember)}).
 * <p>
 * Usually inherited members in a classifier do not require any special code, because they will be accessed via the
 * native prototype chain mechanism of Javascript. However, there are special cases when some special code has to be
 * generated for an inherited member in order to properly access that inherited member, because it is not available
 * via the ordinary prototype chain.
 */
@SuppressWarnings("all")
public class DelegationAssistant extends TransformationAssistant {
  @Inject
  private TypeAssistant typeAssistant;
  
  /**
   * Creates a new delegating member intended to be inserted into classifier <code>origin</code> in order to delegate
   * from <code>origin</code> to the given member <code>target</code>. The target member is assumed to be an inherited
   * or consumed member of classifier <code>origin</code>, i.e. it is assumed to be located in one of the ancestor
   * classes of <code>origin</code> or one of its directly or indirectly implemented interfaces (but not in origin
   * itself!).
   * <p>
   * Throws exceptions in case of invalid arguments or an invalid internal state, see implementation for details.
   */
  public DelegatingMember createDelegatingMember(final TClassifier origin, final TMember target) {
    ContainerType<?> _containingType = target.getContainingType();
    boolean _tripleEquals = (_containingType == origin);
    if (_tripleEquals) {
      throw new IllegalArgumentException("no point in delegating to an owned member");
    }
    EObject _switchResult = null;
    boolean _matched = false;
    if (target instanceof TField) {
      _matched=true;
      throw new IllegalArgumentException("delegation to fields not supported yet");
    }
    if (!_matched) {
      if (target instanceof TGetter) {
        _matched=true;
        _switchResult = ImFactory.eINSTANCE.createDelegatingGetterDeclaration();
      }
    }
    if (!_matched) {
      if (target instanceof TSetter) {
        _matched=true;
        _switchResult = ImFactory.eINSTANCE.createDelegatingSetterDeclaration();
      }
    }
    if (!_matched) {
      if (target instanceof TMethod) {
        _matched=true;
        _switchResult = ImFactory.eINSTANCE.createDelegatingMethodDeclaration();
      }
    }
    final EObject result = ((EObject)_switchResult);
    ((PropertyNameOwner)result).setDeclaredName(TranspilerBuilderBlocks._LiteralOrComputedPropertyName(target.getName()));
    ((DelegatingMember)result).setDelegationTarget(this.getSymbolTableEntryOriginal(target, true));
    boolean _isStatic = target.isStatic();
    if (_isStatic) {
      EList<N4Modifier> _declaredModifiers = ((DelegatingMember)result).getDeclaredModifiers();
      _declaredModifiers.add(N4Modifier.STATIC);
    }
    if ((origin instanceof TInterface)) {
      EObject _eContainer = target.eContainer();
      boolean _not = (!(_eContainer instanceof TInterface));
      if (_not) {
        throw new IllegalArgumentException("cannot delegate from an interface to member of a class");
      }
      final ContainerType<?> tSuper = this.getDirectSuperTypeBequestingMember(origin, target);
      ((DelegatingMember)result).setDelegationBaseType(this.getSymbolTableEntryOriginal(tSuper, false));
      ((DelegatingMember)result).setDelegationSuperClassSteps(0);
    } else {
      if ((origin instanceof TClass)) {
        final TClass tAncestor = this.getAncestorClassBequestingMember(((TClass)origin), target);
        if ((tAncestor != origin)) {
          Type _declaredType = ((TClass)origin).getSuperClassRef().getDeclaredType();
          final TClass tSuper_1 = ((TClass) _declaredType);
          ((DelegatingMember)result).setDelegationBaseType(this.getSymbolTableEntryOriginal(tSuper_1, false));
          int _distanceToAncestorClass = DelegationAssistant.getDistanceToAncestorClass(((TClass)origin), tAncestor);
          int _minus = (_distanceToAncestorClass - 1);
          ((DelegatingMember)result).setDelegationSuperClassSteps(_minus);
        } else {
          if ((tAncestor != null)) {
            final ContainerType<?> tSuper_2 = this.getDirectSuperTypeBequestingMember(origin, target);
            ((DelegatingMember)result).setDelegationBaseType(this.getSymbolTableEntryOriginal(tSuper_2, false));
            ((DelegatingMember)result).setDelegationSuperClassSteps(0);
          } else {
            throw new IllegalStateException("cannot find target (probably not an inherited member)");
          }
        }
      } else {
        String _name = origin.eClass().getName();
        String _plus = ("unsupported subtype of TClassifier: " + _name);
        throw new IllegalArgumentException(_plus);
      }
    }
    ((DelegatingMember)result).setDelegationTargetIsAbstract(target.isAbstract());
    boolean _isAbstract = target.isAbstract();
    boolean _not_1 = (!_isAbstract);
    if (_not_1) {
      ((FunctionOrFieldAccessor)result).setBody(TranspilerBuilderBlocks._Block());
    }
    return ((DelegatingMember)result);
  }
  
  /**
   * Creates code to establish the actual member delegation for the given delegating member. It returns an expression
   * that evaluates to the delegation target from within the context of the delegation origin (here, "target" and
   * "origin" refers to the two arguments of method {@link #createDelegatingMember(TClassifier, TMember)}). The code
   * returned by this method is intended to be used in the member definitions passed to the <code>$makeClass</code>
   * and <code>$makeInterface</code> calls.
   */
  public Expression createDelegationCode(final DelegatingMember delegator) {
    final SymbolTableEntryOriginal baseSTE = delegator.getDelegationBaseType();
    IdentifiableElement _originalTarget = null;
    if (baseSTE!=null) {
      _originalTarget=baseSTE.getOriginalTarget();
    }
    final boolean baseIsInterface = (_originalTarget instanceof TInterface);
    if (baseIsInterface) {
      final String targetName = delegator.getDelegationTarget().getName();
      final boolean targetIsSymbol = ((targetName != null) && targetName.startsWith(N4JSLanguageUtils.SYMBOL_IDENTIFIER_PREFIX));
      final SymbolTableEntryOriginal targetSTE = delegator.getDelegationTarget();
      Expression _xifexpression = null;
      boolean _isStatic = delegator.isStatic();
      boolean _not = (!_isStatic);
      if (_not) {
        ConditionalExpression _xblockexpression = null;
        {
          final SymbolTableEntryInternal $methodsSTE = this.steFor_$methods();
          final SymbolTableEntry valueSTE = this.getPropertyDescriptorValueProperty(delegator);
          ConditionalExpression _xifexpression_1 = null;
          if ((!targetIsSymbol)) {
            ReturnStatement __ReturnStmnt = TranspilerBuilderBlocks._ReturnStmnt(
              TranspilerBuilderBlocks._CallExpr(
                this.__NSSafe_PropertyAccessExpr(baseSTE, $methodsSTE, targetSTE, valueSTE, 
                  this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.functionType(this.getState().G), "apply", false, false, true)), 
                TranspilerBuilderBlocks._ThisLiteral(), 
                TranspilerBuilderBlocks._IdentRef(this.steFor_arguments())));
            _xifexpression_1 = TranspilerBuilderBlocks._ConditionalExpr(
              TranspilerBuilderBlocks._AND(this.__NSSafe_IdentRef(baseSTE), this.__NSSafe_PropertyAccessExpr(baseSTE, $methodsSTE)), 
              this.__NSSafe_PropertyAccessExpr(baseSTE, $methodsSTE, targetSTE, valueSTE), 
              TranspilerBuilderBlocks._FunExpr(false, new Statement[] { __ReturnStmnt }));
          } else {
            ReturnStatement __ReturnStmnt_1 = TranspilerBuilderBlocks._ReturnStmnt(
              TranspilerBuilderBlocks._CallExpr(
                TranspilerBuilderBlocks._PropertyAccessExpr(
                  TranspilerBuilderBlocks._IndexAccessExpr(
                    this.__NSSafe_PropertyAccessExpr(baseSTE, $methodsSTE), 
                    this.typeAssistant.getMemberNameAsSymbol(targetName)), valueSTE, 
                  this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.functionType(this.getState().G), "apply", false, false, true)), 
                TranspilerBuilderBlocks._ThisLiteral(), 
                TranspilerBuilderBlocks._IdentRef(this.steFor_arguments())));
            _xifexpression_1 = TranspilerBuilderBlocks._ConditionalExpr(
              TranspilerBuilderBlocks._AND(this.__NSSafe_IdentRef(baseSTE), this.__NSSafe_PropertyAccessExpr(baseSTE, $methodsSTE)), 
              TranspilerBuilderBlocks._PropertyAccessExpr(
                TranspilerBuilderBlocks._IndexAccessExpr(
                  this.__NSSafe_PropertyAccessExpr(baseSTE, $methodsSTE), 
                  this.typeAssistant.getMemberNameAsSymbol(targetName)), valueSTE), 
              TranspilerBuilderBlocks._FunExpr(false, new Statement[] { __ReturnStmnt_1 }));
          }
          _xblockexpression = _xifexpression_1;
        }
        _xifexpression = _xblockexpression;
      } else {
        EObject _xifexpression_1 = null;
        if ((!targetIsSymbol)) {
          _xifexpression_1 = this.__NSSafe_PropertyAccessExpr(baseSTE, targetSTE);
        } else {
          _xifexpression_1 = TranspilerBuilderBlocks._IndexAccessExpr(this.__NSSafe_IdentRef(baseSTE), this.typeAssistant.getMemberNameAsSymbol(targetName));
        }
        _xifexpression = ((Expression)_xifexpression_1);
      }
      return _xifexpression;
    } else {
      final Expression ctorOfClassOfTarget = this.createAccessToClassConstructor(baseSTE, delegator.getDelegationSuperClassSteps());
      final Expression result = this.createAccessToMemberFunction(ctorOfClassOfTarget, false, delegator);
      return result;
    }
  }
  
  /**
   * Creates an expression that will evaluate to the constructor of the class denoted by the given symbol table
   * entry or one of its super classes (depending on argument <code>superClassSteps</code>).
   * <p>
   * For example, if <code>classSTE</code> denotes a class "C", then this will produce an expression like
   * <table>
   * <tr><td><code>C</code></td><td>for <code>superClassSteps</code> == 0</td></tr>
   * <tr><td><code>Object.getPrototypeOf(C)</code></td><td>for <code>superClassSteps</code> == 1</td></tr>
   * <tr><td><code>Object.getPrototypeOf(Object.getPrototypeOf(C))</code></td><td>for <code>superClassSteps</code> == 2</td></tr>
   * </table>
   */
  public Expression createAccessToClassConstructor(final SymbolTableEntry classSTE, final int superClassSteps) {
    final TClassifier objectType = RuleEnvironmentExtensions.objectType(this.getState().G);
    final SymbolTableEntryOriginal objectSTE = this.getSymbolTableEntryOriginal(objectType, true);
    Expression result = this.__NSSafe_IdentRef(classSTE);
    if ((superClassSteps > 0)) {
      final SymbolTableEntryOriginal getPrototypeOfSTE = this.getSymbolTableEntryForMember(objectType, "getPrototypeOf", false, true, true);
      ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, superClassSteps, true);
      for (final Integer n : _doubleDotLessThan) {
        result = TranspilerBuilderBlocks._CallExpr(
          TranspilerBuilderBlocks._PropertyAccessExpr(objectSTE, getPrototypeOfSTE), result);
      }
    }
    return result;
  }
  
  /**
   * Same as {@link #createAccessToMemberDefinition(Expression, boolean, N4MemberDeclaration) createAccessToMemberDefinition()},
   * but will add a property access to the Javascript function representing the member, which is stored in the member
   * definition (by appending ".get", ".set", or ".value" depending on the type of member).
   * <p>
   * Since fields do not have such a function this will throw an exception if given member is a field.
   */
  public Expression createAccessToMemberFunction(final Expression protoOrCtorExpr, final boolean exprIsProto, final N4MemberDeclaration member) {
    if ((member instanceof N4FieldDeclaration)) {
      throw new IllegalArgumentException("no member function available for fields");
    }
    final Expression accessToMemberDefinition = this.createAccessToMemberDefinition(protoOrCtorExpr, exprIsProto, member);
    final ParameterizedPropertyAccessExpression_IM result = TranspilerBuilderBlocks._PropertyAccessExpr(accessToMemberDefinition, this.getPropertyDescriptorValueProperty(member));
    return result;
  }
  
  /**
   * Given an expression that will evaluate to a prototype or constructor, this method returns an expression that
   * will evaluate to the member definition of a particular member.
   * 
   * @param protoOrCtorExpr
   *         an expression that is expected to evaluate to a prototype or a constructor.
   * @param exprIsProto
   *         tells whether argument <code>protoOrCtorExpr</code> will evaluate to a prototype or a constructor:
   *         if true, it will evaluate to a prototype, if false it will evaluate to a constructor.
   * @param member
   *         the member.
   */
  public Expression createAccessToMemberDefinition(final Expression protoOrCtorExpr, final boolean exprIsProto, final N4MemberDeclaration member) {
    final String memberName = member.getName();
    final boolean memberIsSymbol = ((memberName != null) && memberName.startsWith(N4JSLanguageUtils.SYMBOL_IDENTIFIER_PREFIX));
    final SymbolTableEntry memberSTE = this.findSymbolTableEntryForElement(member, true);
    final TClassifier objectType = RuleEnvironmentExtensions.objectType(this.getState().G);
    final SymbolTableEntryOriginal objectSTE = this.getSymbolTableEntryOriginal(objectType, true);
    final SymbolTableEntryOriginal getOwnPropertyDescriptorSTE = this.getSymbolTableEntryForMember(objectType, "getOwnPropertyDescriptor", false, true, true);
    final boolean isStatic = member.isStatic();
    Expression arg0 = protoOrCtorExpr;
    if (((!exprIsProto) && (!isStatic))) {
      final SymbolTableEntryOriginal prototypeSTE = this.getSymbolTableEntryForMember(objectType, "prototype", false, true, true);
      arg0 = TranspilerBuilderBlocks._PropertyAccessExpr(arg0, prototypeSTE);
    } else {
      if ((exprIsProto && isStatic)) {
        final SymbolTableEntryOriginal constructorSTE = this.getSymbolTableEntryForMember(objectType, "constructor", false, false, true);
        arg0 = TranspilerBuilderBlocks._PropertyAccessExpr(arg0, constructorSTE);
      }
    }
    Expression _xifexpression = null;
    if ((!memberIsSymbol)) {
      _xifexpression = TranspilerBuilderBlocks._StringLiteralForSTE(memberSTE);
    } else {
      _xifexpression = this.typeAssistant.getMemberNameAsSymbol(memberName);
    }
    final Expression arg1 = _xifexpression;
    final ParameterizedCallExpression result = TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._PropertyAccessExpr(objectSTE, getOwnPropertyDescriptorSTE), arg0, arg1);
    return result;
  }
  
  /**
   * Creates a property access to the immediate super class of the given <code>baseClassDecl</code>, i.e. in the
   * non-static case:
   * <pre>
   * S.prototype
   * </pre>
   * and in the static case:
   * <pre>
   * S
   * </pre>
   * (with S being the direct super class).
   */
  public Expression createAccessToSuperClass(final N4ClassDeclaration baseClassDecl, final boolean isStatic) {
    final SymbolTableEntryOriginal superClassSTE = this.typeAssistant.getSuperClassSTE(baseClassDecl);
    final SymbolTableEntryOriginal prototypeSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.objectType(this.getState().G), "prototype", false, true, true);
    ReferencingElementExpression_IM _xifexpression = null;
    if ((!isStatic)) {
      _xifexpression = this.__NSSafe_PropertyAccessExpr(superClassSTE, prototypeSTE);
    } else {
      _xifexpression = this.__NSSafe_IdentRef(superClassSTE);
    }
    return _xifexpression;
  }
  
  /**
   * Like {@link DelegationAssistant#createAccessToSuperClass(N4ClassDeclaration,boolean)}, but follows the property
   * chain until the correct class for the given member is reached (either the containing class or, if the member is
   * consumed from an interface, the first super class that consumed the member).
   */
  public Expression createAccessToSuperClassBequestingMember(final N4ClassDeclaration baseClassDecl, final boolean isStatic, final SymbolTableEntryOriginal memberSTE) {
    final TClass tClassBase = this.getState().info.getOriginalDefinedType(baseClassDecl);
    IdentifiableElement _originalTarget = memberSTE.getOriginalTarget();
    final TMember member = ((TMember) _originalTarget);
    final TClass tClassTarget = this.getAncestorClassBequestingMember(tClassBase, member);
    final int dist = DelegationAssistant.getDistanceToAncestorClass(tClassBase, tClassTarget);
    final SymbolTableEntryInternal __proto__STE = this.steFor___proto__();
    Expression superAccess = this.createAccessToSuperClass(baseClassDecl, isStatic);
    ExclusiveRange _doubleDotLessThan = new ExclusiveRange(1, dist, true);
    for (final Integer i : _doubleDotLessThan) {
      superAccess = TranspilerBuilderBlocks._PropertyAccessExpr(superAccess, __proto__STE);
    }
    return superAccess;
  }
  
  /**
   * Returns the direct super type (i.e. immediate super class or directly implemented interface) of the given
   * classifier through which the given classifier inherits the given inherited member. Fails fast in case of
   * inconsistencies.
   */
  private ContainerType<?> getDirectSuperTypeBequestingMember(final TClassifier classifier, final TMember inheritedMember) {
    return this.getState().memberCollector.directSuperTypeBequestingMember(classifier, inheritedMember);
  }
  
  /**
   * Returns the ancestor class (i.e. direct or indirect super class) of the given classifier that either contains
   * the given inherited member (if given member is contained in a class) or consumes the given member (if given
   * member is contained in an interface).
   * <p>
   * For example:
   * <pre>
   * interface I {
   *     m() {}
   * }
   * class A implements I {}
   * class B extends A {}
   * class C extends B {}
   * </pre>
   * With the above type declarations, for arguments <code>C</code> and <code>m</code> this method would return class
   * <code>A</code>.
   */
  private TClass getAncestorClassBequestingMember(final TClass classifier, final TMember inheritedOrConsumedMember) {
    TClass _xblockexpression = null;
    {
      final ContainerType<?> containingType = inheritedOrConsumedMember.getContainingType();
      TClass _xifexpression = null;
      if ((containingType == classifier)) {
        return classifier;
      } else {
        TClass _xifexpression_1 = null;
        if ((containingType instanceof TInterface)) {
          _xifexpression_1 = SuperInterfacesIterable.of(classifier).findClassImplementingInterface(((TInterface)containingType));
        } else {
          TClass _xifexpression_2 = null;
          if ((containingType instanceof TClass)) {
            _xifexpression_2 = ((TClass)containingType);
          } else {
            String _name = containingType.eClass().getName();
            String _plus = ("unsupported subtype of TClassifier: " + _name);
            throw new IllegalArgumentException(_plus);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns distance to given ancestor or 0 if second argument is 'base' or not an ancestor or <code>null</code>.
   */
  private static int getDistanceToAncestorClass(final TClass base, final TClass ancestorClass) {
    if (((ancestorClass == null) || (ancestorClass == base))) {
      return 0;
    }
    final RecursionGuard<TClass> guard = new RecursionGuard<TClass>();
    int result = 0;
    TClass curr = base;
    while (((curr != null) && (curr != ancestorClass))) {
      boolean _tryNext = guard.tryNext(curr);
      if (_tryNext) {
        result++;
        ParameterizedTypeRef _superClassRef = curr.getSuperClassRef();
        Type _declaredType = null;
        if (_superClassRef!=null) {
          _declaredType=_superClassRef.getDeclaredType();
        }
        curr = ((TClass) _declaredType);
      }
    }
    int _xifexpression = (int) 0;
    if ((curr != null)) {
      _xifexpression = result;
    } else {
      _xifexpression = 0;
    }
    return _xifexpression;
  }
  
  /**
   * Returns symbol table entry for the property in a Javascript property descriptor that holds the actual value,
   * i.e. the function expression implementing the member.
   */
  private SymbolTableEntry getPropertyDescriptorValueProperty(final N4MemberDeclaration delegator) {
    SymbolTableEntryInternal _switchResult = null;
    boolean _matched = false;
    if (delegator instanceof N4GetterDeclaration) {
      _matched=true;
      _switchResult = this.steFor_get();
    }
    if (!_matched) {
      if (delegator instanceof N4SetterDeclaration) {
        _matched=true;
        _switchResult = this.steFor_set();
      }
    }
    if (!_matched) {
      if (delegator instanceof N4MethodDeclaration) {
        _matched=true;
        _switchResult = this.steFor_value();
      }
    }
    return _switchResult;
  }
}
