/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ts.scoping;

import com.google.common.base.Predicate;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.name.Named;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.n4js.ts.scoping.TypesScopeFilter;
import org.eclipse.n4js.ts.scoping.ValidatingScope;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeDefs;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.xtext.resource.EObjectDescription;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.IScopeProvider;
import org.eclipse.xtext.scoping.Scopes;
import org.eclipse.xtext.scoping.impl.AbstractScopeProvider;
import org.eclipse.xtext.scoping.impl.IDelegatingScopeProvider;
import org.eclipse.xtext.scoping.impl.MapBasedScope;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * This class contains custom scoping description.
 * 
 * see : http://www.eclipse.org/Xtext/documentation.html#scoping
 * on how and when to use it
 */
@SuppressWarnings("all")
public class TypesScopeProvider extends AbstractScopeProvider implements IDelegatingScopeProvider {
  public static final String NAMED_DELEGATE = "org.eclipse.xtext.scoping.impl.AbstractDeclarativeScopeProvider.delegate";
  
  @Inject
  private IResourceScopeCache cache;
  
  @Inject
  @Extension
  private TypesScopeFilter _typesScopeFilter;
  
  @Inject
  @Named(TypesScopeProvider.NAMED_DELEGATE)
  private IScopeProvider delegate;
  
  protected IScope delegateGetScope(final EObject context, final EReference reference) {
    return this.delegate.getScope(context, reference);
  }
  
  public void setDelegate(final IScopeProvider delegate) {
    this.delegate = delegate;
  }
  
  @Override
  public IScopeProvider getDelegate() {
    return this.delegate;
  }
  
  public TypesScopeProvider() {
  }
  
  @Override
  public IScope getScope(final EObject context, final EReference reference) {
    boolean _isSuperTypeOf = TypesPackage.Literals.TYPE.isSuperTypeOf(reference.getEReferenceType());
    if (_isSuperTypeOf) {
      IScope _typeScope = this.getTypeScope(context, reference);
      Predicate<? super IEObjectDescription> _typesFilterCriteria = this._typesScopeFilter.getTypesFilterCriteria(context, reference);
      return new ValidatingScope(_typeScope, _typesFilterCriteria);
    }
    return this.delegateGetScope(context, reference);
  }
  
  private IScope getTypeScope(final EObject context, final EReference reference) {
    boolean _matched = false;
    if (context instanceof TModule) {
      _matched=true;
      return this.delegateGetScope(context, reference);
    }
    if (!_matched) {
      if (context instanceof TypeDefs) {
        _matched=true;
        return this.getLocallyKnownTypes(((TypeDefs)context), reference);
      }
    }
    if (!_matched) {
      if (context instanceof Type) {
        _matched=true;
        final IScope scope = this.getTypeScope(((Type)context).eContainer(), reference);
        final EList<TypeVariable> declaredTypeVars = ((Type)context).getTypeVars();
        boolean _isEmpty = declaredTypeVars.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          return Scopes.scopeFor(declaredTypeVars, scope);
        }
        return scope;
      }
    }
    return this.getTypeScope(context.eContainer(), reference);
  }
  
  private IScope getLocallyKnownTypes(final TypeDefs typeDefs, final EReference reference) {
    Pair<TypeDefs, String> _mappedTo = Pair.<TypeDefs, String>of(typeDefs, "locallyKnownTypes");
    final Provider<IScope> _function = () -> {
      IScope parent = this.delegateGetScope(typeDefs, reference);
      final Function1<Type, Boolean> _function_1 = (Type it) -> {
        String _name = it.getName();
        return Boolean.valueOf((_name != null));
      };
      final Function1<Type, IEObjectDescription> _function_2 = (Type it) -> {
        return EObjectDescription.create(it.getName(), it);
      };
      return MapBasedScope.createScope(parent, 
        IterableExtensions.<Type, IEObjectDescription>map(IterableExtensions.<Type>filter(typeDefs.getTypes(), _function_1), _function_2));
    };
    return this.cache.<IScope>get(_mappedTo, typeDefs.eResource(), _function);
  }
}
