/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.scoping;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.ts.typeRefs.Versionable;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TVersionable;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * An {@link N4IDLVersionableFilter} can be used to filter versionable elements
 * according to the N4IDL versioning rules.
 * 
 * The filter ignores any non-versionable elements and will never filter them out.
 */
@SuppressWarnings("all")
public class N4IDLVersionableFilter {
  private final int contextVersion;
  
  public N4IDLVersionableFilter(final int contextVersion) {
    this.contextVersion = contextVersion;
  }
  
  /**
   * Returns a filtered view of the given iterable that only contains the descriptions of elements such that either of the following conditions hold:
   * <ul>
   * <li>The element is not an instance of {@link TClassifier}, i.e., it is not versionable.</li>
   * <li>The element is an instance of {@link TClassifier} and its version is contained in the requested version range and its version is maximal among
   * all such elements.</li>
   * </ul>
   */
  public Iterable<IEObjectDescription> filterElements(final Iterable<IEObjectDescription> descriptions) {
    final int maxVersion = this.getMaxVersion(descriptions);
    final Function1<IEObjectDescription, Boolean> _function = (IEObjectDescription d) -> {
      final EObject o = d.getEObjectOrProxy();
      return Boolean.valueOf(((!(o instanceof TVersionable)) || (((Versionable) o).getVersion() == maxVersion)));
    };
    final Iterable<IEObjectDescription> result = IterableExtensions.<IEObjectDescription>filter(descriptions, _function);
    return result;
  }
  
  /**
   * Returns the maximum version no of any versioned element described by the given descriptions that is contained
   * in the requested version range. If the given iterable does not contain a description of a versioned element
   * whose version is in the requested version range, then <code>-1</code> is returned.
   */
  public int getMaxVersion(final Iterable<IEObjectDescription> descriptions) {
    final Function1<IEObjectDescription, Boolean> _function = (IEObjectDescription it) -> {
      EObject _eObjectOrProxy = it.getEObjectOrProxy();
      return Boolean.valueOf((_eObjectOrProxy instanceof TVersionable));
    };
    final Function1<IEObjectDescription, TVersionable> _function_1 = (IEObjectDescription it) -> {
      EObject _eObjectOrProxy = it.getEObjectOrProxy();
      return ((TVersionable) _eObjectOrProxy);
    };
    final Function1<TVersionable, Boolean> _function_2 = (TVersionable it) -> {
      int _version = it.getVersion();
      return Boolean.valueOf((_version <= this.contextVersion));
    };
    final Function2<Integer, TVersionable, Integer> _function_3 = (Integer u, TVersionable c) -> {
      return Integer.valueOf(Integer.max((u).intValue(), c.getVersion()));
    };
    final Integer max = IterableExtensions.<TVersionable, Integer>fold(IterableExtensions.<TVersionable>filter(IterableExtensions.<IEObjectDescription, TVersionable>map(IterableExtensions.<IEObjectDescription>filter(descriptions, _function), _function_1), _function_2), Integer.valueOf((-1)), _function_3);
    return (max).intValue();
  }
  
  public int getContextVersion() {
    return this.contextVersion;
  }
}
