/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.organize.imports;

import com.google.common.base.Objects;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.organize.imports.ImportProvidedElement;
import org.eclipse.n4js.organize.imports.ImportSpecifiersUtil;
import org.eclipse.n4js.organize.imports.RecordingImportState;
import org.eclipse.n4js.organize.imports.ScriptDependency;
import org.eclipse.n4js.organize.imports.ScriptDependencyResolver;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.utils.Log;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Analyzes all imports in a script. Builds up a data structure of {@link RecordingImportState} to capture the findings.
 */
@Log
@SuppressWarnings("all")
public class ImportStateCalculator {
  /**
   * Algorithm to check the Model for Issues with Imports.
   * @returns {@link RecordingImportState}
   */
  public RecordingImportState calculateImportstate(final Script script) {
    final RecordingImportState reg = new RecordingImportState();
    final Iterable<ImportDeclaration> importDeclarationsALL = Iterables.<ImportDeclaration>filter(script.getScriptElements(), ImportDeclaration.class);
    this.registerDuplicatedImoprtDeclarationsFrom(reg, importDeclarationsALL);
    final Function1<ImportDeclaration, Boolean> _function = (ImportDeclaration it) -> {
      boolean _isDuplicatingImportDeclaration = reg.isDuplicatingImportDeclaration(it);
      return Boolean.valueOf((!_isDuplicatingImportDeclaration));
    };
    final Function1<ImportDeclaration, EList<ImportSpecifier>> _function_1 = (ImportDeclaration it) -> {
      return it.getImportSpecifiers();
    };
    final List<ImportSpecifier> importSpecifiersUnAnalyzed = IterableExtensions.<ImportSpecifier>toList(Iterables.<ImportSpecifier>concat(IterableExtensions.<ImportDeclaration, EList<ImportSpecifier>>map(IterableExtensions.<ImportDeclaration>filter(importDeclarationsALL, _function), _function_1)));
    this.registerUnusedAndBrokenImports(reg, importSpecifiersUnAnalyzed);
    final Function1<ImportSpecifier, Boolean> _function_2 = (ImportSpecifier it) -> {
      boolean _contains = reg.brokenImports.contains(it);
      return Boolean.valueOf((!_contains));
    };
    final List<ImportProvidedElement> importProvidedElements = ImportSpecifiersUtil.mapToImportProvidedElements(IterableExtensions.<ImportSpecifier>toList(IterableExtensions.<ImportSpecifier>filter(importSpecifiersUnAnalyzed, _function_2)));
    final List<Pair<String, List<ImportProvidedElement>>> lN2IPE = CollectionLiterals.<Pair<String, List<ImportProvidedElement>>>newArrayList();
    final List<Pair<TModule, List<ImportProvidedElement>>> lM2IPE = CollectionLiterals.<Pair<TModule, List<ImportProvidedElement>>>newArrayList();
    for (final ImportProvidedElement ipe : importProvidedElements) {
      {
        final Function1<Pair<String, List<ImportProvidedElement>>, Boolean> _function_3 = (Pair<String, List<ImportProvidedElement>> it) -> {
          String _key = it.getKey();
          String _localName = ipe.getLocalName();
          return Boolean.valueOf(Objects.equal(_key, _localName));
        };
        final Pair<String, List<ImportProvidedElement>> pN2IPE = IterableExtensions.<Pair<String, List<ImportProvidedElement>>>findFirst(lN2IPE, _function_3);
        if ((pN2IPE != null)) {
          pN2IPE.getValue().add(ipe);
        } else {
          String _localName = ipe.getLocalName();
          ArrayList<ImportProvidedElement> _newArrayList = CollectionLiterals.<ImportProvidedElement>newArrayList(ipe);
          Pair<String, List<ImportProvidedElement>> _mappedTo = Pair.<String, List<ImportProvidedElement>>of(_localName, _newArrayList);
          lN2IPE.add(_mappedTo);
        }
        final Function1<Pair<TModule, List<ImportProvidedElement>>, Boolean> _function_4 = (Pair<TModule, List<ImportProvidedElement>> it) -> {
          TModule _key = it.getKey();
          TModule _importedModule = ipe.getImportedModule();
          return Boolean.valueOf(Objects.equal(_key, _importedModule));
        };
        final Pair<TModule, List<ImportProvidedElement>> pM2IPE = IterableExtensions.<Pair<TModule, List<ImportProvidedElement>>>findFirst(lM2IPE, _function_4);
        if ((pM2IPE != null)) {
          pM2IPE.getValue().add(ipe);
        } else {
          TModule _importedModule = ipe.getImportedModule();
          ArrayList<ImportProvidedElement> _newArrayList_1 = CollectionLiterals.<ImportProvidedElement>newArrayList(ipe);
          Pair<TModule, List<ImportProvidedElement>> _mappedTo_1 = Pair.<TModule, List<ImportProvidedElement>>of(_importedModule, _newArrayList_1);
          lM2IPE.add(_mappedTo_1);
        }
      }
    }
    this.registerUsedImportsLocalNamesCollisions(reg, lN2IPE);
    this.registerUsedImportsDuplicatedImportedElements(reg, lM2IPE);
    reg.registerAllUsedTypeNameToSpecifierTuples(importProvidedElements);
    final List<ScriptDependency> externalDep = ScriptDependencyResolver.usedDependenciesTypeRefs(script);
    for (final ScriptDependency scriptDep : externalDep) {
      {
        final TModule mod = scriptDep.dependencyModule;
        final Function1<Pair<TModule, List<ImportProvidedElement>>, Boolean> _function_3 = (Pair<TModule, List<ImportProvidedElement>> it) -> {
          TModule _key = it.getKey();
          return Boolean.valueOf(Objects.equal(_key, mod));
        };
        final Pair<TModule, List<ImportProvidedElement>> pM2IPE = IterableExtensions.<Pair<TModule, List<ImportProvidedElement>>>findFirst(lM2IPE, _function_3);
        if ((pM2IPE != null)) {
          final Function1<ImportProvidedElement, Boolean> _function_4 = (ImportProvidedElement it) -> {
            return Boolean.valueOf((Objects.equal(it.getExportedName(), scriptDep.actualName) && Objects.equal(it.getLocalName(), scriptDep.localName)));
          };
          final Consumer<ImportProvidedElement> _function_5 = (ImportProvidedElement it) -> {
            it.markUsed();
          };
          IterableExtensions.<ImportProvidedElement>filter(pM2IPE.getValue(), _function_4).forEach(_function_5);
        }
      }
    }
    return reg;
  }
  
  /**
   * Registers conflicting or duplicate (based on imported elements) imports in the provided {@link RecordingImportState}
   */
  private void registerUsedImportsDuplicatedImportedElements(final RecordingImportState reg, final List<Pair<TModule, List<ImportProvidedElement>>> module2imported) {
    for (final Pair<TModule, List<ImportProvidedElement>> pair : module2imported) {
      {
        final List<ImportProvidedElement> fromMod = pair.getValue();
        final ArrayListMultimap<String, ImportProvidedElement> actname2Import = ArrayListMultimap.<String, ImportProvidedElement>create();
        for (final ImportProvidedElement ipe : fromMod) {
          actname2Import.put(ipe.getExportedName(), ipe);
        }
        Set<String> _keySet = actname2Import.keySet();
        for (final String act : _keySet) {
          {
            final List<ImportProvidedElement> v = IterableExtensions.<ImportProvidedElement>toList(actname2Import.get(act));
            final Function1<ImportProvidedElement, Boolean> _function = (ImportProvidedElement internalIPE) -> {
              boolean _xblockexpression = false;
              {
                final ImportSpecifier specifier = internalIPE.getImportSpecifier();
                boolean _xifexpression = false;
                if ((specifier instanceof NamespaceImportSpecifier)) {
                  String _exportedName = internalIPE.getExportedName();
                  String _computeNamespaceActualName = ImportSpecifiersUtil.computeNamespaceActualName(((NamespaceImportSpecifier)specifier));
                  _xifexpression = (!Objects.equal(_exportedName, _computeNamespaceActualName));
                } else {
                  _xifexpression = true;
                }
                _xblockexpression = _xifexpression;
              }
              return Boolean.valueOf(_xblockexpression);
            };
            final List<ImportProvidedElement> x = IterableExtensions.<ImportProvidedElement>toList(IterableExtensions.<ImportProvidedElement>filter(v, _function));
            int _size = x.size();
            boolean _greaterThan = (_size > 1);
            if (_greaterThan) {
              reg.registerDuplicateImportsOfSameElement(act, pair.getKey(), x);
            }
          }
        }
      }
    }
  }
  
  /**
   * Registers conflicting or duplicate (based on local name checks) imports in the provided {@link RecordingImportState}
   */
  private void registerUsedImportsLocalNamesCollisions(final RecordingImportState reg, final List<Pair<String, List<ImportProvidedElement>>> localname2importprovider) {
    for (final Pair<String, List<ImportProvidedElement>> pair : localname2importprovider) {
      int _size = pair.getValue().size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        reg.registerLocalNameCollision(pair.getKey(), pair.getValue());
      }
    }
  }
  
  /**
   * analyzes provided {@link ImportDeclaration}s, if it finds *exact* duplicates, adds them to the {@link RecordingImportState#duplicatedImportDeclarations}
   */
  private void registerDuplicatedImoprtDeclarationsFrom(final RecordingImportState reg, final Iterable<ImportDeclaration> importDeclarations) {
    final Function1<ImportDeclaration, Boolean> _function = (ImportDeclaration id) -> {
      final Function1<ImportSpecifier, Boolean> _function_1 = (ImportSpecifier is) -> {
        return Boolean.valueOf((is instanceof NamespaceImportSpecifier));
      };
      ImportSpecifier _findFirst = IterableExtensions.<ImportSpecifier>findFirst(id.getImportSpecifiers(), _function_1);
      return Boolean.valueOf((_findFirst != null));
    };
    final Function1<ImportDeclaration, TModule> _function_1 = (ImportDeclaration it) -> {
      return it.getModule();
    };
    final BiConsumer<TModule, List<ImportDeclaration>> _function_2 = (TModule module, List<ImportDeclaration> impDecls) -> {
      this.registerImportDeclarationsWithNamespaceImportsForModule(impDecls, reg);
    };
    IterableExtensions.<TModule, ImportDeclaration>groupBy(IterableExtensions.<ImportDeclaration>filter(importDeclarations, _function), _function_1).forEach(_function_2);
    final Function1<ImportDeclaration, Boolean> _function_3 = (ImportDeclaration id) -> {
      final Function1<ImportSpecifier, Boolean> _function_4 = (ImportSpecifier is) -> {
        return Boolean.valueOf((is instanceof NamedImportSpecifier));
      };
      ImportSpecifier _findFirst = IterableExtensions.<ImportSpecifier>findFirst(id.getImportSpecifiers(), _function_4);
      return Boolean.valueOf((_findFirst != null));
    };
    final Function1<ImportDeclaration, TModule> _function_4 = (ImportDeclaration it) -> {
      return it.getModule();
    };
    final BiConsumer<TModule, List<ImportDeclaration>> _function_5 = (TModule module, List<ImportDeclaration> impDecls) -> {
      this.registerImportDeclarationsWithNamedImportsForModule(impDecls, reg);
    };
    IterableExtensions.<TModule, ImportDeclaration>groupBy(IterableExtensions.<ImportDeclaration>filter(importDeclarations, _function_3), _function_4).forEach(_function_5);
  }
  
  private void registerImportDeclarationsWithNamespaceImportsForModule(final List<ImportDeclaration> importDeclarations, final RecordingImportState reg) {
    int _size = importDeclarations.size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      return;
    }
    final ArrayList<ImportDeclaration> duplicates = CollectionLiterals.<ImportDeclaration>newArrayList();
    final ImportDeclaration firstDeclaration = IterableExtensions.<ImportDeclaration>head(importDeclarations);
    final String firstNamespaceName = IterableExtensions.<NamespaceImportSpecifier>head(Iterables.<NamespaceImportSpecifier>filter(firstDeclaration.getImportSpecifiers(), NamespaceImportSpecifier.class)).getAlias();
    final Consumer<ImportDeclaration> _function = (ImportDeclaration importDeclaration) -> {
      final String followingNamespaceName = IterableExtensions.<NamespaceImportSpecifier>head(Iterables.<NamespaceImportSpecifier>filter(importDeclaration.getImportSpecifiers(), NamespaceImportSpecifier.class)).getAlias();
      boolean _equals = Objects.equal(firstNamespaceName, followingNamespaceName);
      if (_equals) {
        duplicates.add(importDeclaration);
      }
    };
    IterableExtensions.<ImportDeclaration>tail(importDeclarations).forEach(_function);
    boolean _isEmpty = duplicates.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      reg.registerDuplicatesOfImportDeclaration(firstDeclaration, duplicates);
    }
  }
  
  private void registerImportDeclarationsWithNamedImportsForModule(final List<ImportDeclaration> importDeclarations, final RecordingImportState reg) {
    int _size = importDeclarations.size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      return;
    }
    final ArrayList<ImportDeclaration> duplicates = CollectionLiterals.<ImportDeclaration>newArrayList();
    final ImportDeclaration firstDeclaration = IterableExtensions.<ImportDeclaration>head(importDeclarations);
    final Iterable<NamedImportSpecifier> firstDeclarationSpecifiers = Iterables.<NamedImportSpecifier>filter(firstDeclaration.getImportSpecifiers(), NamedImportSpecifier.class);
    final Consumer<ImportDeclaration> _function = (ImportDeclaration importDeclaration) -> {
      final Iterable<NamedImportSpecifier> followingDeclarationSpecifiers = Iterables.<NamedImportSpecifier>filter(importDeclaration.getImportSpecifiers(), NamedImportSpecifier.class);
      if (((!IterableExtensions.isEmpty(firstDeclarationSpecifiers)) && 
        (IterableExtensions.size(firstDeclarationSpecifiers) == IterableExtensions.size(followingDeclarationSpecifiers)))) {
        boolean _allFollowingMatchByNameAndAlias = this.allFollowingMatchByNameAndAlias(firstDeclarationSpecifiers, followingDeclarationSpecifiers);
        if (_allFollowingMatchByNameAndAlias) {
          duplicates.add(importDeclaration);
        }
      }
    };
    IterableExtensions.<ImportDeclaration>tail(importDeclarations).forEach(_function);
    boolean _isEmpty = duplicates.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      reg.registerDuplicatesOfImportDeclaration(firstDeclaration, duplicates);
    }
  }
  
  private boolean allFollowingMatchByNameAndAlias(final Iterable<NamedImportSpecifier> firstDeclarationSpecifiers, final Iterable<NamedImportSpecifier> followingDeclarationSpecifiers) {
    final Function1<NamedImportSpecifier, Boolean> _function = (NamedImportSpecifier namedImportSpecifier) -> {
      final Function1<NamedImportSpecifier, Boolean> _function_1 = (NamedImportSpecifier otherNamedImportSpecifier) -> {
        return Boolean.valueOf((Objects.equal(namedImportSpecifier.getAlias(), otherNamedImportSpecifier.getAlias()) && 
          Objects.equal(namedImportSpecifier.getImportedElement().getName(), otherNamedImportSpecifier.getImportedElement().getName())));
      };
      return Boolean.valueOf(IterableExtensions.<NamedImportSpecifier>exists(followingDeclarationSpecifiers, _function_1));
    };
    return IterableExtensions.<NamedImportSpecifier>forall(firstDeclarationSpecifiers, _function);
  }
  
  /**
   * Registers unused or broken (missing or unresolved imported module) import specifiers in the provided {@link RecordingImportState}
   */
  private void registerUnusedAndBrokenImports(final RecordingImportState reg, final List<ImportSpecifier> importSpecifiers) {
    for (final ImportSpecifier is : importSpecifiers) {
      boolean _isFlaggedUsedInCode = is.isFlaggedUsedInCode();
      boolean _not = (!_isFlaggedUsedInCode);
      if (_not) {
        reg.registerUnusedImport(is);
        boolean _isBrokenImport = ImportSpecifiersUtil.isBrokenImport(is);
        if (_isBrokenImport) {
          reg.registerBrokenImport(is);
        }
      }
    }
  }
  
  private static final Logger logger = Logger.getLogger(ImportStateCalculator.class);
}
