/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.packagejson.model.edit;

import java.util.Arrays;
import java.util.Collection;
import java.util.function.Supplier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.json.JSON.JSONArray;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.JSON.JSONFactory;
import org.eclipse.n4js.json.JSON.JSONObject;
import org.eclipse.n4js.json.JSON.JSONStringLiteral;
import org.eclipse.n4js.json.JSON.JSONValue;
import org.eclipse.n4js.json.model.utils.JSONModelUtils;
import org.eclipse.n4js.packagejson.PackageJsonProperties;
import org.eclipse.n4js.packagejson.model.edit.IJSONDocumentModification;

/**
 * This class provides basic edit operations for N4JS package.json files
 * in terms of {@link IJSONDocumentModification}s.
 */
@SuppressWarnings("all")
public class PackageJsonModificationProvider {
  /**
   * Returns a semantic modification that appends the given list of new project dependencies to the "dependencies"
   * section of a package.json file.
   * 
   * @param dependencies
   * 			The list of project dependencies to insert.
   */
  public static IJSONDocumentModification insertProjectDependencies(final Collection<String> dependencies) {
    return new IJSONDocumentModification() {
      @Override
      public void apply(final JSONDocument document) {
        boolean _isEmpty = dependencies.isEmpty();
        if (_isEmpty) {
          return;
        }
        final JSONObject root = PackageJsonModificationProvider.getDocumentRoot(document);
        for (final String dependency : dependencies) {
          PackageJsonModificationProvider.addProjectDependency(root, dependency, "*");
        }
      }
    };
  }
  
  /**
   * Returns a semantic modification that appends the given list of new required runtime libraries to the required-runtime-libraries-section
   * of a package.json file.
   * 
   * @param resourceUri
   * 			The URI of the package.json file.
   * @param runtimeLibraries
   * 			The list of newly required runtime libraries.
   */
  public static IJSONDocumentModification insertRequiredRuntimeLibraries(final Collection<String> runtimeLibraries) {
    return new IJSONDocumentModification() {
      @Override
      public void apply(final JSONDocument document) {
        boolean _isEmpty = runtimeLibraries.isEmpty();
        if (_isEmpty) {
          return;
        }
        final JSONObject root = PackageJsonModificationProvider.getDocumentRoot(document);
        final JSONObject n4jsSection = PackageJsonModificationProvider.getOrCreateN4JSSection(root);
        final JSONArray requiredRuntimeLibrariesList = PackageJsonModificationProvider.getOrCreateArray(n4jsSection, PackageJsonProperties.REQUIRED_RUNTIME_LIBRARIES);
        for (final String library : runtimeLibraries) {
          requiredRuntimeLibrariesList.getElements().add(JSONModelUtils.createStringLiteral(library));
        }
      }
    };
  }
  
  /**
   * Returns a semantic modification that sets the name of the extended runtime environment to the given string.
   * 
   * @param runtimeEnvironment
   * 			The new extended runtime environment.
   */
  public static IJSONDocumentModification setExtendedRuntimeEnvironment(final String runtimeEnvironment) {
    return new IJSONDocumentModification() {
      @Override
      public void apply(final JSONDocument document) {
        final JSONObject root = PackageJsonModificationProvider.getDocumentRoot(document);
        final JSONObject n4jsSection = PackageJsonModificationProvider.getOrCreateN4JSSection(root);
        JSONModelUtils.setProperty(n4jsSection, PackageJsonProperties.EXTENDED_RUNTIME_ENVIRONMENT.name, runtimeEnvironment);
      }
    };
  }
  
  /**
   * Returns change instance to set the ProjectType to the given value.
   * 
   * @param manifestResource The manifest resource
   * @param runtimeEnvironment The runtime environment to set
   * @param projectDescription The project description object of the manifest
   */
  public static IJSONDocumentModification setProjectType(final String projectType) {
    return new IJSONDocumentModification() {
      @Override
      public void apply(final JSONDocument document) {
        final JSONObject root = PackageJsonModificationProvider.getDocumentRoot(document);
        final JSONObject n4jsSection = PackageJsonModificationProvider.getOrCreateN4JSSection(root);
        JSONModelUtils.setProperty(n4jsSection, PackageJsonProperties.PROJECT_TYPE.name, projectType);
      }
    };
  }
  
  private static JSONObject getDocumentRoot(final JSONDocument document) {
    final JSONValue content = document.getContent();
    if ((!(content instanceof JSONObject))) {
      URI _uRI = document.eResource().getURI();
      String _plus = (("The given resource does not represent a valid package.json file." + 
        "Make sure the document root is a JSON object. (URI=") + _uRI);
      String _plus_1 = (_plus + ")");
      throw new IllegalArgumentException(_plus_1);
    }
    return ((JSONObject) content);
  }
  
  public static void addProjectDependency(final JSONObject root, final String projectName, final String versionConstraint) {
    JSONModelUtils.<JSONStringLiteral>setPath(root, Arrays.<String>asList(PackageJsonProperties.DEPENDENCIES.name, projectName), 
      JSONModelUtils.createStringLiteral(versionConstraint));
  }
  
  private static JSONArray getOrCreateArray(final JSONObject root, final PackageJsonProperties property) {
    final Supplier<JSONValue> _function = () -> {
      return JSONModelUtils.<JSONArray>addProperty(root, property.name, JSONFactory.eINSTANCE.createJSONArray());
    };
    JSONValue _orElseGet = JSONModelUtils.getProperty(root, property.name).orElseGet(_function);
    return ((JSONArray) _orElseGet);
  }
  
  private static JSONObject getOrCreateObject(final JSONObject root, final PackageJsonProperties property) {
    final Supplier<JSONValue> _function = () -> {
      return JSONModelUtils.<JSONObject>addProperty(root, property.name, JSONFactory.eINSTANCE.createJSONObject());
    };
    JSONValue _orElseGet = JSONModelUtils.getProperty(root, property.name).orElseGet(_function);
    return ((JSONObject) _orElseGet);
  }
  
  private static JSONObject getOrCreateN4JSSection(final JSONObject root) {
    return PackageJsonModificationProvider.getOrCreateObject(root, PackageJsonProperties.N4JS);
  }
}
