/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.base.Throwables;
import com.google.inject.Inject;
import java.util.function.BooleanSupplier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * Provides some common base functionality used across all processors (e.g. logging). See {@link ASTProcessor} for more
 * details on processors and post-processing of {@link N4JSResource}s.
 */
@SuppressWarnings("all")
abstract class AbstractProcessor {
  private static final boolean DEBUG_LOG = false;
  
  private static final boolean DEBUG_LOG_RESULT = false;
  
  private static final boolean DEBUG_RIGID = false;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private OperationCanceledManager operationCanceledManager;
  
  /**
   * Convenience method. Same as {@link OperationCanceledManager#checkCanceled(CancelIndicator)}, using the cancel
   * indicator of the given rule environment.
   */
  protected void checkCanceled(final RuleEnvironment G) {
    this.operationCanceledManager.checkCanceled(RuleEnvironmentExtensions.getCancelIndicator(G));
  }
  
  /**
   * Processors can call this method to directly invoke the 'type' judgment, i.e. invoke method {@code TypeJudgment#apply()}
   * via facade method {@link N4JSTypeSystem#use_type_judgment_from_PostProcessors(RuleEnvironment, TypableElement)
   * use_type_judgment_from_PostProcessors()}. Normally, this should only be required by {@link TypeProcessor}, so use
   * this sparingly (however, sometimes it can be helpful to avoid duplication of logic).
   */
  protected TypeRef invokeTypeJudgmentToInferType(final RuleEnvironment G, final TypableElement elem) {
    boolean _eIsProxy = elem.eIsProxy();
    if (_eIsProxy) {
      return TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
    }
    TStructMember _xifexpression = null;
    if ((elem instanceof TStructMember)) {
      _xifexpression = ((TStructMember)elem).getDefinedMember();
    }
    final TStructMember definedMember = _xifexpression;
    if (((definedMember != null) && N4JSLanguageUtils.isASTNode(elem))) {
      return this.invokeTypeJudgmentToInferType(G, definedMember);
    }
    return this.ts.use_type_judgment_from_PostProcessors(G, elem);
  }
  
  /**
   * Some special handling for async functions (including methods): we have to wrap their inner return type
   * <code>R</code> into a <code>Promise&lt;R,?></code> and use that as their actual, outer return type. This means
   * for async functions, the types builder will create a <code>TFunction</code> with the inner return type and during
   * post-processing this method will change that return type to a <code>Promise</code> (only the return type of the
   * TFunction in the types model is changed; the declared return type in the AST remains unchanged).
   * <p>
   * In addition, a return type of <code>void</code> will be replaced by <code>undefined</code>, i.e. will produce an
   * outer return type of <code>Promise&lt;undefined,?></code>. This will be taken care of by method
   * {@link TypeUtils#createPromiseTypeRef(BuiltInTypeScope,TypeArgument,TypeArgument)}.
   * <p>
   * NOTES:
   * <ol>
   * <li>normally, this wrapping could easily be done in the types builder, but because we have to check if the inner
   * return type is <code>void</code> we need to resolve proxies, which is not allowed in the types builder.
   * </ol>
   */
  protected void handleAsyncFunctionDefinition(final RuleEnvironment G, final FunctionDefinition funDef, final ASTMetaInfoCache cache) {
    boolean _isAsync = funDef.isAsync();
    if (_isAsync) {
      final Type tFunction = funDef.getDefinedType();
      if ((tFunction instanceof TFunction)) {
        final TypeRef innerReturnTypeRef = ((TFunction)tFunction).getReturnTypeRef();
        if (((innerReturnTypeRef != null) && (!(innerReturnTypeRef instanceof DeferredTypeRef)))) {
          final BuiltInTypeScope scope = RuleEnvironmentExtensions.getBuiltInTypeScope(G);
          boolean _isPromise = TypeUtils.isPromise(innerReturnTypeRef, scope);
          boolean _not = (!_isPromise);
          if (_not) {
            final ParameterizedTypeRef outerReturnTypeRef = TypeUtils.createPromiseTypeRef(scope, innerReturnTypeRef, null);
            final Procedure0 _function = () -> {
              ((TFunction)tFunction).setReturnTypeRef(outerReturnTypeRef);
            };
            EcoreUtilN4.doWithDeliver(false, _function, tFunction);
          }
        }
      }
    }
  }
  
  /**
   * Some special handling for generator functions (including methods): we have to wrap their inner return type
   * <code>R</code> into a {@code Generator<R,TReturn,TNext>} and use that as their actual, outer return type. This means
   * for generator functions, the types builder will create a <code>TFunction</code> with the inner return type and during
   * post-processing this method will change that return type to a <code>Generator</code> (only the return type of the
   * TFunction in the types model is changed; the declared return type in the AST remains unchanged).
   * <p>
   * In addition, a return type of <code>void</code> will be replaced by <code>undefined</code>, i.e. will produce an
   * outer return type of {@code Generator<undefined,undefined,TNext>}. This will be taken care of by method
   * {@link TypeUtils#createGeneratorTypeRef(BuiltInTypeScope,FunctionDefinition)}.
   * <p>
   * NOTES:
   * <ol>
   * <li>normally, this wrapping could easily be done in the types builder, but because we have to check if the inner
   * return type is <code>void</code> we need to resolve proxies, which is not allowed in the types builder.
   * </ol>
   */
  protected void handleGeneratorFunctionDefinition(final RuleEnvironment G, final FunctionDefinition funDef, final ASTMetaInfoCache cache) {
    boolean _isGenerator = funDef.isGenerator();
    if (_isGenerator) {
      final Type tFunction = funDef.getDefinedType();
      if ((tFunction instanceof TFunction)) {
        final TypeRef innerReturnTypeRef = ((TFunction)tFunction).getReturnTypeRef();
        if (((innerReturnTypeRef != null) && (!(innerReturnTypeRef instanceof DeferredTypeRef)))) {
          final BuiltInTypeScope scope = RuleEnvironmentExtensions.getBuiltInTypeScope(G);
          boolean _isGenerator_1 = TypeUtils.isGenerator(innerReturnTypeRef, scope);
          boolean _not = (!_isGenerator_1);
          if (_not) {
            final ParameterizedTypeRef outerReturnTypeRef = TypeUtils.createGeneratorTypeRef(scope, funDef);
            final Procedure0 _function = () -> {
              ((TFunction)tFunction).setReturnTypeRef(outerReturnTypeRef);
            };
            EcoreUtilN4.doWithDeliver(false, _function, tFunction);
          }
        }
      }
    }
  }
  
  protected static String getObjectInfo(final EObject obj) {
    String _xifexpression = null;
    if ((obj == null)) {
      _xifexpression = "<null>";
    } else {
      String _xifexpression_1 = null;
      if ((obj instanceof IdentifierRef)) {
        String _tokenText = NodeModelUtils.getTokenText(NodeModelUtils.findActualNodeFor(obj));
        String _plus = ("IdentifierRef \"" + _tokenText);
        _xifexpression_1 = (_plus + "\"");
      } else {
        String _xblockexpression = null;
        {
          final String name = AbstractProcessor.getName(obj);
          String _xifexpression_2 = null;
          if ((name != null)) {
            String _name = obj.eClass().getName();
            String _plus_1 = (_name + " \"");
            String _plus_2 = (_plus_1 + name);
            _xifexpression_2 = (_plus_2 + "\"");
          } else {
            _xifexpression_2 = obj.eClass().getName();
          }
          _xblockexpression = _xifexpression_2;
        }
        _xifexpression_1 = _xblockexpression;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  protected static String getName(final EObject obj) {
    String _switchResult = null;
    boolean _matched = false;
    if (obj instanceof NamedElement) {
      _matched=true;
      _switchResult = ((NamedElement)obj).getName();
    }
    if (!_matched) {
      if (obj instanceof IdentifiableElement) {
        _matched=true;
        _switchResult = ((IdentifiableElement)obj).getName();
      }
    }
    return _switchResult;
  }
  
  protected static void log(final int indentLevel, final TypeRef result) {
    boolean _isDEBUG_LOG = AbstractProcessor.isDEBUG_LOG();
    boolean _not = (!_isDEBUG_LOG);
    if (_not) {
      return;
    }
    AbstractProcessor.log(indentLevel, result.getTypeRefAsString());
  }
  
  protected static void log(final int indentLevel, final EObject astNode, final ASTMetaInfoCache cache) {
    boolean _isDEBUG_LOG = AbstractProcessor.isDEBUG_LOG();
    boolean _not = (!_isDEBUG_LOG);
    if (_not) {
      return;
    }
    boolean _isTypableNode = N4JSLanguageUtils.isTypableNode(astNode);
    if (_isTypableNode) {
      final TypeRef result = cache.getTypeFailSafe(((TypableElement) astNode));
      String _xifexpression = null;
      if ((result != null)) {
        _xifexpression = result.getTypeRefAsString();
      } else {
        _xifexpression = "*** MISSING ***";
      }
      final String resultStr = _xifexpression;
      String _objectInfo = AbstractProcessor.getObjectInfo(astNode);
      String _plus = (_objectInfo + " ");
      String _plus_1 = (_plus + resultStr);
      AbstractProcessor.log(indentLevel, _plus_1);
    } else {
      AbstractProcessor.log(indentLevel, AbstractProcessor.getObjectInfo(astNode));
    }
    EList<EObject> _eContents = astNode.eContents();
    for (final EObject childNode : _eContents) {
      AbstractProcessor.log((indentLevel + 1), childNode, cache);
    }
  }
  
  protected static void log(final int indentLevel, final String msg) {
    boolean _isDEBUG_LOG = AbstractProcessor.isDEBUG_LOG();
    boolean _not = (!_isDEBUG_LOG);
    if (_not) {
      return;
    }
    String _indent = AbstractProcessor.indent(indentLevel);
    String _plus = (_indent + msg);
    InputOutput.<String>println(_plus);
  }
  
  protected static void logErr(final String msg) {
    System.out.flush();
    System.err.println(msg);
    System.err.flush();
  }
  
  protected static Throwable logException(final Throwable th) {
    th.printStackTrace();
    return th;
  }
  
  protected static void assertTrueIfRigid(final ASTMetaInfoCache cache, final String message, final BooleanSupplier check) {
    boolean _isDEBUG_RIGID = AbstractProcessor.isDEBUG_RIGID();
    if (_isDEBUG_RIGID) {
      AbstractProcessor.assertTrueIfRigid(cache, message, check.getAsBoolean());
    }
  }
  
  protected static void assertTrueIfRigid(final ASTMetaInfoCache cache, final String message, final boolean actual) {
    if ((AbstractProcessor.isDEBUG_RIGID() && (!actual))) {
      final Error e = new Error(message);
      boolean _hasBrokenAST = cache.hasBrokenAST();
      boolean _not = (!_hasBrokenAST);
      if (_not) {
        UtilN4.<Error>reportError(e);
      }
      Throwables.throwIfUnchecked(e);
      throw new RuntimeException(e);
    }
  }
  
  protected static boolean isDEBUG_LOG() {
    return AbstractProcessor.DEBUG_LOG;
  }
  
  protected static boolean isDEBUG_LOG_RESULT() {
    return AbstractProcessor.DEBUG_LOG_RESULT;
  }
  
  protected static boolean isDEBUG_RIGID() {
    return AbstractProcessor.DEBUG_RIGID;
  }
  
  protected static String indent(final int indentLevel) {
    final Function1<Integer, String> _function = (Integer it) -> {
      return "    ";
    };
    return IterableExtensions.join(IterableExtensions.<Integer, String>map(new ExclusiveRange(0, indentLevel, true), _function));
  }
}
