/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.RelationalExpression;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.AbstractPolyProcessor;
import org.eclipse.n4js.postprocessing.PolyProcessor_ArrayLiteral;
import org.eclipse.n4js.postprocessing.PolyProcessor_CallExpression;
import org.eclipse.n4js.postprocessing.PolyProcessor_FunctionExpression;
import org.eclipse.n4js.postprocessing.PolyProcessor_ObjectLiteral;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.typesystem.constraints.TypeConstraint;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.DestructureHelper;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.util.CancelIndicator;

/**
 * The main poly processor responsible for typing poly expressions using a constraint-based approach.
 * <p>
 * It tells other processors which AST nodes it is responsible for (see {@link PolyProcessor#isResponsibleFor(TypableElement) isResponsibleFor()})
 * and which AST nodes are an entry point to constraint-based type inference (see {@link PolyProcessor#isEntryPoint(TypableElement) isEntryPoint()}).
 * For those "entry points" method {@link PolyProcessor#inferType(RuleEnvironment,Expression,ASTMetaInfoCache) inferType()}
 * should be invoked by the other processors (mainly the TypeProcessor).
 */
@Singleton
@SuppressWarnings("all")
class PolyProcessor extends AbstractPolyProcessor {
  @Inject
  private PolyProcessor_ArrayLiteral arrayLiteralProcessor;
  
  @Inject
  private PolyProcessor_ObjectLiteral objectLiteralProcessor;
  
  @Inject
  private PolyProcessor_FunctionExpression functionExpressionProcessor;
  
  @Inject
  private PolyProcessor_CallExpression callExpressionProcessor;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private DestructureHelper destructureHelper;
  
  @Inject
  private OperationCanceledManager operationCanceledManager;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * Tells if the given AST node's type should be inferred through constraint-based type inference. In that case,
   * no other processor is allowed to add a type for this node to the {@link ASTMetaInfoCache}!
   */
  boolean isResponsibleFor(final TypableElement astNode) {
    return (((((this.isPoly(astNode) || (((astNode instanceof Argument) && (astNode.eContainer() instanceof ParameterizedCallExpression)) && this.isPoly(astNode.eContainer()))) || (((astNode instanceof FormalParameter) && (astNode.eContainer() instanceof FunctionExpression)) && this.isPoly(astNode.eContainer()))) || (((astNode instanceof FormalParameter) && (astNode.eContainer() instanceof PropertyMethodDeclaration)) && this.isPoly(astNode.eContainer()))) || (((astNode instanceof ArrayElement) && (astNode.eContainer() instanceof ArrayLiteral)) && this.isPoly(astNode.eContainer()))) || (((astNode instanceof PropertyAssignment) && (astNode.eContainer() instanceof ObjectLiteral)) && this.isPoly(astNode.eContainer())));
  }
  
  /**
   * Tells if the given AST node is an entry point to constraint-based type inference. In that case, and only in that
   * case, method {@link #inferType(RuleEnvironment,Expression,ASTMetaInfoCache) inferType()} must be invoked for this
   * AST node.
   */
  boolean isEntryPoint(final TypableElement astNode) {
    return this.isRootPoly(astNode);
  }
  
  /**
   * Main method for inferring the type of poly expressions, i.e. for constraint-based type inference. It should be
   * invoked for all AST nodes for which method {@link #isEntryPoint(TypableElement) isEntryPoint()}
   * returns <code>true</code> (and only for such nodes!). This will ensure that this method will be called for all
   * roots of trees of nested poly expressions (including such trees that only consist of a root without children) but
   * not for the nested children.
   * <p>
   * This method, together with its delegates, is responsible for adding to the cache types for all the following
   * AST nodes:
   * <ol>
   * <li>the given root poly expression <code>rootPoly</code>,
   * <li>all nested child poly expressions,
   * <li>some nested elements that aren't expressions but closely belong to one of the above expressions, e.g. formal
   * parameters contained in a function expression (see code of {@link #isResponsibleFor(TypableElement)} for which
   * elements are included here).
   * </ol>
   * <p>
   * The overall process of constraint-based type inference is as follows:
   * <ol>
   * <li>create a new, empty {@link InferenceContext} called <code>IC</code>.
   * <li>invoke method {@link #processExpr(RuleEnvironment,Expression,TypeRef,InferenceContext,ASTMetaInfoCache) #processExpr()}
   * for the given root poly expression and all its direct and indirect child poly expressions. This will ...
   *     <ol type="a">
   *     <li>add to <code>IC</code> (i) inference variables for all types to be inferred and (ii) appropriate
   *         constraints derived from the poly expressions and their relations.
   *     <li>register <code>onSolved</code> handlers to <code>IC</code> (see below what these handlers are doing).
   *     </ol>
   * <li>solve the entire constraint system, i.e. invoke {@link #solve()} on <code>IC</code>.
   * <li>once solution is done (no matter if successful or failed) <code>IC</code> will automatically trigger the
   *     <code>onSolved</code> handlers:
   *     <ul>
   *     <li>in the success case, the handlers will use the solution in <code>IC</code> to add types to the cache for
   *         the given root poly expression and all its nested child poly expressions (and also for contained, typable
   *         elements such as fpars of function expressions).
   *     <li>in the failure case, the handlers will add fall-back types to the cache.
   *     </ul>
   * </ol>
   */
  void inferType(final RuleEnvironment G, final Expression rootPoly, final ASTMetaInfoCache cache) {
    CancelIndicator _cancelIndicator = RuleEnvironmentExtensions.getCancelIndicator(G);
    final InferenceContext infCtx = new InferenceContext(this.ts, this.tsh, this.operationCanceledManager, _cancelIndicator, G);
    boolean _doomTypeInference = this.jsVariantHelper.doomTypeInference(rootPoly);
    if (_doomTypeInference) {
      infCtx.addConstraint(TypeConstraint.FALSE);
    }
    final TypeRef expectedTypeOfPoly = this.destructureHelper.calculateExpectedType(rootPoly, G, infCtx);
    TypeRef _xifexpression = null;
    if ((expectedTypeOfPoly != null)) {
      _xifexpression = expectedTypeOfPoly;
    } else {
      TypeRef _xifexpression_1 = null;
      boolean _isProblematicCaseOfExpectedType = this.isProblematicCaseOfExpectedType(rootPoly);
      boolean _not = (!_isProblematicCaseOfExpectedType);
      if (_not) {
        _xifexpression_1 = this.ts.expectedType(G, rootPoly.eContainer(), rootPoly);
      }
      _xifexpression = _xifexpression_1;
    }
    final TypeRef expectedTypeRef = _xifexpression;
    final TypeRef typeRef = this.processExpr(G, rootPoly, expectedTypeRef, infCtx, cache);
    boolean _isVoid = TypeUtils.isVoid(typeRef);
    boolean _not_1 = (!_isVoid);
    if (_not_1) {
      if ((expectedTypeRef != null)) {
        infCtx.addConstraint(typeRef, expectedTypeRef, Variance.CO);
      }
    }
    infCtx.solve();
  }
  
  /**
   * Key method for handling poly expressions.
   * <p>
   * It has the following responsibilities:
   * <ul>
   * <li>if given expression is non-poly: simply return its type<br>
   *     (note: in <em>this</em> case this method won't process nested expressions in any way).
   * <li>if given expression is poly:
   *     <ol>
   *     <li>introduce a new inference variable to the given inference context for each type to be inferred for the
   *         given poly expression (usually only 1, but may be several, e.g. for a function expression we introduce an
   *         inference variable for the return type and each fpar),
   *     <li>add appropriate constraints to the given inference context,
   *     <li>recursively invoke this method for nested expressions (no matter if poly or non-poly).
   *     <li>register to the given inference context an <code>onSolved</code> handler that will - after the inference
   *         context will have been solved - add all required <b>final types</b> for 'expr' and its non-expression
   *         children (e.g. fpars) to the typing cache.
   *     <li>return <b>temporary type</b> of the given expression <code>expr</code>.
   *     </ol>
   * </ul>
   * IMPORTANT: the "temporary" type may contain inference variables; the "final" types must be proper, i.e. must not
   * contain any inference variables!
   */
  protected TypeRef processExpr(final RuleEnvironment G, final Expression expr, final TypeRef expectedTypeRef, final InferenceContext infCtx, final ASTMetaInfoCache cache) {
    boolean _isPoly = this.isPoly(expr);
    if (_isPoly) {
      TypeRef _switchResult = null;
      boolean _matched = false;
      if (expr instanceof ArrayLiteral) {
        _matched=true;
        _switchResult = this.arrayLiteralProcessor.processArrayLiteral(G, ((ArrayLiteral)expr), expectedTypeRef, infCtx, cache);
      }
      if (!_matched) {
        if (expr instanceof ObjectLiteral) {
          _matched=true;
          _switchResult = this.objectLiteralProcessor.processObjectLiteral(G, ((ObjectLiteral)expr), expectedTypeRef, infCtx, cache);
        }
      }
      if (!_matched) {
        if (expr instanceof FunctionExpression) {
          _matched=true;
          _switchResult = this.functionExpressionProcessor.processFunctionExpression(G, ((FunctionExpression)expr), expectedTypeRef, infCtx, cache);
        }
      }
      if (!_matched) {
        if (expr instanceof ParameterizedCallExpression) {
          _matched=true;
          _switchResult = this.callExpressionProcessor.processCallExpression(G, ((ParameterizedCallExpression)expr), expectedTypeRef, infCtx, cache);
        }
      }
      if (!_matched) {
        throw new IllegalArgumentException(("missing case in #processExpr() for poly expression: " + expr));
      }
      return _switchResult;
    } else {
      final TypeRef result = this.ts.type(G, expr);
      return result;
    }
  }
  
  /**
   * Returns true if we are not allowed to ask for the expected type of 'node', because this would lead to illegal
   * forward references (temporary).
   */
  private boolean isProblematicCaseOfExpectedType(final EObject node) {
    EObject _eContainer = null;
    if (node!=null) {
      _eContainer=node.eContainer();
    }
    return (_eContainer instanceof RelationalExpression);
  }
}
