/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.resource;

import com.google.inject.Singleton;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.validation.ASTStructureValidator;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

/**
 * This class performs some early pre-processing of the AST. This happens after {@link N4JSLinker lazy linking} and
 * before {@link ASTStructureValidator AST structure validation}.
 */
@Singleton
@SuppressWarnings("all")
final class N4JSPreProcessor {
  /**
   * Performs an early processing of the AST, e.g. initialization of transient helper values.
   * <b>This method assumes that it is allowed to change the AST!</b> Thus, it should be invoked from an "exec without
   * cache clear" handler, see {@code OnChangeEvictingCache#execWithoutCacheClear(N4JSResource,IUnitOfWork)}.
   */
  public void process(final Script script, final N4JSResource resource) {
    final ResourceSet resourceSet = resource.getResourceSet();
    if ((resourceSet == null)) {
      return;
    }
    final BuiltInTypeScope builtInTypes = BuiltInTypeScope.get(resourceSet);
    Iterable<EObject> _iterable = IteratorExtensions.<EObject>toIterable(resource.getScript().eAllContents());
    for (final EObject node : _iterable) {
      this.processNode(node, resource, builtInTypes);
    }
  }
  
  private void _processNode(final EObject astNode, final N4JSResource resource, final BuiltInTypeScope builtInTypes) {
  }
  
  /**
   * Support for new array type syntax:
   * <pre>
   * let arr: string[];
   * </pre>
   * and tuple syntax:
   * <pre>
   * let tup: [string, int];
   * </pre>
   */
  private void _processNode(final ParameterizedTypeRef typeRef, final N4JSResource resource, final BuiltInTypeScope builtInTypes) {
    boolean _isArrayTypeExpression = typeRef.isArrayTypeExpression();
    if (_isArrayTypeExpression) {
      typeRef.setDeclaredType(builtInTypes.getArrayType());
    } else {
      boolean _isIterableTypeExpression = typeRef.isIterableTypeExpression();
      if (_isIterableTypeExpression) {
        final int n = typeRef.getTypeArgs().size();
        if ((n < 2)) {
          typeRef.setDeclaredType(builtInTypes.getArrayType());
        } else {
          if ((n <= BuiltInTypeScope.ITERABLE_N__MAX_LEN)) {
            typeRef.setDeclaredType(builtInTypes.getIterableNType(n));
          } else {
            typeRef.setDeclaredType(builtInTypes.getIterableNType(BuiltInTypeScope.ITERABLE_N__MAX_LEN));
          }
        }
      }
    }
  }
  
  private void processNode(final EObject typeRef, final N4JSResource resource, final BuiltInTypeScope builtInTypes) {
    if (typeRef instanceof ParameterizedTypeRef) {
      _processNode((ParameterizedTypeRef)typeRef, resource, builtInTypes);
      return;
    } else if (typeRef != null) {
      _processNode(typeRef, resource, builtInTypes);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(typeRef, resource, builtInTypes).toString());
    }
  }
}
