/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.resource.packagejson;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.JSON.JSONPackage;
import org.eclipse.n4js.json.extension.IJSONResourceDescriptionExtension;
import org.eclipse.n4js.projectDescription.ProjectDependency;
import org.eclipse.n4js.projectDescription.ProjectDescription;
import org.eclipse.n4js.projectDescription.ProjectReference;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.semver.Semver.VersionNumber;
import org.eclipse.n4js.semver.model.SemverSerializer;
import org.eclipse.n4js.utils.ProjectDescriptionLoader;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.EObjectDescription;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.util.IAcceptor;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * {@link IJSONResourceDescriptionExtension} implementation that provides custom resource descriptions of
 * {@code package.json} resources.
 */
@SuppressWarnings("all")
public class PackageJsonResourceDescriptionExtension implements IJSONResourceDescriptionExtension {
  /**
   * Separator that is used to serialize multiple project identifiers as a string.
   */
  private static final String SEPARATOR = "/";
  
  /**
   * The key of the user data for retrieving the project name.
   */
  public static final String PROJECT_NAME_KEY = "projectName";
  
  /**
   * The key of the user data for retrieving the project version.
   */
  public static final String PROJECT_VERSION_KEY = "projectVersion";
  
  /**
   * The key of the user data for retrieving the project type as a string.
   */
  private static final String PROJECT_TYPE_KEY = "projectType";
  
  /**
   * Key for the project implementation ID value. {@code null} value will be mapped to empty string.
   */
  private static final String IMPLEMENTATION_ID_KEY = "implementationId";
  
  /**
   * Key for storing the test project names.
   * If a project does not have any tested projects this key will be missing from the user data.
   * The values are separated with the {@link PackageJsonResourceDescriptionExtension#SEPARATOR} character.
   */
  private static final String TESTED_PROJECT_NAMES_KEY = "testedProjectNames";
  
  /**
   * Key for storing the implemented project names.
   * If a project does not implement any projects this key will be missing from the user data.
   * The values are separated with the {@link PackageJsonResourceDescriptionExtension#SEPARATOR} character.
   */
  private static final String IMPLEMENTED_PROJECT_NAMES_KEY = "implementedProjectNames";
  
  /**
   * Key for storing the project names of all direct dependencies.
   * If a project does not have any direct projects this key will be missing from the user data.
   * The values are separated with the {@link PackageJsonResourceDescriptionExtension#SEPARATOR} character.
   */
  private static final String PROJECT_DEPENDENCY_NAMES_KEY = "projectDependencyNames";
  
  /**
   * Key for storing the project names of all provided runtime libraries.
   * If the project does not provide any runtime libraries, then this value will be omitted form the user data.
   * Multiple values are separated with the {@link PackageJsonResourceDescriptionExtension#SEPARATOR} character.
   */
  private static final String PROVIDED_RUNTIME_LIBRARY_NAMES_KEY = "providedRuntimeLibraryNames";
  
  /**
   * Key for storing the project names of all required runtime libraries.
   * If the project does not have any runtime library requirement, this value will not be present in the user data.
   * Multiple values will be joined with the {@link PackageJsonResourceDescriptionExtension#SEPARATOR} separator.
   */
  private static final String REQUIRED_RUNTIME_LIBRARY_NAMES_KEY = "requiredRuntimeLibraryNames";
  
  /**
   * Key for storing the unique project identifier of the extended runtime environment. If the project does not
   * extend any runtime environment, then this value will not exist in the user data.
   */
  private static final String EXTENDED_RUNTIME_ENVIRONMENT_NAME_KEY = "extendedRuntimeEnvironmentName";
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private IQualifiedNameProvider qualifiedNameProvider;
  
  @Inject
  private ProjectDescriptionLoader projectDescriptionLoader;
  
  private static final Logger LOGGER = Logger.getLogger(PackageJsonResourceDescriptionExtension.class);
  
  @Override
  public boolean isToBeBuilt(final URI uri, final Resource resource) {
    boolean _isPackageJSON = PackageJsonResourceDescriptionExtension.isPackageJSON(uri);
    boolean _not = (!_isPackageJSON);
    if (_not) {
      return false;
    }
    int _depthOfLocation = this.n4jsCore.getDepthOfLocation(uri);
    return (_depthOfLocation == 1);
  }
  
  @Override
  public QualifiedName getFullyQualifiedName(final EObject obj) {
    boolean _isPackageJSON = PackageJsonResourceDescriptionExtension.isPackageJSON(obj);
    boolean _not = (!_isPackageJSON);
    if (_not) {
      return null;
    }
    return this.qualifiedNameProvider.getFullyQualifiedName(obj);
  }
  
  @Override
  public boolean isAffected(final Collection<IResourceDescription.Delta> deltas, final IResourceDescription candidate, final IResourceDescriptions context) {
    boolean _isPackageJSON = PackageJsonResourceDescriptionExtension.isPackageJSON(candidate);
    boolean _not = (!_isPackageJSON);
    if (_not) {
      return false;
    }
    final Function1<IResourceDescription.Delta, URI> _function = (IResourceDescription.Delta it) -> {
      return it.getUri();
    };
    final Function1<URI, Boolean> _function_1 = (URI it) -> {
      return Boolean.valueOf(PackageJsonResourceDescriptionExtension.isPackageJSON(it));
    };
    final Function1<URI, String> _function_2 = (URI it) -> {
      return PackageJsonResourceDescriptionExtension.getProjectNameFromPackageJSONUri(it);
    };
    final Set<String> changedProjectNames = IterableExtensions.<String>toSet(IterableExtensions.<URI, String>map(IterableExtensions.<URI>filter(IterableExtensions.<IResourceDescription.Delta, URI>map(deltas, _function), _function_1), _function_2));
    final LinkedList<String> referencedProjectNames = CollectionLiterals.<String>newLinkedList();
    final Consumer<IEObjectDescription> _function_3 = (IEObjectDescription it) -> {
      referencedProjectNames.addAll(PackageJsonResourceDescriptionExtension.getTestedProjectNames(it));
      referencedProjectNames.addAll(PackageJsonResourceDescriptionExtension.getImplementedProjectNames(it));
      referencedProjectNames.addAll(PackageJsonResourceDescriptionExtension.getProjectDependencyNames(it));
      referencedProjectNames.addAll(PackageJsonResourceDescriptionExtension.getProvidedRuntimeLibraryNames(it));
      referencedProjectNames.addAll(PackageJsonResourceDescriptionExtension.getRequiredRuntimeLibraryNames(it));
      final String extRuntimeEnvironmentId = PackageJsonResourceDescriptionExtension.getExtendedRuntimeEnvironmentName(it);
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(extRuntimeEnvironmentId);
      boolean _not_1 = (!_isNullOrEmpty);
      if (_not_1) {
        referencedProjectNames.add(extRuntimeEnvironmentId);
      }
    };
    candidate.getExportedObjectsByType(JSONPackage.Literals.JSON_DOCUMENT).forEach(_function_3);
    for (final String referencedProjectName : referencedProjectNames) {
      boolean _contains = changedProjectNames.contains(referencedProjectName);
      if (_contains) {
        return true;
      }
    }
    return false;
  }
  
  @Override
  public void createJSONDocumentDescriptions(final JSONDocument document, final IAcceptor<IEObjectDescription> acceptor) {
    final QualifiedName qualifiedName = this.getFullyQualifiedName(document);
    if ((qualifiedName == null)) {
      return;
    }
    Resource _eResource = null;
    if (document!=null) {
      _eResource=document.eResource();
    }
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    URI _trimSegments = null;
    if (_uRI!=null) {
      _trimSegments=_uRI.trimSegments(1);
    }
    final URI projectLocation = _trimSegments;
    if ((projectLocation == null)) {
      PackageJsonResourceDescriptionExtension.LOGGER.error("creation of EObjectDescriptions failed: cannot derive project location from document");
      return;
    }
    final ProjectDescription description = this.projectDescriptionLoader.loadProjectDescriptionAtLocation(projectLocation, document);
    if ((description == null)) {
      PackageJsonResourceDescriptionExtension.LOGGER.error(("creation of EObjectDescriptions failed: cannot load project description at location: " + projectLocation));
      return;
    }
    final Map<String, String> userData = this.createProjectDescriptionUserData(description);
    EObjectDescription _eObjectDescription = new EObjectDescription(qualifiedName, document, userData);
    acceptor.accept(_eObjectDescription);
  }
  
  /**
   * Creates the user data of a {@link ProjectDescription} {@link IEObjectDescription}.
   */
  private Map<String, String> createProjectDescriptionUserData(final ProjectDescription it) {
    final ImmutableMap.Builder<String, String> builder = ImmutableMap.<String, String>builder();
    StringConcatenation _builder = new StringConcatenation();
    ProjectType _projectType = it.getProjectType();
    _builder.append(_projectType);
    builder.put(PackageJsonResourceDescriptionExtension.PROJECT_TYPE_KEY, _builder.toString());
    builder.put(PackageJsonResourceDescriptionExtension.PROJECT_NAME_KEY, Strings.nullToEmpty(it.getProjectName()));
    builder.put(PackageJsonResourceDescriptionExtension.IMPLEMENTATION_ID_KEY, Strings.nullToEmpty(it.getImplementationId()));
    final VersionNumber vers = it.getProjectVersion();
    if ((vers != null)) {
      final String versionStr = SemverSerializer.serialize(vers);
      builder.put(PackageJsonResourceDescriptionExtension.PROJECT_VERSION_KEY, versionStr);
    }
    final EList<ProjectReference> testedProjects = it.getTestedProjects();
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(testedProjects);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      builder.put(PackageJsonResourceDescriptionExtension.TESTED_PROJECT_NAMES_KEY, PackageJsonResourceDescriptionExtension.asString(testedProjects));
    }
    final EList<ProjectReference> implementedProjects = it.getImplementedProjects();
    boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(implementedProjects);
    boolean _not_1 = (!_isNullOrEmpty_1);
    if (_not_1) {
      builder.put(PackageJsonResourceDescriptionExtension.IMPLEMENTED_PROJECT_NAMES_KEY, PackageJsonResourceDescriptionExtension.asString(implementedProjects));
    }
    final EList<ProjectDependency> projectDependencies = it.getProjectDependencies();
    boolean _isNullOrEmpty_2 = IterableExtensions.isNullOrEmpty(projectDependencies);
    boolean _not_2 = (!_isNullOrEmpty_2);
    if (_not_2) {
      builder.put(PackageJsonResourceDescriptionExtension.PROJECT_DEPENDENCY_NAMES_KEY, PackageJsonResourceDescriptionExtension.asString(projectDependencies));
    }
    final EList<ProjectReference> providedRuntimeLibraries = it.getProvidedRuntimeLibraries();
    boolean _isNullOrEmpty_3 = IterableExtensions.isNullOrEmpty(providedRuntimeLibraries);
    boolean _not_3 = (!_isNullOrEmpty_3);
    if (_not_3) {
      builder.put(PackageJsonResourceDescriptionExtension.PROVIDED_RUNTIME_LIBRARY_NAMES_KEY, PackageJsonResourceDescriptionExtension.asString(providedRuntimeLibraries));
    }
    final EList<ProjectReference> requiredRuntimeLibraries = it.getRequiredRuntimeLibraries();
    boolean _isNullOrEmpty_4 = IterableExtensions.isNullOrEmpty(requiredRuntimeLibraries);
    boolean _not_4 = (!_isNullOrEmpty_4);
    if (_not_4) {
      builder.put(PackageJsonResourceDescriptionExtension.REQUIRED_RUNTIME_LIBRARY_NAMES_KEY, PackageJsonResourceDescriptionExtension.asString(requiredRuntimeLibraries));
    }
    final ProjectReference extRuntimeEnvironment = it.getExtendedRuntimeEnvironment();
    if ((extRuntimeEnvironment != null)) {
      builder.put(PackageJsonResourceDescriptionExtension.EXTENDED_RUNTIME_ENVIRONMENT_NAME_KEY, PackageJsonResourceDescriptionExtension.asString(Collections.<ProjectReference>singleton(it.getExtendedRuntimeEnvironment())));
    }
    return builder.build();
  }
  
  /**
   * Optionally returns with the project type extracted from the user data of the given EObject description argument.
   */
  public static ProjectType getProjectType(final IEObjectDescription it) {
    if ((it == null)) {
      return null;
    }
    final String typeLiteral = it.getUserData(PackageJsonResourceDescriptionExtension.PROJECT_TYPE_KEY);
    if ((typeLiteral == null)) {
      return null;
    }
    return ProjectType.get(typeLiteral);
  }
  
  /**
   * Optionally returns with the project name extracted from the user data of the given EObject description argument.
   */
  public static String getProjectName(final IEObjectDescription it) {
    if ((it == null)) {
      return null;
    }
    return it.getUserData(PackageJsonResourceDescriptionExtension.PROJECT_NAME_KEY);
  }
  
  /**
   * Returns with a collection of distinct IDs of the tested projects. Never returns with {@code null}.
   */
  public static Set<String> getTestedProjectNames(final IEObjectDescription it) {
    return PackageJsonResourceDescriptionExtension.getProjectNamesUserDataOf(it, PackageJsonResourceDescriptionExtension.TESTED_PROJECT_NAMES_KEY);
  }
  
  /**
   * Returns with a collection of distinct IDs of the implemented projects. Never returns with {@code null}.
   */
  public static Set<String> getImplementedProjectNames(final IEObjectDescription it) {
    return PackageJsonResourceDescriptionExtension.getProjectNamesUserDataOf(it, PackageJsonResourceDescriptionExtension.IMPLEMENTED_PROJECT_NAMES_KEY);
  }
  
  /**
   * Returns with a collection of distinct IDs of the project dependencies. Never returns with {@code null}.
   */
  public static Set<String> getProjectDependencyNames(final IEObjectDescription it) {
    return PackageJsonResourceDescriptionExtension.getProjectNamesUserDataOf(it, PackageJsonResourceDescriptionExtension.PROJECT_DEPENDENCY_NAMES_KEY);
  }
  
  /**
   * Returns with a collection of distinct IDs of the provided runtime libraries. Never returns with {@code null}.
   */
  public static Set<String> getProvidedRuntimeLibraryNames(final IEObjectDescription it) {
    return PackageJsonResourceDescriptionExtension.getProjectNamesUserDataOf(it, PackageJsonResourceDescriptionExtension.PROVIDED_RUNTIME_LIBRARY_NAMES_KEY);
  }
  
  /**
   * Returns with a collection of distinct IDs of the required runtime libraries. Never returns with {@code null}.
   */
  public static Set<String> getRequiredRuntimeLibraryNames(final IEObjectDescription it) {
    return PackageJsonResourceDescriptionExtension.getProjectNamesUserDataOf(it, PackageJsonResourceDescriptionExtension.REQUIRED_RUNTIME_LIBRARY_NAMES_KEY);
  }
  
  /**
   * Returns with the ID of the extended runtime environment. May return with {@code null} if argument is {@code null}
   * or if the value of the user data key is {@code null}. In a nutshell, if a project does not extend a RE.
   */
  public static String getExtendedRuntimeEnvironmentName(final IEObjectDescription it) {
    if ((it == null)) {
      return null;
    }
    return it.getUserData(PackageJsonResourceDescriptionExtension.EXTENDED_RUNTIME_ENVIRONMENT_NAME_KEY);
  }
  
  /**
   * Returns with a collection of distinct project IDs extracted from the user data. Never returns with {@code null}.
   */
  private static Set<String> getProjectNamesUserDataOf(final IEObjectDescription it, final String key) {
    if ((it == null)) {
      return CollectionLiterals.<String>emptySet();
    }
    return IterableExtensions.<String>toSet(((Iterable<String>)Conversions.doWrapArray(Strings.nullToEmpty(it.getUserData(key)).split(PackageJsonResourceDescriptionExtension.SEPARATOR))));
  }
  
  private static String asString(final Iterable<? extends ProjectReference> it) {
    final Function1<ProjectReference, String> _function = (ProjectReference it_1) -> {
      return it_1.getProjectName();
    };
    return IterableExtensions.join(IterableExtensions.<String>filterNull(IterableExtensions.map(IterableExtensions.filterNull(it), _function)), PackageJsonResourceDescriptionExtension.SEPARATOR);
  }
  
  /**
   * Returns with the projectName of an N4JS project by appending the second segment from the end of a N4JS manifest URI argument.
   * This method only works for N4JS manifest URIs and throws {@link IllegalArgumentException} for all other URIs.
   * Since this method accepts only N4JS manifest URIs it is guaranteed to get the container project name as the second URI
   * segment from the end. We cannot simply grab and return with the first segment as the project name, because external
   * projects have a file URI with an absolute path that can be any arbitrary location on the file system.
   * 
   * The ultimate solution would be to look up the container N4JS project from the nested URI argument and simply get
   * the project ID of the project but due to plug-in dependency issues N4JS core service is not available from here.
   */
  private static String getProjectNameFromPackageJSONUri(final URI uri) {
    boolean _isPackageJSON = PackageJsonResourceDescriptionExtension.isPackageJSON(uri);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Expected URI with ");
    _builder.append(N4JSGlobals.PACKAGE_JSON);
    _builder.append(" as last segment. Was: ");
    _builder.append(uri);
    Preconditions.checkArgument(_isPackageJSON, _builder);
    int _segmentCount = uri.segmentCount();
    int _minus = (_segmentCount - 2);
    return uri.segment(_minus);
  }
  
  private static boolean isPackageJSON(final IResourceDescription desc) {
    return ((desc != null) && PackageJsonResourceDescriptionExtension.isPackageJSON(desc.getURI()));
  }
  
  private static boolean isPackageJSON(final EObject obj) {
    return ((obj != null) && PackageJsonResourceDescriptionExtension.isPackageJSON(obj.eResource()));
  }
  
  private static boolean isPackageJSON(final Resource res) {
    return ((res != null) && PackageJsonResourceDescriptionExtension.isPackageJSON(res.getURI()));
  }
  
  private static boolean isPackageJSON(final URI uri) {
    return ((uri != null) && Objects.equal(uri.lastSegment(), N4JSGlobals.PACKAGE_JSON));
  }
}
