/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.ts.typeRefs.StructuralTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.versions.VersionableUtils;
import org.eclipse.n4js.utils.Log;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A types builder to create and initialize {@link TMigration} instances based on {@link FunctionDeclaration}s.
 */
@Log
@SuppressWarnings("all")
public class N4IDLMigrationTypesBuilder {
  /**
   * Initializes a {@link TMigration} instance based on a {@link FunctionDeclaration}.
   * 
   * This method assumes that all {@link TFunction} related attributes of {@code tMigration} were already
   * initialized appropriately.
   * 
   * This method also assumes that {@code functionDecl} is a migration (and therefore annotated as {@code @Migration}.
   */
  public void initialiseTMigration(final FunctionDeclaration functionDecl, final TMigration tMigration, final boolean preLinkingPhase) {
    final TAnnotation migrationAnno = AnnotationDefinition.MIGRATION.getAnnotation(tMigration);
    if ((!preLinkingPhase)) {
      tMigration.getSourceTypeRefs().addAll(N4IDLMigrationTypesBuilder.computeSourceTypeRefs(tMigration));
      tMigration.getTargetTypeRefs().addAll(N4IDLMigrationTypesBuilder.computeTargetTypeRefs(tMigration));
    }
    int _length = ((Object[])Conversions.unwrapArray(migrationAnno.getArgs(), Object.class)).length;
    boolean _equals = (_length == 2);
    if (_equals) {
      try {
        final Integer sourceVersion = Integer.valueOf(Integer.parseInt(migrationAnno.getArgs().get(0).getArgAsString()));
        final Integer targetVersion = Integer.valueOf(Integer.parseInt(migrationAnno.getArgs().get(1).getArgAsString()));
        tMigration.setSourceVersion((sourceVersion).intValue());
        tMigration.setTargetVersion((targetVersion).intValue());
        tMigration.setHasDeclaredSourceAndTargetVersion(true);
      } catch (final Throwable _t) {
        if (_t instanceof NumberFormatException) {
          final NumberFormatException e = (NumberFormatException)_t;
          String _name = tMigration.getName();
          String _plus = ("Failed to infer source/target version for migration " + _name);
          String _plus_1 = (_plus + " in file ");
          URI _uRI = tMigration.eResource().getURI();
          String _plus_2 = (_plus_1 + _uRI);
          N4IDLMigrationTypesBuilder.logger.error(_plus_2, e);
          tMigration.setSourceVersion(0);
          tMigration.setSourceVersion(0);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } else {
      tMigration.setSourceVersion(N4IDLMigrationTypesBuilder.computeVersion(tMigration.getSourceTypeRefs()));
      tMigration.setTargetVersion(N4IDLMigrationTypesBuilder.computeVersion(tMigration.getTargetTypeRefs()));
      tMigration.setHasDeclaredSourceAndTargetVersion(false);
    }
  }
  
  /**
   * Returns a new instance of {@link TMigration}.
   */
  public TMigration createTMigration() {
    return TypesFactory.eINSTANCE.createTMigration();
  }
  
  /**
   * Computes the list of source types of the given migration.
   */
  private static List<TypeRef> computeSourceTypeRefs(final TMigration migration) {
    EList<TFormalParameter> _fpars = migration.getFpars();
    boolean _tripleEquals = (null == _fpars);
    if (_tripleEquals) {
      return Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList());
    }
    if (((!migration.getFpars().isEmpty()) && (null == IterableExtensions.<TFormalParameter>head(migration.getFpars()).getName()))) {
      return Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList());
    }
    final Function1<TFormalParameter, TypeRef> _function = (TFormalParameter fpar) -> {
      return fpar.getTypeRef();
    };
    return IterableExtensions.<TypeRef>toList(ListExtensions.<TFormalParameter, TypeRef>map(migration.getFpars(), _function));
  }
  
  /**
   * Computes the list of target types of the given migration.
   */
  private static List<TypeRef> computeTargetTypeRefs(final TMigration migration) {
    final TypeRef returnTypeRef = migration.getReturnTypeRef();
    if ((returnTypeRef == null)) {
      return CollectionLiterals.<TypeRef>emptyList();
    }
    if ((returnTypeRef instanceof StructuralTypeRef)) {
      final Function1<TStructField, TypeRef> _function = (TStructField f) -> {
        return f.getTypeRef();
      };
      return IterableExtensions.<TypeRef>toList(IterableExtensions.<TypeRef>filterNull(IterableExtensions.<TStructField, TypeRef>map(Iterables.<TStructField>filter(migration.getReturnTypeRef().getStructuralMembers(), TStructField.class), _function)));
    } else {
      TypeRef _returnTypeRef = migration.getReturnTypeRef();
      boolean _tripleEquals = (null == _returnTypeRef);
      if (_tripleEquals) {
        return CollectionLiterals.<TypeRef>emptyList();
      } else {
        TypeRef _returnTypeRef_1 = migration.getReturnTypeRef();
        return Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList(_returnTypeRef_1));
      }
    }
  }
  
  /**
   * Computes the version of the given {@link TypeRef}s.
   * 
   * Returns {@code 0} if none of the {@link TypeRef}s declare a version.
   * 
   * If the given {@link TypeRef}s declared multiple different versions, this method may
   * yield an inaccurate result. At this point however, there is no other sensible value
   * to populate the type model with.
   */
  private static int computeVersion(final List<TypeRef> typeRefs) {
    return VersionableUtils.getVersion(typeRefs);
  }
  
  private static final Logger logger = Logger.getLogger(N4IDLMigrationTypesBuilder.class);
}
