/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.common.unicode.CharTypes;
import org.eclipse.n4js.compileTime.CompileTimeValue;
import org.eclipse.n4js.conversion.IdentifierValueConverter;
import org.eclipse.n4js.n4JS.AbstractAnnotationList;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.ConditionalExpression;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4EnumLiteral;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4MemberAnnotationList;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.NullLiteral;
import org.eclipse.n4js.n4JS.NumericLiteral;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyAssignmentAnnotationList;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameKind;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.n4JS.UnaryExpression;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.resource.XpectAwareFileExtensionCalculator;
import org.eclipse.n4js.ts.conversions.ComputedPropertyNameValueConverter;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ExistentialTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.OptionalFieldStrategy;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TAnnotableElement;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.util.AllSuperTypesCollector;
import org.eclipse.n4js.ts.types.util.ExtendedClassesIterable;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Intended for small, static utility methods that
 * <ul>
 * <li>need both the AST and types model or code from the main n4js bundle, and can therefore not be put into
 *     {@link N4JSASTUtils} and {@link TypeUtils}.
 * <li>implement a fundamental rule or logic of the core language that defines an important part of N4JS semantics.
 * </ul>
 * 
 * @see N4JSASTUtils
 * @see TypeUtils
 */
@SuppressWarnings("all")
public class N4JSLanguageUtils {
  /**
   * See {@link ComputedPropertyNameValueConverter#SYMBOL_IDENTIFIER_PREFIX}.
   */
  public static final String SYMBOL_IDENTIFIER_PREFIX = ComputedPropertyNameValueConverter.SYMBOL_IDENTIFIER_PREFIX;
  
  /**
   * If the given function definition is asynchronous, will wrap given return type into a Promise.
   * Otherwise, returns given return type unchanged. A return type of <code>void</code> is changed to
   * <code>undefined</code>.
   */
  public static TypeRef makePromiseIfAsync(final FunctionDefinition funDef, final TypeRef returnTypeRef, final BuiltInTypeScope builtInTypeScope) {
    if (((funDef != null) && (returnTypeRef != null))) {
      boolean _isAsync = funDef.isAsync();
      if (_isAsync) {
        return TypeUtils.createPromiseTypeRef(builtInTypeScope, returnTypeRef, null);
      }
      return returnTypeRef;
    }
    return null;
  }
  
  /**
   * If the given function definition is a generator function, will wrap given return type into a Generator.
   * Otherwise, returns given return type unchanged. A return type of <code>void</code> is changed to
   * <code>undefined</code>.
   */
  public static TypeRef makeGeneratorIfGeneratorFunction(final FunctionDefinition funDef, final TypeRef returnTypeRef, final BuiltInTypeScope builtInTypeScope) {
    if (((funDef != null) && (returnTypeRef != null))) {
      boolean _isGenerator = funDef.isGenerator();
      if (_isGenerator) {
        return TypeUtils.createGeneratorTypeRef(builtInTypeScope, funDef);
      }
      return returnTypeRef;
    }
    return null;
  }
  
  /**
   * Tells if given object is an <em>AST node</em>, i.e. contained below a {@link Script} element.
   * <p>
   * Note that it is not possible to tell AST nodes from type model elements only based on the object's type, because
   * there exist type model entities that may appear as a node in the AST (e.g. some TypeRefs, TStructField).
   */
  public static boolean isASTNode(final EObject obj) {
    Script _containerOfType = EcoreUtil2.<Script>getContainerOfType(obj, Script.class);
    return (_containerOfType != null);
  }
  
  /**
   * Tells if given expression denotes the value 'undefined'.
   */
  public static boolean isUndefinedLiteral(final RuleEnvironment G, final Expression expr) {
    if ((expr instanceof IdentifierRef)) {
      IdentifiableElement _id = ((IdentifierRef)expr).getId();
      TField _fieldUndefined = RuleEnvironmentExtensions.getGlobalObjectScope(G).getFieldUndefined();
      return (_id == _fieldUndefined);
    }
    return false;
  }
  
  /**
   * Tells if given object is a <em>type model element</em>, i.e. is contained below a {@link TModule} element.
   * <p>
   * Note that it is not possible to tell AST nodes from type model elements only based on the object's type, because
   * there exist type model entities that may appear as a node in the AST (e.g. some TypeRefs, TStructField).
   */
  public static boolean isTypeModelElement(final EObject obj) {
    TModule _containerOfType = EcoreUtil2.<TModule>getContainerOfType(obj, TModule.class);
    return (_containerOfType != null);
  }
  
  /**
   * Tells if given AST node is a typable AST node, i.e. a node that has an (actual) type that can be inferred
   * using the type system. When <code>true</code> is returned, the given AST node can safely be casted to
   * {@link TypableElement}.
   * <p>
   * For performance reasons, this method will simply assume {@code astNode} to be an AST node (i.e. contained below
   * a {@link Script} element) and will not check this again.
   */
  public static boolean isTypableNode(final EObject astNode) {
    return ((astNode instanceof TypableElement) && (!(astNode instanceof AbstractAnnotationList)));
  }
  
  public static boolean isIdentifiableSubtree(final EObject astNode) {
    return ((((astNode instanceof IdentifiableElement) || (N4JSLanguageUtils.getDefinedTypeModelElement(astNode) instanceof IdentifiableElement)) || (astNode instanceof FunctionDeclaration)) || (astNode instanceof N4ClassDeclaration));
  }
  
  public static boolean isTypeModelElementDefiningASTNode(final EObject astNode) {
    return (((((((astNode instanceof ExportedVariableDeclaration) || (astNode instanceof TypeDefiningElement)) || ((astNode instanceof N4MemberDeclaration) && (!(astNode instanceof N4MemberAnnotationList)))) || ((astNode instanceof PropertyAssignment) && (!(astNode instanceof PropertyAssignmentAnnotationList)))) || (astNode instanceof FormalParameter)) || (astNode instanceof TStructMember)) || (astNode instanceof N4EnumLiteral));
  }
  
  public static EObject getDefinedTypeModelElement(final EObject astNode) {
    IdentifiableElement _switchResult = null;
    boolean _matched = false;
    if (astNode instanceof ExportedVariableDeclaration) {
      _matched=true;
      _switchResult = ((ExportedVariableDeclaration)astNode).getDefinedVariable();
    }
    if (!_matched) {
      if (astNode instanceof PropertyMethodDeclaration) {
        _matched=true;
        _switchResult = ((PropertyMethodDeclaration)astNode).getDefinedMember();
      }
    }
    if (!_matched) {
      if (astNode instanceof TypeDefiningElement) {
        _matched=true;
        _switchResult = ((TypeDefiningElement)astNode).getDefinedType();
      }
    }
    if (!_matched) {
      if (astNode instanceof N4MemberDeclaration) {
        if ((!(astNode instanceof N4MemberAnnotationList))) {
          _matched=true;
          _switchResult = ((N4MemberDeclaration)astNode).getDefinedTypeElement();
        }
      }
    }
    if (!_matched) {
      if (astNode instanceof PropertyAssignment) {
        if ((!(astNode instanceof PropertyAssignmentAnnotationList))) {
          _matched=true;
          _switchResult = ((PropertyAssignment)astNode).getDefinedMember();
        }
      }
    }
    if (!_matched) {
      if (astNode instanceof FormalParameter) {
        _matched=true;
        _switchResult = ((FormalParameter)astNode).getDefinedTypeElement();
      }
    }
    if (!_matched) {
      if (astNode instanceof TStructMember) {
        boolean _isASTNode = N4JSLanguageUtils.isASTNode(astNode);
        if (_isASTNode) {
          _matched=true;
          _switchResult = ((TStructMember)astNode).getDefinedMember();
        }
      }
    }
    if (!_matched) {
      if (astNode instanceof N4EnumLiteral) {
        _matched=true;
        _switchResult = ((N4EnumLiteral)astNode).getDefinedLiteral();
      }
    }
    if (!_matched) {
      if (astNode instanceof TypeVariable) {
        boolean _isASTNode = N4JSLanguageUtils.isASTNode(astNode);
        if (_isASTNode) {
          _matched=true;
          _switchResult = ((TypeVariable)astNode).getDefinedTypeVariable();
        }
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns with {@code true} if the {@link TMember member} argument represents a constructor.
   * More precisely, when the argument is an instance of {@link TMethod} and its {@link TMethod#getName() name}
   * is {@code constructor}. Otherwise returns with {@code false}.
   */
  public static boolean isConstructor(final TMember it) {
    return ((it instanceof TMethod) && Objects.equal(N4JSLanguageConstants.CONSTRUCTOR, it.getName()));
  }
  
  /**
   * Returns with {@code true} if the member argument is a {@link TField} instance and the field is
   * {@link TField#isWriteable() writable}, otherwise returns with {@code false}.
   */
  public static boolean isWriteableField(final TMember m) {
    return ((m instanceof TField) && m.isWriteable());
  }
  
  /**
   * Returns with {@code true} if the member argument is a {@link TField} instance and the field is
   * <b>NOT</b> {@link TField#isWriteable() writable}, otherwise returns with {@code false}.
   */
  public static boolean isReadOnlyField(final TMember m) {
    return ((m instanceof TField) && (!m.isWriteable()));
  }
  
  /**
   * Tells if the given identifiable element is exported.
   */
  public static boolean isExported(final IdentifiableElement elem) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (elem instanceof ExportedVariableDeclaration) {
      _matched=true;
      _switchResult = true;
    }
    if (!_matched) {
      if (elem instanceof TVariable) {
        _matched=true;
        _switchResult = ((TVariable)elem).isExported();
      }
    }
    if (!_matched) {
      if (elem instanceof Type) {
        _matched=true;
        _switchResult = ((Type)elem).isExported();
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  /**
   * Is the given TFunction tagged ASYNC, and moreover does it return Promise?
   */
  public static boolean isAsync(final TFunction tfunction, final BuiltInTypeScope scope) {
    boolean _isDeclaredAsync = tfunction.isDeclaredAsync();
    if (_isDeclaredAsync) {
      TypeRef _returnTypeRef = tfunction.getReturnTypeRef();
      if ((_returnTypeRef instanceof ParameterizedTypeRef)) {
        return TypeUtils.isPromise(tfunction.getReturnTypeRef(), scope);
      }
    }
    return false;
  }
  
  /**
   * Does the given function-type denote an async function?
   * (two cases: declared type available or not, in the latter case heuristically assume Promise-returning implies async).
   * <p>
   * The declared type (ie, a TFunction) is usually but not always available.
   */
  public static boolean isAsync(final FunctionTypeExprOrRef fteor, final RuleEnvironment G) {
    final TFunction tfunction = fteor.getFunctionType();
    final BuiltInTypeScope tscope = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope;
    if ((null == tfunction)) {
      return TypeUtils.isPromise(fteor.getReturnTypeRef(), tscope);
    } else {
      return N4JSLanguageUtils.isAsync(tfunction, tscope);
    }
  }
  
  /**
   * Tells if a value of the given type can be instantiated, i.e. whether
   * <pre>
   * new value();
   * </pre>
   * is legal, given a variable {@code value} of type {@code ctorTypeRef}.
   */
  public static boolean isInstantiable(final TypeTypeRef typeTypeRef) {
    final TypeArgument typeArg = typeTypeRef.getTypeArg();
    if (((typeArg instanceof Wildcard) || (typeArg instanceof ExistentialTypeRef))) {
      return false;
    }
    final Type pseudoStaticType = ((TypeRef) typeArg).getDeclaredType();
    return ((pseudoStaticType instanceof TN4Classifier) || (pseudoStaticType instanceof TObjectPrototype));
  }
  
  /**
   * Returns the variance of the given type reference's position within its containing classifier declaration or
   * <code>null</code> if
   * <ol>
   * <li>it is located at a position where type variables of any variance may be located, i.e. a position that need
   *     not be checked (e.g. type of a local variable in a method, type of a private field),
   * <li>it is not contained in a classifier declaration,
   * <li>it is <code>null</code>, or
   * <li>some error occurred, e.g. invalid TModule, broken AST.
   * </ol>
   */
  public static Variance getVarianceOfPosition(final TypeRef typeRef) {
    final Variance v1 = N4JSLanguageUtils.getVarianceOfPositionInClassifier(typeRef);
    if (((v1 == null) || (v1 == Variance.INV))) {
      return v1;
    }
    final Variance v2 = N4JSLanguageUtils.getVarianceOfPositionRelativeToItsRoot(typeRef);
    return v1.mult(v2);
  }
  
  /**
   * Most client code should use {@link #getVarianceOfPosition(ParameterizedTypeRef)}!
   * <p>
   * Same as {@link #getVarianceOfPosition(TypeRef)}, but <b>does not take into account nesting of type references
   * within other type references.</b> This is covered by method {@link #getVarianceOfPositionRelativeToItsRoot(TypeRef)}.
   */
  public static Variance getVarianceOfPositionInClassifier(final TypeRef typeRef) {
    if ((typeRef == null)) {
      return null;
    }
    final TypeRef rootTypeRef = TypeUtils.getRootTypeRef(typeRef);
    N4ClassifierDeclaration _containerOfType = EcoreUtil2.<N4ClassifierDeclaration>getContainerOfType(rootTypeRef, N4ClassifierDeclaration.class);
    Type _definedType = null;
    if (_containerOfType!=null) {
      _definedType=_containerOfType.getDefinedType();
    }
    final TClassifier tClassifier = ((TClassifier) _definedType);
    if ((tClassifier == null)) {
      return null;
    }
    final EObject parent = rootTypeRef.eContainer();
    EObject _eContainer = null;
    if (parent!=null) {
      _eContainer=parent.eContainer();
    }
    final EObject grandParent = _eContainer;
    Variance _switchResult = null;
    boolean _matched = false;
    if (parent instanceof FormalParameter) {
      if (((((FormalParameter)parent).getDeclaredTypeRef() == rootTypeRef) && N4JSLanguageUtils.isNonPrivateMemberOf(grandParent, tClassifier))) {
        _matched=true;
        _switchResult = Variance.CONTRA;
      }
    }
    if (!_matched) {
      if (parent instanceof N4MethodDeclaration) {
        if (((((N4MethodDeclaration)parent).getReturnTypeRef() == rootTypeRef) && N4JSLanguageUtils.isNonPrivateMemberOf(parent, tClassifier))) {
          _matched=true;
          _switchResult = Variance.CO;
        }
      }
    }
    if (!_matched) {
      if (parent instanceof N4GetterDeclaration) {
        if (((((N4GetterDeclaration)parent).getDeclaredTypeRef() == rootTypeRef) && N4JSLanguageUtils.isNonPrivateMemberOf(parent, tClassifier))) {
          _matched=true;
          _switchResult = Variance.CO;
        }
      }
    }
    if (!_matched) {
      if (parent instanceof N4FieldDeclaration) {
        if (((((N4FieldDeclaration)parent).getDeclaredTypeRef() == rootTypeRef) && N4JSLanguageUtils.isNonPrivateMemberOf(parent, tClassifier))) {
          _matched=true;
          Variance _xblockexpression = null;
          {
            final TField tField = ((N4FieldDeclaration)parent).getDefinedField();
            Variance _xifexpression = null;
            boolean _isFinal = tField.isFinal();
            if (_isFinal) {
              _xifexpression = Variance.CO;
            } else {
              _xifexpression = Variance.INV;
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
    }
    if (!_matched) {
      if (parent instanceof N4ClassifierDeclaration) {
        final Function1<ParameterizedTypeRef, Boolean> _function = (ParameterizedTypeRef it) -> {
          return Boolean.valueOf((it == rootTypeRef));
        };
        boolean _exists = IterableExtensions.<ParameterizedTypeRef>exists(((N4ClassifierDeclaration)parent).getSuperClassifierRefs(), _function);
        if (_exists) {
          _matched=true;
          _switchResult = Variance.CO;
        }
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  /**
   * Most client code should use {@link #getVarianceOfPosition(ParameterizedTypeRef)}!
   * <p>
   * Returns variance of the given type reference's position relative to its root type reference as defined by
   * {@link TypeUtils#getRootTypeRef(TypeRef)}. In case of error, a best effort is made. Never returns
   * <code>null</code>.
   */
  public static Variance getVarianceOfPositionRelativeToItsRoot(final TypeRef typeRef) {
    Variance v = Variance.CO;
    TypeRef curr = typeRef;
    while ((curr != null)) {
      {
        final TypeRef parent = EcoreUtil2.<TypeRef>getContainerOfType(curr.eContainer(), TypeRef.class);
        if ((parent != null)) {
          Variance vFactor = null;
          final Type parentDeclType = parent.getDeclaredType();
          final EList<TypeArgument> parentTypeArgs = parent.getTypeArgs();
          final int parentTypeArgsSize = parentTypeArgs.size();
          for (int idx = 0; ((vFactor == null) && (idx < parentTypeArgsSize)); idx++) {
            {
              final TypeArgument arg = parentTypeArgs.get(idx);
              Variance _xifexpression = null;
              if ((arg == curr)) {
                Variance _xblockexpression = null;
                {
                  TypeVariable _xifexpression_1 = null;
                  if (((idx >= 0) && (idx < parentDeclType.getTypeVars().size()))) {
                    _xifexpression_1 = parentDeclType.getTypeVars().get(idx);
                  } else {
                    _xifexpression_1 = null;
                  }
                  final TypeVariable correspondingTypeVar = _xifexpression_1;
                  Variance _elvis = null;
                  Variance _variance = null;
                  if (correspondingTypeVar!=null) {
                    _variance=correspondingTypeVar.getVariance();
                  }
                  if (_variance != null) {
                    _elvis = _variance;
                  } else {
                    _elvis = Variance.CO;
                  }
                  final Variance incomingVariance = _elvis;
                  _xblockexpression = incomingVariance;
                }
                _xifexpression = _xblockexpression;
              } else {
                Variance _xifexpression_1 = null;
                if ((arg instanceof Wildcard)) {
                  Variance _xifexpression_2 = null;
                  TypeRef _declaredUpperBound = ((Wildcard)arg).getDeclaredUpperBound();
                  boolean _tripleEquals = (_declaredUpperBound == curr);
                  if (_tripleEquals) {
                    _xifexpression_2 = Variance.CO;
                  } else {
                    Variance _xifexpression_3 = null;
                    TypeRef _declaredLowerBound = ((Wildcard)arg).getDeclaredLowerBound();
                    boolean _tripleEquals_1 = (_declaredLowerBound == curr);
                    if (_tripleEquals_1) {
                      _xifexpression_3 = Variance.CONTRA;
                    }
                    _xifexpression_2 = _xifexpression_3;
                  }
                  _xifexpression_1 = _xifexpression_2;
                }
                _xifexpression = _xifexpression_1;
              }
              vFactor = _xifexpression;
            }
          }
          if ((vFactor == null)) {
            final TypeRef currFixed = curr;
            Variance _switchResult = null;
            boolean _matched = false;
            if (parent instanceof ComposedTypeRef) {
              boolean _contains = ((ComposedTypeRef)parent).getTypeRefs().contains(curr);
              if (_contains) {
                _matched=true;
                _switchResult = Variance.CO;
              }
            }
            if (!_matched) {
              if (parent instanceof TypeTypeRef) {
                TypeArgument _typeArg = ((TypeTypeRef)parent).getTypeArg();
                boolean _tripleEquals = (_typeArg == curr);
                if (_tripleEquals) {
                  _matched=true;
                  Variance _xifexpression = null;
                  boolean _isConstructorRef = ((TypeTypeRef)parent).isConstructorRef();
                  if (_isConstructorRef) {
                    _xifexpression = Variance.INV;
                  } else {
                    _xifexpression = Variance.CO;
                  }
                  _switchResult = _xifexpression;
                }
              }
            }
            if (!_matched) {
              if (parent instanceof TypeTypeRef) {
                TypeArgument _typeArg = ((TypeTypeRef)parent).getTypeArg();
                if ((_typeArg instanceof Wildcard)) {
                  _matched=true;
                  Variance _xblockexpression = null;
                  {
                    TypeArgument _typeArg_1 = ((TypeTypeRef)parent).getTypeArg();
                    final Wildcard wc = ((Wildcard) _typeArg_1);
                    Variance _xifexpression = null;
                    TypeRef _declaredUpperBound = wc.getDeclaredUpperBound();
                    boolean _tripleEquals = (_declaredUpperBound == curr);
                    if (_tripleEquals) {
                      _xifexpression = Variance.CO;
                    } else {
                      Variance _xifexpression_1 = null;
                      TypeRef _declaredLowerBound = wc.getDeclaredLowerBound();
                      boolean _tripleEquals_1 = (_declaredLowerBound == curr);
                      if (_tripleEquals_1) {
                        _xifexpression_1 = Variance.CONTRA;
                      }
                      _xifexpression = _xifexpression_1;
                    }
                    _xblockexpression = _xifexpression;
                  }
                  _switchResult = _xblockexpression;
                }
              }
            }
            if (!_matched) {
              if (parent instanceof BoundThisTypeRef) {
                ParameterizedTypeRef _actualThisTypeRef = ((BoundThisTypeRef)parent).getActualThisTypeRef();
                boolean _tripleEquals = (_actualThisTypeRef == curr);
                if (_tripleEquals) {
                  _matched=true;
                  _switchResult = Variance.CO;
                }
              }
            }
            if (!_matched) {
              if (parent instanceof FunctionTypeExpression) {
                TypeRef _returnTypeRef = ((FunctionTypeExpression)parent).getReturnTypeRef();
                boolean _tripleEquals = (_returnTypeRef == curr);
                if (_tripleEquals) {
                  _matched=true;
                  _switchResult = Variance.CO;
                }
              }
            }
            if (!_matched) {
              if (parent instanceof FunctionTypeExpression) {
                final Function1<TFormalParameter, Boolean> _function = (TFormalParameter it) -> {
                  TypeRef _typeRef = it.getTypeRef();
                  return Boolean.valueOf((_typeRef == currFixed));
                };
                boolean _exists = IterableExtensions.<TFormalParameter>exists(((FunctionTypeExpression)parent).getFpars(), _function);
                if (_exists) {
                  _matched=true;
                  _switchResult = Variance.CONTRA;
                }
              }
            }
            vFactor = _switchResult;
          }
          if ((vFactor == null)) {
            throw new IllegalStateException("internal error: unsupported case of containment of one typeRef in another (maybe types model has changed?)");
          }
          v = v.mult(vFactor);
          if ((v == Variance.INV)) {
            return v;
          }
        }
        curr = parent;
      }
    }
    return v;
  }
  
  private static boolean isNonPrivateMemberOf(final EObject member, final TClassifier tClassifier) {
    if ((member instanceof N4MemberDeclaration)) {
      final TMember tMember = ((N4MemberDeclaration)member).getDefinedTypeElement();
      return ((((tMember != null) && (!tMember.isConstructor())) && (tMember.getMemberAccessModifier() != MemberAccessModifier.PRIVATE)) && (tMember.getContainingType() == tClassifier));
    }
    return false;
  }
  
  /**
   * Tells if the given numeric literal is a Javascript int32.
   */
  public static boolean isIntLiteral(final NumericLiteral numLit) {
    final EObject parent = numLit.eContainer();
    final ICompositeNode node = NodeModelUtils.findActualNodeFor(numLit);
    final String text = NodeModelUtils.getTokenText(node);
    final int result = N4JSLanguageUtils.isIntLiteral(text);
    if ((result == 2)) {
      return ((parent instanceof UnaryExpression) && (((UnaryExpression) parent).getOp() == UnaryOperator.NEG));
    }
    return (result == 1);
  }
  
  /**
   * Tells if the given string represents a Javascript int32. Returns 0 if not, 1 if it does, and 2 if the literal
   * represents a number that is an int32 only if it is negative, but not if it is positive (only for literal
   * "2147483648" and equivalent literals).
   * <p>
   * Some notes:
   * <ol>
   * <li>the range of int32 is asymmetric: [ -2147483648, 2147483647 ]
   * <li>in Java, 1E0 etc. are always of type double, so we follow the same rule below.
   * <li>hexadecimal and octal literals are always interpreted as positive integers (important difference to Java).
   * </ol>
   * See N4JS Specification, Section 8.1.3.1 for details.
   */
  public static int isIntLiteral(final String numLitStr) {
    if (((numLitStr == null) || (numLitStr.length() == 0))) {
      return 0;
    }
    final boolean hasFractionOrExponent = N4JSLanguageUtils.containsOneOf(numLitStr, '.', 'e', 'E');
    if (hasFractionOrExponent) {
      return 0;
    }
    try {
      final boolean isHex = (numLitStr.startsWith("0x") || numLitStr.startsWith("0X"));
      final boolean isOct = ((((!isHex) && numLitStr.startsWith("0")) && (numLitStr.length() > 1)) && (!N4JSLanguageUtils.containsOneOf(numLitStr, '8', '9')));
      long _xifexpression = (long) 0;
      if (isHex) {
        _xifexpression = Long.parseLong(numLitStr.substring(2), 16);
      } else {
        long _xifexpression_1 = (long) 0;
        if (isOct) {
          _xifexpression_1 = Long.parseLong(numLitStr.substring(1), 8);
        } else {
          _xifexpression_1 = Long.parseLong(numLitStr);
        }
        _xifexpression = _xifexpression_1;
      }
      final long value = _xifexpression;
      if ((value == 2147483648L)) {
        return 2;
      }
      if (((Integer.MIN_VALUE <= value) && (value <= Integer.MAX_VALUE))) {
        return 1;
      }
      return 0;
    } catch (final Throwable _t) {
      if (_t instanceof NumberFormatException) {
        return 0;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private static boolean containsOneOf(final String str, final char... ch) {
    final int len = str.length();
    for (int i = 0; (i < len); i++) {
      {
        final char chStr = str.charAt(i);
        for (int j = 0; (j < ch.length); j++) {
          char _get = ch[j];
          boolean _tripleEquals = (chStr == _get);
          if (_tripleEquals) {
            return true;
          }
        }
      }
    }
    return false;
  }
  
  /**
   * Checks presence of {@link AnnotationDefinition#POLYFILL} annotation. See also {@link N4JSLanguageUtils#isStaticPolyfill(AnnotableElement) }
   */
  public static boolean isPolyfill(final AnnotableElement astElement) {
    return AnnotationDefinition.POLYFILL.hasAnnotation(astElement);
  }
  
  /**
   * Checks presence of {@link AnnotationDefinition#STATIC_POLYFILL} annotation. See also {@link N4JSLanguageUtils#isPolyfill(AnnotableElement) }
   */
  public static boolean isStaticPolyfill(final AnnotableElement astElement) {
    return AnnotationDefinition.STATIC_POLYFILL.hasAnnotation(astElement);
  }
  
  /**
   * Checks presence of {@link AnnotationDefinition#STATIC_POLYFILL_MODULE} annotation on the containing module.
   * See also {@link N4JSLanguageUtils#isContainedInStaticPolyfillAware(AnnotableElement) }
   */
  public static boolean isContainedInStaticPolyfillModule(final AnnotableElement astElement) {
    return AnnotationDefinition.STATIC_POLYFILL_MODULE.hasAnnotation(astElement);
  }
  
  /**
   * Checks presence of {@link AnnotationDefinition#STATIC_POLYFILL_MODULE} annotation on the containing module.
   * See also {@link N4JSLanguageUtils#isContainedInStaticPolyfillAware(TAnnotableElement) }
   */
  public static boolean isContainedInStaticPolyfillModule(final TAnnotableElement tsElement) {
    return AnnotationDefinition.STATIC_POLYFILL_MODULE.hasAnnotation(tsElement);
  }
  
  /**
   * Checks presence of {@link AnnotationDefinition#STATIC_POLYFILL_AWARE} annotation on the containing module.
   * See also {@link N4JSLanguageUtils#isContainedInStaticPolyfillModule(AnnotableElement) }
   */
  public static boolean isContainedInStaticPolyfillAware(final AnnotableElement astElement) {
    return AnnotationDefinition.STATIC_POLYFILL_AWARE.hasAnnotation(astElement);
  }
  
  /**
   * Checks presence of {@link AnnotationDefinition#STATIC_POLYFILL_AWARE} annotation on the containing module.
   * See also {@link N4JSLanguageUtils#isContainedInStaticPolyfillModule(TAnnotableElement) }
   */
  public static boolean isContainedInStaticPolyfillAware(final TAnnotableElement tsElement) {
    return AnnotationDefinition.STATIC_POLYFILL_AWARE.hasAnnotation(tsElement);
  }
  
  /**
   * checks if the qualifiedName has a last segment named 'default' {@link N4JSLanguageConstants#EXPORT_DEFAULT_NAME}
   */
  public static boolean isDefaultExport(final QualifiedName qualifiedName) {
    return (((qualifiedName != null) && (qualifiedName.getSegmentCount() > 1)) && Objects.equal(qualifiedName.getLastSegment(), N4JSLanguageConstants.EXPORT_DEFAULT_NAME));
  }
  
  /**
   * Returns the semantically important last part of a qualified name. This is commonly the last segment except for 'default' exports, where it is the second last segment.
   */
  public static String lastSegmentOrDefaultHost(final QualifiedName qualifiedName) {
    boolean _isDefaultExport = N4JSLanguageUtils.isDefaultExport(qualifiedName);
    if (_isDefaultExport) {
      int _segmentCount = qualifiedName.getSegmentCount();
      int _minus = (_segmentCount - 2);
      return qualifiedName.getSegment(_minus);
    }
    return qualifiedName.getLastSegment();
  }
  
  /**
   * Returns <code>true</code> if the character {@code c} is a valid JS identifier start.
   * 
   * Moved from {@link IdentifierValueConverter}.
   */
  public static boolean isValidIdentifierStart(final char c) {
    return ((CharTypes.isLetter(c) || N4JSLanguageUtils.isChar(c, "_")) || N4JSLanguageUtils.isChar(c, "$"));
  }
  
  /**
   * Returns <code>true</code> if the character {@code c} is a valid JS identifier part.
   * 
   * Moved from {@link IdentifierValueConverter}.
   */
  public static boolean isValidIdentifierPart(final char c) {
    return (((((N4JSLanguageUtils.isValidIdentifierStart(c) || CharTypes.isDigit(c)) || CharTypes.isConnectorPunctuation(c)) || CharTypes.isCombiningMark(c)) || N4JSLanguageUtils.isChar(c, "‌")) || N4JSLanguageUtils.isChar(c, "‍"));
  }
  
  /**
   * Returns <code>true</code> if the given identifier is a valid N4JS identifier.
   */
  public static boolean isValidIdentifier(final String identifier) {
    final int[] characters = identifier.chars().toArray();
    int _length = characters.length;
    ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _length, true);
    for (final Integer i : _doubleDotLessThan) {
      {
        int _get = characters[(i).intValue()];
        final char c = ((char) _get);
        if (((i).intValue() == 0)) {
          boolean _isValidIdentifierStart = N4JSLanguageUtils.isValidIdentifierStart(c);
          boolean _not = (!_isValidIdentifierStart);
          if (_not) {
            return false;
          }
        } else {
          boolean _isValidIdentifierPart = N4JSLanguageUtils.isValidIdentifierPart(c);
          boolean _not_1 = (!_isValidIdentifierPart);
          if (_not_1) {
            return false;
          }
        }
      }
    }
    return true;
  }
  
  /**
   * Helper method to overcome missing xtend support for character literals
   */
  private static boolean isChar(final char c1, final String c2) {
    char _charAt = c2.charAt(0);
    return (c1 == _charAt);
  }
  
  /**
   * If the given expression is a property access to one of the fields in {@code Symbol}, then this method returns the
   * referenced field, otherwise <code>null</code>. This method may perform proxy resolution.
   */
  public static TField getAccessedBuiltInSymbol(final RuleEnvironment G, final Expression expr) {
    return N4JSLanguageUtils.getAccessedBuiltInSymbol(G, expr, true);
  }
  
  /**
   * Same as {@link #getAccessedBuiltInSymbol(RuleEnvironment, Expression)}, but proxy resolution can be disallowed.
   * However, if proxy resolution is disallowed, this method will only support a "direct access" to built-in symbols,
   * i.e. the target must be an {@link IdentifierRef} directly pointing to built-in object 'Symbol'.
   */
  public static TField getAccessedBuiltInSymbol(final RuleEnvironment G, final Expression expr, final boolean allowProxyResolution) {
    if ((expr instanceof ParameterizedPropertyAccessExpression)) {
      final TObjectPrototype sym = RuleEnvironmentExtensions.symbolObjectType(G);
      if (allowProxyResolution) {
        final IdentifiableElement prop = ((ParameterizedPropertyAccessExpression)expr).getProperty();
        if (((prop instanceof TField) && (prop.eContainer() == sym))) {
          return ((TField) prop);
        }
      } else {
        final Expression targetExpr = ((ParameterizedPropertyAccessExpression)expr).getTarget();
        IdentifiableElement _xifexpression = null;
        if ((targetExpr instanceof IdentifierRef)) {
          _xifexpression = ((IdentifierRef)targetExpr).getId();
        }
        final IdentifiableElement targetElem = _xifexpression;
        if ((targetElem == sym)) {
          final String propName = ((ParameterizedPropertyAccessExpression)expr).getPropertyAsText();
          final Function1<TField, Boolean> _function = (TField it) -> {
            return Boolean.valueOf((it.isStatic() && Objects.equal(it.getName(), propName)));
          };
          return IterableExtensions.<TField>findFirst(Iterables.<TField>filter(sym.getOwnedMembers(), TField.class), _function);
        }
      }
    }
    return null;
  }
  
  /**
   * Tells if the given class has a covariant constructor, cf. {@link AnnotationDefinition#COVARIANT_CONSTRUCTOR}, or
   * the given interface requires all implementing classes to have a covariant constructor.
   */
  public static boolean hasCovariantConstructor(final TClassifier tClassifier) {
    return (tClassifier.isDeclaredCovariantConstructor() || IterableExtensions.<TClassifier>exists(AllSuperTypesCollector.collect(tClassifier), ((Function1<TClassifier, Boolean>) (TClassifier it) -> {
      return Boolean.valueOf(it.isDeclaredCovariantConstructor());
    })));
  }
  
  /**
   * Returns the nearest super class that is itself explicitly annotated with &#64;CovariantConstructor or has an
   * owned constructor explicitly annotated with &#64;CovariantConstructor.
   */
  public static TClass findCovariantConstructorDeclarator(final TClass tClass) {
    final Function1<TClass, Boolean> _function = (TClass it) -> {
      return Boolean.valueOf(it.isDeclaredCovariantConstructor());
    };
    return IterableExtensions.<TClass>findFirst(new ExtendedClassesIterable(tClass), _function);
  }
  
  /**
   * Creates a new type reference representing the implicit upper bound to be used for type variables without an
   * explicitly declared upper bound.
   */
  public static TypeRef getTypeVariableImplicitUpperBound(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.anyTypeRef(G);
  }
  
  /**
   * Tells if the given expression is evaluated during post-processing as a compile-time expression and has its
   * {@link CompileTimeValue value} stored in the {@link ASTMetaInfoCache}.
   * <p>
   * Note that every expression may be a compile-time expression (in fact, every number or string literal is a
   * compile-time expression) but being a compile-time expression only has a special effect in case of certain
   * expressions. This method returns <code>true</code> for these expressions.
   * <p>
   * IMPORTANT: this method will return <code>true</code> only for root expressions directly processed as
   * compile-time expressions, not for expressions directly or indirectly nested in such an expression.
   */
  public static boolean isProcessedAsCompileTimeExpression(final Expression expr) {
    boolean _isMandatoryCompileTimeExpression = N4JSLanguageUtils.isMandatoryCompileTimeExpression(expr);
    if (_isMandatoryCompileTimeExpression) {
      return true;
    }
    final EObject parent = expr.eContainer();
    return ((parent instanceof ExportedVariableDeclaration) || (parent instanceof N4FieldDeclaration));
  }
  
  /**
   * Tells if the given expression is required to be a compile-time expression, according to the N4JS language
   * specification.
   * <p>
   * IMPORTANT: this method will return <code>true</code> only for root expressions directly required to be
   * compile-time expressions, not for expressions directly or indirectly nested in such an expression.
   */
  public static boolean isMandatoryCompileTimeExpression(final Expression expr) {
    final EObject parent = expr.eContainer();
    if ((parent instanceof LiteralOrComputedPropertyName)) {
      return ((((LiteralOrComputedPropertyName)parent).getKind() == PropertyNameKind.COMPUTED) && (((LiteralOrComputedPropertyName)parent).getExpression() == expr));
    } else {
      if ((parent instanceof IndexedAccessExpression)) {
        Expression _index = ((IndexedAccessExpression)parent).getIndex();
        return (_index == expr);
      }
    }
    return false;
  }
  
  /**
   * Returns the property/member name to use for the given compile-time value or <code>null</code> if the value is
   * invalid. This is used to derive a property/member from a compile-time expression in computed property names and
   * index access expressions.
   * <p>
   * IMPLEMENTATION NOTE: we can simply use #toString() on a valid value, even if we have a ValueBoolean, ValueNumber,
   * or ValueSymbol.
   * <p>
   * For undefined, null, NaN, Infinity, booleans, and numbers: they are equivalent to their corresponding string
   * literal, as illustrated in this snippet:
   * 
   * <pre>
   *     // plain Javascript
   * 
   *     var obj = {
   *         [undefined]: 'a',
   *         [null]     : 'b',
   *         41         : 'c',
   *         [42]       : 'd',
   *         [false]    : 'e',
   *         [NaN]      : 'f',
   *         [Infinity] : 'g'
   *     };
   * 
   *     console.log( obj[undefined] === obj['undefined']); // will print true!
   *     console.log( obj[null]      === obj['null']     ); // will print true!
   *     console.log( obj[41]        === obj['41']       ); // will print true!
   *     console.log( obj[42]        === obj['42']       ); // will print true!
   *     console.log( obj[false]     === obj['false']    ); // will print true!
   *     console.log( obj[NaN]       === obj['NaN']      ); // will print true!
   *     console.log( obj[Infinity]  === obj['Infinity'] ); // will print true!
   * </pre>
   * <p>
   * For symbols: the #toString() method in ValueSymbol prepends {@link N4JSLanguageUtils#SYMBOL_IDENTIFIER_PREFIX},
   * so we can simply use that.
   */
  public static String derivePropertyNameFromCompileTimeValue(final CompileTimeValue value) {
    String _xifexpression = null;
    if (((value != null) && value.isValid())) {
      _xifexpression = value.toString();
    } else {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  /**
   * Calculate the optional field strategy of the given expression
   * 
   * @param expr
   * 			the expression to calculate.
   * @return the optional field strategy of the expression.
   */
  public static OptionalFieldStrategy calculateOptionalFieldStrategy(final TypableElement expr, final TypeRef typeRef) {
    boolean _isConstTransitiveObjectLiteral = N4JSLanguageUtils.isConstTransitiveObjectLiteral(expr);
    if (_isConstTransitiveObjectLiteral) {
      return OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL;
    }
    boolean _isConstTransitiveNewExpressionOrFinalNominalClassInstance = N4JSLanguageUtils.isConstTransitiveNewExpressionOrFinalNominalClassInstance(expr, typeRef);
    if (_isConstTransitiveNewExpressionOrFinalNominalClassInstance) {
      return OptionalFieldStrategy.GETTERS_OPTIONAL;
    }
    if ((expr instanceof ConditionalExpression)) {
      final OptionalFieldStrategy optionalStrategyForTrueExpr = N4JSLanguageUtils.calculateOptionalFieldStrategy(((ConditionalExpression)expr).getTrueExpression(), typeRef);
      final OptionalFieldStrategy optionalStrategyForFalseExpr = N4JSLanguageUtils.calculateOptionalFieldStrategy(((ConditionalExpression)expr).getFalseExpression(), typeRef);
      return N4JSLanguageUtils.minOptionalityFieldStrategy(optionalStrategyForTrueExpr, optionalStrategyForFalseExpr);
    }
    return OptionalFieldStrategy.OFF;
  }
  
  /**
   * Checks whether the given expression is an object literal or references an object literal
   * transitively through a const variable.
   * 
   * @param expr
   * 			the expression to check.
   * @return true if the expression is an object literal or references an object literal
   * 			transitively through a const variable.
   */
  private static boolean isConstTransitiveObjectLiteral(final TypableElement expr) {
    if ((expr instanceof NullLiteral)) {
      return true;
    }
    if ((expr instanceof ObjectLiteral)) {
      return true;
    }
    if ((expr instanceof IdentifierRef)) {
      final IdentifiableElement idElem = ((IdentifierRef)expr).getId();
      if ((idElem instanceof VariableDeclaration)) {
        boolean _isConst = ((VariableDeclaration)idElem).isConst();
        if (_isConst) {
          Expression _expression = ((VariableDeclaration)idElem).getExpression();
          return (_expression instanceof ObjectLiteral);
        }
      } else {
        if ((idElem instanceof TVariable)) {
          return ((TVariable)idElem).isObjectLiteral();
        }
      }
    }
    return false;
  }
  
  /**
   * Checks whether the given expression is a new expression, an expression of a final and nominal type,
   * or references these expressions transitively through a const variable.
   * 
   * @param expr
   * 			the expression to check.
   * @return true if the expression is a new expression, an expression of a final and nominal type,
   * 			or references these expressions transitively through a const variable.
   */
  private static boolean isConstTransitiveNewExpressionOrFinalNominalClassInstance(final TypableElement expr, final TypeRef typeRef) {
    if ((expr instanceof NullLiteral)) {
      return true;
    }
    if ((expr instanceof NewExpression)) {
      return true;
    }
    if ((typeRef != null)) {
      final Type declType = typeRef.getDeclaredType();
      if ((((declType != null) && declType.isFinal()) && Objects.equal(typeRef.getTypingStrategy(), TypingStrategy.NOMINAL))) {
        return true;
      }
    }
    if ((expr instanceof IdentifierRef)) {
      final IdentifiableElement idElem = ((IdentifierRef)expr).getId();
      if ((idElem instanceof VariableDeclaration)) {
        boolean _isConst = ((VariableDeclaration)idElem).isConst();
        if (_isConst) {
          Expression _expression = ((VariableDeclaration)idElem).getExpression();
          return (_expression instanceof NewExpression);
        }
      } else {
        if ((idElem instanceof TVariable)) {
          return ((TVariable)idElem).isNewExpression();
        }
      }
    }
    return false;
  }
  
  /**
   * Check if the interface is built-in or an external without N4JS annotation.
   * 
   * @param tinf
   *            The interface.
   * @return true if the interface is either built-in, provided by runtime or an external interface without N4JS annotation. Return false
   *         otherwise.
   */
  public static boolean builtInOrProvidedByRuntimeOrExternalWithoutN4JSAnnotation(final TInterface tinf) {
    final Function1<TAnnotation, Boolean> _function = (TAnnotation it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(AnnotationDefinition.N4JS.name, _name));
    };
    final boolean hasN4JSAnnotation = IterableExtensions.<TAnnotation>exists(tinf.getAnnotations(), _function);
    final TypingStrategy ts = tinf.getTypingStrategy();
    final boolean isDefStructural = ((!Objects.equal(ts, TypingStrategy.NOMINAL)) && (!Objects.equal(ts, TypingStrategy.DEFAULT)));
    final XpectAwareFileExtensionCalculator fileExtensionCalculator = new XpectAwareFileExtensionCalculator();
    final String fileExt = fileExtensionCalculator.getXpectAwareFileExtension(tinf);
    return (((TypeUtils.isBuiltIn(tinf) || tinf.isProvidedByRuntime()) || (tinf.isExternal() && (!hasN4JSAnnotation))) || (isDefStructural && Objects.equal(fileExt, N4JSGlobals.N4JSD_FILE_EXTENSION)));
  }
  
  /**
   * Check if an optional field strategy is less restricted than or equal to another optional field strategy.
   * 
   * @param s1
   *            The first optional field strategy.
   * @param s2
   *            The second optional field strategy.
   * @return true if the first optional field strategy is less restricted than or equal to the second optional field
   *         strategy, false otherwise.
   */
  public static boolean isOptionalityLessRestrictedOrEqual(final OptionalFieldStrategy s1, final OptionalFieldStrategy s2) {
    boolean _equals = Objects.equal(s1, s2);
    if (_equals) {
      return true;
    }
    boolean _switchResult = false;
    if (s1 != null) {
      switch (s1) {
        case FIELDS_AND_ACCESSORS_OPTIONAL:
          _switchResult = true;
          break;
        case OFF:
          _switchResult = Objects.equal(s2, OptionalFieldStrategy.OFF);
          break;
        case GETTERS_OPTIONAL:
          _switchResult = (Objects.equal(s2, OptionalFieldStrategy.GETTERS_OPTIONAL) || Objects.equal(s2, OptionalFieldStrategy.OFF));
          break;
        default:
          throw new RuntimeException(("Invalid enum value " + s1));
      }
    } else {
      throw new RuntimeException(("Invalid enum value " + s1));
    }
    final boolean result = _switchResult;
    return result;
  }
  
  /**
   * Calculate the minimum of two given optional field strategies.
   * 
   * @param s1
   *            The first optional field strategy.
   * @param s2
   *            The second optional field strategy.
   * @return the minimum optional field strategy.
   */
  public static OptionalFieldStrategy minOptionalityFieldStrategy(final OptionalFieldStrategy s1, final OptionalFieldStrategy s2) {
    if ((Objects.equal(s1, OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL) && Objects.equal(s2, OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL))) {
      return OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL;
    }
    if (((!Objects.equal(s1, OptionalFieldStrategy.OFF)) && (!Objects.equal(s2, OptionalFieldStrategy.OFF)))) {
      return OptionalFieldStrategy.GETTERS_OPTIONAL;
    }
    return OptionalFieldStrategy.OFF;
  }
}
