/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import javax.inject.Inject;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.EmptyStatement;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.organize.imports.ImportProvidedElement;
import org.eclipse.n4js.organize.imports.ImportSpecifiersUtil;
import org.eclipse.n4js.organize.imports.ImportStateCalculator;
import org.eclipse.n4js.organize.imports.RecordingImportState;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.utils.Log;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Validations for the import statements.
 */
@Log
@SuppressWarnings("all")
public class N4JSImportValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private ImportStateCalculator importStateCalculator;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkMultipleDefaultExports(final Script script) {
    final Function1<ExportDeclaration, Boolean> _function = (ExportDeclaration it) -> {
      return Boolean.valueOf(it.isDefaultExport());
    };
    final List<ExportDeclaration> defaultExports = IteratorExtensions.<ExportDeclaration>toList(IteratorExtensions.<ExportDeclaration>filter(Iterators.<ExportDeclaration>filter(script.eAllContents(), ExportDeclaration.class), _function));
    final Consumer<ExportDeclaration> _function_1 = (ExportDeclaration it) -> {
      this.addIssue(IssueCodes.getMessageForIMP_DEFAULT_EXPORT_DUPLICATE(), it, N4JSPackage.eINSTANCE.getExportDeclaration_DefaultExport(), IssueCodes.IMP_DEFAULT_EXPORT_DUPLICATE);
    };
    IterableExtensions.<ExportDeclaration>tail(defaultExports).forEach(_function_1);
  }
  
  @Check
  public void checkConflictingImports(final Script script) {
    final HashMap<EObject, String> eObjectToIssueCode = CollectionLiterals.<EObject, String>newHashMap();
    this.analyzeAndCheckConflictingImports(script, eObjectToIssueCode);
    this.markScatteredImports(script);
  }
  
  @Check
  public void checkStaticVsDynamicImport(final NamespaceImportSpecifier importSpecifier) {
    final Type type = importSpecifier.getDefinedType();
    if ((type instanceof ModuleNamespaceVirtualType)) {
      TModule _module = ((ModuleNamespaceVirtualType)type).getModule();
      boolean _tripleNotEquals = (_module != null);
      if (_tripleNotEquals) {
        boolean _isDeclaredDynamic = importSpecifier.isDeclaredDynamic();
        if (_isDeclaredDynamic) {
          boolean _isN4JSMode = this.jsVariantHelper.isN4JSMode(((ModuleNamespaceVirtualType)type).getModule());
          if (_isN4JSMode) {
            this.addIssue(
              IssueCodes.getMessageForIMP_DYNAMIC_NAMESPACE_IMPORT_N4JS(((ModuleNamespaceVirtualType)type).getModule().getModuleSpecifier()), importSpecifier, IssueCodes.IMP_DYNAMIC_NAMESPACE_IMPORT_N4JS);
          } else {
            boolean _isExternalMode = this.jsVariantHelper.isExternalMode(((ModuleNamespaceVirtualType)type).getModule());
            if (_isExternalMode) {
              this.addIssue(
                IssueCodes.getMessageForIMP_DYNAMIC_NAMESPACE_IMPORT_N4JSD(((ModuleNamespaceVirtualType)type).getModule().getModuleSpecifier()), importSpecifier, IssueCodes.IMP_DYNAMIC_NAMESPACE_IMPORT_N4JSD);
            }
          }
        } else {
          boolean _isPlainJS = this.jsVariantHelper.isPlainJS(((ModuleNamespaceVirtualType)type).getModule());
          if (_isPlainJS) {
            this.addIssue(
              IssueCodes.getMessageForIMP_STATIC_NAMESPACE_IMPORT_PLAIN_JS(((ModuleNamespaceVirtualType)type).getModule().getModuleSpecifier()), importSpecifier, IssueCodes.IMP_STATIC_NAMESPACE_IMPORT_PLAIN_JS);
          }
        }
      }
    }
  }
  
  /**
   * Algorithm to check the Model for Issues with Imports.
   */
  private void analyzeAndCheckConflictingImports(final Script script, final Map<EObject, String> eObjectToIssueCode) {
    final RecordingImportState reg = this.importStateCalculator.calculateImportstate(script);
    final Consumer<Pair<ImportDeclaration, List<ImportDeclaration>>> _function = (Pair<ImportDeclaration, List<ImportDeclaration>> it) -> {
      this.handleDuplicatedImportDeclarations(it, eObjectToIssueCode);
    };
    reg.duplicatedImportDeclarations.forEach(_function);
    this.handleNameCollisions(reg.localNameCollision, eObjectToIssueCode);
    this.handleTypeCollisions(reg.duplicateImportsOfSameElement, eObjectToIssueCode);
    this.handleBrokenImports(reg.brokenImports, eObjectToIssueCode);
    this.handleUnusedImports(reg.unusedImports, eObjectToIssueCode);
    this.handleNotImportedTypeRefs(script, IterableExtensions.<ImportSpecifier>toList(Iterables.<ImportSpecifier>filter(eObjectToIssueCode.keySet(), ImportSpecifier.class)), eObjectToIssueCode);
  }
  
  /**
   * Create issue (warning) 'unused import' if  import doesn't contribute to referenced Types.
   * @param specifier import to be marked
   */
  private void handleBrokenImports(final List<ImportSpecifier> importSpecifiers, final Map<EObject, String> eObjectToIssueCode) {
    for (final ImportSpecifier is : importSpecifiers) {
      boolean _contains = eObjectToIssueCode.keySet().contains(is);
      boolean _not = (!_contains);
      if (_not) {
        this.addIssueUnresolved(is, eObjectToIssueCode);
      }
    }
  }
  
  /**
   * Create issue (warning) 'unused import' if  import doesn't contribute to referenced Types.
   * @param specifier import to be marked
   */
  private void handleUnusedImports(final List<ImportSpecifier> importSpecifiers, final Map<EObject, String> eObjectToIssueCode) {
    for (final ImportSpecifier is : importSpecifiers) {
      boolean _contains = eObjectToIssueCode.keySet().contains(is);
      boolean _not = (!_contains);
      if (_not) {
        this.addIssueUnusedImport(is, eObjectToIssueCode);
      }
    }
  }
  
  /**
   * Computes a user-facing name for the given {@link NamedImportSpecifier}
   * that can be used in error/warning messages.
   * 
   * @param spec
   * @return name from NamedImportSpecifier or AST-text if unresolved.
   */
  private String computeImportSpecifierName(final NamedImportSpecifier spec) {
    String _xifexpression = null;
    boolean _isBrokenImport = ImportSpecifiersUtil.isBrokenImport(spec);
    if (_isBrokenImport) {
      _xifexpression = NodeModelUtils.findActualNodeFor(spec).getText().trim();
    } else {
      _xifexpression = ImportSpecifiersUtil.importedElementName(spec);
    }
    return _xifexpression;
  }
  
  /**
   * Mark all imports that don't appear in the header.
   * @param script the script
   */
  private void markScatteredImports(final Script script) {
    boolean stillInHeader = true;
    EList<ScriptElement> _scriptElements = script.getScriptElements();
    for (final ScriptElement se : _scriptElements) {
      if (stillInHeader) {
        boolean _not = (!(((se instanceof ImportDeclaration) || (se instanceof EmptyStatement)) || N4JSASTUtils.isStringLiteralExpression(se)));
        if (_not) {
          stillInHeader = false;
        }
      } else {
        if ((se instanceof ImportDeclaration)) {
          this.handleScatteredImport(((ImportDeclaration)se));
        }
      }
    }
  }
  
  /**
   * Mark with discouraged_placement it not yet marked with other error.
   * @param importDecl - statement to mark
   */
  private void handleScatteredImport(final ImportDeclaration importDecl) {
    this.addIssueScatteredImport(importDecl);
  }
  
  private void handleDuplicatedImportDeclarations(final Pair<ImportDeclaration, List<ImportDeclaration>> duplicatePair, final Map<EObject, String> eObjectToIssueCode) {
    final ImportSpecifier specifier = IterableExtensions.<ImportSpecifier>head(duplicatePair.getKey().getImportSpecifiers());
    final List<ImportDeclaration> duplicates = duplicatePair.getValue();
    final Consumer<ImportDeclaration> _function = (ImportDeclaration duplicateImportDeclaration) -> {
      final ImportSpecifier duplicate = IterableExtensions.<ImportSpecifier>head(duplicateImportDeclaration.getImportSpecifiers());
      boolean _matched = false;
      if (specifier instanceof NamespaceImportSpecifier) {
        _matched=true;
        boolean _matched_1 = false;
        if (duplicate instanceof NamespaceImportSpecifier) {
          _matched_1=true;
          this.addIssueDuplicateNamespaceImportDeclaration(((NamespaceImportSpecifier)specifier), ((NamespaceImportSpecifier)duplicate), duplicateImportDeclaration, eObjectToIssueCode);
        }
        if (!_matched_1) {
          N4JSImportValidator.logger.error(
            "cannot register duplicate import declaration for different kinds of specifiers");
        }
      }
      if (!_matched) {
        if (specifier instanceof NamedImportSpecifier) {
          _matched=true;
          boolean _matched_1 = false;
          if (duplicate instanceof NamedImportSpecifier) {
            _matched_1=true;
            this.addIssueDuplicateNamedImportDeclaration(((NamedImportSpecifier)specifier), ((NamedImportSpecifier)duplicate), duplicateImportDeclaration, eObjectToIssueCode);
          }
          if (!_matched_1) {
            N4JSImportValidator.logger.error(
              "cannot register duplicate import declaration for different kinds of specifiers");
          }
        }
      }
      if (!_matched) {
        N4JSImportValidator.logger.error("cannot register duplicate import declaration for  specifiers");
      }
    };
    duplicates.forEach(_function);
  }
  
  private void handleNameCollisions(final List<Pair<String, List<ImportProvidedElement>>> multimap, final Map<EObject, String> eObjectToIssueCode) {
    final Consumer<Pair<String, List<ImportProvidedElement>>> _function = (Pair<String, List<ImportProvidedElement>> pair) -> {
      final String name = pair.getKey();
      final List<ImportProvidedElement> providers = pair.getValue();
      final ImportSpecifier first = IterableExtensions.<ImportProvidedElement>head(providers).getImportSpecifier();
      Pair<String, String> _switchResult = null;
      boolean _matched = false;
      if (first instanceof NamespaceImportSpecifier) {
        _matched=true;
        String _string = ImportSpecifiersUtil.importedModule(first).getQualifiedName().toString();
        String _plus = ("namespace name for " + _string);
        _switchResult = Pair.<String, String>of(name, _plus);
      }
      if (!_matched) {
        if (first instanceof NamedImportSpecifier) {
          _matched=true;
          Pair<String, String> _xifexpression = null;
          String _alias = ((NamedImportSpecifier)first).getAlias();
          boolean _tripleNotEquals = (_alias != null);
          if (_tripleNotEquals) {
            String _name = ((NamedImportSpecifier)first).getImportedElement().getName();
            String _plus = ("alias name for named import " + _name);
            String _plus_1 = (_plus + " from ");
            String _string = ImportSpecifiersUtil.importedModule(first).getQualifiedName().toString();
            String _plus_2 = (_plus_1 + _string);
            _xifexpression = Pair.<String, String>of(name, _plus_2);
          } else {
            String _name_1 = ((NamedImportSpecifier)first).getImportedElement().getName();
            String _plus_3 = ("name for named import " + _name_1);
            String _plus_4 = (_plus_3 + " from ");
            String _string_1 = ImportSpecifiersUtil.importedModule(first).getQualifiedName().toString();
            String _plus_5 = (_plus_4 + _string_1);
            _xifexpression = Pair.<String, String>of(name, _plus_5);
          }
          _switchResult = _xifexpression;
        }
      }
      if (!_matched) {
        {
          String _name = ImportSpecifier.class.getName();
          String _plus = ("unhandled type of " + _name);
          N4JSImportValidator.logger.error(_plus);
          throw new RuntimeException("Cannot validate local name collisions");
        }
      }
      final Pair<String, String> name2reason = _switchResult;
      final Consumer<ImportProvidedElement> _function_1 = (ImportProvidedElement importProvidedElement) -> {
        this.addLocalNameCollision(importProvidedElement.getImportSpecifier(), name2reason.getKey(), name2reason.getValue(), eObjectToIssueCode);
      };
      IterableExtensions.<ImportProvidedElement>tail(providers).forEach(_function_1);
    };
    multimap.forEach(_function);
  }
  
  private void handleTypeCollisions(final List<Pair<Pair<String, TModule>, List<ImportProvidedElement>>> duplicateslist, final Map<EObject, String> eObjectToIssueCode) {
    final Consumer<Pair<Pair<String, TModule>, List<ImportProvidedElement>>> _function = (Pair<Pair<String, TModule>, List<ImportProvidedElement>> duplicateEntry) -> {
      final Pair<String, TModule> entry = duplicateEntry.getKey();
      final String entryName = entry.getKey();
      final TModule entryModule = entry.getValue();
      final List<ImportProvidedElement> imports = duplicateEntry.getValue();
      final ImportSpecifier firstImportSpecifier = IterableExtensions.<ImportProvidedElement>head(imports).getImportSpecifier();
      String _switchResult = null;
      boolean _matched = false;
      if (firstImportSpecifier instanceof NamespaceImportSpecifier) {
        _matched=true;
        String _alias = ((NamespaceImportSpecifier)firstImportSpecifier).getAlias();
        String _plus = (_alias + ".");
        _switchResult = (_plus + entryName);
      }
      if (!_matched) {
        if (firstImportSpecifier instanceof NamedImportSpecifier) {
          _matched=true;
          String _elvis = null;
          String _alias = ((NamedImportSpecifier)firstImportSpecifier).getAlias();
          if (_alias != null) {
            _elvis = _alias;
          } else {
            _elvis = entryName;
          }
          _switchResult = _elvis;
        }
      }
      final String firstImportName = _switchResult;
      boolean _xifexpression = false;
      if ((firstImportSpecifier instanceof NamedImportSpecifier)) {
        _xifexpression = ((NamedImportSpecifier)firstImportSpecifier).isDefaultImport();
      }
      final boolean firstImportIsDefault = _xifexpression;
      final Consumer<ImportProvidedElement> _function_1 = (ImportProvidedElement dupe) -> {
        final ImportSpecifier duplicateImportSpecifier = dupe.getImportSpecifier();
        if ((firstImportIsDefault && (duplicateImportSpecifier instanceof NamespaceImportSpecifier))) {
          this.addIssueDuplicate(firstImportSpecifier, entryName, entryModule, firstImportName, eObjectToIssueCode);
        }
        if (((firstImportSpecifier instanceof NamespaceImportSpecifier) && 
          (duplicateImportSpecifier instanceof NamespaceImportSpecifier))) {
          this.addIssueDuplicateNamespace(((NamespaceImportSpecifier) duplicateImportSpecifier), 
            ((NamespaceImportSpecifier) firstImportSpecifier), eObjectToIssueCode);
        } else {
          this.addIssueDuplicate(dupe.getImportSpecifier(), entryName, entryModule, firstImportName, eObjectToIssueCode);
        }
      };
      IterableExtensions.<ImportProvidedElement>tail(imports).forEach(_function_1);
    };
    duplicateslist.forEach(_function);
  }
  
  private void regUnresolvedImport(final ParameterizedTypeRef ref, final String name, final Map<EObject, String> eObjectToIssueCode) {
    final String issueCode = IssueCodes.IMP_UNRESOLVED;
    String _put = eObjectToIssueCode.put(ref, issueCode);
    boolean _tripleEquals = (_put == null);
    if (_tripleEquals) {
      final String message = IssueCodes.getMessageForIMP_UNRESOLVED(name);
      this.addIssue(message, ref, issueCode);
    }
  }
  
  private void handleNotImportedTypeRefs(final Script script, final List<ImportSpecifier> specifiersWithIssues, final Map<EObject, String> eObjectToIssueCode) {
    final Function1<ImportProvidedElement, TModule> _function = (ImportProvidedElement it) -> {
      return it.getImportedModule();
    };
    final Map<TModule, List<ImportProvidedElement>> importedProvidedElementsWithIssuesByModule = IterableExtensions.<TModule, ImportProvidedElement>groupBy(ImportSpecifiersUtil.mapToImportProvidedElements(specifiersWithIssues), _function);
    final Function1<ParameterizedTypeRef, Boolean> _function_1 = (ParameterizedTypeRef it) -> {
      return Boolean.valueOf(((it.getDeclaredType() != null) && (it.getDeclaredType().getContainingModule() != null)));
    };
    final Function1<ParameterizedTypeRef, TModule> _function_2 = (ParameterizedTypeRef it) -> {
      return it.getDeclaredType().getContainingModule();
    };
    final Map<TModule, List<ParameterizedTypeRef>> potentiallyAffectedTypeRefs = IteratorExtensions.<TModule, ParameterizedTypeRef>groupBy(IteratorExtensions.<ParameterizedTypeRef>filter(Iterators.<ParameterizedTypeRef>filter(script.eAllContents(), ParameterizedTypeRef.class), _function_1), _function_2);
    final BiConsumer<TModule, List<ParameterizedTypeRef>> _function_3 = (TModule module, List<ParameterizedTypeRef> typeRefs) -> {
      final List<ImportProvidedElement> conflict = importedProvidedElementsWithIssuesByModule.get(module);
      if ((conflict != null)) {
        final Consumer<ParameterizedTypeRef> _function_4 = (ParameterizedTypeRef typeRef) -> {
          final String typeRefName = this.typeRefUsedName(typeRef);
          final Function1<ImportProvidedElement, Boolean> _function_5 = (ImportProvidedElement ipe) -> {
            String _localName = ipe.getLocalName();
            return Boolean.valueOf(Objects.equal(_localName, typeRefName));
          };
          boolean _exists = IterableExtensions.<ImportProvidedElement>exists(conflict, _function_5);
          if (_exists) {
            this.regUnresolvedImport(typeRef, typeRefName, eObjectToIssueCode);
          }
        };
        typeRefs.forEach(_function_4);
      }
    };
    potentiallyAffectedTypeRefs.forEach(_function_3);
  }
  
  private String typeRefUsedName(final ParameterizedTypeRef ref) {
    return NodeModelUtils.getTokenText(NodeModelUtils.findActualNodeFor(ref));
  }
  
  /**
   * Notice, that unlike other issues {@link IssueCodes#IMP_DISCOURAGED_PLACEMENT} is always added, even if there are other issues
   * reported for given import declaration.
   */
  private void addIssueScatteredImport(final ImportDeclaration importDecl) {
    final String issueCode = IssueCodes.IMP_DISCOURAGED_PLACEMENT;
    final String message = IssueCodes.getMessageForIMP_DISCOURAGED_PLACEMENT();
    this.addIssue(message, importDecl, issueCode);
  }
  
  private String addLocalNameCollision(final ImportSpecifier duplicate, final String name, final String reason, final Map<EObject, String> eObjectToIssueCode) {
    String _xblockexpression = null;
    {
      final String issueCode = IssueCodes.IMP_LOCAL_NAME_CONFLICT;
      String _xifexpression = null;
      String _get = eObjectToIssueCode.get(duplicate);
      boolean _tripleEquals = (_get == null);
      if (_tripleEquals) {
        String _xblockexpression_1 = null;
        {
          final String message = IssueCodes.getMessageForIMP_LOCAL_NAME_CONFLICT(name, reason);
          this.addIssue(message, duplicate, issueCode);
          _xblockexpression_1 = eObjectToIssueCode.put(duplicate, issueCode);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private void addIssueDuplicateNamespaceImportDeclaration(final NamespaceImportSpecifier specifier, final NamespaceImportSpecifier duplicate, final ImportDeclaration duplicateImportDeclaration, final Map<EObject, String> eObjectToIssueCode) {
    final String issueCode = IssueCodes.IMP_STMT_DUPLICATE_NAMESPACE;
    String _get = eObjectToIssueCode.get(specifier);
    boolean _tripleEquals = (_get == null);
    if (_tripleEquals) {
      final String message = IssueCodes.getMessageForIMP_STMT_DUPLICATE_NAMESPACE(specifier.getAlias(), 
        ImportSpecifiersUtil.importedModule(duplicate).getQualifiedName());
      this.addIssue(message, duplicateImportDeclaration, issueCode);
    }
    final Consumer<ImportSpecifier> _function = (ImportSpecifier is) -> {
      eObjectToIssueCode.put(specifier, issueCode);
    };
    duplicateImportDeclaration.getImportSpecifiers().forEach(_function);
  }
  
  /**
   * Adds an issue for duplicate named import specifiers.
   * 
   * @param specifier The first import of the element
   * @param duplicate The duplicated import of the element
   * @param duplicateImportDeclaration The import declaration of the duplicated import
   * @param eObjectToIssueCode A map to keep track of all added issues
   */
  private void addIssueDuplicateNamedImportDeclaration(final NamedImportSpecifier specifier, final NamedImportSpecifier duplicate, final ImportDeclaration duplicateImportDeclaration, final Map<EObject, String> eObjectToIssueCode) {
    final String issueCode = IssueCodes.IMP_STMT_DUPLICATE_NAMED;
    String _get = eObjectToIssueCode.get(specifier);
    boolean _tripleEquals = (_get == null);
    if (_tripleEquals) {
      final String message = IssueCodes.getMessageForIMP_STMT_DUPLICATE_NAMED(ImportSpecifiersUtil.usedName(specifier), 
        ImportSpecifiersUtil.importedModule(duplicate).getQualifiedName());
      this.addIssue(message, duplicateImportDeclaration, issueCode);
    }
    final Consumer<ImportSpecifier> _function = (ImportSpecifier is) -> {
      eObjectToIssueCode.put(specifier, issueCode);
    };
    duplicateImportDeclaration.getImportSpecifiers().forEach(_function);
  }
  
  private String addIssueDuplicateNamespace(final NamespaceImportSpecifier duplicateImportSpecifier, final NamespaceImportSpecifier firstImportSpecifier, final Map<EObject, String> eObjectToIssueCode) {
    String _xifexpression = null;
    String _get = eObjectToIssueCode.get(duplicateImportSpecifier);
    boolean _tripleEquals = (_get == null);
    if (_tripleEquals) {
      String _xblockexpression = null;
      {
        final String issueCode = IssueCodes.IMP_DUPLICATE_NAMESPACE;
        final String msg = IssueCodes.getMessageForIMP_DUPLICATE_NAMESPACE(
          ImportSpecifiersUtil.importedModule(firstImportSpecifier).getQualifiedName(), firstImportSpecifier.getAlias());
        this.addIssue(msg, duplicateImportSpecifier, issueCode);
        _xblockexpression = eObjectToIssueCode.put(duplicateImportSpecifier, issueCode);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  private String addIssueDuplicate(final ImportSpecifier specifier, final String actualName, final TModule module, final String firstImportName, final Map<EObject, String> eObjectToIssueCode) {
    String _xblockexpression = null;
    {
      String issueCode = IssueCodes.IMP_DUPLICATE;
      String _xifexpression = null;
      String _get = eObjectToIssueCode.get(specifier);
      boolean _tripleEquals = (_get == null);
      if (_tripleEquals) {
        String _xblockexpression_1 = null;
        {
          final String message = IssueCodes.getMessageForIMP_DUPLICATE(actualName, module.getQualifiedName(), firstImportName);
          this.addIssue(message, specifier, issueCode);
          _xblockexpression_1 = eObjectToIssueCode.put(specifier, issueCode);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private String addIssueUnresolved(final ImportSpecifier specifier, final Map<EObject, String> eObjectToIssueCode) {
    String _xblockexpression = null;
    {
      String issueCode = IssueCodes.IMP_UNRESOLVED;
      String _xifexpression = null;
      String _get = eObjectToIssueCode.get(specifier);
      boolean _tripleEquals = (_get == null);
      if (_tripleEquals) {
        String _xblockexpression_1 = null;
        {
          final String message = IssueCodes.getMessageForIMP_UNRESOLVED(this.computeUnusedOrUnresolvedMessage(specifier));
          this.addIssue(message, specifier, issueCode);
          _xblockexpression_1 = eObjectToIssueCode.put(specifier, issueCode);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private String addIssueUnusedImport(final ImportSpecifier specifier, final Map<EObject, String> eObjectToIssueCode) {
    String _xblockexpression = null;
    {
      final String issueCode = IssueCodes.IMP_UNUSED_IMPORT;
      String _xifexpression = null;
      String _get = eObjectToIssueCode.get(specifier);
      boolean _tripleEquals = (_get == null);
      if (_tripleEquals) {
        String _xblockexpression_1 = null;
        {
          final String message = IssueCodes.getMessageForIMP_UNUSED_IMPORT(this.computeUnusedOrUnresolvedMessage(specifier));
          this.addIssue(message, specifier, issueCode);
          _xblockexpression_1 = eObjectToIssueCode.put(specifier, issueCode);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private String computeUnusedOrUnresolvedMessage(final ImportSpecifier specifier) {
    String _switchResult = null;
    boolean _matched = false;
    if (specifier instanceof NamedImportSpecifier) {
      _matched=true;
      _switchResult = this.computeImportSpecifierName(((NamedImportSpecifier)specifier));
    }
    if (!_matched) {
      if (specifier instanceof NamespaceImportSpecifier) {
        _matched=true;
        String _alias = ((NamespaceImportSpecifier)specifier).getAlias();
        String _plus = ("* as " + _alias);
        String _plus_1 = (_plus + " from ");
        String _computeModuleSpecifier = this.computeModuleSpecifier(((NamespaceImportSpecifier)specifier));
        _switchResult = (_plus_1 + _computeModuleSpecifier);
      }
    }
    return _switchResult;
  }
  
  private String computeModuleSpecifier(final NamespaceImportSpecifier specifier) {
    String _xblockexpression = null;
    {
      final TModule importedModule = ImportSpecifiersUtil.importedModule(specifier);
      String _xifexpression = null;
      if (((importedModule != null) && (!importedModule.eIsProxy()))) {
        _xifexpression = importedModule.getModuleSpecifier();
      } else {
        _xifexpression = "module was a proxy";
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private static final Logger logger = Logger.getLogger(N4JSImportValidator.class);
}
