/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.IntPredicate;
import java.util.stream.IntStream;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.n4JS.BooleanLiteral;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.JSXAbstractElement;
import org.eclipse.n4js.n4JS.JSXAttribute;
import org.eclipse.n4js.n4JS.JSXChild;
import org.eclipse.n4js.n4JS.JSXElement;
import org.eclipse.n4js.n4JS.JSXExpression;
import org.eclipse.n4js.n4JS.JSXFragment;
import org.eclipse.n4js.n4JS.JSXPropertyAttribute;
import org.eclipse.n4js.n4JS.JSXSpreadAttribute;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.NullLiteral;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4jsx.ReactHelper;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.Script_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.utils.ResourceType;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Transforms JSX tags to output code according to JSX/React conventions.
 * <p>
 * For example:
 * <pre>
 * &lt;div attr="value">&lt;/div>
 * </pre>
 * will be transformed to
 * <pre>
 * React.createElement('div', Object.assign({attr: "value"}));
 * </pre>
 */
@SuppressWarnings("all")
public class JSXTransformation extends Transformation {
  private SymbolTableEntryOriginal steForJsxBackendNamespace;
  
  private SymbolTableEntryOriginal steForJsxBackendElementFactoryFunction;
  
  private SymbolTableEntryOriginal steForJsxBackendFragmentComponent;
  
  @Inject
  private ReactHelper reactHelper;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  /**
   * IMPORTANT: our strategy for handling nested JSXElements is as follows:
   * 1) direct child JSXElements will be handled together with their parent JSXElement
   * 2) indirect children that are contained in nested expressions will be handled via a separate invocation to
   * method #transformJSXElement(JSXElement)
   * 
   * Example for case 1:
   * <pre>
   * let elem1 = &lt;div>&lt;a>&lt;/a>&lt;/div>; // &lt;a> is a direct child
   * </pre>
   * Example for case 2:
   * <pre>
   * let elem2 = &lt;div>{function() {return &lt;a>&lt;/a>;}}&lt;/div>; // &lt;a> is the child of a nested expression!
   * </pre>
   * More examples for case 2:
   * <pre>
   * let elem3 = &lt;div>{&lt;a>&lt;/a>}&lt;/div>;
   * let elem4 = &lt;div prop={&lt;a>&lt;/a>}>&lt;/div>;
   * </pre>
   */
  @Override
  public void transform() {
    final ResourceType resourceType = ResourceType.getResourceType(this.getState().resource);
    final boolean inJSX = ((resourceType == ResourceType.JSX) || (resourceType == ResourceType.N4JSX));
    if ((!inJSX)) {
      return;
    }
    final List<JSXAbstractElement> jsxAbstractElements = this.<JSXAbstractElement>collectNodes(this.getState().im, JSXAbstractElement.class, true);
    boolean _isEmpty = jsxAbstractElements.isEmpty();
    if (_isEmpty) {
      return;
    }
    this.steForJsxBackendNamespace = this.prepareImportOfJsxBackend();
    this.steForJsxBackendElementFactoryFunction = this.prepareElementFactoryFunction();
    this.steForJsxBackendFragmentComponent = this.prepareFragmentComponent();
    final Consumer<JSXAbstractElement> _function = (JSXAbstractElement it) -> {
      this.transformJSXAbstractElement(it);
    };
    jsxAbstractElements.forEach(_function);
  }
  
  private SymbolTableEntryOriginal prepareImportOfJsxBackend() {
    final TModule jsxBackendModule = this.reactHelper.getJsxBackendModule(this.getState().resource);
    if ((jsxBackendModule == null)) {
      URI _uRI = this.getState().resource.getURI();
      String _plus = ("cannot locate JSX backend for N4JSX resource " + _uRI);
      throw new RuntimeException(_plus);
    }
    final Function1<ImportDeclaration, Boolean> _function = (ImportDeclaration impDeclIM) -> {
      TModule _importedModule = this.getState().info.getImportedModule(impDeclIM);
      return Boolean.valueOf((_importedModule == jsxBackendModule));
    };
    final Function1<ImportDeclaration, EList<ImportSpecifier>> _function_1 = (ImportDeclaration it) -> {
      return it.getImportSpecifiers();
    };
    final NamespaceImportSpecifier existingNamespaceImportOfReactIM = IterableExtensions.<NamespaceImportSpecifier>head(Iterables.<NamespaceImportSpecifier>filter(Iterables.<ImportSpecifier>concat(IterableExtensions.<ImportDeclaration, EList<ImportSpecifier>>map(IterableExtensions.<ImportDeclaration>filter(Iterables.<ImportDeclaration>filter(this.getState().im.getScriptElements(), ImportDeclaration.class), _function), _function_1)), NamespaceImportSpecifier.class));
    if ((existingNamespaceImportOfReactIM != null)) {
      existingNamespaceImportOfReactIM.setFlaggedUsedInCode(true);
      return this.findSymbolTableEntryForNamespaceImport(existingNamespaceImportOfReactIM);
    }
    return this.addNamespaceImport(jsxBackendModule, this.reactHelper.getJsxBackendNamespaceName());
  }
  
  private SymbolTableEntryOriginal prepareElementFactoryFunction() {
    final TFunction elementFactoryFunction = this.reactHelper.getJsxBackendElementFactoryFunction(this.getState().resource);
    if ((elementFactoryFunction == null)) {
      URI _uRI = this.getState().resource.getURI();
      String _plus = ("cannot locate element factory function of JSX backend for N4JSX resource " + _uRI);
      throw new RuntimeException(_plus);
    }
    return this.getSymbolTableEntryOriginal(elementFactoryFunction, true);
  }
  
  private SymbolTableEntryOriginal prepareFragmentComponent() {
    final TClass fragmentComponent = this.reactHelper.getJsxBackendFragmentComponent(this.getState().resource);
    if ((fragmentComponent == null)) {
      URI _uRI = this.getState().resource.getURI();
      String _plus = ("cannot locate fragment component of JSX backend for N4JSX resource " + _uRI);
      throw new RuntimeException(_plus);
    }
    return this.getSymbolTableEntryOriginal(fragmentComponent, true);
  }
  
  private void transformJSXAbstractElement(final JSXAbstractElement elem) {
    Script_IM _containerOfType = EcoreUtil2.<Script_IM>getContainerOfType(elem, Script_IM.class);
    boolean _tripleEquals = (_containerOfType == null);
    if (_tripleEquals) {
      return;
    }
    this.replace(elem, this.convertJSXAbstractElement(elem));
  }
  
  private ParameterizedCallExpression convertJSXAbstractElement(final JSXAbstractElement elem) {
    List<? extends Expression> _xifexpression = null;
    if ((elem instanceof JSXElement)) {
      Expression _tagNameFromElement = this.getTagNameFromElement(((JSXElement)elem));
      Expression _convertJSXAttributes = this.convertJSXAttributes(((JSXElement)elem).getJsxAttributes());
      _xifexpression = Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(_tagNameFromElement, _convertJSXAttributes));
    } else {
      ParameterizedPropertyAccessExpression_IM __PropertyAccessExpr = TranspilerBuilderBlocks._PropertyAccessExpr(this.steForJsxBackendNamespace, this.steForJsxBackendFragmentComponent);
      NullLiteral __NULL = TranspilerBuilderBlocks._NULL();
      _xifexpression = Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__PropertyAccessExpr, __NULL));
    }
    final List<? extends Expression> firstParams = _xifexpression;
    ParameterizedPropertyAccessExpression_IM __PropertyAccessExpr_1 = TranspilerBuilderBlocks._PropertyAccessExpr(this.steForJsxBackendNamespace, this.steForJsxBackendElementFactoryFunction);
    final Function1<JSXChild, Expression> _function = (JSXChild it) -> {
      return this.convertJSXChild(it);
    };
    List<Expression> _map = ListExtensions.<JSXChild, Expression>map(elem.getJsxChildren(), _function);
    Iterable<Expression> _plus = Iterables.<Expression>concat(firstParams, _map);
    return TranspilerBuilderBlocks._CallExpr(__PropertyAccessExpr_1, ((Expression[])Conversions.unwrapArray(_plus, Expression.class)));
  }
  
  private Expression convertJSXChild(final JSXChild child) {
    Expression _switchResult = null;
    boolean _matched = false;
    if (child instanceof JSXElement) {
      _matched=true;
      _switchResult = this.convertJSXAbstractElement(((JSXAbstractElement)child));
    }
    if (!_matched) {
      if (child instanceof JSXFragment) {
        _matched=true;
        _switchResult = this.convertJSXAbstractElement(((JSXAbstractElement)child));
      }
    }
    if (!_matched) {
      if (child instanceof JSXExpression) {
        _matched=true;
        _switchResult = ((JSXExpression)child).getExpression();
      }
    }
    return _switchResult;
  }
  
  private Expression convertJSXAttributes(final List<JSXAttribute> attrs) {
    boolean _isEmpty = attrs.isEmpty();
    if (_isEmpty) {
      return TranspilerBuilderBlocks._NULL();
    } else {
      if (((attrs.size() == 1) && (attrs.get(0) instanceof JSXSpreadAttribute))) {
        JSXAttribute _get = attrs.get(0);
        return ((JSXSpreadAttribute) _get).getExpression();
      } else {
        final IntPredicate _function = (int i) -> {
          JSXAttribute _get_1 = attrs.get(i);
          return (_get_1 instanceof JSXSpreadAttribute);
        };
        final int[] spreadIndices = IntStream.range(0, attrs.size()).filter(_function).toArray();
        ObjectLiteral _xifexpression = null;
        JSXAttribute _get_1 = attrs.get(0);
        if ((_get_1 instanceof JSXSpreadAttribute)) {
          _xifexpression = TranspilerBuilderBlocks._ObjLit();
        } else {
          ObjectLiteral _xblockexpression = null;
          {
            int _xifexpression_1 = (int) 0;
            boolean _isEmpty_1 = ((List<Integer>)Conversions.doWrapArray(spreadIndices)).isEmpty();
            boolean _not = (!_isEmpty_1);
            if (_not) {
              _xifexpression_1 = spreadIndices[0];
            } else {
              _xifexpression_1 = attrs.size();
            }
            final int firstSpreadIndex = _xifexpression_1;
            final Function1<JSXAttribute, JSXPropertyAttribute> _function_1 = (JSXAttribute it) -> {
              return ((JSXPropertyAttribute) it);
            };
            List<JSXPropertyAttribute> firstProps = ListExtensions.<JSXAttribute, JSXPropertyAttribute>map(attrs.subList(0, firstSpreadIndex), _function_1);
            final Function1<JSXPropertyAttribute, PropertyNameValuePair> _function_2 = (JSXPropertyAttribute it) -> {
              return this.convertJSXAttribute(it);
            };
            _xblockexpression = TranspilerBuilderBlocks._ObjLit(((PropertyAssignment[])Conversions.unwrapArray(ListExtensions.<JSXPropertyAttribute, PropertyNameValuePair>map(firstProps, _function_2), PropertyAssignment.class)));
          }
          _xifexpression = _xblockexpression;
        }
        final ObjectLiteral target = _xifexpression;
        ArrayList<Expression> parameters = new ArrayList<Expression>();
        parameters.add(target);
        for (int i = 0; (i < spreadIndices.length); i++) {
          {
            final int curSpreadIdx = spreadIndices[i];
            JSXAttribute _get_2 = attrs.get(curSpreadIdx);
            parameters.add(((JSXSpreadAttribute) _get_2).getExpression());
            int _xifexpression_1 = (int) 0;
            int _length = spreadIndices.length;
            int _minus = (_length - 1);
            boolean _lessThan = (i < _minus);
            if (_lessThan) {
              _xifexpression_1 = spreadIndices[(i + 1)];
            } else {
              _xifexpression_1 = ((Object[])Conversions.unwrapArray(attrs, Object.class)).length;
            }
            final int nextSpreadIdx = _xifexpression_1;
            final List<JSXAttribute> propsBetweenTwoSpreads = attrs.subList((curSpreadIdx + 1), nextSpreadIdx);
            boolean _isEmpty_1 = propsBetweenTwoSpreads.isEmpty();
            boolean _not = (!_isEmpty_1);
            if (_not) {
              final Function1<JSXAttribute, PropertyNameValuePair> _function_1 = (JSXAttribute it) -> {
                return this.convertJSXAttribute(((JSXPropertyAttribute) it));
              };
              parameters.add(TranspilerBuilderBlocks._ObjLit(((PropertyAssignment[])Conversions.unwrapArray(ListExtensions.<JSXAttribute, PropertyNameValuePair>map(propsBetweenTwoSpreads, _function_1), PropertyAssignment.class))));
            }
          }
        }
        final ArrayList<Expression> _converted_parameters = (ArrayList<Expression>)parameters;
        return TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._PropertyAccessExpr(this.steFor_Object(), this.steFor_assign()), ((Expression[])Conversions.unwrapArray(_converted_parameters, Expression.class)));
      }
    }
  }
  
  private PropertyNameValuePair convertJSXAttribute(final JSXPropertyAttribute attr) {
    return TranspilerBuilderBlocks._PropertyNameValuePair(
      this.getNameFromPropertyAttribute(attr), 
      this.getValueExpressionFromPropertyAttribute(attr));
  }
  
  private Expression getTagNameFromElement(final JSXElement elem) {
    final Expression nameExpr = elem.getJsxElementName().getExpression();
    if ((nameExpr instanceof IdentifierRef_IM)) {
      final SymbolTableEntry id = ((IdentifierRef_IM)nameExpr).getId_IM();
      if ((id == null)) {
        return TranspilerBuilderBlocks._StringLiteral(((IdentifierRef_IM)nameExpr).getIdAsText());
      }
    }
    return nameExpr;
  }
  
  private String getNameFromPropertyAttribute(final JSXPropertyAttribute attr) {
    final IdentifiableElement prop = attr.getProperty();
    if (((prop != null) && (!prop.eIsProxy()))) {
      return prop.getName();
    }
    return attr.getPropertyAsText();
  }
  
  private Expression getValueExpressionFromPropertyAttribute(final JSXPropertyAttribute attr) {
    Expression _elvis = null;
    Expression _jsxAttributeValue = attr.getJsxAttributeValue();
    if (_jsxAttributeValue != null) {
      _elvis = _jsxAttributeValue;
    } else {
      BooleanLiteral __TRUE = TranspilerBuilderBlocks._TRUE();
      _elvis = __TRUE;
    }
    return _elvis;
  }
}
