/**
 * Copyright (c) 2019 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ModifiableElement;
import org.eclipse.n4js.n4JS.ModuleSpecifierForm;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * This transformation will prepare the output code for module loading. Since dropping support for commonjs and SystemJS
 * and instead using ECMAScript 2015 imports/exports in the output code, this transformation is no longer doing much.
 */
@SuppressWarnings("all")
public class ModuleWrappingTransformation extends Transformation {
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  private String[] localModuleSpecifierSegments = null;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
    final TModule localModule = this.getState().resource.getModule();
    final String localModuleSpecifier = this.resourceNameComputer.getCompleteModuleSpecifier(localModule);
    this.localModuleSpecifierSegments = localModuleSpecifier.split("/", (-1));
  }
  
  @Override
  public void transform() {
    final Consumer<ImportDeclaration> _function = (ImportDeclaration it) -> {
      this.transformImportDecl(it);
    };
    this.<ImportDeclaration>collectNodes(this.getState().im, ImportDeclaration.class, false).forEach(_function);
    final Function1<ExportDeclaration, ExportableElement> _function_1 = (ExportDeclaration it) -> {
      return it.getExportedElement();
    };
    final Consumer<ModifiableElement> _function_2 = (ModifiableElement it) -> {
      it.getDeclaredModifiers().clear();
    };
    Iterables.<ModifiableElement>filter(ListExtensions.<ExportDeclaration, ExportableElement>map(this.<ExportDeclaration>collectNodes(this.getState().im, ExportDeclaration.class, false), _function_1), ModifiableElement.class).forEach(_function_2);
    final Consumer<ExportDeclaration> _function_3 = (ExportDeclaration it) -> {
      this.splitDefaultExportFromVarDecl(it);
    };
    this.<ExportDeclaration>collectNodes(this.getState().im, ExportDeclaration.class, false).forEach(_function_3);
    this.addEmptyImport(N4JSGlobals.N4JS_RUNTIME);
  }
  
  private void transformImportDecl(final ImportDeclaration importDeclIM) {
    final String moduleSpecifier = this.computeModuleSpecifierForOutputCode(importDeclIM);
    final String moduleSpecifierNormalized = moduleSpecifier.replace("/./", "/");
    importDeclIM.setModuleSpecifierAsText(moduleSpecifierNormalized);
  }
  
  /**
   * For the following reasons, we cannot simply reuse the module specifier from the N4JS source code
   * in the generated output code:
   * <ol>
   * <li>in N4JS, module specifiers are always absolute whereas in plain Javascript module specifiers
   * must be relative (i.e. start with a segment '.' or '..') when importing from a module within the
   * same npm package. N4JS does not even support relative module specifiers.
   * <li>in N4JS, the project name as the first segment of an absolute module specifier is optional
   * (see {@link ModuleSpecifierForm#PLAIN} vs. {@link ModuleSpecifierForm#COMPLETE}); this is not
   * supported by plain Javascript.
   * <li>in N4JS, module specifiers do not contain the path to the output folder, whereas in plain
   * Javascript absolute module specifiers must always contain the full path from a project's root
   * folder to the module.
   * </ol>
   * Importing from a runtime library is an exception to the above: in this case we must never include
   * the runtime library's project name nor its path to the output folder in the module specifier.
   */
  private String computeModuleSpecifierForOutputCode(final ImportDeclaration importDeclIM) {
    final TModule targetModule = this.getState().info.getImportedModule(importDeclIM);
    final IN4JSProject targetProject = this.n4jsCore.findProject(targetModule.eResource().getURI()).orNull();
    ProjectType _projectType = targetProject.getProjectType();
    boolean _tripleEquals = (_projectType == ProjectType.RUNTIME_LIBRARY);
    if (_tripleEquals) {
      return targetModule.getModuleSpecifier();
    }
    URI _location = targetProject.getLocation();
    URI _location_1 = this.getState().project.getLocation();
    final boolean importingFromModuleInSameProject = Objects.equal(_location, _location_1);
    if (importingFromModuleInSameProject) {
      return this.createRelativeModuleSpecifier(targetModule);
    }
    final ModuleSpecifierForm moduleSpecifierForm = importDeclIM.getModuleSpecifierForm();
    if (((moduleSpecifierForm == ModuleSpecifierForm.PROJECT) || (moduleSpecifierForm == ModuleSpecifierForm.PROJECT_NO_MAIN))) {
      return this.getActualProjectName(targetProject);
    }
    return this.createAbsoluteModuleSpecifier(targetProject, targetModule);
  }
  
  private String createRelativeModuleSpecifier(final TModule targetModule) {
    final String targetModuleSpecifier = this.resourceNameComputer.getCompleteModuleSpecifier(targetModule);
    final String[] targetSegments = targetModuleSpecifier.split("/", (-1));
    final int l = Math.min(targetSegments.length, this.localModuleSpecifierSegments.length);
    int i = 0;
    while (((i < l) && java.util.Objects.equals(targetSegments[i], this.localModuleSpecifierSegments[i]))) {
      i++;
    }
    final String[] differingSegments = Arrays.<String>copyOfRange(targetSegments, i, targetSegments.length);
    int _length = this.localModuleSpecifierSegments.length;
    int _minus = (_length - 1);
    final int goUpCount = (_minus - i);
    String _xifexpression = null;
    if ((goUpCount > 0)) {
      _xifexpression = "../".repeat(goUpCount);
    } else {
      _xifexpression = "./";
    }
    String _join = Joiner.on("/").join(differingSegments);
    final String result = (_xifexpression + _join);
    return result;
  }
  
  private String createAbsoluteModuleSpecifier(final IN4JSProject targetProject, final TModule targetModule) {
    final StringBuilder sb = new StringBuilder();
    final String targetProjectName = this.getActualProjectName(targetProject);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(targetProjectName);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      sb.append(targetProjectName);
    }
    String outputPath = targetProject.getOutputPath();
    boolean _isNullOrEmpty_1 = StringExtensions.isNullOrEmpty(outputPath);
    boolean _not_1 = (!_isNullOrEmpty_1);
    if (_not_1) {
      boolean _startsWith = outputPath.startsWith("/");
      boolean _not_2 = (!_startsWith);
      if (_not_2) {
        sb.append("/");
      }
      sb.append(outputPath);
      boolean _endsWith = outputPath.endsWith("/");
      boolean _not_3 = (!_endsWith);
      if (_not_3) {
        sb.append("/");
      }
    } else {
      int _length = sb.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        sb.append("/");
      }
    }
    final String targetModuleSpecifier = this.resourceNameComputer.getCompleteModuleSpecifier(targetModule);
    sb.append(targetModuleSpecifier);
    return sb.toString();
  }
  
  private String getActualProjectName(final IN4JSProject project) {
    ProjectType _projectType = project.getProjectType();
    boolean _tripleEquals = (_projectType == ProjectType.DEFINITION);
    if (_tripleEquals) {
      final String definedProjectName = project.getDefinesPackageName();
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(definedProjectName);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        return definedProjectName;
      }
    }
    return project.getProjectName();
  }
  
  /**
   * Turns
   * <pre>
   * export default var|let|const C = ...
   * </pre>
   * into
   * <pre>
   * var|let|const C = ...
   * export default C;
   * </pre>
   */
  private void splitDefaultExportFromVarDecl(final ExportDeclaration exportDecl) {
    boolean _isDefaultExport = exportDecl.isDefaultExport();
    if (_isDefaultExport) {
      final ExportableElement exportedElement = exportDecl.getExportedElement();
      if ((exportedElement instanceof VariableStatement)) {
        boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<VariableBinding>filter(((VariableStatement)exportedElement).getVarDeclsOrBindings(), VariableBinding.class));
        boolean _not = (!_isEmpty);
        if (_not) {
          throw new UnsupportedOperationException("unsupported: default-exported variable binding");
        }
        int _size = ((VariableStatement)exportedElement).getVarDeclsOrBindings().size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          throw new UnsupportedOperationException(
            "unsupported: several default-exported variable declarations in a single export declaration");
        }
        VariableDeclarationOrBinding _head = IterableExtensions.<VariableDeclarationOrBinding>head(((VariableStatement)exportedElement).getVarDeclsOrBindings());
        final VariableDeclaration varDecl = ((VariableDeclaration) _head);
        final SymbolTableEntry varDeclSTE = this.findSymbolTableEntryForElement(varDecl, true);
        this.insertBefore(exportDecl, exportedElement);
        exportDecl.setDefaultExportedExpression(TranspilerBuilderBlocks._IdentRef(varDeclSTE));
      }
    }
  }
}
