/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.n4js.conversion.ValueConverterUtils;
import org.eclipse.n4js.generator.GeneratorOption;
import org.eclipse.n4js.n4JS.AdditiveOperator;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ParameterizedAccess;
import org.eclipse.n4js.n4JS.PrimaryExpression;
import org.eclipse.n4js.n4JS.TemplateLiteral;
import org.eclipse.n4js.n4JS.TemplateSegment;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Transforms ES2015 template literals to a corresponding ES5 equivalent.
 */
@TransformationDependency.Optional(GeneratorOption.TemplateStringLiterals)
@SuppressWarnings("all")
public class TemplateStringTransformation extends Transformation {
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<TemplateLiteral> _function = (TemplateLiteral it) -> {
      this.transformTemplateLiteral(it);
    };
    this.<TemplateLiteral>collectNodes(this.getState().im, TemplateLiteral.class, true).forEach(_function);
  }
  
  private void transformTemplateLiteral(final TemplateLiteral template) {
    ArrayList<Expression> _newArrayList = Lists.<Expression>newArrayList(template.getSegments());
    for (final Expression segment : _newArrayList) {
      if ((segment instanceof TemplateSegment)) {
        this.replace(segment, TranspilerBuilderBlocks._StringLiteral(((TemplateSegment)segment).getValueAsString(), TemplateStringTransformation.wrapAndQuote(((TemplateSegment)segment).getValueAsString())));
      } else {
      }
    }
    Expression _switchResult = null;
    int _size = template.getSegments().size();
    switch (_size) {
      case 0:
        _switchResult = TranspilerBuilderBlocks._StringLiteral("");
        break;
      case 1:
        _switchResult = template.getSegments().get(0);
        break;
      default:
        _switchResult = TranspilerBuilderBlocks._Parenthesis(
          TranspilerBuilderBlocks._AdditiveExpression(AdditiveOperator.ADD, ((Expression[])Conversions.unwrapArray(TemplateStringTransformation.wrapIfRequired(template.getSegments()), Expression.class))));
        break;
    }
    final Expression replacement = _switchResult;
    this.replace(template, replacement);
  }
  
  private static Iterable<Expression> wrapIfRequired(final Iterable<Expression> expressions) {
    List<Expression> _list = IterableExtensions.<Expression>toList(expressions);
    final ArrayList<Expression> copy = new ArrayList<Expression>(_list);
    final Function1<Expression, Expression> _function = (Expression expr) -> {
      final boolean needParentheses = (!((expr instanceof PrimaryExpression) || (expr instanceof ParameterizedAccess)));
      if (needParentheses) {
        return TranspilerBuilderBlocks._Parenthesis(expr);
      } else {
        return expr;
      }
    };
    return ListExtensions.<Expression, Expression>map(copy, _function);
  }
  
  /**
   * put raw into double quote and escape all existing double-quotes {@code '"' -> '\"' } and newlines {@code '\n' -> '\\n'}.
   */
  private static String wrapAndQuote(final String raw) {
    String _convertToEscapedString = ValueConverterUtils.convertToEscapedString(raw, true);
    String _plus = ("\"" + _convertToEscapedString);
    return (_plus + "\"");
  }
}
