/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.handler;

import com.google.inject.Inject;
import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.n4js.ui.dialog.TypeInformationPopup;
import org.eclipse.n4js.ui.labeling.N4JSTypeInformationHoverProvider;
import org.eclipse.n4js.ui.selection.AstSelectionProvider2;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.xtext.ui.editor.XtextEditor;
import org.eclipse.xtext.ui.editor.hover.html.XtextBrowserInformationControlInput;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.utils.EditorUtils;
import org.eclipse.xtext.util.TextRegion;
import org.eclipse.xtext.xbase.lib.Extension;

/**
 * Handler for revealing the type information of a particular variable or an expression based on the current selection of
 * the active N4JS editor.
 */
@SuppressWarnings("all")
public class ShowTypeInformationHandler extends AbstractHandler {
  @Inject
  @Extension
  private N4JSTypeInformationHoverProvider _n4JSTypeInformationHoverProvider;
  
  @Inject
  @Extension
  private AstSelectionProvider2 _astSelectionProvider2;
  
  @Override
  public Object execute(final ExecutionEvent event) throws ExecutionException {
    final XtextEditor editor = EditorUtils.getActiveXtextEditor(event);
    if ((null != editor)) {
      ISelection _selection = editor.getSelectionProvider().getSelection();
      final ITextSelection selection = ((ITextSelection) _selection);
      final EObject node = this.getNodeElementFromSelection(editor.getDocument(), selection);
      if ((null != node)) {
        int _offset = selection.getOffset();
        int _length = selection.getLength();
        final Region region = new Region(_offset, _length);
        Object _info = this._n4JSTypeInformationHoverProvider.getHoverInfo(node, editor.getInternalSourceViewer(), region).getInfo();
        final XtextBrowserInformationControlInput info = ((XtextBrowserInformationControlInput) _info);
        if ((null != info)) {
          this.raiseTypeInformationPopup(event, info.getHtml());
        }
      }
    }
    return null;
  }
  
  private EObject getNodeElementFromSelection(final IXtextDocument document, final ITextSelection selection) {
    int _offset = selection.getOffset();
    int _length = selection.getLength();
    TextRegion _textRegion = new TextRegion(_offset, _length);
    return this._astSelectionProvider2.getSelectedAstElement(document, _textRegion);
  }
  
  private int raiseTypeInformationPopup(final ExecutionEvent event, final String html) {
    Shell _activeShell = HandlerUtil.getActiveShell(event);
    Point _popupAnchor = this.getPopupAnchor(event);
    return new TypeInformationPopup(_activeShell, _popupAnchor, html).open();
  }
  
  /**
   * Calculates the anchor for the popup dialog. It will be right bottom of the selection.
   */
  private Point getPopupAnchor(final ExecutionEvent event) {
    final StyledText textWidget = this.getStyledText(event);
    if ((null == textWidget)) {
      return null;
    }
    final Point docRange = textWidget.getSelectionRange();
    final int midOffset = (docRange.x + (docRange.y / 2));
    Point point = textWidget.getLocationAtOffset(midOffset);
    point = textWidget.toDisplay(point);
    final GC gc = new GC(textWidget);
    gc.setFont(textWidget.getFont());
    final int height = gc.getFontMetrics().getHeight();
    gc.dispose();
    int _y = point.y;
    point.y = (_y + height);
    return point;
  }
  
  /**
   * Adapts the currently active workbench part to extract the StyledText from it.
   */
  private StyledText getStyledText(final ExecutionEvent event) {
    final IWorkbenchPart part = HandlerUtil.getActiveWorkbenchWindow(event).getActivePage().getActivePart();
    final ITextViewer viewer = part.<ITextViewer>getAdapter(ITextViewer.class);
    if ((viewer == null)) {
      final Control control = part.<Control>getAdapter(Control.class);
      if ((control instanceof StyledText)) {
        return ((StyledText)control);
      }
    } else {
      return viewer.getTextWidget();
    }
    return null;
  }
}
