/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.wizard.classes;

import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.ui.wizard.classes.N4JSClassWizardModel;
import org.eclipse.n4js.ui.wizard.classifiers.N4JSNewClassifierWizardGenerator;
import org.eclipse.n4js.ui.wizard.generator.ImportRequirement;
import org.eclipse.n4js.ui.wizard.generator.N4JSImportRequirementResolver;
import org.eclipse.n4js.ui.wizard.generator.WizardGeneratorHelper;
import org.eclipse.n4js.ui.wizard.model.ClassifierReference;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A file generator for {@link N4JSClassWizardModel}
 */
@SuppressWarnings("all")
public class N4JSNewClassWizardGenerator extends N4JSNewClassifierWizardGenerator<N4JSClassWizardModel> {
  @Inject
  @Extension
  private WizardGeneratorHelper generatorUtils;
  
  @Override
  protected String generateClassifierCode(final N4JSClassWizardModel model, final Map<URI, String> aliasBindings) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isFinalAnnotated = model.isFinalAnnotated();
      if (_isFinalAnnotated) {
        _builder.append("@Final");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      boolean _isN4jsAnnotated = model.isN4jsAnnotated();
      if (_isN4jsAnnotated) {
        _builder.append("@N4JS");
      }
    }
    _builder.newLineIfNotEmpty();
    {
      boolean _isInternal = model.isInternal();
      if (_isInternal) {
        _builder.append("@Internal ");
      }
    }
    String _addSpace = this.generatorUtils.addSpace(this.generatorUtils.exportStatement(model.getAccessModifier()));
    _builder.append(_addSpace);
    {
      boolean _isDefinitionFile = model.isDefinitionFile();
      if (_isDefinitionFile) {
        _builder.append("external ");
      }
    }
    String _addSpace_1 = this.generatorUtils.addSpace(model.getAccessModifier().toCodeRepresentation());
    _builder.append(_addSpace_1);
    _builder.append("class ");
    String _name = model.getName();
    _builder.append(_name);
    {
      boolean _isComplete = model.getSuperClass().isComplete();
      if (_isComplete) {
        _builder.append(" extends ");
        String _realOrAliasName = this.generatorUtils.realOrAliasName(model.getSuperClass(), aliasBindings);
        _builder.append(_realOrAliasName);
      }
    }
    {
      int _length = ((Object[])Conversions.unwrapArray(model.getInterfaces(), Object.class)).length;
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        _builder.append(" implements ");
      }
    }
    {
      List<ClassifierReference> _interfaces = model.getInterfaces();
      boolean _hasElements = false;
      for(final ClassifierReference iface : _interfaces) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        String _realOrAliasName_1 = this.generatorUtils.realOrAliasName(iface, aliasBindings);
        _builder.append(_realOrAliasName_1);
      }
    }
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    return _builder.toString();
  }
  
  @Override
  protected List<IN4JSProject> getReferencedProjects(final N4JSClassWizardModel model) {
    final Function1<ClassifierReference, IN4JSProject> _function = (ClassifierReference it) -> {
      return this.generatorUtils.projectOfUri(it.uri);
    };
    List<IN4JSProject> _map = ListExtensions.<ClassifierReference, IN4JSProject>map(model.getInterfaces(), _function);
    ArrayList<IN4JSProject> referencedProjects = new ArrayList<IN4JSProject>(_map);
    boolean _isComplete = model.getSuperClass().isComplete();
    if (_isComplete) {
      referencedProjects.add(this.generatorUtils.projectOfUri(model.getSuperClass().uri));
    }
    return referencedProjects;
  }
  
  @Override
  protected List<ImportRequirement> getImportRequirements(final N4JSClassWizardModel model) {
    ArrayList<ImportRequirement> _xblockexpression = null;
    {
      ArrayList<ImportRequirement> demandedImports = new ArrayList<ImportRequirement>();
      boolean _isComplete = model.getSuperClass().isComplete();
      if (_isComplete) {
        demandedImports.add(N4JSImportRequirementResolver.classifierReferenceToImportRequirement(model.getSuperClass()));
      }
      boolean _isEmpty = model.getInterfaces().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        demandedImports.addAll(N4JSImportRequirementResolver.classifierReferencesToImportRequirements(model.getInterfaces()));
      }
      _xblockexpression = demandedImports;
    }
    return _xblockexpression;
  }
}
