/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.wizard.generator;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.common.io.Files;
import com.google.inject.Inject;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.model.utils.JSONModelUtils;
import org.eclipse.n4js.packagejson.model.edit.IJSONDocumentModification;
import org.eclipse.n4js.packagejson.model.edit.PackageJsonModificationProvider;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.ui.changes.ChangeManager;
import org.eclipse.n4js.ui.changes.IAtomicChange;
import org.eclipse.n4js.ui.organize.imports.Interaction;
import org.eclipse.n4js.ui.organize.imports.OrganizeImportsService;
import org.eclipse.n4js.ui.wizard.generator.ImportRequirement;
import org.eclipse.n4js.ui.wizard.generator.N4JSImportRequirementResolver;
import org.eclipse.n4js.ui.wizard.model.AccessModifier;
import org.eclipse.n4js.ui.wizard.model.ClassifierReference;
import org.eclipse.n4js.ui.wizard.workspace.WorkspaceWizardModel;
import org.eclipse.n4js.utils.Log;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.xtext.resource.SaveOptions;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.ui.editor.model.XtextDocumentProvider;
import org.eclipse.xtext.util.concurrent.IUnitOfWork;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class contains commonly used methods when writing wizard generators.
 */
@Log
@SuppressWarnings("all")
public class WizardGeneratorHelper {
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private ChangeManager changeManager;
  
  @Inject
  private XtextDocumentProvider docProvider;
  
  @Inject
  private N4JSImportRequirementResolver requirementResolver;
  
  public static final String LINEBREAK = "\n";
  
  /**
   * Return the last character of a given file.
   */
  public String lastCharacterInFile(final IFile file) {
    try {
      final String contents = this.readFileAsString(file);
      int _length = contents.length();
      int _minus = (_length - 1);
      return Character.valueOf(contents.charAt(_minus)).toString();
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        return "";
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Returns the given string with a trailing line break.
   * 
   * If the string is empty no line break is added.
   */
  public String addLineBreak(final String str) {
    String _xifexpression = null;
    boolean _isEmpty = str.isEmpty();
    if (_isEmpty) {
      _xifexpression = str;
    } else {
      _xifexpression = (str + WizardGeneratorHelper.LINEBREAK);
    }
    return _xifexpression;
  }
  
  /**
   * Returns the given string with a trailing space.
   * 
   * If the string is empty an empty string is returned.
   */
  public String addSpace(final String str) {
    String _xifexpression = null;
    boolean _isEmpty = str.isEmpty();
    if (_isEmpty) {
      _xifexpression = str;
    } else {
      _xifexpression = (str + " ");
    }
    return _xifexpression;
  }
  
  /**
   * Returns the export statement if the modifier
   * requires it or an empty string if not.
   */
  public String exportStatement(final AccessModifier modifier) {
    String _xifexpression = null;
    if ((Objects.equal(modifier, AccessModifier.PROJECT) || Objects.equal(modifier, AccessModifier.PUBLIC))) {
      _xifexpression = "export";
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }
  
  /**
   * Returns the content of the file as a string.
   */
  public String readFileAsString(final IFile file) throws IOException, CoreException, UnsupportedEncodingException {
    return Files.toString(file.getLocation().toFile(), Charset.defaultCharset());
  }
  
  /**
   * Inserts the corresponding import statements for the given model into an existing file.
   * 
   * <p>The method tries to find the files import region and append the import statements to it</p>
   */
  public void insertImportStatements(final XtextResource moduleResource, final List<ImportRequirement> importRequirements) {
    final IAtomicChange importReplacement = this.requirementResolver.getImportStatementChanges(moduleResource, importRequirements);
    this.applyChanges(moduleResource, Collections.<IAtomicChange>unmodifiableList(CollectionLiterals.<IAtomicChange>newArrayList(importReplacement)));
  }
  
  /**
   * Returns true if the given path exists in the workspace.
   * 
   * Note that the path must contain a project segment and at least one additional segment.
   */
  public boolean exists(final IPath path) {
    boolean _xblockexpression = false;
    {
      if ((null == path)) {
        return false;
      }
      final IResource member = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
      if ((null == member)) {
        return false;
      }
      _xblockexpression = member.exists();
    }
    return _xblockexpression;
  }
  
  /**
   * Load and return the {@link XtextResource} at the given URI
   */
  public XtextResource getResource(final URI moduleURI) {
    Object _xblockexpression = null;
    {
      final ResourceSet resourceSet = this.n4jsCore.createResourceSet(Optional.<IN4JSProject>fromNullable(null));
      final Resource moduleResource = resourceSet.getResource(moduleURI, true);
      if ((moduleResource instanceof XtextResource)) {
        return ((XtextResource)moduleResource);
      }
      _xblockexpression = null;
    }
    return ((XtextResource)_xblockexpression);
  }
  
  /**
   * Retrieve the XtextDocument for the given resource and apply the changes
   * 
   * @param resource The XtextResource to modify
   * @param changes The changes to apply
   */
  public boolean applyChanges(final XtextResource resource, final Collection<? extends IAtomicChange> changes) {
    String _string = resource.getURI().toString();
    Path _path = new Path(_string);
    Path _path_1 = new Path("platform:/resource/");
    final IPath resourcePath = _path.makeRelativeTo(_path_1);
    final IFile resourceFile = ResourcesPlugin.getWorkspace().getRoot().getFile(resourcePath);
    boolean _exists = resourceFile.exists();
    if (_exists) {
      try {
        final FileEditorInput fileInput = new FileEditorInput(resourceFile);
        this.docProvider.connect(fileInput);
        IDocument _document = this.docProvider.getDocument(fileInput);
        final IXtextDocument document = ((IXtextDocument) _document);
        this.docProvider.aboutToChange(fileInput);
        document.<Object>modify(
          new IUnitOfWork.Void<XtextResource>() {
            @Override
            public void process(final XtextResource state) throws Exception {
              try {
                final Function1<IAtomicChange, Integer> _function = (IAtomicChange it) -> {
                  return Integer.valueOf(it.getOffset());
                };
                WizardGeneratorHelper.this.changeManager.applyAllInSameDocument(ListExtensions.reverse(IterableExtensions.sortBy(changes, _function)), document);
              } catch (final Throwable _t) {
                if (_t instanceof BadLocationException) {
                  return;
                } else {
                  throw Exceptions.sneakyThrow(_t);
                }
              }
            }
          });
        this.docProvider.saveDocument(null, fileInput, document, true);
        this.docProvider.changed(fileInput);
        this.docProvider.disconnect(fileInput);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          return false;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } else {
      return false;
    }
    return true;
  }
  
  /**
   * Applies the given list of {@link IJSONDocumentModification}s to the given JSON resource.
   * 
   * Runs the JSON formatter on the whole file after applying the modification.
   * 
   * @param resource
   * 			The XtextResource to modify.
   * @param changes
   * 			The JSON document modifications to apply.
   */
  public boolean applyJSONModifications(final XtextResource resource, final Collection<? extends IJSONDocumentModification> modifications) {
    String _string = resource.getURI().toString();
    Path _path = new Path(_string);
    Path _path_1 = new Path("platform:/resource/");
    final IPath resourcePath = _path.makeRelativeTo(_path_1);
    final IFile resourceFile = ResourcesPlugin.getWorkspace().getRoot().getFile(resourcePath);
    final JSONDocument jsonDocument = JSONModelUtils.getDocument(resource);
    boolean _exists = resourceFile.exists();
    if (_exists) {
      try {
        for (final IJSONDocumentModification modification : modifications) {
          modification.apply(jsonDocument);
        }
        resource.save(SaveOptions.newBuilder().format().getOptions().toOptionsMap());
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          return false;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } else {
      return false;
    }
    return true;
  }
  
  /**
   * Run organize import on the given file and save it.
   * 
   * This method works in the background without opening the graphical editor.
   */
  public void organizeImports(final IFile file, final IProgressMonitor mon) throws CoreException {
    OrganizeImportsService.organizeImportsInFile(file, Interaction.takeFirst, mon);
  }
  
  /**
   * Return the project name of the containing project of the given uri.
   */
  public IN4JSProject projectOfUri(final URI uri) {
    final Optional<? extends IN4JSProject> projectOptional = this.n4jsCore.findProject(uri);
    boolean _isPresent = projectOptional.isPresent();
    if (_isPresent) {
      return projectOptional.get();
    }
    return null;
  }
  
  /**
   * Returns the real or bound name of the classifier reference.
   * 
   * Always prioritizes alias name over real name.
   * 
   * @param reference The classifier reference
   * @param aliasBindings The alias bindings, may be null
   */
  public String realOrAliasName(final ClassifierReference reference, final Map<URI, String> aliasBindings) {
    if (((aliasBindings != null) && aliasBindings.containsKey(reference.uri))) {
      return aliasBindings.get(reference.uri);
    }
    return reference.classifierName;
  }
  
  /**
   * Return the package.json modifications which need to be applied in order to allow referencing of
   * the given projects/runtime libraries.
   * 
   * @param packageJson
   * 			The package.json resource
   * @param model
   * 			The workspace wizard model
   * @param referencedProjects
   * 			A list of the projects to be referenced
   * @param moduleURI
   * 			The platform uri of the target module
   * 
   * @returns A list of {@link IAtomicChange}s for the manifest resource.
   */
  public Collection<IJSONDocumentModification> projectDescriptionModifications(final Resource packageJson, final WorkspaceWizardModel model, final Collection<IN4JSProject> referencedProjects, final URI moduleURI) {
    final ArrayList<IJSONDocumentModification> modifications = new ArrayList<IJSONDocumentModification>();
    final Function1<IN4JSProject, Boolean> _function = (IN4JSProject it) -> {
      boolean _equals = it.getProjectName().equals(model.getProject().lastSegment());
      return Boolean.valueOf((!_equals));
    };
    final Iterable<IN4JSProject> referencedProjectsExceptContainer = IterableExtensions.<IN4JSProject>filter(referencedProjects, _function);
    final HashSet<IN4JSProject> referencedProjectsSet = new HashSet<IN4JSProject>();
    Iterables.<IN4JSProject>addAll(referencedProjectsSet, referencedProjectsExceptContainer);
    final Function1<IN4JSProject, String> _function_1 = (IN4JSProject it) -> {
      return it.getProjectName();
    };
    modifications.add(
      PackageJsonModificationProvider.insertProjectDependencies(
        IterableExtensions.<String>toList(IterableExtensions.<IN4JSProject, String>map(referencedProjectsSet, _function_1))));
    final Function1<IN4JSProject, Boolean> _function_2 = (IN4JSProject it) -> {
      ProjectType _projectType = it.getProjectType();
      return Boolean.valueOf(Objects.equal(_projectType, ProjectType.RUNTIME_LIBRARY));
    };
    final Function1<IN4JSProject, String> _function_3 = (IN4JSProject it) -> {
      return it.getProjectName();
    };
    modifications.add(PackageJsonModificationProvider.insertRequiredRuntimeLibraries(IterableExtensions.<String>toList(IterableExtensions.<IN4JSProject, String>map(IterableExtensions.<IN4JSProject>filter(referencedProjectsSet, _function_2), _function_3))));
    return modifications;
  }
  
  private static final Logger logger = Logger.getLogger(WizardGeneratorHelper.class);
}
