/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.wizard.project;

import java.util.function.Consumer;
import org.eclipse.n4js.packagejson.PackageJsonBuilder;
import org.eclipse.n4js.packagejson.PackageJsonProperties;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectDescription.SourceContainerType;
import org.eclipse.n4js.ui.wizard.project.N4JSProjectInfo;
import org.eclipse.n4js.utils.ProjectDescriptionUtils;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * Basic Xtend templates for new project wizard.
 */
@SuppressWarnings("all")
public class N4JSNewProjectFileTemplates {
  /**
   * Returns the contents of the greeter class module for given projectName
   * 
   * @param projectName The name of the project
   * @param safeProjectName The name of the project transformed to be safe to use as class identifier
   */
  public static CharSequence getSourceFileWithGreeterClass(final String projectName, final String safeProjectName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("export class GreeterFor_");
    _builder.append(safeProjectName);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("    ");
    _builder.append("public greet() {");
    _builder.newLine();
    _builder.append("        ");
    _builder.append("console.log(\"Hello World from \'");
    _builder.append(projectName, "        ");
    _builder.append("\'!\");");
    _builder.newLineIfNotEmpty();
    _builder.append("    ");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("var greeter = new GreeterFor_");
    _builder.append(safeProjectName);
    _builder.append("();");
    _builder.newLineIfNotEmpty();
    _builder.append("greeter.greet();");
    _builder.newLine();
    _builder.append("//right click this module and select \"Run As -> Launch in Node.js\" to see");
    _builder.newLine();
    _builder.append("//\"Hello World from \'");
    _builder.append(projectName);
    _builder.append("\'!\"");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    return _builder;
  }
  
  /**
   * Returns the contents of the greeter test class module for given projectName
   * 
   * @param safeProjectName The name of the project transformed to be safe to use as class identifier
   */
  public static CharSequence getSourceFileWithTestGreeter(final String safeProjectName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("import { Assert } from \"org/eclipse/n4js/mangelhaft/assert/Assert\"");
    _builder.newLine();
    _builder.newLine();
    _builder.append("export public class Test_");
    _builder.append(safeProjectName);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("@Test");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("testPass() {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("Assert.isTrue(true, \"Testing works!\");");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("@Test");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("testFail() {");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("Assert.fail(\"Test failed.\");");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    _builder.append("//right click this module and select \"Run As -> Test in Node.js\" to see the test results");
    _builder.newLine();
    return _builder;
  }
  
  /**
   * Returns the project description file contents for the given project info (package.json).
   */
  public static String getProjectDescriptionContents(final N4JSProjectInfo projectInfo) {
    final String projectName = ProjectDescriptionUtils.convertEclipseProjectNameToN4JSProjectName(projectInfo.getProjectName());
    final PackageJsonBuilder builder = PackageJsonBuilder.newBuilder().withName(projectName).withVersion(PackageJsonProperties.VERSION.defaultValue).withType(projectInfo.getProjectType()).withOutput(projectInfo.getOutputFolder()).withVendorId(projectInfo.getVendorId());
    final Consumer<String> _function = (String sourceFolder) -> {
      builder.withSourceContainer(SourceContainerType.SOURCE, sourceFolder);
    };
    projectInfo.getSourceFolders().forEach(_function);
    final Consumer<String> _function_1 = (String sourceFolder) -> {
      builder.withSourceContainer(SourceContainerType.EXTERNAL, sourceFolder);
    };
    projectInfo.getExternalSourceFolders().forEach(_function_1);
    final Consumer<String> _function_2 = (String sourceFolder) -> {
      builder.withSourceContainer(SourceContainerType.TEST, sourceFolder);
    };
    projectInfo.getTestSourceFolders().forEach(_function_2);
    boolean _equals = ProjectType.LIBRARY.equals(projectInfo.getProjectType());
    if (_equals) {
      String _implementationId = projectInfo.getImplementationId();
      boolean _tripleNotEquals = (_implementationId != null);
      if (_tripleNotEquals) {
        builder.withImplementationId(projectInfo.getImplementationId());
      }
      final Consumer<String> _function_3 = (String p) -> {
        builder.withImplementedProject(p);
      };
      projectInfo.getImplementedProjects().forEach(_function_3);
    }
    boolean _equals_1 = ProjectType.TEST.equals(projectInfo.getProjectType());
    if (_equals_1) {
      final Consumer<String> _function_4 = (String p) -> {
        builder.withTestedProject(p);
      };
      projectInfo.getTestedProjects().forEach(_function_4);
    }
    final Consumer<String> _function_5 = (String dep) -> {
      builder.withDependency(dep);
    };
    projectInfo.getProjectDependencies().forEach(_function_5);
    final Consumer<String> _function_6 = (String dep) -> {
      builder.withDevDependency(dep);
    };
    projectInfo.getProjectDevDependencies().forEach(_function_6);
    return builder.build();
  }
}
