/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4JS.extensions;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.n4JS.VariableEnvironmentElement;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Extensions for source element, in particular for statements.
 * Because this extension requires the type inferencer, it is part of the n4js bundle rather than the n4js.model bundle.
 */
@Singleton
@SuppressWarnings("all")
public class SourceElementExtensions {
  @Inject
  @Extension
  private IResourceScopeCache cache;
  
  /**
   * Collects all elements visible from the given element. This will include the element itself, if it is either a
   * {@link FunctionExpression} or a {@link IdentifiableElement}. Note that a {@code N4Class} is not variable environment,
   * so you cannot scope elements in its content. All the children of given element will also be searched for visible
   * identifiable elements.
   * 
   * this element will be walked through
   * @param element the given element
   * @return the list of EObjects visible
   */
  public List<IdentifiableElement> collectVisibleIdentifiableElements(final VariableEnvironmentElement element) {
    Pair<String, VariableEnvironmentElement> _mappedTo = Pair.<String, VariableEnvironmentElement>of("collectVisibleIdentifiableElements", element);
    final Provider<ArrayList<IdentifiableElement>> _function = () -> {
      ArrayList<IdentifiableElement> _xblockexpression = null;
      {
        final ArrayList<IdentifiableElement> result = CollectionLiterals.<IdentifiableElement>newArrayList();
        boolean _matched = false;
        if (element instanceof IdentifiableElement) {
          _matched=true;
          result.add(((IdentifiableElement) element));
        }
        if (!_matched) {
          if (element instanceof FunctionExpression) {
            String _name = ((FunctionExpression)element).getName();
            boolean _tripleNotEquals = (_name != null);
            if (_tripleNotEquals) {
              _matched=true;
              this.<FunctionExpression>collectVisibleTypedElement(((FunctionExpression)element), result);
            }
          }
        }
        this.doCollectVisibleIdentifiableElements(element, element, true, result);
        _xblockexpression = result;
      }
      return _xblockexpression;
    };
    return this.cache.<ArrayList<IdentifiableElement>>get(_mappedTo, element.eResource(), _function);
  }
  
  /**
   * collect only the local arguments variable - if available.
   * @param element - an possible provider of an arguments variable (function,function-expression, getter/setter, mehtod)
   * @return list with single entry of an arguments variable or empty list.
   */
  public List<IdentifiableElement> collectLocalArguments(final VariableEnvironmentElement element) {
    Pair<String, VariableEnvironmentElement> _mappedTo = Pair.<String, VariableEnvironmentElement>of("collectLocalArguments", element);
    final Provider<ArrayList<IdentifiableElement>> _function = () -> {
      ArrayList<IdentifiableElement> _xblockexpression = null;
      {
        final ArrayList<IdentifiableElement> result = CollectionLiterals.<IdentifiableElement>newArrayList();
        if ((element instanceof FunctionOrFieldAccessor)) {
          if ((!(element instanceof ArrowFunction))) {
            result.add(((FunctionOrFieldAccessor)element).getLocalArgumentsVariable());
          }
        }
        _xblockexpression = result;
      }
      return _xblockexpression;
    };
    return this.cache.<ArrayList<IdentifiableElement>>get(_mappedTo, element.eResource(), _function);
  }
  
  /**
   * @return the defined type of the given element or the declared type of the corresponding polyfilled class
   */
  public Type getTypeOrPolyfilledType(final TypeDefiningElement tde) {
    if ((tde instanceof N4ClassDeclaration)) {
      if ((N4JSLanguageUtils.isPolyfill(((AnnotableElement)tde)) || N4JSLanguageUtils.isStaticPolyfill(((AnnotableElement)tde)))) {
        TClass _definedTypeAsClass = ((N4ClassDeclaration)tde).getDefinedTypeAsClass();
        ParameterizedTypeRef _superClassRef = null;
        if (_definedTypeAsClass!=null) {
          _superClassRef=_definedTypeAsClass.getSuperClassRef();
        }
        Type _declaredType = null;
        if (_superClassRef!=null) {
          _declaredType=_superClassRef.getDeclaredType();
        }
        final Type filledType = _declaredType;
        return filledType;
      }
    }
    return tde.getDefinedType();
  }
  
  private <T extends TypeDefiningElement> boolean collectVisibleTypedElement(final T element, final List<? super IdentifiableElement> addHere) {
    final Function1<T, Type> _function = (T e) -> {
      return e.getDefinedType();
    };
    return this.<T, Type>collectVisibleIdentifiableElement(element, addHere, _function);
  }
  
  private <T extends ExportedVariableDeclaration> boolean collectVisibleVariable(final T element, final List<? super IdentifiableElement> addHere) {
    final Function1<T, TVariable> _function = (T e) -> {
      return e.getDefinedVariable();
    };
    return this.<T, TVariable>collectVisibleIdentifiableElement(element, addHere, _function);
  }
  
  /**
   * Determines the defined for the given element, if there is no one, it is tried to infer it via
   * {@link N4JSTypeSystem#tau(EObject)}. If a type could be inferred or was already there it
   * will be collected.
   */
  private <T extends TypableElement, U extends IdentifiableElement> boolean collectVisibleIdentifiableElement(final T element, final List<? super IdentifiableElement> addHere, final Function1<? super T, ? extends U> calculateType) {
    boolean _xifexpression = false;
    if ((calculateType != null)) {
      boolean _xblockexpression = false;
      {
        final U type = calculateType.apply(element);
        boolean _xifexpression_1 = false;
        if ((type != null)) {
          _xifexpression_1 = addHere.add(type);
        }
        _xblockexpression = _xifexpression_1;
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  /**
   * Walk through all contents list of the given element. If the entry in the list is a N4ClassDefinition,
   * a N4EnumDeclaration, a ExportedVariableDeclaration or a FunctionDefinition its children are walked, too.
   * If the entry in the list is a IdentifiableElement this is also collected. If the entry in the list is a
   * before mentioned element or a Expression the iterator is pruned as it is clear that below these elements no
   * further visible identifiable elements can be found.
   */
  private void doCollectVisibleIdentifiableElements(final VariableEnvironmentElement start, final EObject element, final boolean includeBlockScopedElements, final List<? super IdentifiableElement> addHere) {
    final TreeIterator<EObject> allContents = element.eAllContents();
    while (allContents.hasNext()) {
      {
        final EObject next = allContents.next();
        boolean _matched = false;
        if (next instanceof N4ClassDeclaration) {
          _matched=true;
          final Type polyfilledOrOriginalType = this.getTypeOrPolyfilledType(((TypeDefiningElement)next));
          N4ClassDeclaration _xifexpression = null;
          if ((polyfilledOrOriginalType instanceof TClass)) {
            N4ClassDeclaration _xblockexpression = null;
            {
              EObject _astElement = null;
              if (((TClass)polyfilledOrOriginalType)!=null) {
                _astElement=((TClass)polyfilledOrOriginalType).getAstElement();
              }
              final N4ClassDeclaration n4ClassDecl = ((N4ClassDeclaration) _astElement);
              N4ClassDeclaration _xifexpression_1 = null;
              if ((n4ClassDecl == null)) {
                _xifexpression_1 = ((N4ClassDeclaration)next);
              } else {
                _xifexpression_1 = n4ClassDecl;
              }
              _xblockexpression = _xifexpression_1;
            }
            _xifexpression = _xblockexpression;
          } else {
            _xifexpression = ((N4ClassDeclaration)next);
          }
          final N4ClassDeclaration nonNullClassDecl = _xifexpression;
          this.<N4ClassDeclaration>collectVisibleTypedElement(nonNullClassDecl, addHere);
          allContents.prune();
        }
        if (!_matched) {
          if (next instanceof N4InterfaceDeclaration) {
            _matched=true;
            this.<N4InterfaceDeclaration>collectVisibleTypedElement(((N4InterfaceDeclaration)next), addHere);
            allContents.prune();
          }
        }
        if (!_matched) {
          if (next instanceof N4EnumDeclaration) {
            _matched=true;
            this.<N4EnumDeclaration>collectVisibleTypedElement(((N4EnumDeclaration)next), addHere);
            allContents.prune();
          }
        }
        if (!_matched) {
          if (next instanceof FunctionDeclaration) {
            _matched=true;
            this.<FunctionDeclaration>collectVisibleTypedElement(((FunctionDeclaration)next), addHere);
            allContents.prune();
          }
        }
        if (!_matched) {
          if (next instanceof ExportedVariableDeclaration) {
            String _name = ((ExportedVariableDeclaration)next).getName();
            boolean _tripleNotEquals = (_name != null);
            if (_tripleNotEquals) {
              _matched=true;
              this.<ExportedVariableDeclaration>collectVisibleVariable(((ExportedVariableDeclaration)next), addHere);
              allContents.prune();
            }
          }
        }
        if (!_matched) {
          if (next instanceof IdentifiableElement) {
            if ((!(next instanceof TypeVariable))) {
              _matched=true;
              boolean _isBlockScoped = N4JSASTUtils.isBlockScoped(((IdentifiableElement)next));
              if (_isBlockScoped) {
                if (includeBlockScopedElements) {
                  addHere.add(((IdentifiableElement)next));
                }
              } else {
                boolean _belongsToScope = this.belongsToScope(((IdentifiableElement)next), start);
                if (_belongsToScope) {
                  addHere.add(((IdentifiableElement)next));
                }
              }
              allContents.prune();
            }
          }
        }
        if (!_matched) {
          if (next instanceof Block) {
            _matched=true;
            this.doCollectVisibleIdentifiableElements(start, next, false, addHere);
            allContents.prune();
          }
        }
        if (!_matched) {
          if (next instanceof VariableEnvironmentElement) {
            _matched=true;
            this.doCollectVisibleIdentifiableElements(start, next, false, addHere);
            allContents.prune();
          }
        }
        if (!_matched) {
          if (next instanceof Expression) {
            _matched=true;
            allContents.prune();
          }
        }
        if (!_matched) {
          if (next instanceof TypeRef) {
            _matched=true;
            allContents.prune();
          }
        }
      }
    }
  }
  
  private boolean belongsToScope(final IdentifiableElement elem, final VariableEnvironmentElement scope) {
    VariableEnvironmentElement _scope = N4JSASTUtils.getScope(elem);
    return (_scope == scope);
  }
}
