/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.naming;

import com.google.common.base.Objects;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.JSON.JSONObject;
import org.eclipse.n4js.json.JSON.JSONStringLiteral;
import org.eclipse.n4js.json.JSON.JSONValue;
import org.eclipse.n4js.json.model.utils.JSONModelUtils;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.packagejson.PackageJsonProperties;
import org.eclipse.n4js.ts.scoping.N4TSQualifiedNameProvider;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.ProjectDescriptionUtils;
import org.eclipse.xtext.naming.QualifiedName;

/**
 * Calculates the fully qualified name for the passed in objects.
 * <p>
 * Be very careful when changing anything here as the FQN affects a lot of concepts, including scoping and even typing.
 * That is, elements are often handled differently if they have a qualified name or not.
 */
@SuppressWarnings("all")
public class N4JSQualifiedNameProvider extends N4TSQualifiedNameProvider {
  /**
   * Last segment of fully qualified names for the root {@link JSONDocument} of package.json files.
   */
  public static final String PACKAGE_JSON_SEGMENT = "!package_json";
  
  /**
   * For the root element (Script) the resource qualified name is used.
   * For all other elements the resource qualified name plus the simple
   * name of the element is used as qualified name.
   * Exceptions are IdentifiableElement, N4ClassExpression and FunctionExpression
   * for which only the simple name is returned.
   * For a ExportDeclaration the qualified name of its contained element is returned.
   * For a TModule the qualified name is just converted from dots to slashes.
   */
  @Override
  public QualifiedName getFullyQualifiedName(final EObject it) {
    QualifiedName _switchResult = null;
    boolean _matched = false;
    if (it instanceof Script) {
      _matched=true;
      _switchResult = this.getFullyQualifiedName(((Script)it).getModule());
    }
    if (!_matched) {
      if (it instanceof TModule) {
        _matched=true;
        QualifiedName _xifexpression = null;
        String _qualifiedName = ((TModule)it).getQualifiedName();
        boolean _tripleNotEquals = (_qualifiedName != null);
        if (_tripleNotEquals) {
          _xifexpression = this.fqnTModule(((TModule)it));
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof N4TypeDeclaration) {
        _matched=true;
        QualifiedName _xifexpression = null;
        String _name = ((N4TypeDeclaration)it).getName();
        boolean _tripleNotEquals = (_name != null);
        if (_tripleNotEquals) {
          _xifexpression = this.fqnTypeDeclaration(((N4TypeDeclaration)it));
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof FunctionDeclaration) {
        _matched=true;
        QualifiedName _xifexpression = null;
        if (((((FunctionDeclaration)it).getName() != null) && (((FunctionDeclaration)it).eContainer() instanceof ExportDeclaration))) {
          QualifiedName _fullyQualifiedName = this.getFullyQualifiedName(EcoreUtil.getRootContainer(it));
          QualifiedName _append = null;
          if (_fullyQualifiedName!=null) {
            _append=_fullyQualifiedName.append(((FunctionDeclaration)it).getName());
          }
          _xifexpression = _append;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof VariableDeclaration) {
        _matched=true;
        QualifiedName _xifexpression = null;
        if (((((VariableDeclaration)it).getName() != null) && (((VariableDeclaration)it).eContainer() instanceof ExportDeclaration))) {
          QualifiedName _fullyQualifiedName = this.getFullyQualifiedName(EcoreUtil.getRootContainer(it));
          QualifiedName _append = null;
          if (_fullyQualifiedName!=null) {
            _append=_fullyQualifiedName.append(((VariableDeclaration)it).getName());
          }
          _xifexpression = _append;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof TClass) {
        _matched=true;
        QualifiedName _xifexpression = null;
        String _name = ((TClass)it).getName();
        boolean _tripleNotEquals = (_name != null);
        if (_tripleNotEquals) {
          _xifexpression = this.fqnTClassifier(((TClassifier)it));
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof TInterface) {
        _matched=true;
        QualifiedName _xifexpression = null;
        String _name = ((TInterface)it).getName();
        boolean _tripleNotEquals = (_name != null);
        if (_tripleNotEquals) {
          _xifexpression = this.fqnTClassifier(((TClassifier)it));
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof TEnum) {
        _matched=true;
        QualifiedName _xifexpression = null;
        String _name = ((TEnum)it).getName();
        boolean _tripleNotEquals = (_name != null);
        if (_tripleNotEquals) {
          QualifiedName _fullyQualifiedName = this.getFullyQualifiedName(EcoreUtil.getRootContainer(it));
          QualifiedName _append = null;
          if (_fullyQualifiedName!=null) {
            String _elvis = null;
            String _exportedName = ((TEnum)it).getExportedName();
            if (_exportedName != null) {
              _elvis = _exportedName;
            } else {
              String _name_1 = ((TEnum)it).getName();
              _elvis = _name_1;
            }
            _append=_fullyQualifiedName.append(_elvis);
          }
          _xifexpression = _append;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof TFunction) {
        _matched=true;
        QualifiedName _xifexpression = null;
        if (((((TFunction)it).getName() != null) && ((TFunction)it).isExported())) {
          QualifiedName _fullyQualifiedName = this.getFullyQualifiedName(EcoreUtil.getRootContainer(it));
          QualifiedName _append = null;
          if (_fullyQualifiedName!=null) {
            _append=_fullyQualifiedName.append(((TFunction)it).getExportedName());
          }
          _xifexpression = _append;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof TVariable) {
        _matched=true;
        QualifiedName _xifexpression = null;
        if (((((TVariable)it).getName() != null) && ((TVariable)it).isExported())) {
          QualifiedName _fullyQualifiedName = this.getFullyQualifiedName(EcoreUtil.getRootContainer(it));
          QualifiedName _append = null;
          if (_fullyQualifiedName!=null) {
            _append=_fullyQualifiedName.append(((TVariable)it).getExportedName());
          }
          _xifexpression = _append;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof ExportDeclaration) {
        _matched=true;
        ExportableElement _exportedElement = ((ExportDeclaration)it).getExportedElement();
        QualifiedName _fullyQualifiedName = null;
        if (_exportedElement!=null) {
          _fullyQualifiedName=this.getFullyQualifiedName(_exportedElement);
        }
        _switchResult = _fullyQualifiedName;
      }
    }
    if (!_matched) {
      if (it instanceof TypeVariable) {
        _matched=true;
        _switchResult = null;
      }
    }
    if (!_matched) {
      if (it instanceof Type) {
        _matched=true;
        QualifiedName _xifexpression = null;
        String _name = ((Type)it).getName();
        boolean _tripleNotEquals = (_name != null);
        if (_tripleNotEquals) {
          _xifexpression = QualifiedName.create(((Type)it).getName());
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      if (it instanceof TMember) {
        _matched=true;
        _switchResult = null;
      }
    }
    if (!_matched) {
      if (it instanceof IdentifiableElement) {
        _matched=true;
        _switchResult = null;
      }
    }
    if (!_matched) {
      if (it instanceof JSONDocument) {
        _matched=true;
        _switchResult = this.fqnJSONDocument(((JSONDocument)it));
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  private QualifiedName fqnTModule(final TModule module) {
    if (((module.getQualifiedName().length() != 0) && (!AnnotationDefinition.GLOBAL.hasAnnotation(module)))) {
      QualifiedName plainQN = this.converter.toQualifiedName(module.getQualifiedName());
      boolean _isStaticPolyfillModule = module.isStaticPolyfillModule();
      if (_isStaticPolyfillModule) {
        return N4TSQualifiedNameProvider.prepend(N4TSQualifiedNameProvider.MODULE_POLYFILL_SEGMENT, plainQN);
      }
      return plainQN;
    } else {
      return QualifiedName.create(N4TSQualifiedNameProvider.GLOBAL_NAMESPACE_SEGMENT);
    }
  }
  
  private QualifiedName fqnTypeDeclaration(final N4TypeDeclaration typeDecl) {
    QualifiedName prefix = this.getFullyQualifiedName(EcoreUtil.getRootContainer(typeDecl));
    if ((N4JSLanguageUtils.isPolyfill(typeDecl) || N4JSLanguageUtils.isStaticPolyfill(typeDecl))) {
      prefix = N4TSQualifiedNameProvider.append(prefix, N4TSQualifiedNameProvider.POLYFILL_SEGMENT);
    }
    String _elvis = null;
    String _exportedName = typeDecl.getExportedName();
    if (_exportedName != null) {
      _elvis = _exportedName;
    } else {
      String _name = typeDecl.getName();
      _elvis = _name;
    }
    final QualifiedName fqn = N4TSQualifiedNameProvider.append(prefix, _elvis);
    return fqn;
  }
  
  private QualifiedName fqnTClassifier(final TClassifier tClassifier) {
    QualifiedName prefix = this.getFullyQualifiedName(EcoreUtil.getRootContainer(tClassifier));
    boolean _isPolyfill = tClassifier.isPolyfill();
    if (_isPolyfill) {
      prefix = N4TSQualifiedNameProvider.append(prefix, N4TSQualifiedNameProvider.POLYFILL_SEGMENT);
    }
    String _elvis = null;
    String _exportedName = tClassifier.getExportedName();
    if (_exportedName != null) {
      _elvis = _exportedName;
    } else {
      String _name = tClassifier.getName();
      _elvis = _name;
    }
    final QualifiedName fqn = N4TSQualifiedNameProvider.append(prefix, _elvis);
    return fqn;
  }
  
  private QualifiedName fqnJSONDocument(final JSONDocument document) {
    final Resource res = document.eResource();
    URI _uRI = null;
    if (res!=null) {
      _uRI=res.getURI();
    }
    final URI uri = _uRI;
    if (((uri == null) || (!Objects.equal(uri.lastSegment(), N4JSGlobals.PACKAGE_JSON)))) {
      return null;
    }
    String projectName = null;
    final JSONValue content = document.getContent();
    if ((content instanceof JSONObject)) {
      final JSONValue value = JSONModelUtils.getProperty(((JSONObject)content), PackageJsonProperties.NAME.name).orElse(null);
      String _xifexpression = null;
      if ((value instanceof JSONStringLiteral)) {
        _xifexpression = ((JSONStringLiteral)value).getValue();
      } else {
        _xifexpression = null;
      }
      projectName = _xifexpression;
    }
    if ((projectName == null)) {
      projectName = ProjectDescriptionUtils.deriveN4JSProjectNameFromURI(uri.trimSegments(1));
    }
    if (((projectName != null) && (!projectName.isEmpty()))) {
      final QualifiedName fqnBase = this.converter.toQualifiedName(projectName);
      if ((fqnBase != null)) {
        return fqnBase.append(N4JSQualifiedNameProvider.PACKAGE_JSON_SEGMENT);
      }
    }
    return null;
  }
}
