/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.compileTime.CompileTimeValue;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.n4JS.PropertySpread;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * Processing of {@link LiteralOrComputedPropertyName}s that have a computed property name, mainly setting property
 * {@link LiteralOrComputedPropertyName#getComputedName() 'computedName'}.
 * <p>
 * For details, see {@link ComputedNameProcessor#processComputedPropertyName(RuleEnvironment, LiteralOrComputedPropertyName, ASTMetaInfoCache, int)}.
 */
@Singleton
@SuppressWarnings("all")
public class ComputedNameProcessor {
  /**
   * If the given 'nameDecl' has a computed property name, this method will
   * <ol>
   * <li>obtain its expression's compile-time value from the cache (the actual evaluation of the expression happened
   * in {@link CompileTimeExpressionProcessor}),
   * <li>derive the actual property name from that value (cf. {@link #getPropertyNameFromExpression(RuleEnvironment, Expression, ASTMetaInfoCache)}),
   * <li>store this name in 'nameDecl' (for later use), and
   * <li>store this name in the corresponding TModule element (if such a TModule element exists).
   * </ol>
   * <p>
   * In case the compile-time value of the expression is invalid (i.e. the expression is not a valid compile-time
   * expression) no actual name will be stored in 'nameDecl' and the corresponding TModule element will be removed
   * from the TModule, entirely (if such a TModule element exists).
   */
  public void processComputedPropertyName(final RuleEnvironment G, final LiteralOrComputedPropertyName nameDecl, final ASTMetaInfoCache cache, final int indentLevel) {
    boolean _hasComputedPropertyName = nameDecl.hasComputedPropertyName();
    if (_hasComputedPropertyName) {
      final CompileTimeValue value = cache.getCompileTimeValue(nameDecl.getExpression());
      final String name = N4JSLanguageUtils.derivePropertyNameFromCompileTimeValue(value);
      if ((name != null)) {
        final Procedure0 _function = () -> {
          nameDecl.setComputedName(name);
        };
        EcoreUtilN4.doWithDeliver(false, _function, nameDecl);
        final EObject owner = nameDecl.eContainer();
        final EObject typeElem = N4JSASTUtils.getCorrespondingTypeModelElement(owner);
        if ((typeElem instanceof IdentifiableElement)) {
          final Procedure0 _function_1 = () -> {
            ((IdentifiableElement)typeElem).setName(name);
          };
          EcoreUtilN4.doWithDeliver(false, _function_1, typeElem);
        }
      } else {
        final EObject owner_1 = nameDecl.eContainer();
        this.discardTypeModelElement(owner_1);
      }
    }
  }
  
  /**
   * Discards the types model element corresponding to the given AST node. Throws exception is given AST node does not
   * have a corresponding types model element.
   */
  private void discardTypeModelElement(final EObject astNode) {
    final EObject elem = N4JSASTUtils.getCorrespondingTypeModelElement(astNode);
    if ((elem == null)) {
      throw new IllegalArgumentException(
        "given AST node does not have a corresponding type model element to discard");
    }
    if ((elem instanceof SyntaxRelatedTElement)) {
      ((SyntaxRelatedTElement)elem).setAstElement(null);
    }
    final Procedure0 _function = () -> {
      boolean _matched = false;
      if (astNode instanceof TypeDefiningElement) {
        _matched=true;
        ((TypeDefiningElement)astNode).setDefinedType(null);
      }
      if (!_matched) {
        if (astNode instanceof N4FieldDeclaration) {
          _matched=true;
          ((N4FieldDeclaration)astNode).setDefinedField(null);
        }
      }
      if (!_matched) {
        if (astNode instanceof N4GetterDeclaration) {
          _matched=true;
          ((N4GetterDeclaration)astNode).setDefinedGetter(null);
        }
      }
      if (!_matched) {
        if (astNode instanceof N4SetterDeclaration) {
          _matched=true;
          ((N4SetterDeclaration)astNode).setDefinedSetter(null);
        }
      }
      if (!_matched) {
        if (astNode instanceof PropertyNameValuePair) {
          _matched=true;
          ((PropertyNameValuePair)astNode).setDefinedField(null);
        }
      }
      if (!_matched) {
        if (astNode instanceof PropertyGetterDeclaration) {
          _matched=true;
          ((PropertyGetterDeclaration)astNode).setDefinedGetter(null);
        }
      }
      if (!_matched) {
        if (astNode instanceof PropertySetterDeclaration) {
          _matched=true;
          ((PropertySetterDeclaration)astNode).setDefinedSetter(null);
        }
      }
      if (!_matched) {
        if (astNode instanceof PropertySpread) {
          _matched=true;
        }
      }
      if (!_matched) {
        throw new UnsupportedOperationException(("switch case missing for: " + astNode));
      }
    };
    EcoreUtilN4.doWithDeliver(false, _function, astNode);
    final Procedure0 _function_1 = () -> {
      EcoreUtil2.remove(elem);
    };
    EcoreUtilN4.doWithDeliver(false, _function_1, elem.eContainer());
  }
}
