/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.scoping;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.name.Named;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.JSXElement;
import org.eclipse.n4js.n4JS.JSXElementName;
import org.eclipse.n4js.n4JS.JSXPropertyAttribute;
import org.eclipse.n4js.n4JS.LabelledStatement;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.MemberAccess;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldAccessor;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableEnvironmentElement;
import org.eclipse.n4js.n4JS.extensions.SourceElementExtensions;
import org.eclipse.n4js.n4idl.scoping.FailedToInferContextVersionWrappingScope;
import org.eclipse.n4js.n4idl.scoping.MigrationScopeHelper;
import org.eclipse.n4js.n4idl.scoping.N4IDLVersionAwareScope;
import org.eclipse.n4js.n4idl.scoping.NonVersionAwareContextScope;
import org.eclipse.n4js.n4idl.versioning.MigrationUtils;
import org.eclipse.n4js.n4idl.versioning.VersionHelper;
import org.eclipse.n4js.n4idl.versioning.VersionUtils;
import org.eclipse.n4js.n4jsx.ReactHelper;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.scoping.IContentAssistScopeProvider;
import org.eclipse.n4js.scoping.TopLevelElementsCollector;
import org.eclipse.n4js.scoping.accessModifiers.MemberVisibilityChecker;
import org.eclipse.n4js.scoping.accessModifiers.VisibilityAwareCtorScope;
import org.eclipse.n4js.scoping.imports.ImportedElementsScopingHelper;
import org.eclipse.n4js.scoping.members.MemberScopingHelper;
import org.eclipse.n4js.scoping.utils.DynamicPseudoScope;
import org.eclipse.n4js.scoping.utils.LocallyKnownTypesScopingHelper;
import org.eclipse.n4js.scoping.utils.MainModuleAwareSelectableBasedScope;
import org.eclipse.n4js.scoping.utils.N4JSTypesScopeFilter;
import org.eclipse.n4js.scoping.utils.ProjectImportEnablingScope;
import org.eclipse.n4js.scoping.utils.ScopesHelper;
import org.eclipse.n4js.ts.scoping.ValidatingScope;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TStructMethod;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeDefs;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.EObjectDescriptionHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.ResourceType;
import org.eclipse.n4js.utils.TameAutoClosable;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.ValidatorMessageHelper;
import org.eclipse.n4js.xtext.scoping.FilteringScope;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.resource.EObjectDescription;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.IScopeProvider;
import org.eclipse.xtext.scoping.Scopes;
import org.eclipse.xtext.scoping.impl.AbstractScopeProvider;
import org.eclipse.xtext.scoping.impl.IDelegatingScopeProvider;
import org.eclipse.xtext.scoping.impl.SimpleScope;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * This class contains custom scoping description.
 * 
 * Although some methods are called according to declarative scope provider, no reflection in
 * {@link #getScope(EObject,EReference)}.
 * 
 * see : http://www.eclipse.org/Xtext/documentation/latest/xtext.html#scoping
 * on how and when to use it
 */
@SuppressWarnings("all")
public class N4JSScopeProvider extends AbstractScopeProvider implements IDelegatingScopeProvider, IContentAssistScopeProvider {
  public static final String NAMED_DELEGATE = "org.eclipse.xtext.scoping.impl.AbstractDeclarativeScopeProvider.delegate";
  
  @Inject
  private IResourceScopeCache cache;
  
  /**
   * The scope provider creating the "parent" scope, i.e. including elements from the index
   */
  @Inject
  @Named(N4JSScopeProvider.NAMED_DELEGATE)
  private IScopeProvider delegate;
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private ResourceDescriptionsProvider resourceDescriptionsProvider;
  
  /**
   * Poor mans filter to reduce number of elements in getAllElements
   */
  @Inject
  @Extension
  private N4JSTypesScopeFilter _n4JSTypesScopeFilter;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private MemberScopingHelper memberScopingHelper;
  
  @Inject
  @Extension
  private LocallyKnownTypesScopingHelper locallyKnownTypesScopingHelper;
  
  @Inject
  @Extension
  private ImportedElementsScopingHelper _importedElementsScopingHelper;
  
  @Inject
  @Extension
  private SourceElementExtensions _sourceElementExtensions;
  
  @Inject
  private EObjectDescriptionHelper descriptionsHelper;
  
  @Inject
  @Extension
  private ReactHelper _reactHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  @Inject
  private MemberVisibilityChecker checker;
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private TopLevelElementsCollector topLevelElementCollector;
  
  @Inject
  private ScopesHelper scopesHelper;
  
  @Inject
  private VersionHelper versionHelper;
  
  @Inject
  private ValidatorMessageHelper messageHelper;
  
  @Inject
  private MigrationScopeHelper migrationScopeHelper;
  
  /**
   * True: Proxies of IdentifierRefs are only resolved within the resource. Otherwise, the proxy is returned.
   */
  private boolean suppressCrossFileResolutionOfIdentifierRef = false;
  
  public TameAutoClosable newCrossFileResolutionSuppressor() {
    abstract class __N4JSScopeProvider_1 implements TameAutoClosable {
      boolean tmpSuppressCrossFileResolutionOfIdentifierRef;
      
      abstract boolean init();
    }
    
    final TameAutoClosable tac = new __N4JSScopeProvider_1() {
      {
        tmpSuppressCrossFileResolutionOfIdentifierRef = this.init();
      }
      boolean init() {
        boolean _xblockexpression = false;
        {
          this.tmpSuppressCrossFileResolutionOfIdentifierRef = N4JSScopeProvider.this.suppressCrossFileResolutionOfIdentifierRef;
          _xblockexpression = N4JSScopeProvider.this.suppressCrossFileResolutionOfIdentifierRef = true;
        }
        return _xblockexpression;
      }
      
      @Override
      public void close() {
        N4JSScopeProvider.this.suppressCrossFileResolutionOfIdentifierRef = this.tmpSuppressCrossFileResolutionOfIdentifierRef;
      }
    };
    return tac;
  }
  
  protected IScope delegateGetScope(final EObject context, final EReference reference) {
    return this.delegate.getScope(context, reference);
  }
  
  @Override
  public IScopeProvider getDelegate() {
    return this.delegate;
  }
  
  /**
   * Dispatches to concrete scopes based on the context and reference inspection
   */
  @Override
  public IScope getScope(final EObject context, final EReference reference) {
    try {
      final ResourceType resourceType = ResourceType.getResourceType(context);
      if (resourceType != null) {
        switch (resourceType) {
          case N4JSX:
            return this.getN4JSXScope(context, reference);
          case JSX:
            return this.getN4JSXScope(context, reference);
          case N4IDL:
            return this.getN4IDLScope(context, reference);
          default:
            return this.getN4JSScope(context, reference);
        }
      } else {
        return this.getN4JSScope(context, reference);
      }
    } catch (final Throwable _t) {
      if (_t instanceof Error) {
        final Error ex = (Error)_t;
        if (((context != null) && context.eResource().getErrors().isEmpty())) {
          throw ex;
        } else {
        }
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return IScope.NULLSCOPE;
  }
  
  /**
   * Returns the N4JS scope for the given context and reference.
   * 
   * The returned scope is not sensitive to any of the language variants of N4JS. In order
   * to obtain a variant-specific scope, please use {@link N4JSScopeProvider#getScope(EObject, EReference)}.
   */
  public IScope getN4JSScope(final EObject context, final EReference reference) {
    final IScope maybeScopeShortcut = this.getScopeByShortcut(context, reference);
    if ((maybeScopeShortcut != IScope.NULLSCOPE)) {
      return maybeScopeShortcut;
    }
    return this.getScopeForContext(context, reference);
  }
  
  /**
   * shortcut to concrete scopes based on reference sniffing. Will return {@link IScope#NULLSCOPE} if no suitable scope found
   */
  private IScope getScopeByShortcut(final EObject context, final EReference reference) {
    boolean _equals = Objects.equal(reference, TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE);
    if (_equals) {
      if ((context instanceof ParameterizedTypeRef)) {
        final ModuleNamespaceVirtualType namespace = ((ParameterizedTypeRef)context).getAstNamespace();
        if ((namespace != null)) {
          return this.createScopeForNamespaceAccess(namespace, context);
        }
      }
    }
    if ((Objects.equal(reference, TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE) || Objects.equal(reference, TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__AST_NAMESPACE))) {
      IScope _typeScope = this.getTypeScope(context, false);
      Predicate<? super IEObjectDescription> _typesFilterCriteria = this._n4JSTypesScopeFilter.getTypesFilterCriteria(context, reference);
      return new ValidatingScope(_typeScope, _typesFilterCriteria);
    }
    EClass _eReferenceType = reference.getEReferenceType();
    boolean _equals_1 = Objects.equal(_eReferenceType, N4JSPackage.Literals.LABELLED_STATEMENT);
    if (_equals_1) {
      return this.scope_LabelledStatement(context);
    }
    return IScope.NULLSCOPE;
  }
  
  /**
   * dispatch to internal methods based on the context
   */
  private IScope getScopeForContext(final EObject context, final EReference reference) {
    boolean _matched = false;
    if (context instanceof ImportDeclaration) {
      _matched=true;
      return this.scope_ImportedModule(((ImportDeclaration)context), reference);
    }
    if (!_matched) {
      if (context instanceof NamedImportSpecifier) {
        _matched=true;
        return this.scope_ImportedElement(((NamedImportSpecifier)context), reference);
      }
    }
    if (!_matched) {
      if (context instanceof IdentifierRef) {
        _matched=true;
        return this.scope_IdentifierRef_id(((IdentifierRef)context), reference);
      }
    }
    if (!_matched) {
      if (context instanceof ParameterizedPropertyAccessExpression) {
        _matched=true;
        return this.scope_PropertyAccessExpression_property(((ParameterizedPropertyAccessExpression)context), reference);
      }
    }
    if (!_matched) {
      if (context instanceof N4FieldAccessor) {
        _matched=true;
        return Scopes.scopeFor(EcoreUtil2.<N4ClassifierDefinition>getContainerOfType(context, N4ClassifierDefinition.class).getOwnedFields());
      }
    }
    return IScope.NULLSCOPE;
  }
  
  @Override
  public IScope getScopeForContentAssist(final EObject context, final EReference reference) {
    final IScope scope = this.getScope(context, reference);
    if ((scope == IScope.NULLSCOPE)) {
      boolean _equals = Objects.equal(reference, N4JSPackage.Literals.IMPORT_DECLARATION__MODULE);
      if (_equals) {
        return this.scope_ImportedAndCurrentModule(context, reference);
      }
      boolean _matched = false;
      if (context instanceof Script) {
        _matched=true;
        return this.scope_EObject_id(context, reference);
      }
      if (!_matched) {
        if (context instanceof N4TypeDeclaration) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof VariableDeclaration) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof Statement) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof NewExpression) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof ParameterizedCallExpression) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof Argument) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof Expression) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
      if (!_matched) {
        if (context instanceof LiteralOrComputedPropertyName) {
          _matched=true;
          return this.scope_EObject_id(context, reference);
        }
      }
    }
    return scope;
  }
  
  /**
   * Returns a scope as created by {@link #getScope(EObject, EReference)} for the 'from' part of an import declaration
   * in the AST, but without the need for providing any AST nodes. This can be used to implement implicit imports
   * without duplicating any logic from the scoping.
   * <p>
   * There are two minor differences to the scope created by {@code #getScope()}:
   * <ol>
   * <li>the current module, i.e. the module represented by the given resource, won't be filtered out, and
   * <li>advanced error reporting will be disabled, i.e. {@link IScope#getSingleElement(QualifiedName)} will simply
   *     return <code>null</code> instead of returning an {@code IEObjectDescriptionWithError}.
   * </ol>
   */
  public IScope getScopeForImplicitImports(final N4JSResource resource) {
    return this.scope_ImportedModule(resource, Optional.<ImportDeclaration>absent());
  }
  
  /**
   * In <pre>continue XYZ</pre>, bind XYZ to label.
   * Bind to ALL labels in script, although not all of them are valid. Later is to be validated in ASTStructureValidator and
   * allows for better error and quick fix handling. However, most inner (and correct) scope is preferred (solving problems in case of duplicate names).
   */
  private IScope scope_LabelledStatement(final EObject context) {
    EObject _rootContainer = EcoreUtil.getRootContainer(context);
    final IScope parent = this.getAllLabels(((Script) _rootContainer));
    final HashSet<String> names = CollectionLiterals.<String>newHashSet();
    final ArrayList<IEObjectDescription> elements = CollectionLiterals.<IEObjectDescription>newArrayList();
    EObject current = context;
    while ((current != null)) {
      {
        boolean _matched = false;
        if (current instanceof LabelledStatement) {
          _matched=true;
          boolean _add = names.add(((LabelledStatement)current).getName());
          if (_add) {
            IEObjectDescription _create = EObjectDescription.create(((LabelledStatement)current).getName(), current);
            elements.add(_create);
          }
        }
        current = current.eContainer();
      }
    }
    boolean _isEmpty = elements.isEmpty();
    if (_isEmpty) {
      return parent;
    }
    final SimpleScope result = new SimpleScope(parent, elements);
    return result;
  }
  
  private IScope getAllLabels(final Script script) {
    return Scopes.scopeFor(IteratorExtensions.<LabelledStatement>toIterable(Iterators.<LabelledStatement>filter(script.eAllContents(), LabelledStatement.class)));
  }
  
  /**
   * E.g. in
   * <pre>import { e1,e2 } from "a/b/importedModule"</pre> bind "a/b/importedModule" to module with qualified name "a.b.importedModule"
   * 
   * @param importDeclaration usually of type ImportDeclaratoin in case of N4JS bindings, but maybe used in JSDoc as well
   */
  private IScope scope_ImportedModule(final ImportDeclaration importDeclaration, final EReference reference) {
    Resource _eResource = importDeclaration.eResource();
    final N4JSResource resource = ((N4JSResource) _eResource);
    final IScope projectImportEnabledScope = this.scope_ImportedModule(resource, Optional.<ImportDeclaration>of(importDeclaration));
    final Predicate<IEObjectDescription> _function = (IEObjectDescription it) -> {
      boolean _xifexpression = false;
      if ((it == null)) {
        _xifexpression = false;
      } else {
        boolean _isDescriptionOfModuleWith = this.descriptionsHelper.isDescriptionOfModuleWith(it, importDeclaration);
        _xifexpression = (!_isDescriptionOfModuleWith);
      }
      return _xifexpression;
    };
    return new FilteringScope(projectImportEnabledScope, _function);
  }
  
  private IScope scope_ImportedModule(final N4JSResource resource, final Optional<ImportDeclaration> importDeclaration) {
    final EReference reference = N4JSPackage.eINSTANCE.getImportDeclaration_Module();
    final IScope initialScope = this.scope_ImportedAndCurrentModule(resource.getScript(), reference);
    final IResourceDescriptions resourceDescriptions = this.resourceDescriptionsProvider.getResourceDescriptions(resource);
    final IScope delegateMainModuleAwareScope = MainModuleAwareSelectableBasedScope.createMainModuleAwareScope(initialScope, resourceDescriptions, reference.getEReferenceType());
    final IScope projectImportEnabledScope = ProjectImportEnablingScope.create(this.n4jsCore, resource, importDeclaration, initialScope, delegateMainModuleAwareScope);
    return projectImportEnabledScope;
  }
  
  /**
   * Same as {@link #scope_ImportedModule(EObject,EReference)}, but also includes the current module.
   */
  private IScope scope_ImportedAndCurrentModule(final EObject importDeclaration, final EReference reference) {
    return this.delegateGetScope(importDeclaration, reference);
  }
  
  /**
   * E.g. in
   * <pre>import { e1,e2 } from "importedModule"</pre>
   * bind e1 or e2 by retrieving all (not only exported, see below!) top level elements of
   * importedModule (variables, types; functions are types!). All elements enables better error handling and quick fixes, as links are not broken.
   */
  protected IScope scope_ImportedElement(final NamedImportSpecifier specifier, final EReference reference) {
    final ImportDeclaration declaration = EcoreUtil2.<ImportDeclaration>getContainerOfType(specifier, ImportDeclaration.class);
    return this.scope_AllTopLevelElementsFromModule(declaration.getModule(), declaration);
  }
  
  /**
   * Called from getScope(), binds an identifier reference.
   */
  private IScope scope_IdentifierRef_id(final IdentifierRef identifierRef, final EReference ref) {
    final VariableEnvironmentElement vee = this.<VariableEnvironmentElement>ancestor(identifierRef, VariableEnvironmentElement.class);
    final IScope scope = this.getLexicalEnvironmentScope(vee, identifierRef, ref);
    EObject _eContainer = identifierRef.eContainer();
    if ((_eContainer instanceof NewExpression)) {
      EObject _eContainer_1 = identifierRef.eContainer();
      final NewExpression newExpr = ((NewExpression) _eContainer_1);
      final VisibilityAwareCtorScope vacs = new VisibilityAwareCtorScope(scope, this.checker, this.containerTypesHelper, newExpr);
      return vacs;
    }
    return scope;
  }
  
  /**
   * Only used for content assist. Modeled after method {@link #scope_IdentifierRef_id(IdentifierRef,EReference)}.
   */
  private IScope scope_EObject_id(final EObject obj, final EReference ref) {
    VariableEnvironmentElement _xifexpression = null;
    if ((obj instanceof VariableEnvironmentElement)) {
      _xifexpression = ((VariableEnvironmentElement)obj);
    } else {
      _xifexpression = this.<VariableEnvironmentElement>ancestor(obj, VariableEnvironmentElement.class);
    }
    final VariableEnvironmentElement vee = _xifexpression;
    return this.getLexicalEnvironmentScope(vee, obj, ref);
  }
  
  private IScope getLexicalEnvironmentScope(final VariableEnvironmentElement vee, final EObject context, final EReference ref) {
    if ((vee == null)) {
      return IScope.NULLSCOPE;
    }
    Pair<String, VariableEnvironmentElement> _mappedTo = Pair.<String, VariableEnvironmentElement>of("scope_IdentifierRef_id", vee);
    final Provider<IScope> _function = () -> {
      return this.buildLexicalEnvironmentScope(vee, context, ref);
    };
    return this.cache.<IScope>get(_mappedTo, 
      vee.eResource(), _function);
  }
  
  /**
   * Builds a lexical environment scope with the given parameters.
   * Filters out primitive types.
   */
  private IScope buildLexicalEnvironmentScope(final VariableEnvironmentElement vee, final EObject context, final EReference reference) {
    final ArrayList<Iterable<IEObjectDescription>> scopeLists = CollectionLiterals.<Iterable<IEObjectDescription>>newArrayList();
    this.collectLexialEnvironmentsScopeLists(vee, scopeLists);
    IScope scope = null;
    if (this.suppressCrossFileResolutionOfIdentifierRef) {
      scope = IScope.NULLSCOPE;
    } else {
      EObject _rootContainer = EcoreUtil.getRootContainer(vee);
      final Script script = ((Script) _rootContainer);
      final IScope baseScope = this.getScriptBaseScope(script, context, reference);
      scope = this._importedElementsScopingHelper.getImportedIdentifiables(baseScope, script);
    }
    List<Iterable<IEObjectDescription>> _reverseView = ListExtensions.<Iterable<IEObjectDescription>>reverseView(scopeLists);
    for (final Iterable<IEObjectDescription> scopeList : _reverseView) {
      scope = this.scopesHelper.mapBasedScopeFor(context, scope, scopeList);
    }
    return scope;
  }
  
  private IScope getScriptBaseScope(final Script script, final EObject context, final EReference ref) {
    final IScope globalScope = this.delegate.getScope(script, ref);
    boolean _activateDynamicPseudoScope = this.jsVariantHelper.activateDynamicPseudoScope(context);
    if (_activateDynamicPseudoScope) {
      return new DynamicPseudoScope(globalScope);
    }
    return globalScope;
  }
  
  private List<Iterable<IEObjectDescription>> collectLexialEnvironmentsScopeLists(final VariableEnvironmentElement vee, final List<Iterable<IEObjectDescription>> result) {
    List<Iterable<IEObjectDescription>> _xblockexpression = null;
    {
      result.add(Scopes.scopedElementsFor(this._sourceElementExtensions.collectVisibleIdentifiableElements(vee)));
      result.add(Scopes.scopedElementsFor(this._sourceElementExtensions.collectLocalArguments(vee)));
      final VariableEnvironmentElement parent = this.<VariableEnvironmentElement>ancestor(vee, VariableEnvironmentElement.class);
      List<Iterable<IEObjectDescription>> _xifexpression = null;
      if ((parent != null)) {
        _xifexpression = this.collectLexialEnvironmentsScopeLists(parent, result);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Creates IScope with all top level elements (variables and types, including functions), from target module.
   * Provided resource is used to check visibility of module elements. Not visible elements are imported too, which
   * allows better error handling and quick fixes, as links are not broken.
   * 
   * Used for elements imported with named import and to access elements vi namespace import.
   * 
   * @param importedModule target {@link TModule} from which elements are imported
   * @param contextResource Receiver context {@link EObject} which is importing elements
   */
  private IScope scope_AllTopLevelElementsFromModule(final TModule importedModule, final EObject context) {
    if ((importedModule == null)) {
      return IScope.NULLSCOPE;
    }
    final IScope topLevelElementsScope = this.scopesHelper.mapBasedScopeFor(importedModule, IScope.NULLSCOPE, 
      this.topLevelElementCollector.getTopLevelElements(importedModule, context.eResource()));
    if (((!this.jsVariantHelper.allowVersionedTypes(context)) && this.jsVariantHelper.allowVersionedTypes(importedModule))) {
      return new NonVersionAwareContextScope(topLevelElementsScope, false, this.messageHelper);
    }
    return topLevelElementsScope;
  }
  
  /**
   * In <pre>receiver.property</pre>, binds "property".
   */
  private IScope scope_PropertyAccessExpression_property(final ParameterizedPropertyAccessExpression propertyAccess, final EReference ref) {
    final Expression receiver = propertyAccess.getTarget();
    if ((receiver instanceof IdentifierRef)) {
      final IdentifiableElement id = ((IdentifierRef)receiver).getId();
      if ((id instanceof ModuleNamespaceVirtualType)) {
        return this.createScopeForNamespaceAccess(((ModuleNamespaceVirtualType)id), propertyAccess);
      }
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(propertyAccess);
    final TypeRef typeRefRaw = this.ts.type(G, receiver);
    final TypeRef typeRef = this.ts.upperBound(G, typeRefRaw);
    final boolean staticAccess = (typeRef instanceof TypeTypeRef);
    TypingStrategy _typingStrategy = typeRef.getTypingStrategy();
    final boolean structFieldInitMode = (_typingStrategy == TypingStrategy.STRUCTURAL_FIELD_INITIALIZER);
    final boolean checkVisibility = true;
    return this.memberScopingHelper.createMemberScope(typeRef, propertyAccess, checkVisibility, staticAccess, structFieldInitMode);
  }
  
  private IScope createScopeForNamespaceAccess(final ModuleNamespaceVirtualType namespace, final EObject context) {
    final TModule module = namespace.getModule();
    IScope _xifexpression = null;
    if (((module != null) && (!module.eIsProxy()))) {
      _xifexpression = this.scope_AllTopLevelElementsFromModule(module, context);
    } else {
      IScope _xifexpression_1 = null;
      boolean _eIsProxy = namespace.eIsProxy();
      if (_eIsProxy) {
        _xifexpression_1 = new DynamicPseudoScope();
      } else {
        _xifexpression_1 = IScope.NULLSCOPE;
      }
      _xifexpression = _xifexpression_1;
    }
    final IScope result = _xifexpression;
    if ((namespace.isDeclaredDynamic() && (!(result instanceof DynamicPseudoScope)))) {
      return new DynamicPseudoScope(result);
    }
    return result;
  }
  
  /**
   * Is entered (and later recursively called) to initially bind "T" in <pre>var x : T;</pre> or other parameterized type references.
   */
  public IScope getTypeScope(final EObject context, final boolean fromStaticContext) {
    boolean _matched = false;
    if (context instanceof Script) {
      _matched=true;
      return this.locallyKnownTypesScopingHelper.scopeWithLocallyKnownTypes(((Script)context), this.delegate);
    }
    if (!_matched) {
      if (context instanceof TModule) {
        _matched=true;
        EObject _astElement = ((TModule)context).getAstElement();
        return this.locallyKnownTypesScopingHelper.scopeWithLocallyKnownTypes(((Script) _astElement), this.delegate);
      }
    }
    if (!_matched) {
      if (context instanceof N4FieldDeclaration) {
        _matched=true;
        final boolean isStaticContext = ((N4FieldDeclaration)context).isStatic();
        return this.getTypeScope(((N4FieldDeclaration)context).eContainer(), isStaticContext);
      }
    }
    if (!_matched) {
      if (context instanceof N4FieldAccessor) {
        _matched=true;
        final boolean isStaticContext = ((N4FieldAccessor)context).isStatic();
        return this.getTypeScope(((N4FieldAccessor)context).eContainer(), isStaticContext);
      }
    }
    if (!_matched) {
      if (context instanceof TypeDefiningElement) {
        _matched=true;
        final boolean isStaticContext = ((context instanceof N4MemberDeclaration) && ((N4MemberDeclaration) context).isStatic());
        final IScope parent = this.getTypeScope(((TypeDefiningElement)context).eContainer(), isStaticContext);
        final Type polyfilledOrOriginalType = this._sourceElementExtensions.getTypeOrPolyfilledType(((TypeDefiningElement)context));
        return this.locallyKnownTypesScopingHelper.scopeWithTypeAndItsTypeVariables(parent, polyfilledOrOriginalType, fromStaticContext);
      }
    }
    if (!_matched) {
      if (context instanceof TStructMethod) {
        _matched=true;
        final IScope parent = this.getTypeScope(((TStructMethod)context).eContainer(), fromStaticContext);
        return this.locallyKnownTypesScopingHelper.scopeWithTypeVarsOfTStructMethod(parent, ((TStructMethod)context));
      }
    }
    if (!_matched) {
      if (context instanceof FunctionTypeExpression) {
        _matched=true;
        final IScope parent = this.getTypeScope(((FunctionTypeExpression)context).eContainer(), fromStaticContext);
        return this.locallyKnownTypesScopingHelper.scopeWithTypeVarsOfFunctionTypeExpression(parent, ((FunctionTypeExpression)context));
      }
    }
    if (!_matched) {
      if (context instanceof TypeDefs) {
        _matched=true;
        return BuiltInTypeScope.get(((TypeDefs)context).eResource().getResourceSet());
      }
    }
    {
      final EObject container = context.eContainer();
      if ((container instanceof N4ClassDeclaration)) {
        if ((N4JSLanguageUtils.isPolyfill(((AnnotableElement)container)) || N4JSLanguageUtils.isStaticPolyfill(((AnnotableElement)container)))) {
          boolean _contains = ((N4ClassDeclaration)container).getTypeVars().contains(context);
          if (_contains) {
            final IScope parent = this.getTypeScope(context.eContainer(), false);
            return this.locallyKnownTypesScopingHelper.scopeWithTypeAndItsTypeVariables(parent, ((N4ClassDeclaration)container).getDefinedType(), fromStaticContext);
          } else {
            ParameterizedTypeRef _superClassRef = ((N4ClassDeclaration)container).getSuperClassRef();
            boolean _tripleEquals = (_superClassRef == context);
            if (_tripleEquals) {
              final Script script = EcoreUtil2.<Script>getContainerOfType(container, Script.class);
              final IScope parent_1 = this.locallyKnownTypesScopingHelper.scopeWithLocallyKnownTypesForPolyfillSuperRef(script, this.delegate, 
                ((N4ClassDeclaration)container).getDefinedType());
              return parent_1;
            }
          }
        }
      }
      return this.getTypeScope(container, fromStaticContext);
    }
  }
  
  /**
   * Returns ancestor of given type, or null, if no such container exists. Note that this method is slightly different from
   * EcoreUtil2::getContainerOfType, as it only returns a container and not the element itself. That is, ancestor is not reflexive here.
   */
  private <T extends EObject> T ancestor(final EObject obj, final Class<T> ancestorType) {
    if ((obj == null)) {
      return null;
    }
    return EcoreUtil2.<T>getContainerOfType(obj.eContainer(), ancestorType);
  }
  
  private IScope getN4IDLScope(final EObject context, final EReference reference) {
    final IScope contextVersionScope = this.getN4IDLContextVersionScope(context, reference);
    boolean _isVersionAwareContext = VersionUtils.isVersionAwareContext(context);
    boolean _not = (!_isVersionAwareContext);
    if (_not) {
      return new NonVersionAwareContextScope(contextVersionScope, true, this.messageHelper);
    } else {
      final java.util.Optional<FunctionDeclaration> migrationDeclaration = MigrationUtils.getMigrationDeclaration(context);
      boolean _isPresent = migrationDeclaration.isPresent();
      if (_isPresent) {
        if (((context instanceof IdentifierRef) && MigrationUtils.isMigrateCallIdentifier(((IdentifierRef) context)))) {
          EObject _eContainer = context.eContainer();
          final ParameterizedCallExpression callExpression = ((ParameterizedCallExpression) _eContainer);
          return this.migrationScopeHelper.migrationsScope(callExpression.getArguments(), context);
        } else {
          return this.migrationScopeHelper.migrationContextAwareScope(migrationDeclaration.get(), contextVersionScope);
        }
      }
    }
    return contextVersionScope;
  }
  
  /**
   * Returns a version-aware scope based on the context version that is specified in {@param context}.
   * 
   * If no context version can be inferred from {@param context}, all versionable results will
   * be wrapped in a {@link FailedToInferContextVersionWrappingScope}.
   */
  private IScope getN4IDLContextVersionScope(final EObject context, final EReference reference) {
    final IScope scope = this.getN4JSScope(context, reference);
    boolean _equals = Objects.equal(scope, IScope.NULLSCOPE);
    if (_equals) {
      return scope;
    }
    if (((reference == TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE) || 
      (reference == N4JSPackage.Literals.IDENTIFIER_REF__ID))) {
      final Optional<Integer> contextVersion = this.versionHelper.computeMaximumVersion(context);
      Integer _or = contextVersion.or(Integer.valueOf(Integer.MAX_VALUE));
      final N4IDLVersionAwareScope versionAwareScope = new N4IDLVersionAwareScope(scope, (_or).intValue());
      boolean _isPresent = contextVersion.isPresent();
      if (_isPresent) {
        return versionAwareScope;
      } else {
        return new FailedToInferContextVersionWrappingScope(scope);
      }
    }
    return scope;
  }
  
  private IScope getN4JSXScope(final EObject context, final EReference reference) {
    final IScope jsxPropertyAttributeScope = this.getJSXPropertyAttributeScope(context, reference);
    if ((jsxPropertyAttributeScope != IScope.NULLSCOPE)) {
      return jsxPropertyAttributeScope;
    }
    final IScope jsxElementScope = this.getJSXElementScope(context, reference);
    if ((jsxElementScope != IScope.NULLSCOPE)) {
      return jsxElementScope;
    }
    return this.getN4JSScope(context, reference);
  }
  
  /**
   * Returns scope for the {@link JSXPropertyAttribute} (obtained from context) or {@link IScope#NULLSCOPE}
   */
  private IScope getJSXPropertyAttributeScope(final EObject context, final EReference reference) {
    boolean _equals = Objects.equal(reference, N4JSPackage.Literals.JSX_PROPERTY_ATTRIBUTE__PROPERTY);
    if (_equals) {
      if ((context instanceof JSXPropertyAttribute)) {
        EObject _eContainer = ((JSXPropertyAttribute)context).eContainer();
        final JSXElement jsxElem = ((JSXElement) _eContainer);
        final TypeRef propsTypeRef = this._reactHelper.getPropsType(jsxElem);
        final boolean checkVisibility = true;
        final boolean staticAccess = false;
        final boolean structFieldInitMode = false;
        if ((propsTypeRef != null)) {
          final IScope memberScope = this.memberScopingHelper.createMemberScope(propsTypeRef, ((MemberAccess)context), checkVisibility, staticAccess, structFieldInitMode);
          return new DynamicPseudoScope(memberScope);
        } else {
          final IScope scope = this.getN4JSScope(context, reference);
          return new DynamicPseudoScope(scope);
        }
      }
    }
    return IScope.NULLSCOPE;
  }
  
  /**
   * Returns scope for the JSXElement (obtained from context) or {@link IScope#NULLSCOPE}
   */
  private IScope getJSXElementScope(final EObject context, final EReference reference) {
    JSXElementName _containerOfType = EcoreUtil2.<JSXElementName>getContainerOfType(context, JSXElementName.class);
    boolean _tripleEquals = (_containerOfType == null);
    if (_tripleEquals) {
      return IScope.NULLSCOPE;
    }
    final IScope scope = this.getN4JSScope(context, reference);
    return new DynamicPseudoScope(scope);
  }
}
