/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.conversion.AbstractN4JSStringValueConverter;
import org.eclipse.n4js.n4JS.AbstractCaseClause;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.ArrayBindingPattern;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.AssignmentOperator;
import org.eclipse.n4js.n4JS.BinaryLogicalExpression;
import org.eclipse.n4js.n4JS.BinaryLogicalOperator;
import org.eclipse.n4js.n4JS.BindingElement;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.BreakStatement;
import org.eclipse.n4js.n4JS.CatchBlock;
import org.eclipse.n4js.n4JS.ContinueStatement;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FieldAccessor;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.IfStatement;
import org.eclipse.n4js.n4JS.ImportCallExpression;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.IterationStatement;
import org.eclipse.n4js.n4JS.LabelRef;
import org.eclipse.n4js.n4JS.LabelledStatement;
import org.eclipse.n4js.n4JS.LegacyOctalIntLiteral;
import org.eclipse.n4js.n4JS.Literal;
import org.eclipse.n4js.n4JS.LocalArgumentsVariable;
import org.eclipse.n4js.n4JS.MethodDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDefinition;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4FieldAccessor;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NewTarget;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.PostfixExpression;
import org.eclipse.n4js.n4JS.PostfixOperator;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameOwner;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertyNameValuePairSingleName;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.n4JS.StrictModeRelevant;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.n4JS.TemplateSegment;
import org.eclipse.n4js.n4JS.UnaryExpression;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4JS.Variable;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4JS.VariableStatementKeyword;
import org.eclipse.n4js.n4JS.WithStatement;
import org.eclipse.n4js.n4JS.YieldExpression;
import org.eclipse.n4js.parser.InternalSemicolonInjectingParser;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.services.N4JSGrammarAccess;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.utils.N4JSLanguageHelper;
import org.eclipse.n4js.validation.ASTStructureDiagnosticProducer;
import org.eclipse.n4js.validation.BaseJavaScriptVariantHelper;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.helper.FunctionValidationHelper;
import org.eclipse.xtend.lib.annotations.ToString;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.Keyword;
import org.eclipse.xtext.diagnostics.DiagnosticMessage;
import org.eclipse.xtext.diagnostics.IDiagnosticConsumer;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.ILeafNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.SyntaxErrorMessage;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure3;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * A utility that validates the structure of the AST in one pass.
 * 
 * Note:
 * The validations here are important for using plain JavaScript,
 * especially the EcmaScript test suite relies on validations here.
 * Validations that are in the package 'validators' are not considered
 * when the EcmaScript test suite is executed.
 */
@SuppressWarnings("all")
public class ASTStructureValidator {
  @ToString
  protected static class Constraints {
    private static final int STRICT = 1;
    
    private static final int N4JS = (ASTStructureValidator.Constraints.STRICT << 1);
    
    private static final int EXTERNAL = (ASTStructureValidator.Constraints.N4JS << 1);
    
    private static final int ALLOW_NESTED_FUNCTION_DECLARATION = (ASTStructureValidator.Constraints.EXTERNAL << 1);
    
    private static final int ALLOW_RETURN = (ASTStructureValidator.Constraints.ALLOW_NESTED_FUNCTION_DECLARATION << 1);
    
    private static final int ALLOW_CONTINUE = (ASTStructureValidator.Constraints.ALLOW_RETURN << 1);
    
    private static final int ALLOW_BREAK = (ASTStructureValidator.Constraints.ALLOW_CONTINUE << 1);
    
    private static final int ALLOW_VAR_WITHOUT_INITIALIZER = (ASTStructureValidator.Constraints.ALLOW_BREAK << 1);
    
    private static final int ALLOW_YIELD_EXPRESSION = (ASTStructureValidator.Constraints.ALLOW_VAR_WITHOUT_INITIALIZER << 1);
    
    private static final int ALLOW_SUPER = (ASTStructureValidator.Constraints.ALLOW_YIELD_EXPRESSION << 1);
    
    private static final int ALLOW_SUPER_CALL = (ASTStructureValidator.Constraints.ALLOW_SUPER << 1);
    
    private static final int IN_FUNCTION_DECLARATION = (ASTStructureValidator.Constraints.ALLOW_SUPER_CALL << 1);
    
    private final int bits;
    
    private static int getIf(final int value, final boolean b) {
      int _xifexpression = (int) 0;
      if (b) {
        _xifexpression = value;
      } else {
        _xifexpression = 0;
      }
      return _xifexpression;
    }
    
    public Constraints(final boolean n4js, final boolean external) {
      this((((ASTStructureValidator.Constraints.getIf(ASTStructureValidator.Constraints.N4JS, n4js) | ASTStructureValidator.Constraints.getIf(ASTStructureValidator.Constraints.EXTERNAL, external)) | ASTStructureValidator.Constraints.ALLOW_VAR_WITHOUT_INITIALIZER) | ASTStructureValidator.Constraints.ALLOW_YIELD_EXPRESSION));
    }
    
    public Constraints(final int bits) {
      this.bits = bits;
    }
    
    private boolean is(final int bit) {
      return ((this.bits & bit) != 0);
    }
    
    public boolean isN4JS() {
      return this.is(ASTStructureValidator.Constraints.N4JS);
    }
    
    public boolean isStrict() {
      return (this.is(ASTStructureValidator.Constraints.N4JS) || this.is(ASTStructureValidator.Constraints.STRICT));
    }
    
    public boolean isExternal() {
      return this.is(ASTStructureValidator.Constraints.EXTERNAL);
    }
    
    public boolean isNestedFunctionAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_NESTED_FUNCTION_DECLARATION);
    }
    
    public boolean isInFunctionDeclaration() {
      return this.is(ASTStructureValidator.Constraints.IN_FUNCTION_DECLARATION);
    }
    
    public boolean isReturnAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_RETURN);
    }
    
    public boolean isBreakAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_BREAK);
    }
    
    public boolean isContinueAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_CONTINUE);
    }
    
    public boolean isVarInitializerRequired() {
      boolean _is = this.is(ASTStructureValidator.Constraints.ALLOW_VAR_WITHOUT_INITIALIZER);
      return (!_is);
    }
    
    public boolean isYieldExpressionAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_YIELD_EXPRESSION);
    }
    
    public boolean isSuperLiteralAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_SUPER);
    }
    
    public boolean isSuperCallAllowed() {
      return this.is(ASTStructureValidator.Constraints.ALLOW_SUPER_CALL);
    }
    
    public ASTStructureValidator.Constraints with(final int bit, final boolean set) {
      int _xifexpression = (int) 0;
      if (set) {
        _xifexpression = (this.bits | bit);
      } else {
        _xifexpression = (this.bits & (~bit));
      }
      final int newBits = _xifexpression;
      if ((newBits == this.bits)) {
        return this;
      }
      return new ASTStructureValidator.Constraints(newBits);
    }
    
    public ASTStructureValidator.Constraints strict(final boolean strict) {
      return this.with(ASTStructureValidator.Constraints.STRICT, strict);
    }
    
    public ASTStructureValidator.Constraints allowNestedFunctions(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_NESTED_FUNCTION_DECLARATION, allow);
    }
    
    public ASTStructureValidator.Constraints allowBreak(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_BREAK, allow);
    }
    
    public ASTStructureValidator.Constraints allowContinue(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_CONTINUE, allow);
    }
    
    public ASTStructureValidator.Constraints allowReturn(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_RETURN, allow);
    }
    
    public ASTStructureValidator.Constraints allowVarWithoutInitializer(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_VAR_WITHOUT_INITIALIZER, allow);
    }
    
    public ASTStructureValidator.Constraints allowYieldExpression(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_YIELD_EXPRESSION, allow);
    }
    
    public ASTStructureValidator.Constraints allowSuperLiteral(final boolean allow) {
      ASTStructureValidator.Constraints _xifexpression = null;
      if ((!allow)) {
        _xifexpression = this.allowSuperCall(false).with(ASTStructureValidator.Constraints.ALLOW_SUPER, allow);
      } else {
        _xifexpression = this.with(ASTStructureValidator.Constraints.ALLOW_SUPER, allow);
      }
      return _xifexpression;
    }
    
    public ASTStructureValidator.Constraints allowSuperCall(final boolean allow) {
      return this.with(ASTStructureValidator.Constraints.ALLOW_SUPER_CALL, allow);
    }
    
    public ASTStructureValidator.Constraints enterFunctionDeclaration() {
      return this.with(ASTStructureValidator.Constraints.IN_FUNCTION_DECLARATION, true);
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("bits", this.bits);
      return b.toString();
    }
  }
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private N4JSGrammarAccess grammarAccess;
  
  @Inject
  private N4JSLanguageHelper languageHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  public void validate(final EObject model, final IDiagnosticConsumer consumer) {
    boolean _and = false;
    Resource _eResource = null;
    if (model!=null) {
      _eResource=model.eResource();
    }
    boolean _tripleNotEquals = (_eResource != null);
    if (!_tripleNotEquals) {
      _and = false;
    } else {
      boolean _isNoValidate = this.n4jsCore.isNoValidate(model.eResource().getURI());
      boolean _not = (!_isNoValidate);
      _and = _not;
    }
    if (_and) {
      final ASTStructureDiagnosticProducer producer = new ASTStructureDiagnosticProducer(consumer);
      HashSet<LabelledStatement> _newHashSetWithExpectedSize = Sets.<LabelledStatement>newHashSetWithExpectedSize(2);
      boolean _isN4JSMode = this.jsVariantHelper.isN4JSMode(model);
      boolean _isExternalMode = this.jsVariantHelper.isExternalMode(model);
      ASTStructureValidator.Constraints _constraints = new ASTStructureValidator.Constraints(_isN4JSMode, _isExternalMode);
      this.validateASTStructure(model, producer, _newHashSetWithExpectedSize, _constraints);
    }
  }
  
  private void recursiveValidateASTStructure(final EObject model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    final Iterator<EObject> content = model.eContents().iterator();
    boolean newStrict = constraints.isStrict();
    boolean first = true;
    while (content.hasNext()) {
      {
        final EObject next = content.next();
        newStrict = (newStrict || (first && this.isUseStrictProlog(model, next)));
        first = (first && this.isProlog(next));
        if ((next instanceof StrictModeRelevant)) {
          ((StrictModeRelevant)next).setStrictMode(newStrict);
        }
        this.validateASTStructure(next, producer, validLabels, 
          constraints.strict(newStrict));
      }
    }
  }
  
  private boolean isProlog(final EObject object) {
    if ((object instanceof ExpressionStatement)) {
      Expression _expression = ((ExpressionStatement)object).getExpression();
      return (_expression instanceof StringLiteral);
    }
    return false;
  }
  
  private boolean isUseStrictProlog(final EObject model, final EObject next) {
    if (((model instanceof Script) || ((model instanceof Block) && (model.eContainer() instanceof FunctionDefinition)))) {
      boolean _matched = false;
      if (next instanceof ExpressionStatement) {
        _matched=true;
        Expression _expression = ((ExpressionStatement)next).getExpression();
        final Expression it = _expression;
        boolean _matched_1 = false;
        if (it instanceof StringLiteral) {
          _matched_1=true;
          String _value = ((StringLiteral)it).getValue();
          return Objects.equal(BaseJavaScriptVariantHelper.STRICT_MODE_LITERAL_VALUE, _value);
        }
      }
    }
    return false;
  }
  
  private void _validateASTStructure(final EObject model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final Script model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.strict(false).allowNestedFunctions(true).allowReturn(false).allowContinue(false).allowBreak(false));
  }
  
  private void _validateASTStructure(final ExportDeclaration model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isDefaultExport = model.isDefaultExport();
    if (_isDefaultExport) {
      final ExportableElement exportedElement = model.getExportedElement();
      if ((exportedElement instanceof VariableStatement)) {
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final N4ClassifierDefinition model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    if ((model instanceof N4ClassifierDeclaration)) {
      if (((((N4ClassifierDeclaration)model).getName() == null) && (!((N4ClassifierDeclaration)model).isExportedAsDefault()))) {
        final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
        producer.setNode(target);
        String _messageForAST_TYPE_DECL_MISSING_NAME = IssueCodes.getMessageForAST_TYPE_DECL_MISSING_NAME();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_TYPE_DECL_MISSING_NAME);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_TYPE_DECL_MISSING_NAME, _defaultSeverity, IssueCodes.AST_TYPE_DECL_MISSING_NAME);
        producer.addDiagnostic(_diagnosticMessage);
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.strict(true).allowNestedFunctions(true).allowReturn(false).allowContinue(false).allowBreak(false));
  }
  
  private void _validateASTStructure(final N4EnumDeclaration model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    if (((model.getName() == null) && (!model.isExportedAsDefault()))) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_TYPE_DECL_MISSING_NAME = IssueCodes.getMessageForAST_TYPE_DECL_MISSING_NAME();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_TYPE_DECL_MISSING_NAME);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_TYPE_DECL_MISSING_NAME, _defaultSeverity, IssueCodes.AST_TYPE_DECL_MISSING_NAME);
      producer.addDiagnostic(_diagnosticMessage);
    }
    boolean _isEmpty = model.getLiterals().isEmpty();
    if (_isEmpty) {
      final ICompositeNode target_1 = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target_1);
      String _messageForENM_WITHOUT_LITERALS = IssueCodes.getMessageForENM_WITHOUT_LITERALS();
      Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.ENM_WITHOUT_LITERALS);
      DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForENM_WITHOUT_LITERALS, _defaultSeverity_1, IssueCodes.ENM_WITHOUT_LITERALS);
      producer.addDiagnostic(_diagnosticMessage_1);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.strict(true).allowNestedFunctions(true).allowReturn(false).allowContinue(false).allowBreak(false));
  }
  
  private void _validateASTStructure(final LegacyOctalIntLiteral model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isStrict = constraints.isStrict();
    if (_isStrict) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_STR_NO_OCTALS = IssueCodes.getMessageForAST_STR_NO_OCTALS();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_STR_NO_OCTALS);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_STR_NO_OCTALS, _defaultSeverity, IssueCodes.AST_STR_NO_OCTALS);
      producer.addDiagnostic(_diagnosticMessage);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final StringLiteral model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isStrict = constraints.isStrict();
    if (_isStrict) {
      this.addErrorForOctalEscapeSequence(model.getRawValue(), model, N4JSPackage.Literals.STRING_LITERAL__VALUE, producer);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final TemplateSegment model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.addErrorForOctalEscapeSequence(model.getRawValue(), model, N4JSPackage.Literals.TEMPLATE_SEGMENT__VALUE, producer);
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void addErrorForOctalEscapeSequence(final String rawValue, final Literal model, final EAttribute valueEAttribute, final ASTStructureDiagnosticProducer producer) {
    final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, valueEAttribute);
    final INode target = IterableExtensions.<INode>head(nodes);
    final SyntaxErrorMessage syntaxError = target.getSyntaxErrorMessage();
    if (((((syntaxError == null) || Objects.equal(syntaxError.getIssueCode(), AbstractN4JSStringValueConverter.WARN_ISSUE_CODE)) || Objects.equal(syntaxError.getIssueCode(), InternalSemicolonInjectingParser.SEMICOLON_INSERTED)) && AbstractN4JSStringValueConverter.hasOctalEscapeSequence(rawValue))) {
      producer.setNode(target);
      String _messageForAST_STR_NO_OCTALS = IssueCodes.getMessageForAST_STR_NO_OCTALS();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_STR_NO_OCTALS);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_STR_NO_OCTALS, _defaultSeverity, IssueCodes.AST_STR_NO_OCTALS);
      producer.addDiagnostic(_diagnosticMessage);
    }
  }
  
  private void _validateASTStructure(final PostfixExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
    final Expression child = model.getExpression();
    boolean _isValidSimpleAssignmentTarget = child.isValidSimpleAssignmentTarget();
    boolean _not = (!_isValidSimpleAssignmentTarget);
    if (_not) {
      final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, N4JSPackage.Literals.POSTFIX_EXPRESSION__EXPRESSION);
      final INode target = IterableExtensions.<INode>head(nodes);
      producer.setNode(target);
      String _xifexpression = null;
      PostfixOperator _op = model.getOp();
      boolean _equals = Objects.equal(_op, PostfixOperator.DEC);
      if (_equals) {
        _xifexpression = "decrement";
      } else {
        _xifexpression = "increment";
      }
      final String operand = _xifexpression;
      String _messageForAST_INVALID_OPERAND = IssueCodes.getMessageForAST_INVALID_OPERAND(operand);
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_OPERAND);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_OPERAND, _defaultSeverity, IssueCodes.AST_INVALID_OPERAND);
      producer.addDiagnostic(_diagnosticMessage);
    }
  }
  
  private void _validateASTStructure(final UnaryExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
    if ((Objects.equal(model.getOp(), UnaryOperator.DEC) || Objects.equal(model.getOp(), UnaryOperator.INC))) {
      final Expression child = model.getExpression();
      if (((child != null) && (!child.isValidSimpleAssignmentTarget()))) {
        final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, 
          N4JSPackage.Literals.POSTFIX_EXPRESSION__EXPRESSION);
        final INode target = IterableExtensions.<INode>head(nodes);
        producer.setNode(target);
        String _xifexpression = null;
        UnaryOperator _op = model.getOp();
        boolean _equals = Objects.equal(_op, UnaryOperator.DEC);
        if (_equals) {
          _xifexpression = "decrement";
        } else {
          _xifexpression = "increment";
        }
        final String operand = _xifexpression;
        String _messageForAST_INVALID_OPERAND = IssueCodes.getMessageForAST_INVALID_OPERAND(operand);
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_OPERAND);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_OPERAND, _defaultSeverity, IssueCodes.AST_INVALID_OPERAND);
        producer.addDiagnostic(_diagnosticMessage);
      }
    }
  }
  
  private void _validateASTStructure(final YieldExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isYieldExpressionAllowed = constraints.isYieldExpressionAllowed();
    boolean _not = (!_isYieldExpressionAllowed);
    if (_not) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_INVALID_YIELD_EXPRESSION = IssueCodes.getMessageForAST_INVALID_YIELD_EXPRESSION();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_YIELD_EXPRESSION);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_YIELD_EXPRESSION, _defaultSeverity, IssueCodes.AST_INVALID_YIELD_EXPRESSION);
      producer.addDiagnostic(_diagnosticMessage);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final AssignmentExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
    final Expression lhs = model.getLhs();
    if (((lhs != null) && (!lhs.isValidSimpleAssignmentTarget()))) {
      if (((model.getOp() != AssignmentOperator.ASSIGN) || (!DestructureUtils.isTopOfDestructuringAssignment(model)))) {
        final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, N4JSPackage.Literals.ASSIGNMENT_EXPRESSION__LHS);
        final INode target = IterableExtensions.<INode>head(nodes);
        producer.setNode(target);
        String _messageForAST_EXP_INVALID_LHS_ASS = IssueCodes.getMessageForAST_EXP_INVALID_LHS_ASS();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_EXP_INVALID_LHS_ASS);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_EXP_INVALID_LHS_ASS, _defaultSeverity, 
          IssueCodes.AST_EXP_INVALID_LHS_ASS);
        producer.addDiagnostic(_diagnosticMessage);
      }
    }
  }
  
  private void _validateASTStructure(final ImportCallExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    int _size = model.getArguments().size();
    boolean _tripleNotEquals = (_size != 1);
    if (_tripleNotEquals) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_IMPORT_CALL_WRONG_NUM_OF_ARGS = IssueCodes.getMessageForAST_IMPORT_CALL_WRONG_NUM_OF_ARGS();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_IMPORT_CALL_WRONG_NUM_OF_ARGS);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_IMPORT_CALL_WRONG_NUM_OF_ARGS, _defaultSeverity, IssueCodes.AST_IMPORT_CALL_WRONG_NUM_OF_ARGS);
      producer.addDiagnostic(_diagnosticMessage);
    }
    if (((!model.getArguments().isEmpty()) && model.getArgument().isSpread())) {
      final ICompositeNode target_1 = NodeModelUtils.findActualNodeFor(model.getArgument());
      producer.setNode(target_1);
      String _messageForAST_IMPORT_CALL_SPREAD = IssueCodes.getMessageForAST_IMPORT_CALL_SPREAD();
      Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_IMPORT_CALL_SPREAD);
      DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_IMPORT_CALL_SPREAD, _defaultSeverity_1, IssueCodes.AST_IMPORT_CALL_SPREAD);
      producer.addDiagnostic(_diagnosticMessage_1);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final IdentifierRef model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    final String name = model.getIdAsText();
    if ((name != null)) {
      if ((constraints.isStrict() && N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name))) {
        this.issueNameDiagnostic(model, producer, name, N4JSPackage.Literals.IDENTIFIER_REF__ID, Severity.ERROR);
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final Variable model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    final String name = model.getName();
    if ((name != null)) {
      if ((Objects.equal(name, N4JSLanguageConstants.LOCAL_ARGUMENTS_VARIABLE_NAME) && (!(model instanceof LocalArgumentsVariable)))) {
        this.issueArgumentsError(model, name, constraints.isStrict(), producer);
      } else {
        if (((!Objects.equal(name, N4JSLanguageConstants.YIELD_KEYWORD)) && 
          ((((((this.languageHelper.getECMAKeywords().contains(name) || "enum".equals(name)) || "await".equals(name)) || "let".equals(name)) || "true".equals(name)) || "false".equals(name)) || "null".equals(name)))) {
          this.issueNameDiagnostic(model, producer, name);
        } else {
          boolean _isStrict = constraints.isStrict();
          if (_isStrict) {
            if ((N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name) || Objects.equal(name, N4JSLanguageConstants.EVAL_NAME))) {
              this.issueNameDiagnostic(model, producer, name);
            }
          }
        }
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final WithStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isStrict = constraints.isStrict();
    if (_isStrict) {
      final Function1<ILeafNode, Boolean> _function = (ILeafNode it) -> {
        EObject _grammarElement = it.getGrammarElement();
        Keyword _withKeyword_0 = this.grammarAccess.getWithStatementAccess().getWithKeyword_0();
        return Boolean.valueOf(Objects.equal(_grammarElement, _withKeyword_0));
      };
      final ILeafNode node = IterableExtensions.<ILeafNode>findFirst(NodeModelUtils.findActualNodeFor(model).getLeafNodes(), _function);
      producer.setNode(node);
      if ((node != null)) {
        String _messageForAST_STR_NO_WITH_STMT = IssueCodes.getMessageForAST_STR_NO_WITH_STMT();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_STR_NO_WITH_STMT);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_STR_NO_WITH_STMT, _defaultSeverity, 
          IssueCodes.AST_STR_NO_WITH_STMT);
        producer.addDiagnostic(_diagnosticMessage);
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final LabelledStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    final String name = model.getName();
    if ((name != null)) {
      if ((constraints.isStrict() && N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name))) {
        this.issueNameDiagnostic(model, producer, name);
      }
    }
    try {
      validLabels.add(model);
      boolean _isStrict = constraints.isStrict();
      boolean _not = (!_isStrict);
      this.recursiveValidateASTStructure(model, producer, validLabels, 
        constraints.allowNestedFunctions(_not).allowBreak(true));
    } finally {
      validLabels.remove(model);
    }
  }
  
  private void _validateASTStructure(final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.validateBlockStructure(
      model.eContainer(), model, producer, validLabels, constraints);
  }
  
  private void _validateBlockStructure(final IfStatement container, final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateBlockStructure(final IterationStatement container, final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(((!constraints.isStrict()) && (!constraints.isInFunctionDeclaration()))));
  }
  
  private void _validateBlockStructure(final FunctionDefinition container, final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(true));
  }
  
  private void _validateBlockStructure(final CatchBlock container, final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(true));
  }
  
  private void _validateBlockStructure(final EObject container, final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isStrict = constraints.isStrict();
    boolean _not = (!_isStrict);
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(_not));
  }
  
  private void _validateASTStructure(final IfStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(((!constraints.isStrict()) && (!constraints.isInFunctionDeclaration()))));
  }
  
  private void _validateASTStructure(final AbstractCaseClause model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowBreak(true));
  }
  
  private void _validateASTStructure(final ForStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isForPlain = model.isForPlain();
    boolean _not = (!_isForPlain);
    if (_not) {
      boolean _isEmpty = model.getVarDeclsOrBindings().isEmpty();
      boolean _not_1 = (!_isEmpty);
      if (_not_1) {
        final Consumer<VariableDeclaration> _function = (VariableDeclaration varDecl) -> {
          if ((((varDecl.getExpression() != null) && (!(varDecl.eContainer() instanceof BindingElement))) && (constraints.isStrict() || (model.getVarStmtKeyword() == VariableStatementKeyword.LET)))) {
            final List<INode> nodes = NodeModelUtils.findNodesForFeature(varDecl, N4JSPackage.Literals.VARIABLE_DECLARATION__EXPRESSION);
            INode _elvis = null;
            INode _head = IterableExtensions.<INode>head(nodes);
            if (_head != null) {
              _elvis = _head;
            } else {
              ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(varDecl);
              _elvis = _findActualNodeFor;
            }
            final INode target = _elvis;
            producer.setNode(target);
            if ((target != null)) {
              String _messageForAST_VAR_DECL_IN_FOR_INVALID_INIT = IssueCodes.getMessageForAST_VAR_DECL_IN_FOR_INVALID_INIT();
              Severity _xifexpression = null;
              if ((constraints.isStrict() || model.isForIn())) {
                _xifexpression = IssueCodes.getDefaultSeverity(IssueCodes.AST_VAR_DECL_IN_FOR_INVALID_INIT);
              } else {
                _xifexpression = Severity.WARNING;
              }
              DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_VAR_DECL_IN_FOR_INVALID_INIT, _xifexpression, 
                IssueCodes.AST_VAR_DECL_IN_FOR_INVALID_INIT);
              producer.addDiagnostic(_diagnosticMessage);
            }
          } else {
            if (((model.getVarStmtKeyword() == VariableStatementKeyword.LET) && Objects.equal(varDecl.getName(), "let"))) {
              final List<INode> nodes_1 = NodeModelUtils.findNodesForFeature(varDecl, TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME);
              INode _elvis_1 = null;
              INode _head_1 = IterableExtensions.<INode>head(nodes_1);
              if (_head_1 != null) {
                _elvis_1 = _head_1;
              } else {
                ICompositeNode _findActualNodeFor_1 = NodeModelUtils.findActualNodeFor(varDecl);
                _elvis_1 = _findActualNodeFor_1;
              }
              final INode target_1 = _elvis_1;
              producer.setNode(target_1);
              if ((target_1 != null)) {
                String _messageForAST_RESERVED_IDENTIFIER = IssueCodes.getMessageForAST_RESERVED_IDENTIFIER(varDecl.getName());
                Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_RESERVED_IDENTIFIER);
                DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_RESERVED_IDENTIFIER, _defaultSeverity, 
                  IssueCodes.AST_RESERVED_IDENTIFIER);
                producer.addDiagnostic(_diagnosticMessage_1);
              }
            }
          }
        };
        model.getVarDecl().forEach(_function);
      } else {
        Expression _initExpr = model.getInitExpr();
        boolean _tripleNotEquals = (_initExpr != null);
        if (_tripleNotEquals) {
          final Expression initExpr = model.getInitExpr();
          if ((initExpr instanceof AssignmentExpression)) {
            final List<INode> nodes = NodeModelUtils.findNodesForFeature(initExpr, 
              TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME);
            INode _elvis = null;
            INode _head = IterableExtensions.<INode>head(nodes);
            if (_head != null) {
              _elvis = _head;
            } else {
              ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(initExpr);
              _elvis = _findActualNodeFor;
            }
            final INode target = _elvis;
            producer.setNode(target);
            if ((target != null)) {
              String _messageForAST_VAR_DECL_IN_FOR_INVALID_INIT = IssueCodes.getMessageForAST_VAR_DECL_IN_FOR_INVALID_INIT();
              Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_VAR_DECL_IN_FOR_INVALID_INIT);
              DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_VAR_DECL_IN_FOR_INVALID_INIT, _defaultSeverity, 
                IssueCodes.AST_VAR_DECL_IN_FOR_INVALID_INIT);
              producer.addDiagnostic(_diagnosticMessage);
            }
          } else {
            if (((!initExpr.isValidSimpleAssignmentTarget()) && (!DestructureUtils.isTopOfDestructuringForStatement(model)))) {
              final List<INode> nodes_1 = NodeModelUtils.findNodesForFeature(model, 
                N4JSPackage.Literals.FOR_STATEMENT__INIT_EXPR);
              INode _elvis_1 = null;
              INode _head_1 = IterableExtensions.<INode>head(nodes_1);
              if (_head_1 != null) {
                _elvis_1 = _head_1;
              } else {
                ICompositeNode _findActualNodeFor_1 = NodeModelUtils.findActualNodeFor(initExpr);
                _elvis_1 = _findActualNodeFor_1;
              }
              final INode target_1 = _elvis_1;
              producer.setNode(target_1);
              if ((target_1 != null)) {
                String _messageForAST_EXP_INVALID_LHS_ASS = IssueCodes.getMessageForAST_EXP_INVALID_LHS_ASS();
                Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_EXP_INVALID_LHS_ASS);
                DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_EXP_INVALID_LHS_ASS, _defaultSeverity_1, 
                  IssueCodes.AST_EXP_INVALID_LHS_ASS);
                producer.addDiagnostic(_diagnosticMessage_1);
              }
            }
          }
        }
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(((!constraints.isStrict()) && (!constraints.isInFunctionDeclaration()))).allowBreak(true).allowContinue(true));
  }
  
  private void _validateASTStructure(final IterationStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowNestedFunctions(((!constraints.isStrict()) && (!constraints.isInFunctionDeclaration()))).allowBreak(true).allowContinue(true));
  }
  
  private void _validateASTStructure(final FormalParameter model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    final EObject container = model.eContainer();
    boolean allowYieldInInit = false;
    if ((container instanceof FunctionDefinition)) {
      boolean _isGenerator = ((FunctionDefinition)container).isGenerator();
      boolean _not = (!_isGenerator);
      allowYieldInInit = _not;
      final Procedure3<String, String, EObject> _function = (String msg, String id, EObject eObj) -> {
        producer.setNode(NodeModelUtils.findActualNodeFor(eObj));
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(id);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(msg, _defaultSeverity, id);
        producer.addDiagnostic(_diagnosticMessage);
      };
      final Procedure3<String, String, EObject> issueConsumer = _function;
      final Predicate<FormalParameter> _function_1 = (FormalParameter it) -> {
        return it.isVariadic();
      };
      final Predicate<FormalParameter> _function_2 = (FormalParameter it) -> {
        return it.isHasInitializerAssignment();
      };
      FunctionValidationHelper.<FormalParameter>internalCheckFormalParameter(
        ((FunctionDefinition)container).getFpars(), model, _function_1, _function_2, new FunctionValidationHelper.TripleConsumer<String, String, EObject>() {
          public void accept(String arg0, String arg1, EObject arg2) {
            issueConsumer.apply(arg0, arg1, arg2);
          }
      });
    }
    this._validateASTStructure(
      ((Variable) model), producer, validLabels, 
      constraints.allowYieldExpression(allowYieldInInit));
  }
  
  private void _validateASTStructure(final NewTarget model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    FunctionDefinition _containerOfType = EcoreUtil2.<FunctionDefinition>getContainerOfType(model, FunctionDefinition.class);
    boolean _tripleEquals = (_containerOfType == null);
    if (_tripleEquals) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      if ((target != null)) {
        producer.setNode(target);
        String _messageForAST_INVALID_NEW_TARGET = IssueCodes.getMessageForAST_INVALID_NEW_TARGET();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_NEW_TARGET);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_NEW_TARGET, _defaultSeverity, 
          IssueCodes.AST_INVALID_NEW_TARGET);
        producer.addDiagnostic(_diagnosticMessage);
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final SuperLiteral model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isInvalidSuperLiteral = this.isInvalidSuperLiteral(model, constraints);
    if (_isInvalidSuperLiteral) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      if ((target != null)) {
        producer.setNode(target);
        EStructuralFeature _eContainingFeature = model.eContainingFeature();
        boolean _tripleEquals = (_eContainingFeature == N4JSPackage.Literals.PARAMETERIZED_CALL_EXPRESSION__TARGET);
        if (_tripleEquals) {
          String _messageForKEY_SUP_CTOR_INVALID_LOC = IssueCodes.getMessageForKEY_SUP_CTOR_INVALID_LOC();
          Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.KEY_SUP_CTOR_INVALID_LOC);
          DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForKEY_SUP_CTOR_INVALID_LOC, _defaultSeverity, 
            IssueCodes.KEY_SUP_CTOR_INVALID_LOC);
          producer.addDiagnostic(_diagnosticMessage);
        } else {
          N4MethodDeclaration _containerOfType = EcoreUtil2.<N4MethodDeclaration>getContainerOfType(model, N4MethodDeclaration.class);
          boolean _tripleEquals_1 = (_containerOfType == null);
          if (_tripleEquals_1) {
            String _messageForKEY_SUP_ACCESS_INVALID_LOC = IssueCodes.getMessageForKEY_SUP_ACCESS_INVALID_LOC();
            Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.KEY_SUP_ACCESS_INVALID_LOC);
            DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForKEY_SUP_ACCESS_INVALID_LOC, _defaultSeverity_1, 
              IssueCodes.KEY_SUP_ACCESS_INVALID_LOC);
            producer.addDiagnostic(_diagnosticMessage_1);
          } else {
            final N4ClassifierDeclaration containingClass = EcoreUtil2.<N4ClassifierDeclaration>getContainerOfType(model, N4ClassifierDeclaration.class);
            if ((containingClass instanceof N4InterfaceDeclaration)) {
              String _messageForKEY_SUP_ACCESS_INVALID_LOC_INTERFACE = IssueCodes.getMessageForKEY_SUP_ACCESS_INVALID_LOC_INTERFACE();
              Severity _defaultSeverity_2 = IssueCodes.getDefaultSeverity(IssueCodes.KEY_SUP_ACCESS_INVALID_LOC_INTERFACE);
              DiagnosticMessage _diagnosticMessage_2 = new DiagnosticMessage(_messageForKEY_SUP_ACCESS_INVALID_LOC_INTERFACE, _defaultSeverity_2, 
                IssueCodes.KEY_SUP_ACCESS_INVALID_LOC_INTERFACE);
              producer.addDiagnostic(_diagnosticMessage_2);
            } else {
              if ((containingClass != null)) {
                boolean _isN4JS = constraints.isN4JS();
                boolean _not = (!_isN4JS);
                if (_not) {
                  String _messageForKEY_SUP_ACCESS_NO_EXTENDS = IssueCodes.getMessageForKEY_SUP_ACCESS_NO_EXTENDS();
                  Severity _defaultSeverity_3 = IssueCodes.getDefaultSeverity(IssueCodes.KEY_SUP_ACCESS_NO_EXTENDS);
                  DiagnosticMessage _diagnosticMessage_3 = new DiagnosticMessage(_messageForKEY_SUP_ACCESS_NO_EXTENDS, _defaultSeverity_3, 
                    IssueCodes.KEY_SUP_ACCESS_NO_EXTENDS);
                  producer.addDiagnostic(_diagnosticMessage_3);
                }
              } else {
                throw new IllegalStateException("a");
              }
            }
          }
        }
      }
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private boolean isInvalidSuperLiteral(final SuperLiteral model, final ASTStructureValidator.Constraints constraints) {
    boolean _isValidContainment = this.isValidContainment(model);
    boolean _not = (!_isValidContainment);
    if (_not) {
      return true;
    }
    boolean _isSuperLiteralAllowed = constraints.isSuperLiteralAllowed();
    boolean _not_1 = (!_isSuperLiteralAllowed);
    if (_not_1) {
      return true;
    }
    boolean _isSuperCallAllowed = constraints.isSuperCallAllowed();
    boolean _not_2 = (!_isSuperCallAllowed);
    if (_not_2) {
      EObject _eContainer = model.eContainer();
      return (_eContainer instanceof ParameterizedCallExpression);
    }
    return false;
  }
  
  private boolean isValidContainment(final SuperLiteral literal) {
    final EObject container = literal.eContainer();
    return (((container instanceof IndexedAccessExpression) || (container instanceof ParameterizedCallExpression)) || (container instanceof ParameterizedPropertyAccessExpression));
  }
  
  private void _validateASTStructure(final FunctionDeclaration model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.validateFunctionDefinition(model, 
      N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, 
      constraints.enterFunctionDeclaration(), 
      model.getName(), producer, validLabels);
  }
  
  private void _validateASTStructure(final FunctionExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    final String name = model.getName();
    if ((name != null)) {
      boolean _equals = Objects.equal(name, N4JSLanguageConstants.LOCAL_ARGUMENTS_VARIABLE_NAME);
      if (_equals) {
        this.issueArgumentsError(model, name, constraints.isStrict(), producer);
      } else {
        boolean _isStrict = constraints.isStrict();
        if (_isStrict) {
          if ((N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name) || Objects.equal(name, N4JSLanguageConstants.EVAL_NAME))) {
            this.issueNameDiagnostic(model, producer, name);
          }
        }
      }
    }
    this.recursiveValidateASTStructure(model, producer, 
      Sets.<LabelledStatement>newHashSetWithExpectedSize(2), 
      constraints.allowNestedFunctions(true).allowReturn(true).allowBreak(false).allowContinue(false));
  }
  
  private void validateFunctionDefinition(final FunctionDefinition model, final EAttribute attribute, final ASTStructureValidator.Constraints constraints, final String name, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels) {
    boolean _isNestedFunctionAllowed = constraints.isNestedFunctionAllowed();
    boolean _not = (!_isNestedFunctionAllowed);
    if (_not) {
      final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, attribute);
      INode _elvis = null;
      INode _head = IterableExtensions.<INode>head(nodes);
      if (_head != null) {
        _elvis = _head;
      } else {
        ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(model);
        _elvis = _findActualNodeFor;
      }
      final INode target = _elvis;
      producer.setNode(target);
      String _messageForAST_STR_FUN_NOT_NESTED = IssueCodes.getMessageForAST_STR_FUN_NOT_NESTED();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_STR_FUN_NOT_NESTED);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_STR_FUN_NOT_NESTED, _defaultSeverity, IssueCodes.AST_STR_FUN_NOT_NESTED);
      producer.addDiagnostic(_diagnosticMessage);
    } else {
      if ((name != null)) {
        boolean _equals = Objects.equal(name, N4JSLanguageConstants.LOCAL_ARGUMENTS_VARIABLE_NAME);
        if (_equals) {
          this.issueArgumentsError(model, name, constraints.isStrict(), producer);
        } else {
          boolean _isStrict = constraints.isStrict();
          if (_isStrict) {
            if ((N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name) || Objects.equal(name, N4JSLanguageConstants.EVAL_NAME))) {
              this.issueNameDiagnostic(model, producer, name);
            }
          }
        }
      }
    }
    this.recursiveValidateASTStructure(model, producer, 
      Sets.<LabelledStatement>newHashSetWithExpectedSize(2), 
      constraints.allowNestedFunctions(true).allowReturn(true).allowContinue(false).allowBreak(false).allowYieldExpression(true));
  }
  
  private void validateName(final PropertyNameOwner model, final ASTStructureValidator.Constraints constraints, final ASTStructureDiagnosticProducer producer) {
    final String name = model.getName();
    if ((name != null)) {
      boolean _isValidName = model.isValidName();
      boolean _not = (!_isValidName);
      if (_not) {
        this.issueNameDiagnostic(model, producer, name);
      } else {
        boolean _isN4JS = constraints.isN4JS();
        if (_isN4JS) {
          boolean _contains = N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name);
          if (_contains) {
            this.issueNameDiagnostic(model, producer, name);
          }
        } else {
          boolean _isStrict = constraints.isStrict();
          if (_isStrict) {
            boolean _contains_1 = N4JSLanguageConstants.RESERVED_WORDS_IN_STRICT_MODE.contains(name);
            if (_contains_1) {
              this.issueNameDiagnostic(model, producer, name, this.getNameFeature(model), Severity.WARNING);
            }
          }
        }
      }
    }
  }
  
  private void issueArgumentsError(final EObject model, final String name, final boolean strict, final ASTStructureDiagnosticProducer producer) {
    EStructuralFeature _nameFeature = this.getNameFeature(model);
    Severity _xifexpression = null;
    if (strict) {
      _xifexpression = Severity.ERROR;
    } else {
      _xifexpression = Severity.WARNING;
    }
    this.issueNameDiagnostic(model, producer, name, _nameFeature, _xifexpression);
  }
  
  private void issueNameDiagnostic(final EObject model, final ASTStructureDiagnosticProducer producer, final String name) {
    this.issueNameDiagnostic(model, producer, name, this.getNameFeature(model), IssueCodes.getDefaultSeverity(IssueCodes.AST_RESERVED_IDENTIFIER));
  }
  
  private void issueNameDiagnostic(final EObject model, final ASTStructureDiagnosticProducer producer, final String name, final EStructuralFeature feature, final Severity severity) {
    final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, feature);
    INode _elvis = null;
    INode _head = IterableExtensions.<INode>head(nodes);
    if (_head != null) {
      _elvis = _head;
    } else {
      ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(model);
      _elvis = _findActualNodeFor;
    }
    final INode target = _elvis;
    producer.setNode(target);
    if ((target != null)) {
      String _messageForAST_RESERVED_IDENTIFIER = IssueCodes.getMessageForAST_RESERVED_IDENTIFIER(name);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_RESERVED_IDENTIFIER, severity, 
        IssueCodes.AST_RESERVED_IDENTIFIER);
      producer.addDiagnostic(_diagnosticMessage);
    }
  }
  
  private EStructuralFeature getNameFeature(final EObject model) {
    EStructuralFeature _elvis = null;
    EStructuralFeature _eStructuralFeature = model.eClass().getEStructuralFeature("name");
    if (_eStructuralFeature != null) {
      _elvis = _eStructuralFeature;
    } else {
      EStructuralFeature _eStructuralFeature_1 = model.eClass().getEStructuralFeature("declaredName");
      _elvis = _eStructuralFeature_1;
    }
    return _elvis;
  }
  
  private void _validateASTStructure(final N4FieldAccessor model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.validateName(model, constraints, producer);
    this.recursiveValidateASTStructure(model, producer, 
      Sets.<LabelledStatement>newHashSetWithExpectedSize(2), 
      constraints.allowReturn(true).allowSuperLiteral(true).allowSuperCall(false));
  }
  
  private void _validateASTStructure(final MethodDeclaration model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.validateName(model, constraints, producer);
    ASTStructureValidator.Constraints _allowSuperLiteral = constraints.allowReturn(true).allowSuperLiteral(true);
    this.recursiveValidateASTStructure(model, producer, 
      Sets.<LabelledStatement>newHashSetWithExpectedSize(2), 
      _allowSuperLiteral.allowSuperCall(((Objects.equal(model.getName(), "constructor") && (!model.isStatic())) && this.canCallSuperConstructor(model, constraints))));
  }
  
  private boolean canCallSuperConstructor(final MethodDeclaration declaration, final ASTStructureValidator.Constraints constraints) {
    final EObject container = declaration.eContainer();
    if ((container instanceof N4ClassDefinition)) {
      boolean _isN4JS = constraints.isN4JS();
      if (_isN4JS) {
        return true;
      }
      return ((((N4ClassDefinition)container).getSuperClassRef() != null) || (((N4ClassDefinition)container).getSuperClassExpression() != null));
    }
    return false;
  }
  
  private void _validateASTStructure(final FieldAccessor model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.validateName(model, constraints, producer);
    this.recursiveValidateASTStructure(model, producer, 
      Sets.<LabelledStatement>newHashSetWithExpectedSize(2), 
      constraints.allowReturn(true).allowSuperLiteral(true).allowSuperCall(false));
  }
  
  private void _validateASTStructure(final PropertyAssignment model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, 
      Sets.<LabelledStatement>newHashSetWithExpectedSize(2), 
      constraints.allowNestedFunctions(false).allowReturn(true).allowContinue(false).allowBreak(false).allowYieldExpression(true));
  }
  
  private void _validateASTStructure(final ReturnStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isReturnAllowed = constraints.isReturnAllowed();
    boolean _not = (!_isReturnAllowed);
    if (_not) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_INVALID_RETURN = IssueCodes.getMessageForAST_INVALID_RETURN();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_RETURN);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_RETURN, _defaultSeverity, IssueCodes.AST_INVALID_RETURN);
      producer.addDiagnostic(_diagnosticMessage);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final ContinueStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isContinueAllowed = constraints.isContinueAllowed();
    boolean _not = (!_isContinueAllowed);
    if (_not) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_INVALID_CONTINUE = IssueCodes.getMessageForAST_INVALID_CONTINUE();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_CONTINUE);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_CONTINUE, _defaultSeverity, IssueCodes.AST_INVALID_CONTINUE);
      producer.addDiagnostic(_diagnosticMessage);
    } else {
      this.validateLabelRef(model, producer, validLabels);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final BreakStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    boolean _isBreakAllowed = constraints.isBreakAllowed();
    boolean _not = (!_isBreakAllowed);
    if (_not) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_INVALID_BREAK = IssueCodes.getMessageForAST_INVALID_BREAK();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_BREAK);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_BREAK, _defaultSeverity, IssueCodes.AST_INVALID_BREAK);
      producer.addDiagnostic(_diagnosticMessage);
    } else {
      this.validateLabelRef(model, producer, validLabels);
    }
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void validateLabelRef(final LabelRef model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels) {
    final String labelAsText = model.getLabelAsText();
    if (((labelAsText != null) && (!IterableExtensions.<LabelledStatement>exists(validLabels, ((Function1<LabelledStatement, Boolean>) (LabelledStatement it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, labelAsText));
    }))))) {
      final ICompositeNode target = NodeModelUtils.findActualNodeFor(model);
      producer.setNode(target);
      String _messageForAST_INVALID_LABEL = IssueCodes.getMessageForAST_INVALID_LABEL();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_LABEL);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_LABEL, _defaultSeverity, IssueCodes.AST_INVALID_LABEL);
      producer.addDiagnostic(_diagnosticMessage);
    }
  }
  
  private void _validateASTStructure(final Expression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final ArrayElement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
    this.validateExpressionInArrayOrObjectLiteral(model, constraints, producer);
    this.validateSpreadInArrayLiteral(model, producer);
  }
  
  /**
   * @param elem must be an ArrayElement or PropertyNameValuePair.
   */
  private void validateExpressionInArrayOrObjectLiteral(final EObject elem, final ASTStructureValidator.Constraints constraints, final ASTStructureDiagnosticProducer producer) {
    if ((!(((elem instanceof ArrayElement) || (elem instanceof PropertyNameValuePair)) || (elem instanceof PropertyMethodDeclaration)))) {
      throw new IllegalArgumentException();
    }
    if ((elem instanceof PropertyNameValuePairSingleName)) {
      final IdentifierRef identifier = ((PropertyNameValuePairSingleName)elem).getIdentifierRef();
      if (((identifier != null) && (!identifier.isValidSimpleAssignmentTarget()))) {
        producer.setNode(NodeModelUtils.findActualNodeFor(identifier));
        String _messageForAST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN = IssueCodes.getMessageForAST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN, _defaultSeverity, IssueCodes.AST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN);
        producer.addDiagnostic(_diagnosticMessage);
      }
      return;
    }
    if (((elem != null) && DestructureUtils.isArrayOrObjectLiteralUsedAsDestructuringPattern(elem.eContainer()))) {
      if ((elem instanceof PropertyMethodDeclaration)) {
        producer.setNode(NodeModelUtils.findActualNodeFor(elem));
        String _messageForAST_INVALID_PROPERTY_METHOD_IN_LHS_DESTRUCTURING_PATTERN = IssueCodes.getMessageForAST_INVALID_PROPERTY_METHOD_IN_LHS_DESTRUCTURING_PATTERN();
        Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_PROPERTY_METHOD_IN_LHS_DESTRUCTURING_PATTERN);
        DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_INVALID_PROPERTY_METHOD_IN_LHS_DESTRUCTURING_PATTERN, _defaultSeverity_1, IssueCodes.AST_INVALID_PROPERTY_METHOD_IN_LHS_DESTRUCTURING_PATTERN);
        producer.addDiagnostic(_diagnosticMessage_1);
      } else {
        Expression _switchResult = null;
        boolean _matched = false;
        if (elem instanceof ArrayElement) {
          _matched=true;
          _switchResult = ((ArrayElement)elem).getExpression();
        }
        if (!_matched) {
          if (elem instanceof PropertyNameValuePair) {
            _matched=true;
            _switchResult = ((PropertyNameValuePair)elem).getExpression();
          }
        }
        final Expression expr = _switchResult;
        if (((expr != null) && (!this.isValidBindingElement(expr, constraints)))) {
          producer.setNode(NodeModelUtils.findActualNodeFor(expr));
          String _messageForAST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN_1 = IssueCodes.getMessageForAST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN();
          Severity _defaultSeverity_2 = IssueCodes.getDefaultSeverity(IssueCodes.AST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN);
          DiagnosticMessage _diagnosticMessage_2 = new DiagnosticMessage(_messageForAST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN_1, _defaultSeverity_2, IssueCodes.AST_INVALID_EXPR_IN_LHS_DESTRUCTURING_PATTERN);
          producer.addDiagnostic(_diagnosticMessage_2);
        }
      }
    }
  }
  
  private boolean isValidBindingElement(final Expression expr, final ASTStructureValidator.Constraints constraints) {
    boolean _isN4JS = constraints.isN4JS();
    if (_isN4JS) {
      return ((((expr.isValidSimpleAssignmentTarget() && (expr instanceof IdentifierRef)) || (expr instanceof AssignmentExpression)) || (expr instanceof ArrayLiteral)) || (expr instanceof ObjectLiteral));
    } else {
      return (((expr.isValidSimpleAssignmentTarget() || (expr instanceof AssignmentExpression)) || (expr instanceof ArrayLiteral)) || (expr instanceof ObjectLiteral));
    }
  }
  
  private void validateSpreadInArrayLiteral(final ArrayElement elem, final ASTStructureDiagnosticProducer producer) {
    if (((elem != null) && elem.isSpread())) {
      boolean _isArrayOrObjectLiteralUsedAsDestructuringPattern = DestructureUtils.isArrayOrObjectLiteralUsedAsDestructuringPattern(elem.eContainer());
      boolean _not = (!_isArrayOrObjectLiteralUsedAsDestructuringPattern);
      if (_not) {
        final List<INode> nodes = NodeModelUtils.findNodesForFeature(elem, N4JSPackage.eINSTANCE.getArrayElement_Spread());
        INode _elvis = null;
        INode _head = IterableExtensions.<INode>head(nodes);
        if (_head != null) {
          _elvis = _head;
        } else {
          ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(elem);
          _elvis = _findActualNodeFor;
        }
        producer.setNode(_elvis);
        String _messageForAST_SPREAD_IN_ARRAY_LITERAL_UNSUPPORTED = IssueCodes.getMessageForAST_SPREAD_IN_ARRAY_LITERAL_UNSUPPORTED();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_SPREAD_IN_ARRAY_LITERAL_UNSUPPORTED);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_SPREAD_IN_ARRAY_LITERAL_UNSUPPORTED, _defaultSeverity, IssueCodes.AST_SPREAD_IN_ARRAY_LITERAL_UNSUPPORTED);
        producer.addDiagnostic(_diagnosticMessage);
      } else {
        final EObject lit = elem.eContainer();
        if ((lit instanceof ArrayLiteral)) {
          if (((IterableExtensions.<ArrayElement>last(((ArrayLiteral)lit).getElements()) != elem) || ((ArrayLiteral)lit).isTrailingComma())) {
            final List<INode> nodes_1 = NodeModelUtils.findNodesForFeature(elem, N4JSPackage.eINSTANCE.getArrayElement_Spread());
            INode _elvis_1 = null;
            INode _head_1 = IterableExtensions.<INode>head(nodes_1);
            if (_head_1 != null) {
              _elvis_1 = _head_1;
            } else {
              ICompositeNode _findActualNodeFor_1 = NodeModelUtils.findActualNodeFor(elem);
              _elvis_1 = _findActualNodeFor_1;
            }
            producer.setNode(_elvis_1);
            String _messageForAST_REST_MUST_APPEAR_AT_END = IssueCodes.getMessageForAST_REST_MUST_APPEAR_AT_END();
            Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_REST_MUST_APPEAR_AT_END);
            DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_REST_MUST_APPEAR_AT_END, _defaultSeverity_1, IssueCodes.AST_REST_MUST_APPEAR_AT_END);
            producer.addDiagnostic(_diagnosticMessage_1);
          }
          Expression _expression = elem.getExpression();
          if ((_expression instanceof AssignmentExpression)) {
            final List<INode> nodes_2 = NodeModelUtils.findNodesForFeature(elem.getExpression(), N4JSPackage.eINSTANCE.getAssignmentExpression_Rhs());
            INode _elvis_2 = null;
            INode _head_2 = IterableExtensions.<INode>head(nodes_2);
            if (_head_2 != null) {
              _elvis_2 = _head_2;
            } else {
              ICompositeNode _findActualNodeFor_2 = NodeModelUtils.findActualNodeFor(elem.getExpression());
              _elvis_2 = _findActualNodeFor_2;
            }
            producer.setNode(_elvis_2);
            String _messageForAST_REST_WITH_INITIALIZER = IssueCodes.getMessageForAST_REST_WITH_INITIALIZER();
            Severity _defaultSeverity_2 = IssueCodes.getDefaultSeverity(IssueCodes.AST_REST_WITH_INITIALIZER);
            DiagnosticMessage _diagnosticMessage_2 = new DiagnosticMessage(_messageForAST_REST_WITH_INITIALIZER, _defaultSeverity_2, IssueCodes.AST_REST_WITH_INITIALIZER);
            producer.addDiagnostic(_diagnosticMessage_2);
          }
        }
      }
    }
  }
  
  private void _validateASTStructure(final PropertyNameValuePair model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowYieldExpression(true));
    this.validateExpressionInArrayOrObjectLiteral(model, constraints, producer);
    this.validateSingleNameInObjectLiteral(model, producer);
  }
  
  private void _validateASTStructure(final PropertyMethodDeclaration model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowReturn(true).allowSuperLiteral(true).allowSuperCall(false).allowYieldExpression(true));
    this.validateExpressionInArrayOrObjectLiteral(model, constraints, producer);
  }
  
  private void validateSingleNameInObjectLiteral(final PropertyNameValuePair elem, final ASTStructureDiagnosticProducer producer) {
    if (((elem instanceof PropertyNameValuePairSingleName) && (!DestructureUtils.isArrayOrObjectLiteralUsedAsDestructuringPattern(elem.eContainer())))) {
      producer.setNode(NodeModelUtils.findActualNodeFor(elem));
      String _messageForAST_SINGLE_NAME_IN_OBJECT_LITERAL_UNSUPPORTED = IssueCodes.getMessageForAST_SINGLE_NAME_IN_OBJECT_LITERAL_UNSUPPORTED();
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_SINGLE_NAME_IN_OBJECT_LITERAL_UNSUPPORTED);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_SINGLE_NAME_IN_OBJECT_LITERAL_UNSUPPORTED, _defaultSeverity, IssueCodes.AST_SINGLE_NAME_IN_OBJECT_LITERAL_UNSUPPORTED);
      producer.addDiagnostic(_diagnosticMessage);
    }
  }
  
  private void _validateASTStructure(final VariableStatement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    VariableStatementKeyword _varStmtKeyword = model.getVarStmtKeyword();
    boolean _tripleEquals = (_varStmtKeyword == VariableStatementKeyword.CONST);
    if (_tripleEquals) {
      boolean _isValidConstOrLetPosition = this.isValidConstOrLetPosition(model);
      boolean _not = (!_isValidConstOrLetPosition);
      if (_not) {
        final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, N4JSPackage.Literals.VARIABLE_DECLARATION_CONTAINER__VAR_STMT_KEYWORD);
        INode _elvis = null;
        INode _head = IterableExtensions.<INode>head(nodes);
        if (_head != null) {
          _elvis = _head;
        } else {
          ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(model);
          _elvis = _findActualNodeFor;
        }
        producer.setNode(_elvis);
        String _messageForAST_CONST_IN_STATEMENT_POSITION = IssueCodes.getMessageForAST_CONST_IN_STATEMENT_POSITION();
        Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_CONST_IN_STATEMENT_POSITION);
        DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_CONST_IN_STATEMENT_POSITION, _defaultSeverity, 
          IssueCodes.AST_CONST_IN_STATEMENT_POSITION);
        producer.addDiagnostic(_diagnosticMessage);
      }
    } else {
      VariableStatementKeyword _varStmtKeyword_1 = model.getVarStmtKeyword();
      boolean _tripleEquals_1 = (_varStmtKeyword_1 == VariableStatementKeyword.LET);
      if (_tripleEquals_1) {
        boolean _isValidConstOrLetPosition_1 = this.isValidConstOrLetPosition(model);
        boolean _not_1 = (!_isValidConstOrLetPosition_1);
        if (_not_1) {
          final List<INode> nodes_1 = NodeModelUtils.findNodesForFeature(model, N4JSPackage.Literals.VARIABLE_DECLARATION_CONTAINER__VAR_STMT_KEYWORD);
          INode _elvis_1 = null;
          INode _head_1 = IterableExtensions.<INode>head(nodes_1);
          if (_head_1 != null) {
            _elvis_1 = _head_1;
          } else {
            ICompositeNode _findActualNodeFor_1 = NodeModelUtils.findActualNodeFor(model);
            _elvis_1 = _findActualNodeFor_1;
          }
          producer.setNode(_elvis_1);
          String _messageForAST_LET_IN_STATEMENT_POSITION = IssueCodes.getMessageForAST_LET_IN_STATEMENT_POSITION();
          Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_LET_IN_STATEMENT_POSITION);
          DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_LET_IN_STATEMENT_POSITION, _defaultSeverity_1, 
            IssueCodes.AST_LET_IN_STATEMENT_POSITION);
          producer.addDiagnostic(_diagnosticMessage_1);
        }
      }
    }
    boolean _isEmpty = model.getVarDeclsOrBindings().isEmpty();
    if (_isEmpty) {
      final List<INode> nodes_2 = NodeModelUtils.findNodesForFeature(model, N4JSPackage.Literals.VARIABLE_DECLARATION_CONTAINER__VAR_STMT_KEYWORD);
      INode _elvis_2 = null;
      INode _head_2 = IterableExtensions.<INode>head(nodes_2);
      if (_head_2 != null) {
        _elvis_2 = _head_2;
      } else {
        ICompositeNode _findActualNodeFor_2 = NodeModelUtils.findActualNodeFor(model);
        _elvis_2 = _findActualNodeFor_2;
      }
      producer.setNode(_elvis_2);
      String _messageForAST_VAR_STMT_NO_DECL = IssueCodes.getMessageForAST_VAR_STMT_NO_DECL();
      Severity _defaultSeverity_2 = IssueCodes.getDefaultSeverity(IssueCodes.AST_VAR_STMT_NO_DECL);
      DiagnosticMessage _diagnosticMessage_2 = new DiagnosticMessage(_messageForAST_VAR_STMT_NO_DECL, _defaultSeverity_2, 
        IssueCodes.AST_VAR_STMT_NO_DECL);
      producer.addDiagnostic(_diagnosticMessage_2);
    }
    final EObject directParent = model.eContainer();
    EObject _xifexpression = null;
    if ((directParent instanceof ExportDeclaration)) {
      _xifexpression = ((ExportDeclaration)directParent).eContainer();
    } else {
      _xifexpression = directParent;
    }
    final EObject parent = _xifexpression;
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowVarWithoutInitializer(((model.getVarStmtKeyword() == VariableStatementKeyword.VAR) || ((model.getVarStmtKeyword() == VariableStatementKeyword.LET) && (((parent instanceof Block) || (parent instanceof Script)) || (parent instanceof AbstractCaseClause))))));
  }
  
  private boolean isValidConstOrLetPosition(final EObject model) {
    if ((((model.eContainer() instanceof Block) || (model.eContainer() instanceof Script)) || (model.eContainer() instanceof AbstractCaseClause))) {
      return true;
    }
    EObject _eContainer = model.eContainer();
    if ((_eContainer instanceof ExportDeclaration)) {
      return this.isValidConstOrLetPosition(model.eContainer());
    }
    return false;
  }
  
  private void _validateASTStructure(final VariableDeclaration model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    if ((((model.getExpression() == null) && constraints.isVarInitializerRequired()) && (!constraints.isExternal()))) {
      final List<INode> nodes = NodeModelUtils.findNodesForFeature(model, TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME);
      INode _elvis = null;
      INode _head = IterableExtensions.<INode>head(nodes);
      if (_head != null) {
        _elvis = _head;
      } else {
        ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(model);
        _elvis = _findActualNodeFor;
      }
      producer.setNode(_elvis);
      String _messageForAST_CONST_HAS_NO_INITIALIZER = IssueCodes.getMessageForAST_CONST_HAS_NO_INITIALIZER(model.getName());
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_CONST_HAS_NO_INITIALIZER);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_CONST_HAS_NO_INITIALIZER, _defaultSeverity, IssueCodes.AST_CONST_HAS_NO_INITIALIZER);
      producer.addDiagnostic(_diagnosticMessage);
    }
    this._validateASTStructure(((Variable) model), producer, validLabels, constraints.allowVarWithoutInitializer(true));
  }
  
  private void _validateASTStructure(final ThisTypeRef thisTypeRef, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(thisTypeRef, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final VariableBinding model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.recursiveValidateASTStructure(model, producer, validLabels, 
      constraints.allowVarWithoutInitializer(true));
  }
  
  private void _validateASTStructure(final BindingElement model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    this.validateRestInBindingPattern(model, producer);
    this.recursiveValidateASTStructure(model, producer, validLabels, constraints);
  }
  
  private void _validateASTStructure(final Annotation model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    EObject _eContainer = model.eContainer();
    if ((_eContainer instanceof Script)) {
      EObject _eContainer_1 = model.eContainer();
      final Script script = ((Script) _eContainer_1);
      int _size = script.getScriptElements().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        final ICompositeNode annotationNode = NodeModelUtils.findActualNodeFor(model);
        final int annotationOffset = annotationNode.getOffset();
        final ScriptElement firstScriptElement = script.getScriptElements().get(0);
        final int scriptElementOffset = NodeModelUtils.findActualNodeFor(firstScriptElement).getOffset();
        if ((annotationOffset > scriptElementOffset)) {
          producer.setNode(annotationNode);
          String _messageForAST_SCRIPT_ANNO_INVALID_PLACEMENT = IssueCodes.getMessageForAST_SCRIPT_ANNO_INVALID_PLACEMENT(model.getName());
          Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_SCRIPT_ANNO_INVALID_PLACEMENT);
          DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_SCRIPT_ANNO_INVALID_PLACEMENT, _defaultSeverity, IssueCodes.AST_SCRIPT_ANNO_INVALID_PLACEMENT);
          producer.addDiagnostic(_diagnosticMessage);
        }
      }
    }
  }
  
  private void validateRestInBindingPattern(final BindingElement elem, final ASTStructureDiagnosticProducer producer) {
    if (((elem != null) && elem.isRest())) {
      final EObject pattern = elem.eContainer();
      if ((pattern instanceof ArrayBindingPattern)) {
        BindingElement _last = IterableExtensions.<BindingElement>last(((ArrayBindingPattern)pattern).getElements());
        boolean _tripleNotEquals = (_last != elem);
        if (_tripleNotEquals) {
          final List<INode> nodes = NodeModelUtils.findNodesForFeature(elem, N4JSPackage.eINSTANCE.getBindingElement_Rest());
          INode _elvis = null;
          INode _head = IterableExtensions.<INode>head(nodes);
          if (_head != null) {
            _elvis = _head;
          } else {
            ICompositeNode _findActualNodeFor = NodeModelUtils.findActualNodeFor(elem);
            _elvis = _findActualNodeFor;
          }
          producer.setNode(_elvis);
          String _messageForAST_REST_MUST_APPEAR_AT_END = IssueCodes.getMessageForAST_REST_MUST_APPEAR_AT_END();
          Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_REST_MUST_APPEAR_AT_END);
          DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_REST_MUST_APPEAR_AT_END, _defaultSeverity, IssueCodes.AST_REST_MUST_APPEAR_AT_END);
          producer.addDiagnostic(_diagnosticMessage);
        }
      }
      Expression _expression = elem.getExpression();
      boolean _tripleNotEquals_1 = (_expression != null);
      if (_tripleNotEquals_1) {
        final List<INode> nodes_1 = NodeModelUtils.findNodesForFeature(elem, N4JSPackage.eINSTANCE.getBindingElement_Expression());
        INode _elvis_1 = null;
        INode _head_1 = IterableExtensions.<INode>head(nodes_1);
        if (_head_1 != null) {
          _elvis_1 = _head_1;
        } else {
          ICompositeNode _findActualNodeFor_1 = NodeModelUtils.findActualNodeFor(elem);
          _elvis_1 = _findActualNodeFor_1;
        }
        producer.setNode(_elvis_1);
        String _messageForAST_REST_WITH_INITIALIZER = IssueCodes.getMessageForAST_REST_WITH_INITIALIZER();
        Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_REST_WITH_INITIALIZER);
        DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_REST_WITH_INITIALIZER, _defaultSeverity_1, IssueCodes.AST_REST_WITH_INITIALIZER);
        producer.addDiagnostic(_diagnosticMessage_1);
      }
    }
  }
  
  private void _validateASTStructure(final BinaryLogicalExpression model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    Expression _lhs = model.getLhs();
    boolean _tripleEquals = (_lhs == null);
    if (_tripleEquals) {
      producer.setNode(NodeModelUtils.findActualNodeFor(model));
      String _messageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART = IssueCodes.getMessageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART("left operand");
      Severity _defaultSeverity = IssueCodes.getDefaultSeverity(IssueCodes.AST_BINARY_LOGICAL_EXPRESSION_MISSING_PART);
      DiagnosticMessage _diagnosticMessage = new DiagnosticMessage(_messageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART, _defaultSeverity, 
        IssueCodes.AST_BINARY_LOGICAL_EXPRESSION_MISSING_PART);
      producer.addDiagnostic(_diagnosticMessage);
    }
    Expression _rhs = model.getRhs();
    boolean _tripleEquals_1 = (_rhs == null);
    if (_tripleEquals_1) {
      producer.setNode(NodeModelUtils.findActualNodeFor(model));
      String _messageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART_1 = IssueCodes.getMessageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART("right operand");
      Severity _defaultSeverity_1 = IssueCodes.getDefaultSeverity(IssueCodes.AST_BINARY_LOGICAL_EXPRESSION_MISSING_PART);
      DiagnosticMessage _diagnosticMessage_1 = new DiagnosticMessage(_messageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART_1, _defaultSeverity_1, 
        IssueCodes.AST_BINARY_LOGICAL_EXPRESSION_MISSING_PART);
      producer.addDiagnostic(_diagnosticMessage_1);
    }
    BinaryLogicalOperator _op = model.getOp();
    boolean _tripleEquals_2 = (_op == null);
    if (_tripleEquals_2) {
      producer.setNode(NodeModelUtils.findActualNodeFor(model));
      String _messageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART_2 = IssueCodes.getMessageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART("operator");
      Severity _defaultSeverity_2 = IssueCodes.getDefaultSeverity(IssueCodes.AST_BINARY_LOGICAL_EXPRESSION_MISSING_PART);
      DiagnosticMessage _diagnosticMessage_2 = new DiagnosticMessage(_messageForAST_BINARY_LOGICAL_EXPRESSION_MISSING_PART_2, _defaultSeverity_2, 
        IssueCodes.AST_BINARY_LOGICAL_EXPRESSION_MISSING_PART);
      producer.addDiagnostic(_diagnosticMessage_2);
    }
  }
  
  private void validateASTStructure(final EObject model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    if (model instanceof LegacyOctalIntLiteral) {
      _validateASTStructure((LegacyOctalIntLiteral)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof N4EnumDeclaration) {
      _validateASTStructure((N4EnumDeclaration)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof PropertyMethodDeclaration) {
      _validateASTStructure((PropertyMethodDeclaration)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof PropertyNameValuePair) {
      _validateASTStructure((PropertyNameValuePair)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof StringLiteral) {
      _validateASTStructure((StringLiteral)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof TemplateSegment) {
      _validateASTStructure((TemplateSegment)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ThisTypeRef) {
      _validateASTStructure((ThisTypeRef)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ForStatement) {
      _validateASTStructure((ForStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof FormalParameter) {
      _validateASTStructure((FormalParameter)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof FunctionDeclaration) {
      _validateASTStructure((FunctionDeclaration)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof FunctionExpression) {
      _validateASTStructure((FunctionExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof IdentifierRef) {
      _validateASTStructure((IdentifierRef)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof MethodDeclaration) {
      _validateASTStructure((MethodDeclaration)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof N4ClassifierDefinition) {
      _validateASTStructure((N4ClassifierDefinition)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof N4FieldAccessor) {
      _validateASTStructure((N4FieldAccessor)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof SuperLiteral) {
      _validateASTStructure((SuperLiteral)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof VariableDeclaration) {
      _validateASTStructure((VariableDeclaration)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof AssignmentExpression) {
      _validateASTStructure((AssignmentExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof BinaryLogicalExpression) {
      _validateASTStructure((BinaryLogicalExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof Block) {
      _validateASTStructure((Block)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof BreakStatement) {
      _validateASTStructure((BreakStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ContinueStatement) {
      _validateASTStructure((ContinueStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ExportDeclaration) {
      _validateASTStructure((ExportDeclaration)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof FieldAccessor) {
      _validateASTStructure((FieldAccessor)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof IfStatement) {
      _validateASTStructure((IfStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ImportCallExpression) {
      _validateASTStructure((ImportCallExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof IterationStatement) {
      _validateASTStructure((IterationStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof LabelledStatement) {
      _validateASTStructure((LabelledStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof NewTarget) {
      _validateASTStructure((NewTarget)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof PostfixExpression) {
      _validateASTStructure((PostfixExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof PropertyAssignment) {
      _validateASTStructure((PropertyAssignment)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ReturnStatement) {
      _validateASTStructure((ReturnStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof UnaryExpression) {
      _validateASTStructure((UnaryExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof Variable) {
      _validateASTStructure((Variable)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof VariableBinding) {
      _validateASTStructure((VariableBinding)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof VariableStatement) {
      _validateASTStructure((VariableStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof WithStatement) {
      _validateASTStructure((WithStatement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof YieldExpression) {
      _validateASTStructure((YieldExpression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof AbstractCaseClause) {
      _validateASTStructure((AbstractCaseClause)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof Annotation) {
      _validateASTStructure((Annotation)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof ArrayElement) {
      _validateASTStructure((ArrayElement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof BindingElement) {
      _validateASTStructure((BindingElement)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof Expression) {
      _validateASTStructure((Expression)model, producer, validLabels, constraints);
      return;
    } else if (model instanceof Script) {
      _validateASTStructure((Script)model, producer, validLabels, constraints);
      return;
    } else if (model != null) {
      _validateASTStructure(model, producer, validLabels, constraints);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(model, producer, validLabels, constraints).toString());
    }
  }
  
  private void validateBlockStructure(final EObject container, final Block model, final ASTStructureDiagnosticProducer producer, final Set<LabelledStatement> validLabels, final ASTStructureValidator.Constraints constraints) {
    if (container instanceof FunctionDefinition) {
      _validateBlockStructure((FunctionDefinition)container, model, producer, validLabels, constraints);
      return;
    } else if (container instanceof IfStatement) {
      _validateBlockStructure((IfStatement)container, model, producer, validLabels, constraints);
      return;
    } else if (container instanceof IterationStatement) {
      _validateBlockStructure((IterationStatement)container, model, producer, validLabels, constraints);
      return;
    } else if (container instanceof CatchBlock) {
      _validateBlockStructure((CatchBlock)container, model, producer, validLabels, constraints);
      return;
    } else if (container != null) {
      _validateBlockStructure(container, model, producer, validLabels, constraints);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(container, model, producer, validLabels, constraints).toString());
    }
  }
}
