/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation;

import com.google.inject.Inject;
import java.lang.reflect.Method;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.BasicDiagnostic;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.URI;
import org.eclipse.n4js.smith.DataCollector;
import org.eclipse.n4js.smith.Measurement;
import org.eclipse.n4js.smith.N4JSDataCollectors;
import org.eclipse.n4js.ts.validation.TypesValidator;
import org.eclipse.n4js.utils.Log;
import org.eclipse.n4js.validation.AbstractMessageAdjustingN4JSValidator;
import org.eclipse.n4js.validation.validators.IDEBUGValidator;
import org.eclipse.n4js.validation.validators.N4IDLMigrationValidator;
import org.eclipse.n4js.validation.validators.N4IDLValidator;
import org.eclipse.n4js.validation.validators.N4JSAccessModifierValidator;
import org.eclipse.n4js.validation.validators.N4JSAnnotationValidator;
import org.eclipse.n4js.validation.validators.N4JSClassValidator;
import org.eclipse.n4js.validation.validators.N4JSClassifierValidator;
import org.eclipse.n4js.validation.validators.N4JSDeclaredNameValidator;
import org.eclipse.n4js.validation.validators.N4JSDependencyInjectionValidator;
import org.eclipse.n4js.validation.validators.N4JSDestructureValidator;
import org.eclipse.n4js.validation.validators.N4JSEnumValidator;
import org.eclipse.n4js.validation.validators.N4JSExpressionValidator;
import org.eclipse.n4js.validation.validators.N4JSExternalValidator;
import org.eclipse.n4js.validation.validators.N4JSFlowgraphValidator;
import org.eclipse.n4js.validation.validators.N4JSFunctionValidator;
import org.eclipse.n4js.validation.validators.N4JSImportValidator;
import org.eclipse.n4js.validation.validators.N4JSInjectorCallsitesValidator;
import org.eclipse.n4js.validation.validators.N4JSInterfaceValidator;
import org.eclipse.n4js.validation.validators.N4JSLambdaValidator;
import org.eclipse.n4js.validation.validators.N4JSMemberRedefinitionValidator;
import org.eclipse.n4js.validation.validators.N4JSMemberValidator;
import org.eclipse.n4js.validation.validators.N4JSModuleValidator;
import org.eclipse.n4js.validation.validators.N4JSNameValidator;
import org.eclipse.n4js.validation.validators.N4JSStatementValidator;
import org.eclipse.n4js.validation.validators.N4JSSuperValidator;
import org.eclipse.n4js.validation.validators.N4JSSyntaxValidator;
import org.eclipse.n4js.validation.validators.N4JSTypeValidator;
import org.eclipse.n4js.validation.validators.N4JSVariableValidator;
import org.eclipse.n4js.validation.validators.N4JSXValidator;
import org.eclipse.n4js.validation.validators.ThirdPartyValidator;
import org.eclipse.n4js.validation.validators.UnsupportedFeatureValidator;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.validation.AbstractDeclarativeValidator;
import org.eclipse.xtext.validation.ComposedChecks;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Validation rules for N4JS.
 * 
 * Validation of type expression is defined in
 * {@link TypesValidator}. However, some context
 * sensitive validations such as type ref of formal parameters or return types
 * have to be repeated here, as the rules are overwritten.
 * <p>
 * Note on contained validators:
 * It is required to override
 * <pre>
 * override register(EValidatorRegistrar registrar) {
 * 	// not needed for classes used as ComposedCheck
 * }
 * </pre>
 * since otherwise they will check everything twice!
 * 
 * @see http://www.eclipse.org/Xtext/documentation.html#validation
 * @see <a name="N4JSSpec">[N4JSSpec]</a> N4JS Specification / NumberFour AG. Berlin, 2013 <a href="https://github.com/NumberFour/specs/">[GitHub]</a>
 * @see TypesValidator
 */
@ComposedChecks(validators = { IDEBUGValidator.class, N4JSAccessModifierValidator.class, N4JSAnnotationValidator.class, N4JSClassifierValidator.class, N4JSClassValidator.class, N4JSDeclaredNameValidator.class, N4JSDependencyInjectionValidator.class, N4JSDestructureValidator.class, N4JSEnumValidator.class, N4JSExpressionValidator.class, N4JSExternalValidator.class, N4JSFlowgraphValidator.class, N4JSFunctionValidator.class, N4JSImportValidator.class, N4JSInjectorCallsitesValidator.class, N4JSInterfaceValidator.class, N4JSLambdaValidator.class, N4JSMemberRedefinitionValidator.class, N4JSMemberValidator.class, N4JSModuleValidator.class, N4JSNameValidator.class, N4JSStatementValidator.class, N4JSSuperValidator.class, N4JSSyntaxValidator.class, N4JSTypeValidator.class, N4JSVariableValidator.class, N4JSXValidator.class, ThirdPartyValidator.class, UnsupportedFeatureValidator.class, N4IDLValidator.class, N4IDLMigrationValidator.class })
@Log
@SuppressWarnings("all")
public class N4JSValidator extends AbstractMessageAdjustingN4JSValidator {
  public static class N4JSMethodWrapperCancelable extends AbstractMessageAdjustingN4JSValidator.MethodWrapperCancelable {
    private OperationCanceledManager operationCanceledManager;
    
    public N4JSMethodWrapperCancelable(final AbstractDeclarativeValidator instance, final Method m, final OperationCanceledManager operationCanceledManager) {
      super(instance, m);
      this.operationCanceledManager = operationCanceledManager;
    }
    
    @Override
    public void handleInvocationTargetException(final Throwable targetException, final AbstractDeclarativeValidator.State state) {
      super.handleInvocationTargetException(targetException, state);
      if ((targetException instanceof NullPointerException)) {
        Exceptions.sneakyThrow(targetException);
      }
    }
    
    @Override
    public void invoke(final AbstractDeclarativeValidator.State state) {
      final String valMethodName = this.getMethod().getName();
      final URI URI = state.currentObject.eResource().getURI();
      this.operationCanceledManager.checkCanceled(this.getCancelIndicator(state));
      final DataCollector dcCheckMethod = N4JSDataCollectors.createDataCollectorForCheckMethod(valMethodName);
      String _string = URI.toString();
      String _plus = ((valMethodName + "_") + _string);
      final Measurement mesVM = dcCheckMethod.getMeasurement(_plus);
      try {
        super.invoke(state);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          this.operationCanceledManager.propagateIfCancelException(e);
          N4JSValidator.logger.error("Error executing EValidator", e);
          String _string_1 = state.currentObject.toString();
          String _message = e.getMessage();
          BasicDiagnostic _basicDiagnostic = new BasicDiagnostic(Diagnostic.ERROR, _string_1, 0, _message, new Object[] { e });
          state.chain.add(_basicDiagnostic);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      } finally {
        mesVM.close();
      }
    }
  }
  
  @Inject
  private OperationCanceledManager operationCanceledManager;
  
  /**
   * Override to improve error message in case of abnormal termination of validation.
   */
  @Override
  public AbstractMessageAdjustingN4JSValidator.MethodWrapperCancelable createMethodWrapper(final AbstractDeclarativeValidator instanceToUse, final Method method) {
    return new N4JSValidator.N4JSMethodWrapperCancelable(instanceToUse, method, this.operationCanceledManager);
  }
  
  private static final Logger logger = Logger.getLogger(N4JSValidator.class);
}
