/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Joiner;
import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.inject.Inject;
import java.util.HashSet;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AbstractAnnotationList;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.AnnotableExpression;
import org.eclipse.n4js.n4JS.AnnotableN4MemberDeclaration;
import org.eclipse.n4js.n4JS.AnnotablePropertyAssignment;
import org.eclipse.n4js.n4JS.AnnotableScriptElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.projectModel.IN4JSSourceContainer;
import org.eclipse.n4js.ts.scoping.N4TSQualifiedNameProvider;
import org.eclipse.n4js.ts.typeRefs.BaseTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.PromisifyHelper;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Annotation validation rules for N4JS.
 */
@SuppressWarnings("all")
public class N4JSAnnotationValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private IQualifiedNameProvider qnProvider;
  
  @Inject
  private ResourceDescriptionsProvider indexAccess;
  
  @Inject
  private PromisifyHelper promisifyHelper;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * check all annotable elements and do various checks for special types.
   */
  @Check
  public void checkAnnotations(final AnnotableElement annotableElement) {
    boolean _allowAnnotation = this.jsVariantHelper.allowAnnotation(annotableElement);
    boolean _not = (!_allowAnnotation);
    if (_not) {
      int _size = annotableElement.getAnnotations().size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        this.addIssue(IssueCodes.getMessageForANN__ONLY_IN_N4JS(), annotableElement, this.annoFeature(annotableElement), IssueCodes.ANN__ONLY_IN_N4JS);
      }
      return;
    }
    if ((annotableElement instanceof AbstractAnnotationList)) {
      return;
    }
    final HashSet<String> foundNames = new HashSet<String>();
    EList<Annotation> _annotations = annotableElement.getAnnotations();
    for (final Annotation a : _annotations) {
      String _name = a.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        final AnnotationDefinition def = AnnotationDefinition.find(a.getName());
        if ((def == null)) {
          this.addIssue(IssueCodes.getMessageForANN_NOT_DEFINED(a.getName()), a, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_NOT_DEFINED);
        } else {
          if (def.repeatable) {
            boolean _add = foundNames.add(a.getName());
            if (_add) {
              final Function1<Annotation, Boolean> _function = (Annotation e) -> {
                String _name_1 = e.getName();
                String _name_2 = a.getName();
                return Boolean.valueOf(Objects.equal(_name_1, _name_2));
              };
              this.internalCheckRepeatableAnnotations(def, IterableExtensions.<Annotation>filter(annotableElement.getAnnotations(), _function));
            }
          } else {
            boolean _add_1 = foundNames.add(a.getName());
            boolean _not_1 = (!_add_1);
            if (_not_1) {
              this.addIssue(IssueCodes.getMessageForANN_NON_REPEATABLE(a.getName()), a, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_NON_REPEATABLE);
            } else {
              this.internalCheckAnnotation(def, a);
            }
          }
        }
      }
    }
  }
  
  /**
   * Checks all repeatable annotations with same name.
   */
  private void internalCheckRepeatableAnnotations(final AnnotationDefinition definition, final Iterable<Annotation> annotations) {
    if ((IterableExtensions.<Annotation>forall(annotations, ((Function1<Annotation, Boolean>) (Annotation it) -> {
      return Boolean.valueOf(this.holdsTargets(definition, it));
    })) && 
      IterableExtensions.<Annotation>forall(annotations, ((Function1<Annotation, Boolean>) (Annotation it) -> {
        return Boolean.valueOf(this.holdsArgumentTypes(definition, it));
      })))) {
      if ((definition.transitive && (!definition.repeatable))) {
        final Consumer<Annotation> _function = (Annotation it) -> {
          this.checkUnnecessaryAnnotation(definition, it);
        };
        annotations.forEach(_function);
      }
    }
  }
  
  /**
   * Checks a single non-repeatable annotation
   */
  private void internalCheckAnnotation(final AnnotationDefinition definition, final Annotation annotation) {
    if (((this.holdsJavaScriptVariants(definition, annotation) && this.holdsTargets(definition, annotation)) && this.holdsArgumentTypes(definition, annotation))) {
      this.checkUnnecessaryAnnotation(definition, annotation);
      String _name = annotation.getName();
      boolean _matched = false;
      if (Objects.equal(_name, AnnotationDefinition.THIS.name)) {
        _matched=true;
        this.internalCheckThis(annotation);
      }
      if (!_matched) {
        if (Objects.equal(_name, AnnotationDefinition.PROVIDED_BY_RUNTIME.name)) {
          _matched=true;
          this.internalCheckProvidedByRuntime(annotation);
        }
      }
      if (!_matched) {
        if (Objects.equal(_name, AnnotationDefinition.IGNORE_IMPLEMENTATION.name)) {
          _matched=true;
          this.internalCheckIgnoreImplementation(annotation);
        }
      }
      if (!_matched) {
        if (Objects.equal(_name, AnnotationDefinition.STATIC_POLYFILL_MODULE.name)) {
          _matched=true;
          this.internalCheckStaticPolyfillModule(annotation);
        }
      }
      if (!_matched) {
        if (Objects.equal(_name, AnnotationDefinition.STATIC_POLYFILL.name)) {
          _matched=true;
          this.internalCheckStaticPolyfill(annotation);
        }
      }
      if (!_matched) {
        if (Objects.equal(_name, AnnotationDefinition.N4JS.name)) {
          _matched=true;
          this.internalCheckN4JS(annotation);
        }
      }
    }
  }
  
  private void checkUnnecessaryAnnotation(final AnnotationDefinition definition, final Annotation annotation) {
    if ((definition.transitive && (!definition.repeatable))) {
      boolean _hasAnnotation = definition.hasAnnotation(
        EcoreUtil2.<AnnotableElement>getContainerOfType(annotation.getAnnotatedElement().eContainer(), AnnotableElement.class));
      if (_hasAnnotation) {
        this.addIssue(IssueCodes.getMessageForANN_UNNECESSARY(annotation.getName()), annotation, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_UNNECESSARY);
      }
    }
  }
  
  private boolean holdsArgumentTypes(final AnnotationDefinition definition, final Annotation annotation) {
    final int actualSize = annotation.getArgs().size();
    final int expectedSize = definition.argtypes.length;
    final boolean variadic = definition.argsVariadic;
    if (variadic) {
      if (((expectedSize - actualSize) >= 2)) {
        this.addIssue(IssueCodes.getMessageForANN_WRONG_NUMBER_OF_ARGUMENTS(definition.name, Integer.valueOf((expectedSize - 1)), Integer.valueOf(actualSize)), annotation, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_WRONG_NUMBER_OF_ARGUMENTS);
        return false;
      }
      if (((1 == (expectedSize - actualSize)) || (expectedSize == actualSize))) {
        return this.validateArgumentTypes(definition, annotation);
      }
      boolean valid = true;
      for (int i = 0; (i < actualSize); i++) {
        {
          final EObject arg = annotation.getArgs().get(i).value();
          EClass _xifexpression = null;
          int _length = definition.argtypes.length;
          boolean _greaterThan = ((i + 1) > _length);
          if (_greaterThan) {
            _xifexpression = IterableExtensions.<EClass>last(((Iterable<EClass>)Conversions.doWrapArray(definition.argtypes)));
          } else {
            _xifexpression = definition.argtypes[i];
          }
          final EClass argType = _xifexpression;
          boolean _isInstance = argType.isInstance(arg);
          boolean _not = (!_isInstance);
          if (_not) {
            this.addIssue(IssueCodes.getMessageForANN_WRONG_ARGUMENT_TYPE(definition.name, argType.getName()), annotation, 
              N4JSPackage.Literals.ANNOTATION__ARGS, i, IssueCodes.ANN_WRONG_ARGUMENT_TYPE);
            valid = false;
          }
        }
      }
      return valid;
    } else {
      if (((actualSize > expectedSize) || ((!definition.argsOptional) && (actualSize != expectedSize)))) {
        this.addIssue(IssueCodes.getMessageForANN_WRONG_NUMBER_OF_ARGUMENTS(definition.name, Integer.valueOf(expectedSize), Integer.valueOf(actualSize)), annotation, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_WRONG_NUMBER_OF_ARGUMENTS);
        return false;
      }
    }
    return this.validateArgumentTypes(definition, annotation);
  }
  
  private boolean validateArgumentTypes(final AnnotationDefinition definition, final Annotation annotation) {
    final int actualSize = annotation.getArgs().size();
    final int expectedSize = definition.argtypes.length;
    boolean valid = true;
    final int min = Math.min(expectedSize, actualSize);
    for (int i = 0; (i < min); i++) {
      {
        final EObject arg = annotation.getArgs().get(i).value();
        final EClass argType = definition.argtypes[i];
        boolean _isInstance = argType.isInstance(arg);
        boolean _not = (!_isInstance);
        if (_not) {
          this.addIssue(IssueCodes.getMessageForANN_WRONG_ARGUMENT_TYPE(definition.name, argType.getName()), annotation, 
            N4JSPackage.Literals.ANNOTATION__ARGS, i, IssueCodes.ANN_WRONG_ARGUMENT_TYPE);
          valid = false;
        }
      }
    }
    return valid;
  }
  
  /**
   * Checks whether the given annotation conforms with the {@link AnnotationDefinition#javaScriptVariants} of the
   * given AnnotationDefinition.
   */
  private boolean holdsJavaScriptVariants(final AnnotationDefinition definition, final Annotation annotation) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(((Iterable<?>)Conversions.doWrapArray(definition.javaScriptVariants)));
    if (_isNullOrEmpty) {
      return true;
    }
    final EObject element = annotation.getAnnotatedElement();
    boolean _contains = ((List<String>)Conversions.doWrapArray(definition.javaScriptVariants)).contains(this.jsVariantHelper.variantMode(element));
    boolean _not = (!_contains);
    if (_not) {
      final Function1<String, String> _function = (String v) -> {
        return this.jsVariantHelper.getVariantName(v);
      };
      final String message = IssueCodes.getMessageForANN_ONL_ALLOWED_IN_VARIANTS(definition.name, 
        this.validatorMessageHelper.orList(ListExtensions.<String, String>map(((List<String>)Conversions.doWrapArray(definition.javaScriptVariants)), _function)));
      this.addIssue(message, annotation, IssueCodes.ANN_ONL_ALLOWED_IN_VARIANTS);
      return false;
    }
    return true;
  }
  
  private boolean holdsTargets(final AnnotationDefinition definition, final Annotation annotation) {
    if (((definition.targets == null) || (((List<EClass>)Conversions.doWrapArray(definition.targets)).size() == 0))) {
      return true;
    }
    final EObject element = annotation.getAnnotatedElement();
    if ((element == null)) {
      return true;
    }
    final boolean isElementAmongValidTargets = (IterableExtensions.<EClass>exists(((Iterable<EClass>)Conversions.doWrapArray(definition.targets)), ((Function1<EClass, Boolean>) (EClass it) -> {
      return Boolean.valueOf(it.isInstance(element));
    })) || 
      IterableExtensions.<EClass>exists(((Iterable<EClass>)Conversions.doWrapArray(definition.targetsWithCustomError)), ((Function1<EClass, Boolean>) (EClass it) -> {
        return Boolean.valueOf(it.isInstance(element));
      })));
    if ((!isElementAmongValidTargets)) {
      ExportableElement _xifexpression = null;
      if ((element instanceof ExportDeclaration)) {
        _xifexpression = ((ExportDeclaration)element).getExportedElement();
      } else {
        _xifexpression = null;
      }
      final ExportableElement exportedElement = _xifexpression;
      final boolean isExportedElementAmongValidTargets = ((exportedElement != null) && 
        (IterableExtensions.<EClass>exists(((Iterable<EClass>)Conversions.doWrapArray(definition.targets)), ((Function1<EClass, Boolean>) (EClass it) -> {
          return Boolean.valueOf(it.isInstance(exportedElement));
        })) || 
          IterableExtensions.<EClass>exists(((Iterable<EClass>)Conversions.doWrapArray(definition.targetsWithCustomError)), ((Function1<EClass, Boolean>) (EClass it) -> {
            return Boolean.valueOf(it.isInstance(exportedElement));
          }))));
      if ((!isExportedElementAmongValidTargets)) {
        this.addWrongLocationIssue(annotation);
        return false;
      }
    }
    String _name = annotation.getName();
    boolean _matched = false;
    if (Objects.equal(_name, AnnotationDefinition.SPEC.name)) {
      _matched=true;
      return this.internalCheckLocationSpec(annotation);
    }
    return true;
  }
  
  private void addWrongLocationIssue(final Annotation annotation) {
    this.addIssue(IssueCodes.getMessageForANN_DISALLOWED_AT_LOCATION(annotation.getName()), annotation, N4JSPackage.Literals.ANNOTATION__NAME, 
      IssueCodes.ANN_DISALLOWED_AT_LOCATION);
  }
  
  /**
   * Mapping from EObject to the Feature containing the Annotations. Used for the correct location of error-markings.
   */
  private EStructuralFeature annoFeature(final AnnotableElement element) {
    EReference _switchResult = null;
    boolean _matched = false;
    if (element instanceof AnnotableExpression) {
      _matched=true;
      _switchResult = N4JSPackage.Literals.ANNOTABLE_EXPRESSION__ANNOTATION_LIST;
    }
    if (!_matched) {
      if (element instanceof AnnotableScriptElement) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.ANNOTABLE_SCRIPT_ELEMENT__ANNOTATION_LIST;
      }
    }
    if (!_matched) {
      if (element instanceof AnnotableN4MemberDeclaration) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.ANNOTABLE_N4_MEMBER_DECLARATION__ANNOTATION_LIST;
      }
    }
    if (!_matched) {
      if (element instanceof AnnotablePropertyAssignment) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.ANNOTABLE_PROPERTY_ASSIGNMENT__ANNOTATION_LIST;
      }
    }
    if (!_matched) {
      if (element instanceof FormalParameter) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.FORMAL_PARAMETER__ANNOTATIONS;
      }
    }
    if (!_matched) {
      if (element instanceof Script) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.SCRIPT__ANNOTATIONS;
      }
    }
    if (!_matched) {
      if (element instanceof VariableDeclaration) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.VARIABLE_DECLARATION__ANNOTATIONS;
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  /**
   * Constraints 91 (Valid Target and Argument for &#64;This Annotation)
   */
  private void internalCheckThis(final Annotation annotation) {
    final EObject element = annotation.getAnnotatedElement();
    if ((element == null)) {
      return;
    }
    if ((element instanceof N4MemberDeclaration)) {
      final TMember tMember = ((N4MemberDeclaration)element).getDefinedTypeElement();
      ContainerType<?> _containingType = null;
      if (tMember!=null) {
        _containingType=tMember.getContainingType();
      }
      final ContainerType<?> containingType = _containingType;
      if (((tMember == null) || (containingType == null))) {
        return;
      }
      if ((tMember.isStatic() && (containingType instanceof TInterface))) {
        final String msg = IssueCodes.getMessageForANN_THIS_DISALLOWED_ON_STATIC_MEMBER_OF_INTERFACE();
        this.addIssue(msg, annotation, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_THIS_DISALLOWED_ON_STATIC_MEMBER_OF_INTERFACE);
        return;
      }
      TypeRef _switchResult = null;
      boolean _matched = false;
      if (tMember instanceof TMethod) {
        _matched=true;
        _switchResult = ((TMethod)tMember).getDeclaredThisType();
      }
      if (!_matched) {
        if (tMember instanceof FieldAccessor) {
          _matched=true;
          _switchResult = ((FieldAccessor)tMember).getDeclaredThisType();
        }
      }
      final TypeRef declThisTypeRef = _switchResult;
      BaseTypeRef _xifexpression = null;
      boolean _isStatic = tMember.isStatic();
      if (_isStatic) {
        _xifexpression = TypeUtils.createTypeTypeRef(TypeUtils.createTypeRef(containingType), false);
      } else {
        _xifexpression = TypeUtils.createTypeRef(containingType, TypingStrategy.DEFAULT, true);
      }
      final BaseTypeRef containingTypeRef = _xifexpression;
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(element);
      boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, declThisTypeRef, containingTypeRef);
      boolean _not = (!_subtypeSucceeded);
      if (_not) {
        final String msg_1 = IssueCodes.getMessageForANN_THIS_NOT_SUBTYPE_OF_CONTAINING_TYPE(this.validatorMessageHelper.description(tMember), 
          this.validatorMessageHelper.description(containingType), containingTypeRef.getTypeRefAsString());
        this.addIssue(msg_1, annotation, N4JSPackage.Literals.ANNOTATION__ARGS, IssueCodes.ANN_THIS_NOT_SUBTYPE_OF_CONTAINING_TYPE);
      }
    }
  }
  
  /**
   * Check N4JS annotation to be in definition file.
   */
  private void internalCheckN4JS(final Annotation annotation) {
    final EObject element = annotation.getAnnotatedElement();
    if ((element == null)) {
      return;
    }
    boolean _isExternalMode = this.jsVariantHelper.isExternalMode(element);
    boolean _not = (!_isExternalMode);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForANN_DISALLOWED_IN_NONDEFINTION_FILE(annotation.getName()), annotation, N4JSPackage.Literals.ANNOTATION__NAME, 
        IssueCodes.ANN_DISALLOWED_IN_NONDEFINTION_FILE);
      return;
    }
  }
  
  /**
   * Check SPEC annotation to be at a formal parameter in a constructor.
   */
  private boolean internalCheckLocationSpec(final Annotation annotation) {
    final EObject element = annotation.getAnnotatedElement();
    if ((element != null)) {
      final EObject parent = element.eContainer();
      if ((parent instanceof N4MethodDeclaration)) {
        boolean _isConstructor = ((N4MethodDeclaration)parent).isConstructor();
        if (_isConstructor) {
          return true;
        }
      }
    }
    final String msg = IssueCodes.getMessageForANN_ONLY_ALLOWED_LOCATION_CONSTRUCTORS(annotation.getName());
    this.addIssue(msg, annotation, N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_ONLY_ALLOWED_LOCATION_CONSTRUCTORS);
    return false;
  }
  
  /**
   * Constraints 120 (Provided By Runtime)
   */
  private void internalCheckProvidedByRuntime(final Annotation annotation) {
    final EObject element = annotation.getAnnotatedElement();
    if ((element == null)) {
      return;
    }
    boolean _isExternalMode = this.jsVariantHelper.isExternalMode(element);
    boolean _not = (!_isExternalMode);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForANN_DISALLOWED_IN_NONDEFINTION_FILE(annotation.getName()), annotation, N4JSPackage.Literals.ANNOTATION__NAME, 
        IssueCodes.ANN_DISALLOWED_IN_NONDEFINTION_FILE);
      return;
    }
    final URI projURI = element.eResource().getURI();
    final Optional<? extends IN4JSProject> project = this.n4jsCore.findProject(projURI);
    boolean _isPresent = project.isPresent();
    boolean _not_1 = (!_isPresent);
    if (_not_1) {
      final String msg = IssueCodes.getMessageForNO_PROJECT_FOUND(projURI);
      final Script script = EcoreUtil2.<Script>getContainerOfType(element, Script.class);
      this.addIssue(msg, script, IssueCodes.NO_PROJECT_FOUND);
    } else {
      IN4JSProject _orNull = project.orNull();
      ProjectType _projectType = null;
      if (_orNull!=null) {
        _projectType=_orNull.getProjectType();
      }
      final ProjectType projectType = _projectType;
      if (((projectType != ProjectType.RUNTIME_ENVIRONMENT) && (projectType != ProjectType.RUNTIME_LIBRARY))) {
        this.addIssue(IssueCodes.getMessageForANN_DISALLOWED_IN_NON_RUNTIME_COMPONENT(annotation.getName()), annotation, 
          N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.ANN_DISALLOWED_IN_NON_RUNTIME_COMPONENT);
        return;
      }
    }
  }
  
  /**
   * 11.1.3. Implementation of External Declarations
   */
  private void internalCheckIgnoreImplementation(final Annotation annotation) {
    final EObject element = annotation.getAnnotatedElement();
    if ((element == null)) {
      return;
    }
    boolean _isExternalMode = this.jsVariantHelper.isExternalMode(element);
    boolean _not = (!_isExternalMode);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForANN_DISALLOWED_IN_NONDEFINTION_FILE(annotation.getName()), annotation, N4JSPackage.Literals.ANNOTATION__NAME, 
        IssueCodes.ANN_DISALLOWED_IN_NONDEFINTION_FILE);
      return;
    }
  }
  
  /**
   * Constraints 140 (Restrictions on static-polyfilling)  §140.3
   */
  private void internalCheckStaticPolyfillModule(final Annotation annotation) {
    final EObject element = annotation.getAnnotatedElement();
    if ((element == null)) {
      return;
    }
    if ((!(element instanceof Script))) {
      return;
    }
    final Script script = ((Script) element);
    boolean _isContainedInStaticPolyfillAware = N4JSLanguageUtils.isContainedInStaticPolyfillAware(script);
    if (_isContainedInStaticPolyfillAware) {
      this.addIssue(IssueCodes.getMessageForANN_POLY_AWARE_AND_MODULE_MUTUAL_EXCLUSIVE(), annotation, N4JSPackage.Literals.ANNOTATION__NAME, 
        IssueCodes.ANN_POLY_AWARE_AND_MODULE_MUTUAL_EXCLUSIVE);
      return;
    }
    final QualifiedName moduleQN = this.qnProvider.getFullyQualifiedName(script.getModule());
    boolean _isModulePolyfill = N4TSQualifiedNameProvider.isModulePolyfill(moduleQN);
    boolean _not = (!_isModulePolyfill);
    if (_not) {
      TModule _module = null;
      if (script!=null) {
        _module=script.getModule();
      }
      Resource _eResource = null;
      if (_module!=null) {
        _eResource=_module.eResource();
      }
      URI _uRI = null;
      if (_eResource!=null) {
        _uRI=_eResource.getURI();
      }
      String _plus = ("### strange should start with \'!MPOLY\' for " + _uRI);
      InputOutput.<String>println(_plus);
      return;
    }
    if ((moduleQN == null)) {
      return;
    }
    final IN4JSProject currentProject = this.n4jsCore.findProject(script.eResource().getURI()).orNull();
    if ((currentProject == null)) {
      return;
    }
    final IResourceDescriptions index = this.indexAccess.getResourceDescriptions(script.eResource().getResourceSet());
    final IResourceDescription currentResDesc = index.getResourceDescription(script.eResource().getURI());
    final Function1<IEObjectDescription, Boolean> _function = (IEObjectDescription it) -> {
      return Boolean.valueOf(((it.getQualifiedName() != null) && it.getQualifiedName().startsWith(moduleQN)));
    };
    final Function1<IEObjectDescription, URI> _function_1 = (IEObjectDescription it) -> {
      return it.getEObjectURI();
    };
    final Function1<URI, URI> _function_2 = (URI it) -> {
      return it.trimFragment();
    };
    final Function1<URI, IResourceDescription> _function_3 = (URI it) -> {
      return index.getResourceDescription(it);
    };
    final Function1<IResourceDescription, Boolean> _function_4 = (IResourceDescription it) -> {
      return Boolean.valueOf((!Objects.equal(currentResDesc, it)));
    };
    final Function1<IResourceDescription, Boolean> _function_5 = (IResourceDescription it) -> {
      boolean _xblockexpression = false;
      {
        final IN4JSProject prj = this.n4jsCore.findProject(it.getURI()).orNull();
        _xblockexpression = Objects.equal(currentProject, prj);
      }
      return Boolean.valueOf(_xblockexpression);
    };
    final List<IResourceDescription> sameResdesc = IterableExtensions.<IResourceDescription>toList(IterableExtensions.<IResourceDescription>filter(IterableExtensions.<IResourceDescription>filter(IterableExtensions.<URI, IResourceDescription>map(IterableExtensions.<URI, URI>map(IterableExtensions.<IEObjectDescription, URI>map(IterableExtensions.<IEObjectDescription>filter(index.getExportedObjectsByType(TypesPackage.Literals.TMODULE), _function), _function_1), _function_2), _function_3), _function_4), _function_5));
    boolean _isEmpty = sameResdesc.isEmpty();
    boolean _not_1 = (!_isEmpty);
    if (_not_1) {
      String _xifexpression = null;
      int _size = sameResdesc.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        _xifexpression = "s";
      } else {
        _xifexpression = "";
      }
      String _plus_1 = ("module" + _xifexpression);
      final String msg_prefix = (_plus_1 + " in ");
      final Function1<IResourceDescription, String> _function_6 = (IResourceDescription it) -> {
        String _xblockexpression = null;
        {
          final Optional<? extends IN4JSSourceContainer> srcCon = this.n4jsCore.findN4JSSourceContainer(it.getURI());
          String _xifexpression_1 = null;
          boolean _isPresent = srcCon.isPresent();
          if (_isPresent) {
            _xifexpression_1 = srcCon.get().getRelativeLocation();
          } else {
            _xifexpression_1 = it.getURI().toString();
          }
          _xblockexpression = _xifexpression_1;
        }
        return _xblockexpression;
      };
      final String clashes = Joiner.on(", ").join(ListExtensions.<IResourceDescription, String>map(sameResdesc, _function_6));
      this.addIssue(IssueCodes.getMessageForPOLY_CLASH_IN_STATIC_POLYFILL_MODULE((msg_prefix + clashes)), annotation, 
        N4JSPackage.Literals.ANNOTATION__NAME, IssueCodes.POLY_CLASH_IN_STATIC_POLYFILL_MODULE);
    }
  }
  
  /**
   * Constraints 139 (static polyfill layout)  §139
   */
  private void internalCheckStaticPolyfill(final Annotation annotation) {
    EObject _annotatedElement = annotation.getAnnotatedElement();
    final AnnotableElement element = ((AnnotableElement) _annotatedElement);
    if ((element == null)) {
      return;
    }
    boolean _isContainedInStaticPolyfillModule = N4JSLanguageUtils.isContainedInStaticPolyfillModule(element);
    boolean _not = (!_isContainedInStaticPolyfillModule);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForANN_POLY_STATIC_POLY_ONLY_IN_POLYFILL_MODULE(), annotation, N4JSPackage.Literals.ANNOTATION__NAME, 
        IssueCodes.ANN_POLY_STATIC_POLY_ONLY_IN_POLYFILL_MODULE);
      return;
    }
    EObject annoTarget = element;
    if ((element instanceof ExportDeclaration)) {
      annoTarget = ((ExportDeclaration)element).getExportedElement();
    }
    if ((!(annoTarget instanceof N4ClassDeclaration))) {
    }
  }
  
  @Check
  public boolean checkPromisifiableMethod(final FunctionDeclaration methodDecl) {
    boolean _xifexpression = false;
    boolean _hasAnnotation = AnnotationDefinition.PROMISIFIABLE.hasAnnotation(methodDecl);
    if (_hasAnnotation) {
      _xifexpression = this.holdsPromisifiablePreconditions(methodDecl);
    }
    return _xifexpression;
  }
  
  @Check
  public boolean checkPromisifiableMethod(final N4MethodDeclaration methodDecl) {
    boolean _xifexpression = false;
    boolean _hasAnnotation = AnnotationDefinition.PROMISIFIABLE.hasAnnotation(methodDecl);
    if (_hasAnnotation) {
      _xifexpression = this.holdsPromisifiablePreconditions(methodDecl);
    }
    return _xifexpression;
  }
  
  private boolean holdsPromisifiablePreconditions(final FunctionDefinition funDef) {
    boolean _switchResult = false;
    PromisifyHelper.CheckResult _checkPromisifiablePreconditions = this.promisifyHelper.checkPromisifiablePreconditions(funDef);
    if (_checkPromisifiablePreconditions != null) {
      switch (_checkPromisifiablePreconditions) {
        case MISSING_CALLBACK:
          boolean _xblockexpression = false;
          {
            this.addIssue(IssueCodes.getMessageForANN_PROMISIFIABLE_MISSING_CALLBACK(), AnnotationDefinition.PROMISIFIABLE.getAnnotation(funDef), 
              IssueCodes.ANN_PROMISIFIABLE_MISSING_CALLBACK);
            _xblockexpression = false;
          }
          _switchResult = _xblockexpression;
          break;
        case BAD_CALLBACK__MORE_THAN_ONE_ERROR:
          boolean _xblockexpression_1 = false;
          {
            this.addIssue(IssueCodes.getMessageForANN_PROMISIFIABLE_BAD_CALLBACK_MORE_THAN_ONE_ERROR(), 
              AnnotationDefinition.PROMISIFIABLE.getAnnotation(funDef), IssueCodes.ANN_PROMISIFIABLE_BAD_CALLBACK_MORE_THAN_ONE_ERROR);
            _xblockexpression_1 = false;
          }
          _switchResult = _xblockexpression_1;
          break;
        case BAD_CALLBACK__ERROR_NOT_FIRST_ARG:
          boolean _xblockexpression_2 = false;
          {
            this.addIssue(IssueCodes.getMessageForANN_PROMISIFIABLE_BAD_CALLBACK_ERROR_NOT_FIRST_ARG(), 
              AnnotationDefinition.PROMISIFIABLE.getAnnotation(funDef), IssueCodes.ANN_PROMISIFIABLE_BAD_CALLBACK_ERROR_NOT_FIRST_ARG);
            _xblockexpression_2 = false;
          }
          _switchResult = _xblockexpression_2;
          break;
        case OK:
          _switchResult = true;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
}
