/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimaps;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.n4js.n4JS.N4ClassDefinition;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.util.Tuples;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Validation of rules that apply to all classifiers w/o examining members of the classifiers.<p>
 */
@SuppressWarnings("all")
public class N4JSClassifierValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  @Extension
  private IQualifiedNameProvider qualifiedNameProvider;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Constraints 38.3: Wildcards may not be used as type argument when binding a supertype’s type parameters.
   */
  @Check
  public void checkWildcardInExtendsImplements(final N4ClassifierDefinition n4ClassifierDef) {
    Iterable<ParameterizedTypeRef> _switchResult = null;
    boolean _matched = false;
    if (n4ClassifierDef instanceof N4ClassDefinition) {
      _matched=true;
      ParameterizedTypeRef _superClassRef = ((N4ClassDefinition)n4ClassifierDef).getSuperClassRef();
      EList<ParameterizedTypeRef> _implementedInterfaceRefs = ((N4ClassDefinition)n4ClassifierDef).getImplementedInterfaceRefs();
      _switchResult = Iterables.<ParameterizedTypeRef>concat(Collections.<ParameterizedTypeRef>unmodifiableList(CollectionLiterals.<ParameterizedTypeRef>newArrayList(_superClassRef)), _implementedInterfaceRefs);
    }
    if (!_matched) {
      if (n4ClassifierDef instanceof N4InterfaceDeclaration) {
        _matched=true;
        _switchResult = ((N4InterfaceDeclaration)n4ClassifierDef).getSuperInterfaceRefs();
      }
    }
    final Iterable<ParameterizedTypeRef> superTypeRefs = IterableExtensions.<ParameterizedTypeRef>filterNull(_switchResult);
    for (final ParameterizedTypeRef typeRef : superTypeRefs) {
      EList<TypeArgument> _typeArgs = typeRef.getTypeArgs();
      for (final TypeArgument typeArg : _typeArgs) {
        if ((typeArg instanceof Wildcard)) {
          this.addIssue(IssueCodes.getMessageForCLF_IMPLEMENT_EXTEND_WITH_WILDCARD(), typeArg, IssueCodes.CLF_IMPLEMENT_EXTEND_WITH_WILDCARD);
        }
      }
    }
  }
  
  /**
   * Constraints 32 (Consuming Roles)
   */
  @Check
  public void checkConsumingRoles(final N4ClassifierDefinition n4ClassifierDefinition) {
    Type _definedType = n4ClassifierDefinition.getDefinedType();
    final TClassifier tClassifier = ((TClassifier) _definedType);
    if ((tClassifier == null)) {
      return;
    }
    this.internalCheckDuplicatedConsumedInterfaces(tClassifier);
  }
  
  /**
   * Delegates further to perform check if classifier is {@link TClass} or {@link TInterface}
   * otherwise does nothing
   */
  private void internalCheckDuplicatedConsumedInterfaces(final TClassifier classifier) {
    boolean _matched = false;
    if (classifier instanceof TClass) {
      _matched=true;
      final Function1<ParameterizedTypeRef, QualifiedName> _function = (ParameterizedTypeRef it) -> {
        Type _declaredType = it.getDeclaredType();
        QualifiedName _fullyQualifiedName = null;
        if (_declaredType!=null) {
          _fullyQualifiedName=this.qualifiedNameProvider.getFullyQualifiedName(_declaredType);
        }
        return _fullyQualifiedName;
      };
      this.doInternalCheckDuplicatedConsumedRoles(IterableExtensions.<QualifiedName>toList(IterableExtensions.<QualifiedName>filterNull(ListExtensions.<ParameterizedTypeRef, QualifiedName>map(((TClass)classifier).getImplementedInterfaceRefs(), _function))), classifier, 
        N4JSPackage.Literals.N4_CLASS_DEFINITION__IMPLEMENTED_INTERFACE_REFS);
    }
    if (!_matched) {
      if (classifier instanceof TInterface) {
        _matched=true;
        final Function1<ParameterizedTypeRef, QualifiedName> _function = (ParameterizedTypeRef it) -> {
          Type _declaredType = it.getDeclaredType();
          QualifiedName _fullyQualifiedName = null;
          if (_declaredType!=null) {
            _fullyQualifiedName=this.qualifiedNameProvider.getFullyQualifiedName(_declaredType);
          }
          return _fullyQualifiedName;
        };
        this.doInternalCheckDuplicatedConsumedRoles(IterableExtensions.<QualifiedName>toList(IterableExtensions.<QualifiedName>filterNull(ListExtensions.<ParameterizedTypeRef, QualifiedName>map(((TInterface)classifier).getSuperInterfaceRefs(), _function))), classifier, 
          N4JSPackage.Literals.N4_INTERFACE_DECLARATION__SUPER_INTERFACE_REFS);
      }
    }
    if (!_matched) {
      return;
    }
  }
  
  private void doInternalCheckDuplicatedConsumedRoles(final List<QualifiedName> names, final SyntaxRelatedTElement source, final EReference eref) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(names);
    if (_isNullOrEmpty) {
      return;
    }
    final Function1<QualifiedName, String> _function = (QualifiedName it) -> {
      return this.qualifiedNameConverter.toString(it);
    };
    final Function1<Pair<String, Integer>, Boolean> _function_1 = (Pair<String, Integer> it) -> {
      Integer _value = it.getValue();
      return Boolean.valueOf(((_value).intValue() > 1));
    };
    final Iterable<Pair<String, Integer>> duplicates = IterableExtensions.<Pair<String, Integer>>filter(N4JSClassifierValidator.computeStringOccurance(ListExtensions.<QualifiedName, String>map(names, _function)), _function_1);
    for (final Pair<String, Integer> dupe : duplicates) {
      {
        final String message = IssueCodes.getMessageForCLF_MULTIPLE_ROLE_CONSUME(dupe.getKey());
        this.addIssue(message, source.getAstElement(), eref, IssueCodes.CLF_MULTIPLE_ROLE_CONSUME);
      }
    }
  }
  
  /**
   * Computes occurrence of every String in the Collection.
   * Returns Iterable&lt;Pair&lt;String, Integer>>, where {@link Pair} keys are
   * items of original collection and values are number of occurrences in collection
   */
  private static Iterable<Pair<String, Integer>> computeStringOccurance(final Collection<String> collection) {
    HashMap<String, Integer> acc = new HashMap<String, Integer>();
    for (final String entry : collection) {
      boolean _containsKey = acc.containsKey(entry);
      if (_containsKey) {
        int _intValue = acc.get(entry).intValue();
        int _plus = (_intValue + 1);
        acc.put(entry, Integer.valueOf(_plus));
      } else {
        acc.put(entry, Integer.valueOf(1));
      }
    }
    final HashMap<String, Integer> finalAcc = acc;
    final Function1<String, Pair<String, Integer>> _function = (String it) -> {
      Integer _get = finalAcc.get(it);
      return Pair.<String, Integer>of(it, _get);
    };
    return IterableExtensions.<String, Pair<String, Integer>>map(acc.keySet(), _function);
  }
  
  /**
   * @see N4JSSpec, 4.18. Members, Constraints 33 (Member Names)
   */
  @Check
  public void checkUniqueOwenedMemberNames(final N4ClassifierDefinition n4ClassifierDefinition) {
    Type _definedType = n4ClassifierDefinition.getDefinedType();
    boolean _not = (!(_definedType instanceof TClassifier));
    if (_not) {
      return;
    }
    Type _definedType_1 = n4ClassifierDefinition.getDefinedType();
    final TClassifier tClassifier = ((TClassifier) _definedType_1);
    final Function<TMember, org.eclipse.xtext.util.Pair<String, Boolean>> _function = (TMember it) -> {
      return Tuples.<String, Boolean>pair(it.getName(), Boolean.valueOf(it.isStatic()));
    };
    final ImmutableMap<org.eclipse.xtext.util.Pair<String, Boolean>, Collection<TMember>> ownedMembersByNameAndStatic = Multimaps.<org.eclipse.xtext.util.Pair<String, Boolean>, TMember>index(tClassifier.getOwnedMembers(), _function).asMap();
    final Consumer<Collection<TMember>> _function_1 = (Collection<TMember> it) -> {
      int _size = it.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.createErrorsForDuplicateOwnedMembers(it);
        TMember _head = IterableExtensions.<TMember>head(it);
        if ((_head instanceof TGetter)) {
          this.createErrorsForDuplicateOwnedMembers(Iterables.<TSetter>filter(it, TSetter.class));
        } else {
          TMember _head_1 = IterableExtensions.<TMember>head(it);
          if ((_head_1 instanceof TSetter)) {
            this.createErrorsForDuplicateOwnedMembers(Iterables.<TGetter>filter(it, TGetter.class));
          }
        }
      }
    };
    ownedMembersByNameAndStatic.values().forEach(_function_1);
  }
  
  /**
   * Assumes that all TMembers in argument 'members' have the same name and will create error messages for them,
   * except for getter/setter pairs.
   */
  private void createErrorsForDuplicateOwnedMembers(final Iterable<? extends TMember> members) {
    final TMember firstDup = IterableExtensions.head(members);
    boolean createErrorForFirst = true;
    Iterable<? extends TMember> _tail = IterableExtensions.tail(members);
    for (final TMember otherDup : _tail) {
      boolean _isFieldAccessorPair = this.isFieldAccessorPair(firstDup, otherDup);
      boolean _not = (!_isFieldAccessorPair);
      if (_not) {
        boolean _isConstructor = firstDup.isConstructor();
        if (_isConstructor) {
          if (createErrorForFirst) {
            final String message = IssueCodes.getMessageForCLF_DUP_CTOR(
              Integer.valueOf(NodeModelUtils.getNode(firstDup.getAstElement()).getStartLine()), 
              Integer.valueOf(NodeModelUtils.getNode(otherDup.getAstElement()).getStartLine()));
            this.addIssue(message, firstDup.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
              IssueCodes.CLF_DUP_CTOR);
            createErrorForFirst = false;
          }
          final String message_1 = IssueCodes.getMessageForCLF_DUP_CTOR(
            Integer.valueOf(NodeModelUtils.getNode(otherDup.getAstElement()).getStartLine()), 
            Integer.valueOf(NodeModelUtils.getNode(firstDup.getAstElement()).getStartLine()));
          this.addIssue(message_1, otherDup.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
            IssueCodes.CLF_DUP_CTOR);
        } else {
          if (createErrorForFirst) {
            final String message_2 = IssueCodes.getMessageForCLF_DUP_MEMBER(this.validatorMessageHelper.descriptionWithLine(firstDup), 
              this.validatorMessageHelper.descriptionWithLine(otherDup));
            this.addIssue(message_2, firstDup.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
              IssueCodes.CLF_DUP_MEMBER);
            createErrorForFirst = false;
          }
          final String message_3 = IssueCodes.getMessageForCLF_DUP_MEMBER(this.validatorMessageHelper.descriptionWithLine(otherDup), 
            this.validatorMessageHelper.descriptionWithLine(firstDup));
          this.addIssue(message_3, otherDup.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
            IssueCodes.CLF_DUP_MEMBER);
        }
      }
    }
  }
  
  @Check
  public void checkUseOfDefinitionSiteVariance(final TypeVariable typeVar) {
    if (((typeVar.isDeclaredCovariant() || typeVar.isDeclaredContravariant()) && 
      (!((typeVar.eContainer() instanceof N4ClassifierDeclaration) && (typeVar.eContainmentFeature() == N4JSPackage.eINSTANCE.getGenericDeclaration_TypeVars()))))) {
      final String message = IssueCodes.getMessageForCLF_DEF_SITE_VARIANCE_ONLY_IN_CLASSIFIER();
      EAttribute _xifexpression = null;
      boolean _isDeclaredCovariant = typeVar.isDeclaredCovariant();
      if (_isDeclaredCovariant) {
        _xifexpression = TypesPackage.eINSTANCE.getTypeVariable_DeclaredCovariant();
      } else {
        _xifexpression = TypesPackage.eINSTANCE.getTypeVariable_DeclaredContravariant();
      }
      final EAttribute feature = _xifexpression;
      this.addIssue(message, typeVar, feature, IssueCodes.CLF_DEF_SITE_VARIANCE_ONLY_IN_CLASSIFIER);
    }
  }
  
  /**
   * Ensures that type variables declared as covariant only appear in covariant positions and type variables declared
   * as contravariant only appear in contravariant positions. Type variables declared to be invariant do not require
   * such a check.
   */
  @Check
  public void checkTypeVariableVariance(final ParameterizedTypeRef typeRefInAST) {
    final Type tv = typeRefInAST.getDeclaredType();
    if ((tv instanceof TypeVariable)) {
      final Variance variance = ((TypeVariable)tv).getVariance();
      if ((variance != Variance.INV)) {
        final Variance varianceOfPos = N4JSLanguageUtils.getVarianceOfPosition(typeRefInAST);
        if (((varianceOfPos != null) && (variance != varianceOfPos))) {
          final String msg = IssueCodes.getMessageForCLF_TYPE_VARIABLE_AT_INVALID_POSITION(variance.getDescriptiveString(true), 
            varianceOfPos.getDescriptiveString(false));
          this.addIssue(msg, typeRefInAST, IssueCodes.CLF_TYPE_VARIABLE_AT_INVALID_POSITION);
        }
      }
    }
  }
}
