/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.CatchVariable;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.LocalArgumentsVariable;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassExpression;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDefinition;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.SetterDeclaration;
import org.eclipse.n4js.n4JS.Variable;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableEnvironmentElement;
import org.eclipse.n4js.n4JS.VersionedElement;
import org.eclipse.n4js.n4JS.extensions.SourceElementExtensions;
import org.eclipse.n4js.n4idl.versioning.VersionUtils;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.scoping.builtin.GlobalObjectScope;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.ValidatorMessageHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class N4JSDeclaredNameValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private IResourceScopeCache cache;
  
  @Inject
  private ValidatorMessageHelper messageHelper;
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private SourceElementExtensions sourceElementExtensions;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  public static final HashSet<String> BASE_JS_TYPES = Sets.<String>newHashSet(
    new String[] { "Object", "Function", "Array", "String", "Boolean", "Number", "Math", "Date", "RegExp", "Error", "JSON" });
  
  public static final HashSet<String> BASE_GLOBAL_NAMES = Sets.<String>newHashSet(
    new String[] { "number", "string", "boolean", "any", "pathSelector", "i18nKey", "typeName", "N4Object", "N4Class", "N4Enum" });
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Check duplicated names of declared properties in the {@link ObjectLiteral}s.
   * Similar to check of duplicates in scope.
   */
  @Check
  public void checkNameConflicts(final ObjectLiteral objectLiteral) {
    final Function1<PropertyAssignment, Boolean> _function = (PropertyAssignment it) -> {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(this.findName(it));
      return Boolean.valueOf((!_isNullOrEmpty));
    };
    final Function<EObject, String> _function_1 = (EObject it) -> {
      return this.findName(it);
    };
    final Function2<String, List<EObject>, Boolean> _function_2 = (String name, List<EObject> lstEO) -> {
      int _size = lstEO.size();
      return Boolean.valueOf((_size > 1));
    };
    final BiConsumer<String, List<EObject>> _function_3 = (String name, List<EObject> lstEO) -> {
      final Comparator<EObject> _function_4 = (EObject e1, EObject e2) -> {
        return Integer.valueOf(NodeModelUtils.getNode(e1).getOffset()).compareTo(
          Integer.valueOf(
            NodeModelUtils.getNode(e2).getOffset()));
      };
      lstEO.sort(_function_4);
      final ListIterator<EObject> iter = lstEO.listIterator();
      final EObject baseEO = iter.next();
      final Consumer<EObject> _function_5 = (EObject dupeEO) -> {
        if ((((baseEO instanceof PropertyGetterDeclaration) && (dupeEO instanceof PropertySetterDeclaration)) || ((baseEO instanceof PropertySetterDeclaration) && (dupeEO instanceof PropertyGetterDeclaration)))) {
          return;
        }
        this.addIssue(
          StringExtensions.toFirstUpper(
            IssueCodes.getMessageForAST_NAME_DUPLICATE_ERR(this.messageHelper.description(dupeEO, name), 
              this.messageHelper.descriptionWithLine(baseEO, name))), dupeEO, this.findNameEAttribute(dupeEO), 
          IssueCodes.AST_NAME_DUPLICATE_ERR);
      };
      iter.forEachRemaining(_function_5);
    };
    MapExtensions.<String, List<EObject>>filter(IterableExtensions.<PropertyAssignment>toList(IterableExtensions.<PropertyAssignment>filter(objectLiteral.getPropertyAssignments(), _function)).stream().collect(
      Collectors.<EObject, String>groupingBy(_function_1)), _function_2).forEach(_function_3);
  }
  
  /**
   * Constraints 126 (Global Definitions), except polyfills
   */
  @Check
  public void checkExportableNameConflictsWithBuiltIn(final ExportableElement exportableElement) {
    if ((exportableElement instanceof AnnotableElement)) {
      boolean _hasAnnotation = AnnotationDefinition.GLOBAL.hasAnnotation(((AnnotableElement)exportableElement));
      if (_hasAnnotation) {
        boolean _isPolyfill = N4JSLanguageUtils.isPolyfill(((AnnotableElement)exportableElement));
        if (_isPolyfill) {
          return;
        }
        final String name = this.getDeclaredName(exportableElement);
        if ((name != null)) {
          boolean _contains = N4JSDeclaredNameValidator.BASE_JS_TYPES.contains(name);
          if (_contains) {
            final IN4JSProject project = this.n4jsCore.findProject(exportableElement.eResource().getURI()).get();
            if (((project == null) || (project.getProjectType() != ProjectType.RUNTIME_ENVIRONMENT))) {
              this.addIssue(IssueCodes.getMessageForAST_GLOBAL_JS_NAME_CONFLICT(name), exportableElement, 
                IssueCodes.AST_GLOBAL_JS_NAME_CONFLICT);
            }
          } else {
            boolean _contains_1 = N4JSDeclaredNameValidator.BASE_GLOBAL_NAMES.contains(name);
            if (_contains_1) {
              this.addIssue(IssueCodes.getMessageForAST_GLOBAL_NAME_CONFLICT(name), exportableElement, 
                IssueCodes.AST_GLOBAL_NAME_CONFLICT);
            }
          }
        }
      }
    }
  }
  
  /**
   * Check name conflicts in script.
   * For every scope ({@link VariableEnvironmentElement} instance) will
   *  - check for name conflict with members of global object
   *  - check for name conflict with implicit function arguments
   *  - check for name conflicts with outer (containing) scope
   */
  @Check
  public void checkNameConflicts(final Script script) {
    this.checkNameConflicts(script, Collections.<String>emptySet());
  }
  
  /**
   * Delegates proper checks for a given scope.
   * After checks are done for current scope, calls itself recursively on children.
   * Recursive call is passing set of used names in current scope, allowing child scope to perform proper comparison checks.
   * Only names are passed, actual {@link EObject}s are resolved only if name clash is detected.
   */
  private void checkNameConflicts(final VariableEnvironmentElement scope, final Set<String> outerNames) {
    final List<String> localNames = IterableExtensions.<String>toList(this.getDeclaredNames(scope));
    this.checkGlobalNamesConflict(scope, IterableExtensions.<String>toList(this.getDeclaredNamesForGlobalScopeComparison(scope)));
    final Set<String> localNamesNoDuplicates = new HashSet<String>(localNames);
    this.checkLocalScopeNamesConflict(scope, localNames, localNamesNoDuplicates);
    final Set<String> allNamesNoDuplicates = new HashSet<String>(outerNames);
    allNamesNoDuplicates.addAll(localNamesNoDuplicates);
    this.checkOuterScopesNamesConflict(scope, localNames, localNamesNoDuplicates, allNamesNoDuplicates, outerNames);
    final Procedure1<VariableEnvironmentElement> _function = (VariableEnvironmentElement it) -> {
      this.checkNameConflicts(it, allNamesNoDuplicates);
    };
    IteratorExtensions.<VariableEnvironmentElement>forEach(this.getNestedScopes(scope), _function);
  }
  
  /**
   * checks if provided localNames of given scope are not conflicting with names in global object
   */
  private void checkGlobalNamesConflict(final VariableEnvironmentElement scope, final List<String> localNames) {
    final List<String> globalNames = this.findGlobalNames(scope.eResource());
    final Set<String> globalNamesConflicts = new HashSet<String>(globalNames);
    boolean _retainAll = globalNamesConflicts.retainAll(localNames);
    if (_retainAll) {
      final Function1<EObject, Boolean> _function = (EObject it) -> {
        return Boolean.valueOf(globalNamesConflicts.contains(this.getDeclaredNameForGlobalScopeComparision(it)));
      };
      final Consumer<EObject> _function_1 = (EObject it) -> {
        final EObject innerScopeObject = it;
        final String name = this.getDeclaredNameForGlobalScopeComparision(innerScopeObject);
        boolean _notEquals = (!Objects.equal(name, "eval"));
        if (_notEquals) {
          final Function1<TMember, Boolean> _function_2 = (TMember m) -> {
            return Boolean.valueOf(name.equals(m.getName()));
          };
          final TMember globalObjectMemeber = IterableExtensions.<TMember>findFirst(this.findGlobalMembers(scope.eResource()), _function_2);
          this.addIssue(
            StringExtensions.toFirstUpper(
              IssueCodes.getMessageForAST_GLOBAL_NAME_SHADOW_ERR(
                this.messageHelper.description(innerScopeObject, name), 
                this.messageHelper.description(globalObjectMemeber, name))), innerScopeObject, 
            this.findNameEAttribute(innerScopeObject), IssueCodes.AST_GLOBAL_NAME_SHADOW_ERR);
        }
      };
      IterableExtensions.<EObject>filter(this.getNameDeclarations(scope), _function).forEach(_function_1);
    }
  }
  
  /**
   * check (pre-computed) localNames of the given scope against each other. When conflict is found EObjects
   * that create conflict are analyzed and if appropriate error marker is issued.
   */
  private void checkLocalScopeNamesConflict(final VariableEnvironmentElement scope, final List<String> localNames, final Set<String> localNamesNoDuplicates) {
    if ((((scope instanceof Block) && (scope.eContainer() instanceof FunctionOrFieldAccessor)) && (((FunctionOrFieldAccessor) scope.eContainer()).getBody() == scope))) {
      EObject _eContainer = scope.eContainer();
      this.checkLocalScopeNamesConflict_letConstSpecialCase(((FunctionOrFieldAccessor) _eContainer));
    }
    int _size = localNamesNoDuplicates.size();
    int _size_1 = localNames.size();
    boolean _lessThan = (_size < _size_1);
    if (_lessThan) {
      final ArrayList<String> temp = new ArrayList<String>(localNames);
      final Consumer<String> _function = (String it) -> {
        temp.remove(it);
      };
      localNamesNoDuplicates.forEach(_function);
      final Set<String> localNamesDuplicatesLocally = new HashSet<String>(temp);
      for (final String n : localNamesDuplicatesLocally) {
        final Function<EObject, String> _function_1 = (EObject it) -> {
          return this.getDeclaredName(it);
        };
        final Function2<String, List<EObject>, Boolean> _function_2 = (String name, List<EObject> lstEO) -> {
          int _size_2 = lstEO.size();
          return Boolean.valueOf((_size_2 > 1));
        };
        final BiConsumer<String, List<EObject>> _function_3 = (String name, List<EObject> lstEO) -> {
          final Comparator<EObject> _function_4 = (EObject e1, EObject e2) -> {
            final ICompositeNode n1 = NodeModelUtils.getNode(e1);
            final ICompositeNode n2 = NodeModelUtils.getNode(e2);
            int _xifexpression = (int) 0;
            if (((n1 == null) || (n2 == null))) {
              _xifexpression = 0;
            } else {
              int _offset = n1.getOffset();
              int _offset_1 = n2.getOffset();
              _xifexpression = (_offset - _offset_1);
            }
            return _xifexpression;
          };
          lstEO.sort(_function_4);
          final ListIterator<EObject> iter = lstEO.listIterator();
          final EObject baseEO = iter.next();
          final Consumer<EObject> _function_5 = (EObject dupeEO) -> {
            if (((baseEO instanceof N4ClassExpression) || (dupeEO instanceof N4ClassExpression))) {
              return;
            }
            if (((baseEO instanceof LocalArgumentsVariable) || (dupeEO instanceof LocalArgumentsVariable))) {
              return;
            }
            boolean _equals = baseEO.equals(scope);
            if (_equals) {
              if ((dupeEO instanceof FormalParameter)) {
                this.addIssue(
                  StringExtensions.toFirstUpper(
                    IssueCodes.getMessageForAST_NAME_SHADOW_ERR(
                      this.messageHelper.description(dupeEO, name), 
                      this.messageHelper.description(baseEO, name))), dupeEO, 
                  this.findNameEAttribute(dupeEO), IssueCodes.AST_NAME_SHADOW_ERR);
              } else {
                this.addIssue(
                  StringExtensions.toFirstUpper(
                    IssueCodes.getMessageForAST_NAME_SHADOW_ERR(
                      this.messageHelper.description(dupeEO, name), 
                      this.messageHelper.descriptionWithLine(baseEO, name))), dupeEO, 
                  this.findNameEAttribute(dupeEO), IssueCodes.AST_NAME_SHADOW_ERR);
              }
              return;
            }
            if ((dupeEO instanceof FormalParameter)) {
              this.addIssue(
                StringExtensions.toFirstUpper(
                  IssueCodes.getMessageForAST_NAME_DUPLICATE_ERR(
                    this.messageHelper.description(dupeEO, name), 
                    this.messageHelper.description(baseEO, name))), dupeEO, 
                this.findNameEAttribute(dupeEO), IssueCodes.AST_NAME_DUPLICATE_ERR);
            } else {
              if ((((dupeEO instanceof NamedImportSpecifier) && 
                (baseEO instanceof NamedImportSpecifier)) || ((dupeEO instanceof NamespaceImportSpecifier) && 
                (baseEO instanceof NamespaceImportSpecifier)))) {
                return;
              } else {
                boolean _not = (!(((dupeEO instanceof N4ClassDeclaration) && (baseEO instanceof ImportSpecifier)) && 
                  N4JSLanguageUtils.isPolyfill(((N4ClassDeclaration) dupeEO))));
                if (_not) {
                  this.addIssue(
                    StringExtensions.toFirstUpper(
                      IssueCodes.getMessageForAST_NAME_DUPLICATE_ERR(
                        this.messageHelper.description(dupeEO, name), 
                        this.messageHelper.descriptionWithLine(baseEO, name))), dupeEO, 
                    this.findNameEAttribute(dupeEO), IssueCodes.AST_NAME_DUPLICATE_ERR);
                }
              }
            }
          };
          iter.forEachRemaining(_function_5);
        };
        MapExtensions.<String, List<EObject>>filter(IterableExtensions.<EObject>toList(this.getNameDeclarations(scope, n)).stream().collect(Collectors.<EObject, String>groupingBy(_function_1)), _function_2).forEach(_function_3);
      }
    }
  }
  
  /**
   * This method handles a special case related to let/const:
   * <pre>
   * function foo1(param) {
   *     let param; // <-- ES6 engines will throw error at runtime!
   * }
   * function foo2(param) {
   *     {
   *         let param; // <-- valid ES6!
   *     }
   * }
   * </pre>
   * This method takes care of producing a validation error in the first case above.
   */
  private void checkLocalScopeNamesConflict_letConstSpecialCase(final FunctionOrFieldAccessor fun) {
    final Block block = fun.getBody();
    List<FormalParameter> _switchResult = null;
    boolean _matched = false;
    if (fun instanceof FunctionDefinition) {
      _matched=true;
      _switchResult = ((FunctionDefinition)fun).getFpars();
    }
    if (!_matched) {
      if (fun instanceof SetterDeclaration) {
        _matched=true;
        FormalParameter _fpar = ((SetterDeclaration)fun).getFpar();
        _switchResult = Collections.<FormalParameter>unmodifiableList(CollectionLiterals.<FormalParameter>newArrayList(_fpar));
      }
    }
    if (!_matched) {
      _switchResult = Collections.<FormalParameter>unmodifiableList(CollectionLiterals.<FormalParameter>newArrayList());
    }
    final List<FormalParameter> fpars = _switchResult;
    final Function1<FormalParameter, String> _function = (FormalParameter it) -> {
      return it.getName();
    };
    final Set<String> fparNames = IterableExtensions.<String>toSet(ListExtensions.<FormalParameter, String>map(fpars, _function));
    final Function1<VariableDeclaration, Boolean> _function_1 = (VariableDeclaration it) -> {
      return Boolean.valueOf(N4JSASTUtils.isBlockScoped(it));
    };
    final Iterable<VariableDeclaration> declaredLetConst = IterableExtensions.<VariableDeclaration>filter(Iterables.<VariableDeclaration>filter(this.getNameDeclarations(block), VariableDeclaration.class), _function_1);
    final Function1<VariableDeclaration, Boolean> _function_2 = (VariableDeclaration it) -> {
      return Boolean.valueOf(fparNames.contains(this.getDeclaredName(it)));
    };
    final Consumer<VariableDeclaration> _function_3 = (VariableDeclaration dupeEO) -> {
      final String name = this.getDeclaredName(dupeEO);
      final Function1<FormalParameter, Boolean> _function_4 = (FormalParameter it) -> {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, name));
      };
      final FormalParameter baseEO = IterableExtensions.<FormalParameter>head(IterableExtensions.<FormalParameter>filter(fpars, _function_4));
      this.addIssue(
        StringExtensions.toFirstUpper(
          IssueCodes.getMessageForAST_NAME_DUPLICATE_ERR(
            this.messageHelper.description(dupeEO, name), 
            this.messageHelper.description(baseEO, name))), dupeEO, 
        this.findNameEAttribute(dupeEO), IssueCodes.AST_NAME_SHADOW_ERR);
    };
    IterableExtensions.<VariableDeclaration>filter(declaredLetConst, _function_2).forEach(_function_3);
  }
  
  /**
   * check (pre-computed) localNames of the given scope with (pre-computed) all names used in parent scopes.
   * If conflict is detected traverse containers of the scope, until {@link EObject} creating using the name is found.
   * If it meets error conditions we issue proper error, otherwise ast is traversed further up.
   * If no {@link EObject} that meets error conditions is found, no error is issued.
   * AST traversing stops at first {@link EObject} that meets error conditions, so no multiple errors should be issued on a given declaration.
   */
  private void checkOuterScopesNamesConflict(final VariableEnvironmentElement scope, final List<String> localNames, final Set<String> localNamesNoDuplicates, final Set<String> allNamesNoDuplicates, final Set<String> outerNames) {
    int _size = allNamesNoDuplicates.size();
    int _size_1 = outerNames.size();
    int _size_2 = localNamesNoDuplicates.size();
    int _plus = (_size_1 + _size_2);
    boolean _lessThan = (_size < _plus);
    if (_lessThan) {
      final ArrayList<String> temp = new ArrayList<String>(localNamesNoDuplicates);
      temp.addAll(outerNames);
      final Consumer<String> _function = (String it) -> {
        temp.remove(it);
      };
      allNamesNoDuplicates.forEach(_function);
      final Set<String> localNamesDuplicatesGlobally = new HashSet<String>(temp);
      for (final String n : localNamesDuplicatesGlobally) {
        {
          boolean _equals = n.equals("arguments");
          if (_equals) {
            return;
          }
          final Consumer<EObject> _function_1 = (EObject it) -> {
            EObject conflict = null;
            EObject conflictContainer = scope.eContainer();
            while ((conflict == null)) {
              {
                if ((conflictContainer == null)) {
                  return;
                }
                final Iterable<EObject> z = this.findOuterDeclaration(conflictContainer, n);
                boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(z);
                if (_isNullOrEmpty) {
                  conflictContainer = conflictContainer.eContainer();
                } else {
                  conflict = IterableExtensions.<EObject>head(z);
                  EObject outerScopeObject = conflict;
                  EObject innerScopeObject = it;
                  String name = n;
                  if (((outerScopeObject instanceof N4ClassExpression) || 
                    (innerScopeObject instanceof N4ClassExpression))) {
                    return;
                  }
                  if (((outerScopeObject instanceof FunctionDeclaration) || 
                    (outerScopeObject instanceof FunctionExpression))) {
                    if (((innerScopeObject instanceof FunctionExpression) || 
                      (innerScopeObject instanceof N4ClassExpression))) {
                      return;
                    }
                    if ((((innerScopeObject instanceof FormalParameter) && (outerScopeObject instanceof FunctionDefinition)) && (innerScopeObject.eContainer() == outerScopeObject))) {
                      this.addIssue(
                        StringExtensions.toFirstUpper(
                          IssueCodes.getMessageForAST_NAME_SHADOW_ERR(
                            this.messageHelper.description(innerScopeObject, name), 
                            this.messageHelper.description(outerScopeObject, name))), innerScopeObject, 
                        this.findNameEAttribute(innerScopeObject), IssueCodes.AST_NAME_SHADOW_ERR);
                      return;
                    }
                    boolean _equals_1 = outerScopeObject.equals(scope);
                    boolean _equals_2 = (_equals_1 == false);
                    if (_equals_2) {
                      return;
                    }
                    this.addIssue(
                      StringExtensions.toFirstUpper(
                        IssueCodes.getMessageForAST_NAME_SHADOW_ERR(
                          this.messageHelper.description(innerScopeObject, name), 
                          this.messageHelper.descriptionWithLine(outerScopeObject, name))), innerScopeObject, this.findNameEAttribute(innerScopeObject), IssueCodes.AST_NAME_SHADOW_ERR);
                    return;
                  }
                  if ((innerScopeObject instanceof CatchVariable)) {
                    if ((outerScopeObject instanceof CatchVariable)) {
                      boolean _isN4JSMode = this.jsVariantHelper.isN4JSMode(innerScopeObject);
                      if (_isN4JSMode) {
                        this.addIssue(
                          StringExtensions.toFirstUpper(
                            IssueCodes.getMessageForAST_NAME_SHADOW_WARN(
                              this.messageHelper.description(innerScopeObject, name), 
                              this.messageHelper.descriptionWithLine(outerScopeObject, name))), innerScopeObject, this.findNameEAttribute(innerScopeObject), IssueCodes.AST_NAME_SHADOW_WARN);
                        return;
                      }
                      return;
                    }
                    this.addIssue(
                      StringExtensions.toFirstUpper(
                        IssueCodes.getMessageForAST_NAME_SHADOW_ERR(
                          this.messageHelper.description(innerScopeObject, name), 
                          this.messageHelper.descriptionWithLine(outerScopeObject, name))), innerScopeObject, this.findNameEAttribute(innerScopeObject), IssueCodes.AST_NAME_SHADOW_ERR);
                    return;
                  }
                  return;
                }
              }
            }
          };
          IterableExtensions.<EObject>toList(this.getNameDeclarations(scope, n)).forEach(_function_1);
        }
      }
    }
  }
  
  private Iterable<EObject> findOuterDeclaration(final EObject scope, final String name) {
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      return Boolean.valueOf(name.equals(this.getDeclaredName(it)));
    };
    final Iterable<EObject> decl = IterableExtensions.<EObject>filter(scope.eContents(), _function);
    return decl;
  }
  
  /**
   * returns {@link List} of names declared in global scope.
   * Returned data is cached, resolved only if new (or changed) {@link Resource}
   */
  public List<String> findGlobalNames(final Resource resource) {
    final Provider<List<String>> _function = () -> {
      final Function1<TMember, Boolean> _function_1 = (TMember it) -> {
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(it.getName());
        return Boolean.valueOf((!_isNullOrEmpty));
      };
      final Function1<TMember, String> _function_2 = (TMember it) -> {
        return it.getName();
      };
      return IterableExtensions.<String>toList(IterableExtensions.<TMember, String>map(IterableExtensions.<TMember>filter(EcoreUtil2.<TMember>getAllContentsOfType(this.getGlobalObject(resource), TMember.class), _function_1), _function_2));
    };
    return this.cache.<List<String>>get("globalObjectNames", resource, _function);
  }
  
  /**
   * returns {@link List} of members of the global object
   * Returned data is cached, resolved only if new (or changed) {@link Resource}
   */
  public List<TMember> findGlobalMembers(final Resource resource) {
    final Provider<List<TMember>> _function = () -> {
      return EcoreUtil2.<TMember>getAllContentsOfType(this.getGlobalObject(resource), TMember.class);
    };
    return this.cache.<List<TMember>>get("globalObjectMembers", resource, _function);
  }
  
  /**
   * returns {@link EObject} instance of the global object
   * Returned data is cached, resolved only if new (or changed) {@link Resource}
   */
  public TClass getGlobalObject(final Resource resource) {
    return GlobalObjectScope.get(resource.getResourceSet()).getGlobalObject();
  }
  
  /**
   * Returns all elements that declare a name in scope 'scope' without(!) considering nested scopes.
   */
  private Iterable<EObject> getNameDeclarations(final VariableEnvironmentElement scope) {
    final Resource resource = scope.eResource();
    final ArrayList<EObject> namedEOs = CollectionLiterals.<EObject>newArrayList();
    boolean _matched = false;
    if (scope instanceof Script) {
      _matched=true;
      List<EObject> _list = IteratorExtensions.<EObject>toList(Iterators.<ImportSpecifier>filter(((Script)scope).eAllContents(), ImportSpecifier.class));
      Iterables.<EObject>addAll(namedEOs, _list);
    }
    if (!_matched) {
      if (scope instanceof FunctionOrFieldAccessor) {
        _matched=true;
        LocalArgumentsVariable _localArgumentsVariable = ((FunctionOrFieldAccessor)scope).getLocalArgumentsVariable();
        namedEOs.add(_localArgumentsVariable);
      }
    }
    final Function1<IdentifiableElement, EObject> _function = (IdentifiableElement it) -> {
      return this.backToAST(it, resource);
    };
    List<EObject> _map = ListExtensions.<IdentifiableElement, EObject>map(this.sourceElementExtensions.collectVisibleIdentifiableElements(scope), _function);
    Iterables.<EObject>addAll(namedEOs, _map);
    final Function1<EObject, Boolean> _function_1 = (EObject it) -> {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(this.getDeclaredName(it));
      return Boolean.valueOf((!_isNullOrEmpty));
    };
    return IterableExtensions.<EObject>filter(namedEOs, _function_1);
  }
  
  /**
   * Returns all elements that declare the given name in scope 'scope' without(!) considering nested scopes.
   */
  private Iterable<EObject> getNameDeclarations(final VariableEnvironmentElement scope, final String name) {
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      String _declaredName = this.getDeclaredName(it);
      return Boolean.valueOf(Objects.equal(_declaredName, name));
    };
    return IterableExtensions.<EObject>filter(this.getNameDeclarations(scope), _function);
  }
  
  /**
   * Returns list of names declared in scope 'scope' without(!) considering nested scopes.
   */
  private Iterable<String> getDeclaredNames(final VariableEnvironmentElement scope) {
    final Function1<EObject, String> _function = (EObject it) -> {
      return this.getDeclaredName(it);
    };
    return IterableExtensions.<EObject, String>map(this.getNameDeclarations(scope), _function);
  }
  
  /**
   * Returns list of names declared in scope 'scope' without(!) considering nested scopes for comparison in global scope.
   */
  private Iterable<String> getDeclaredNamesForGlobalScopeComparison(final VariableEnvironmentElement scope) {
    final Function1<EObject, String> _function = (EObject it) -> {
      return this.getDeclaredNameForGlobalScopeComparision(it);
    };
    return IterableExtensions.<EObject, String>map(this.getNameDeclarations(scope), _function);
  }
  
  /**
   * Returns nested scopes of 'scope'. Only direct sub-scopes of 'scope' are returned, no sub-sub-scopes,
   * i.e. sub-scopes of sub-scopes.
   */
  private Iterator<VariableEnvironmentElement> getNestedScopes(final VariableEnvironmentElement scope) {
    final Predicate<EObject> _function = (EObject it) -> {
      return (!(this.createsScope(it.eContainer()) && (it.eContainer() != scope)));
    };
    final Function1<EObject, Boolean> _function_1 = (EObject it) -> {
      return Boolean.valueOf(this.createsScope(it));
    };
    return Iterators.<VariableEnvironmentElement>filter(IteratorExtensions.<EObject>filter(EcoreUtilN4.getAllContentsFiltered(scope, _function), _function_1), VariableEnvironmentElement.class);
  }
  
  private boolean createsScope(final EObject eo) {
    return (eo instanceof VariableEnvironmentElement);
  }
  
  /**
   * Resolves name of given {@link EObject} if possible, null otherwise.
   * Will not return actual name for all objects, only the ones interesting from duplicates/shadowing point of view.
   * (so variable declarations, function definitions, type definitions...). In case of named import specifiers returns alias
   * if possible, otherwise name of imported element.
   * returns null in other cases.
   * Does not check value of the returned name, so it can be null or empty string.
   */
  private String getDeclaredName(final EObject eo) {
    if ((VersionUtils.isVersioned(eo) && (eo instanceof NamedElement))) {
      String _name = ((NamedElement) eo).getName();
      String _plus = (_name + "#");
      BigDecimal _declaredVersion = ((VersionedElement) eo).getDeclaredVersion();
      return (_plus + _declaredVersion);
    }
    if (((((eo instanceof FunctionDeclaration) || (eo instanceof FunctionExpression)) || (eo instanceof N4TypeDefinition)) || 
      (eo instanceof Variable))) {
      return this.findName(eo);
    }
    if ((eo instanceof NamedImportSpecifier)) {
      final NamedImportSpecifier namedIS = ((NamedImportSpecifier)eo);
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(namedIS.getAlias());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        return namedIS.getAlias();
      } else {
        final TExportableElement importedElem = namedIS.getImportedElement();
        if ((importedElem == null)) {
          return null;
        }
        final String n = this.findName(importedElem);
        return n;
      }
    }
    if ((eo instanceof NamespaceImportSpecifier)) {
      return ((NamespaceImportSpecifier)eo).getAlias();
    }
    return null;
  }
  
  private String getDeclaredNameForGlobalScopeComparision(final EObject eo) {
    String _switchResult = null;
    boolean _matched = false;
    if (eo instanceof N4ClassDeclaration) {
      _matched=true;
      _switchResult = ((N4ClassDeclaration)eo).getName();
    }
    if (!_matched) {
      if (eo instanceof N4InterfaceDeclaration) {
        _matched=true;
        _switchResult = ((N4InterfaceDeclaration)eo).getName();
      }
    }
    if (!_matched) {
      if (eo instanceof N4EnumDeclaration) {
        _matched=true;
        _switchResult = ((N4EnumDeclaration)eo).getName();
      }
    }
    if (!_matched) {
      _switchResult = this.getDeclaredName(eo);
    }
    return _switchResult;
  }
  
  /**
   * helper dispatch because we lack one uniform interface for getName
   */
  private String _findName(final NamedElement it) {
    return it.getName();
  }
  
  /**
   * helper dispatch because we lack one uniform interface for getName
   */
  private String _findName(final IdentifiableElement it) {
    return it.getName();
  }
  
  /**
   * returns attribute holding name for given EObject. Throws error if provided EObject does not define name
   * attribute.
   */
  private EStructuralFeature findNameEAttribute(final EObject eo) {
    if ((eo instanceof N4TypeDeclaration)) {
      return N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME;
    }
    if ((eo instanceof N4ClassExpression)) {
      return N4JSPackage.Literals.N4_CLASS_EXPRESSION__NAME;
    }
    if ((eo instanceof FunctionExpression)) {
      return N4JSPackage.Literals.FUNCTION_EXPRESSION__NAME;
    }
    if ((eo instanceof FunctionDeclaration)) {
      return N4JSPackage.Literals.FUNCTION_DECLARATION__NAME;
    }
    if ((eo instanceof NamedImportSpecifier)) {
      String _alias = ((NamedImportSpecifier)eo).getAlias();
      boolean _tripleNotEquals = (_alias != null);
      if (_tripleNotEquals) {
        return N4JSPackage.Literals.NAMED_IMPORT_SPECIFIER__ALIAS;
      }
      String _name = ((NamedImportSpecifier)eo).getImportedElement().getName();
      boolean _tripleNotEquals_1 = (_name != null);
      if (_tripleNotEquals_1) {
        return TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME;
      }
    }
    if ((eo instanceof NamespaceImportSpecifier)) {
      String _alias_1 = ((NamespaceImportSpecifier)eo).getAlias();
      boolean _tripleNotEquals_2 = (_alias_1 != null);
      if (_tripleNotEquals_2) {
        return N4JSPackage.Literals.NAMESPACE_IMPORT_SPECIFIER__ALIAS;
      }
    }
    if ((eo instanceof IdentifiableElement)) {
      return TypesPackage.Literals.IDENTIFIABLE_ELEMENT__NAME;
    }
    if ((eo instanceof PropertyAssignment)) {
      return N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME;
    }
    throw new RuntimeException(("cannot obtain name attribute for " + eo));
  }
  
  /**
   * If <code>potentialTModuleElement</code> is a TModule element that has a related AST node within
   * <code>resource</code>, then return this AST node, otherwise return <code>potentialTModuleElement</code> itself.
   */
  private EObject backToAST(final EObject potentialTModuleElement, final Resource resource) {
    if ((potentialTModuleElement instanceof SyntaxRelatedTElement)) {
      Resource _eResource = ((SyntaxRelatedTElement)potentialTModuleElement).eResource();
      boolean _tripleEquals = (_eResource == resource);
      if (_tripleEquals) {
        return ((SyntaxRelatedTElement)potentialTModuleElement).getAstElement();
      }
    }
    return potentialTModuleElement;
  }
  
  private String findName(final EObject it) {
    if (it instanceof IdentifiableElement) {
      return _findName((IdentifiableElement)it);
    } else if (it instanceof NamedElement) {
      return _findName((NamedElement)it);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it).toString());
    }
  }
}
