/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.compileTime.CompileTimeEvaluationError;
import org.eclipse.n4js.compileTime.CompileTimeEvaluator;
import org.eclipse.n4js.compileTime.CompileTimeValue;
import org.eclipse.n4js.n4JS.AdditiveExpression;
import org.eclipse.n4js.n4JS.AdditiveOperator;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.AwaitExpression;
import org.eclipse.n4js.n4JS.BinaryLogicalExpression;
import org.eclipse.n4js.n4JS.BooleanLiteral;
import org.eclipse.n4js.n4JS.CastExpression;
import org.eclipse.n4js.n4JS.ConditionalExpression;
import org.eclipse.n4js.n4JS.EqualityExpression;
import org.eclipse.n4js.n4JS.EqualityOperator;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.MultiplicativeExpression;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.NumericLiteral;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.PostfixExpression;
import org.eclipse.n4js.n4JS.PromisifyExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.RelationalExpression;
import org.eclipse.n4js.n4JS.RelationalOperator;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ShiftExpression;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.n4JS.ThisArgProvider;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.n4JS.UnaryExpression;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.extensions.ExpressionExtensions;
import org.eclipse.n4js.postprocessing.ASTMetaInfoUtils;
import org.eclipse.n4js.scoping.members.MemberScopingHelper;
import org.eclipse.n4js.ts.conversions.ComputedPropertyNameValueConverter;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ExistentialTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.BuiltInType;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.NullType;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructuralType;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeDefs;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.UndefinedType;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.PromisifyHelper;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.n4js.validation.ValidatorMessageHelper;
import org.eclipse.n4js.validation.validators.ConstBoolean;
import org.eclipse.n4js.xtext.scoping.IEObjectDescriptionWithError;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure4;

@SuppressWarnings("all")
public class N4JSExpressionValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  protected N4JSTypeSystem ts;
  
  @Inject
  protected TypeSystemHelper tsh;
  
  @Inject
  @Extension
  private N4JSElementKeywordProvider _n4JSElementKeywordProvider;
  
  @Inject
  @Extension
  private ValidatorMessageHelper _validatorMessageHelper;
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private MemberScopingHelper memberScopingHelper;
  
  @Inject
  private PromisifyHelper promisifyHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkAwaitExpression(final AwaitExpression awaitExpression) {
    final FunctionDefinition containerFunDef = EcoreUtil2.<FunctionDefinition>getContainerOfType(awaitExpression, FunctionDefinition.class);
    if (((containerFunDef == null) || (Boolean.valueOf(containerFunDef.isAsync()) == Boolean.valueOf(false)))) {
      final String message = IssueCodes.getMessageForEXP_MISPLACED_AWAIT_EXPRESSION("await", "async");
      this.addIssue(message, awaitExpression, IssueCodes.EXP_MISPLACED_AWAIT_EXPRESSION);
    }
    Expression _expression = awaitExpression.getExpression();
    boolean _tripleEquals = (_expression == null);
    if (_tripleEquals) {
      return;
    }
    this.internalCheckAwaitingAPromise(awaitExpression);
  }
  
  private void internalCheckAwaitingAPromise(final AwaitExpression awaitExpression) {
    final Expression subExpr = awaitExpression.getExpression();
    if ((subExpr == null)) {
      return;
    }
    final TypeRef typeRef = this.ts.tau(subExpr, awaitExpression);
    if ((typeRef == null)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(awaitExpression);
    final BuiltInTypeScope scope = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope;
    final TypeRefsFactory einst = TypeRefsFactory.eINSTANCE;
    final UnionTypeExpression union1 = einst.createUnionTypeExpression();
    union1.getTypeRefs().add(scope.getVoidTypeRef());
    union1.getTypeRefs().add(scope.getAnyTypeRef());
    final UnionTypeExpression union2 = einst.createUnionTypeExpression();
    union2.getTypeRefs().add(scope.getVoidTypeRef());
    union2.getTypeRefs().add(scope.getAnyTypeRef());
    final ParameterizedTypeRef promUni = TypeUtils.createPromiseTypeRef(scope, union1, union2);
    final boolean stUnions = this.ts.subtypeSucceeded(G, typeRef, promUni);
    Type _declaredType = typeRef.getDeclaredType();
    UndefinedType _undefinedType = RuleEnvironmentExtensions.undefinedType(G);
    final boolean isUndef = Objects.equal(_declaredType, _undefinedType);
    Type _declaredType_1 = typeRef.getDeclaredType();
    NullType _nullType = RuleEnvironmentExtensions.nullType(G);
    final boolean isNull = Objects.equal(_declaredType_1, _nullType);
    if ((!stUnions)) {
      final String message = IssueCodes.getMessageForEXP_AWAIT_NON_ASYNC();
      this.addIssue(message, awaitExpression, IssueCodes.EXP_AWAIT_NON_ASYNC);
    }
    if ((isUndef || isNull)) {
      final String message_1 = IssueCodes.getMessageForEXP_AWAIT_NON_ASYNC_SPECIAL(typeRef.getDeclaredType().getName());
      this.addIssue(message_1, awaitExpression, IssueCodes.EXP_AWAIT_NON_ASYNC_SPECIAL);
    }
  }
  
  @Check
  public void checkPropertyAccesssExpression(final ParameterizedPropertyAccessExpression propAccessExpression) {
    boolean _or = false;
    Expression _target = null;
    if (propAccessExpression!=null) {
      _target=propAccessExpression.getTarget();
    }
    boolean _tripleEquals = (_target == null);
    if (_tripleEquals) {
      _or = true;
    } else {
      IdentifiableElement _property = propAccessExpression.getProperty();
      boolean _tripleEquals_1 = (_property == null);
      _or = _tripleEquals_1;
    }
    if (_or) {
      return;
    }
    final IdentifiableElement prop = propAccessExpression.getProperty();
    List<TypeVariable> _xifexpression = null;
    if ((prop instanceof Type)) {
      _xifexpression = ((Type)prop).getTypeVars();
    } else {
      _xifexpression = Collections.<TypeVariable>unmodifiableList(CollectionLiterals.<TypeVariable>newArrayList());
    }
    final List<TypeVariable> typeVars = _xifexpression;
    this.internalCheckTypeArguments(typeVars, propAccessExpression.getTypeArgs(), true, prop, propAccessExpression, 
      N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Property());
    this.internalCheckTargetSubtypeOfDeclaredThisType(propAccessExpression);
    this.internalCheckMethodReference(propAccessExpression);
    this.internalCheckAccessToStaticMemberOfInterface(propAccessExpression);
  }
  
  private void internalCheckTargetSubtypeOfDeclaredThisType(final ParameterizedPropertyAccessExpression propAccessExpr) {
    final IdentifiableElement prop = propAccessExpr.getProperty();
    boolean _eIsProxy = prop.eIsProxy();
    if (_eIsProxy) {
      return;
    }
    TypeRef _switchResult = null;
    boolean _matched = false;
    if (prop instanceof TMethod) {
      _matched=true;
      _switchResult = ((TMethod)prop).getDeclaredThisType();
    }
    if (!_matched) {
      if (prop instanceof FieldAccessor) {
        _matched=true;
        _switchResult = ((FieldAccessor)prop).getDeclaredThisType();
      }
    }
    final TypeRef declThisTypeRef = _switchResult;
    if ((declThisTypeRef != null)) {
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(propAccessExpr);
      final TypeRef targetTypeRef = this.ts.type(G, propAccessExpr.getTarget());
      boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, targetTypeRef, declThisTypeRef);
      boolean _not = (!_subtypeSucceeded);
      if (_not) {
        final String msg = IssueCodes.getMessageForEXP_ACCESS_INVALID_TYPE_OF_TARGET(this._validatorMessageHelper.description(prop), 
          targetTypeRef.getTypeRefAsString(), declThisTypeRef.getTypeRefAsString());
        this.addIssue(msg, propAccessExpr, N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Property(), 
          IssueCodes.TYS_NO_SUBTYPE);
      }
    }
  }
  
  /**
   * Fixes IDE-1048, Method Assignment: Methods can't be assigned to variables.
   * 
   * <p>
   * If allowed, that variable could be used later
   * to invoke the function it holds with a wrong this-instance
   * (for example, with an instance of a superclass while the function,
   * defined in a subclass, requires members private to that subclass).
   * 
   * <p>
   * To be safe, we warn on expressions out of which a method reference might escape
   * and become assigned to a variable. An example where a method reference is consumed
   * before escaping is <code>typeof method-ref-expr</code>, for which no warning is raised.
   * 
   * @see N4JSSpec, 5.2.1
   */
  private void internalCheckMethodReference(final ParameterizedPropertyAccessExpression propAccessExpression) {
    boolean _checkMethodReference = this.jsVariantHelper.checkMethodReference(propAccessExpression);
    boolean _not = (!_checkMethodReference);
    if (_not) {
      return;
    }
    final IdentifiableElement prop = propAccessExpression.getProperty();
    if ((!(prop instanceof TMethod))) {
      return;
    }
    final TMethod method = ((TMethod) prop);
    if (((!method.isStatic()) && Objects.equal("constructor", method.getName()))) {
      return;
    }
    final EObject enclosing = propAccessExpression.eContainer();
    boolean _switchResult = false;
    boolean _matched = false;
    if (enclosing instanceof ParameterizedCallExpression) {
      _matched=true;
      Expression _target = ((ParameterizedCallExpression)enclosing).getTarget();
      _switchResult = (_target != propAccessExpression);
    }
    if (!_matched) {
      if (enclosing instanceof ParameterizedPropertyAccessExpression) {
        _matched=true;
        _switchResult = false;
      }
    }
    if (!_matched) {
      if (enclosing instanceof UnaryExpression) {
        _matched=true;
        UnaryOperator _op = ((UnaryExpression)enclosing).getOp();
        _switchResult = (_op != UnaryOperator.TYPEOF);
      }
    }
    if (!_matched) {
      if (enclosing instanceof EqualityExpression) {
        _matched=true;
        _switchResult = false;
      }
    }
    if (!_matched) {
      if (enclosing instanceof ExpressionStatement) {
        _matched=true;
        _switchResult = false;
      }
    }
    if (!_matched) {
      _switchResult = true;
    }
    final boolean shouldWarn = _switchResult;
    if ((!shouldWarn)) {
      return;
    }
    if ((this.isMethodEffectivelyFinal(method) && method.isLacksThisOrSuperUsage())) {
      return;
    }
    final String message = IssueCodes.getMessageForEXP_METHOD_REF_UNATTACHED_FROM_RECEIVER(method.getName());
    final ParameterizedPropertyAccessExpression source = propAccessExpression;
    final EReference feature = N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Property();
    this.warning(message, source, feature, IssueCodes.EXP_METHOD_REF_UNATTACHED_FROM_RECEIVER);
  }
  
  private boolean isMethodEffectivelyFinal(final TMethod method) {
    if ((method.isFinal() || Objects.equal(method.getMemberAccessModifier(), MemberAccessModifier.PRIVATE))) {
      return true;
    }
    final ContainerType<?> containingType = method.getContainingType();
    if (((containingType != null) && containingType.isFinal())) {
      return true;
    }
    return false;
  }
  
  /**
   * Static members of interfaces may only be accessed directly via the type name of the containing interface.
   * This is required, because there is no inheritance of static members of interfaces.
   */
  private void internalCheckAccessToStaticMemberOfInterface(final ParameterizedPropertyAccessExpression propAccessExpr) {
    final IdentifiableElement prop = propAccessExpr.getProperty();
    if ((prop instanceof TMember)) {
      if ((((prop != null) && ((TMember)prop).isStatic()) && (((TMember)prop).eContainer() instanceof TInterface))) {
        final Expression target = propAccessExpr.getTarget();
        IdentifierRef _xifexpression = null;
        if ((target instanceof IdentifierRef)) {
          _xifexpression = ((IdentifierRef)target);
        } else {
          _xifexpression = null;
        }
        final IdentifierRef targetIdRef = _xifexpression;
        final boolean isExceptionCase = (target instanceof ThisLiteral);
        boolean _and = false;
        IdentifiableElement _id = null;
        if (targetIdRef!=null) {
          _id=targetIdRef.getId();
        }
        EObject _eContainer = ((TMember)prop).eContainer();
        boolean _tripleNotEquals = (_id != _eContainer);
        if (!_tripleNotEquals) {
          _and = false;
        } else {
          _and = (!isExceptionCase);
        }
        if (_and) {
          final String message = IssueCodes.getMessageForCLF_INVALID_ACCESS_OF_STATIC_MEMBER_OF_INTERFACE();
          this.addIssue(message, propAccessExpr, 
            N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Target(), 
            IssueCodes.CLF_INVALID_ACCESS_OF_STATIC_MEMBER_OF_INTERFACE);
        }
      }
    }
  }
  
  @Check
  public void checkCallExpression(final ParameterizedCallExpression callExpression) {
    boolean _checkCallExpression = this.jsVariantHelper.checkCallExpression(callExpression);
    boolean _not = (!_checkCallExpression);
    if (_not) {
      return;
    }
    Expression _target = null;
    if (callExpression!=null) {
      _target=callExpression.getTarget();
    }
    boolean _tripleEquals = (_target == null);
    if (_tripleEquals) {
      return;
    }
    final TypeRef typeRef = this.ts.tau(callExpression.getTarget());
    if ((typeRef == null)) {
      return;
    }
    if ((typeRef instanceof UnknownTypeRef)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(callExpression);
    if (((!(callExpression.getTarget() instanceof SuperLiteral)) && (!this.tsh.isCallable(G, typeRef)))) {
      if ((this.tsh.isClassConstructorFunction(G, typeRef) || this.isClassifierTypeRefToAbstractClass(G, typeRef))) {
        final String message = IssueCodes.getMessageForEXP_CALL_CLASS_CTOR();
        this.addIssue(message, callExpression.getTarget(), null, IssueCodes.EXP_CALL_CLASS_CTOR);
      } else {
        final String message_1 = IssueCodes.getMessageForEXP_CALL_NOT_A_FUNCTION(typeRef.getTypeRefAsString());
        this.addIssue(message_1, callExpression.getTarget(), null, IssueCodes.EXP_CALL_NOT_A_FUNCTION);
      }
      return;
    }
    if ((typeRef instanceof FunctionTypeExprOrRef)) {
      this.internalCheckTypeArguments(((FunctionTypeExprOrRef)typeRef).getTypeVars(), callExpression.getTypeArgs(), true, ((FunctionTypeExprOrRef)typeRef).getDeclaredType(), callExpression, N4JSPackage.eINSTANCE.getParameterizedCallExpression_Target());
      this.internalCheckCallingAsyncFunWithoutAwaitingForIt(((FunctionTypeExprOrRef)typeRef), callExpression);
      final Expression trgt = callExpression.getTarget();
      boolean _matched = false;
      if (trgt instanceof IdentifierRef) {
        _matched=true;
        final Procedure4<String, EObject, EStructuralFeature, String> _function = (String message_2, EObject source, EStructuralFeature feature, String issueCode) -> {
          this.addIssue(message_2, source, feature, issueCode);
        };
        N4JSExpressionValidator.internalCheckNameRestrictionInMethodBodies(((IdentifierRef)trgt), _function);
      }
    }
  }
  
  private boolean isClassifierTypeRefToAbstractClass(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef instanceof TypeTypeRef)) {
      final Type staticType = this.tsh.getStaticType(G, ((TypeTypeRef)typeRef));
      if ((staticType instanceof TClass)) {
        return ((TClass)staticType).isAbstract();
      }
    }
    return false;
  }
  
  /**
   * If the given function-type is async and not awaited-for, issue a warning unless the return-type (Promise) is made explicit.
   * "Made explicit" either by:
   * <ul>
   * <li>the invocation is the RHS of a variable (declaration or assignment) where the LHS makes explicit the Promise type.</li>
   * <li>the invocation is made at the top-level for its side-effects.</li>
   * <li>the invocation is given as argument to {@code Promise.all()}, {@code Promise.race()}, or {@code Promise.resolve()}.</li>
   * </ul>
   * To clarify, a not-awaited-for call is perfectly valid, after all sometimes only the promise is of interest, but more commonly an await was forgotten.
   */
  public void internalCheckCallingAsyncFunWithoutAwaitingForIt(final FunctionTypeExprOrRef fteor, final ParameterizedCallExpression callExpression) {
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(callExpression);
    boolean _isAsync = N4JSLanguageUtils.isAsync(fteor, G);
    boolean _not = (!_isAsync);
    if (_not) {
      return;
    }
    EObject container = callExpression.eContainer();
    while ((container instanceof ParenExpression)) {
      container = ((ParenExpression)container).eContainer();
    }
    final boolean isAwaitedFor = (container instanceof AwaitExpression);
    final boolean isTopLevel = ((container instanceof ExpressionStatement) && (container.eContainer() instanceof Script));
    if ((isAwaitedFor || isTopLevel)) {
      return;
    }
    boolean _xifexpression = false;
    if ((container instanceof VariableDeclaration)) {
      _xifexpression = ((((VariableDeclaration)container).getExpression() == callExpression) && (((VariableDeclaration)container).getDeclaredTypeRef() != null));
    } else {
      boolean _xifexpression_1 = false;
      if ((container instanceof AssignmentExpression)) {
        Expression _rhs = ((AssignmentExpression)container).getRhs();
        _xifexpression_1 = (_rhs == callExpression);
      } else {
        boolean _xifexpression_2 = false;
        boolean _isArgumentToPromiseUtilityMethod = this.isArgumentToPromiseUtilityMethod(callExpression, container, G);
        if (_isArgumentToPromiseUtilityMethod) {
          _xifexpression_2 = true;
        } else {
          _xifexpression_2 = false;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    final boolean isPromiseExplict = _xifexpression;
    final boolean shouldWarn = (!isPromiseExplict);
    if (shouldWarn) {
      final String message = IssueCodes.getMessageForEXP_MISSNG_AWAIT_FOR_ASYNC_TARGET();
      this.addIssue(message, callExpression.getTarget(), IssueCodes.EXP_MISSNG_AWAIT_FOR_ASYNC_TARGET);
    }
  }
  
  /**
   * Does the given AST-node occur as argument to {@code Promise.all()}, {@code Promise.race()}, or {@code Promise.resolve()} ?
   */
  private boolean isArgumentToPromiseUtilityMethod(final ParameterizedCallExpression asyncInvocation, final EObject container, final RuleEnvironment G) {
    EObject utilityCall = container;
    final boolean isArrayElem = ((container instanceof ArrayElement) && (container.eContainer() instanceof ArrayLiteral));
    if (isArrayElem) {
      utilityCall = container.eContainer().eContainer();
    }
    if ((utilityCall instanceof Argument)) {
      utilityCall = ((Argument)utilityCall).eContainer();
    }
    if ((utilityCall instanceof ParameterizedCallExpression)) {
      Expression _target = ((ParameterizedCallExpression)utilityCall).getTarget();
      if ((_target instanceof ParameterizedPropertyAccessExpression)) {
        Expression _target_1 = ((ParameterizedCallExpression)utilityCall).getTarget();
        final ParameterizedPropertyAccessExpression utilityAccess = ((ParameterizedPropertyAccessExpression) _target_1);
        boolean _isPromiseUtilityPropertyAccess = this.isPromiseUtilityPropertyAccess(utilityAccess, G);
        if (_isPromiseUtilityPropertyAccess) {
          final Function1<Argument, Boolean> _function = (Argument arg) -> {
            Expression _expression = arg.getExpression();
            return Boolean.valueOf((_expression == asyncInvocation));
          };
          final boolean isDirectArg = IterableExtensions.<Argument>exists(((ParameterizedCallExpression)utilityCall).getArguments(), _function);
          if (isDirectArg) {
            return true;
          }
          final String name = utilityAccess.getProperty().getName();
          if ((isArrayElem && (Objects.equal(name, "all") || Objects.equal(name, "race")))) {
            final Function1<Argument, Boolean> _function_1 = (Argument arg) -> {
              Expression _expression = arg.getExpression();
              EObject _eContainer = container.eContainer();
              return Boolean.valueOf((_expression == _eContainer));
            };
            final boolean argOccursInArray = IterableExtensions.<Argument>exists(((ParameterizedCallExpression)utilityCall).getArguments(), _function_1);
            return argOccursInArray;
          }
        }
      }
    }
    return false;
  }
  
  /**
   * Does 'utilityAccess' stand for 'Promise.{all/race/resolve}' ?
   */
  private boolean isPromiseUtilityPropertyAccess(final ParameterizedPropertyAccessExpression utilityAccess, final RuleEnvironment G) {
    final IdentifiableElement invokedUtility = utilityAccess.getProperty();
    if ((invokedUtility instanceof TMethod)) {
      final boolean isStaticUtility = ((TMethod)invokedUtility).isStatic();
      final boolean hasNameOfInterest = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("all", "race", "resolve")).contains(((TMethod)invokedUtility).getName());
      if ((isStaticUtility && hasNameOfInterest)) {
        final BuiltInTypeScope tscope = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope;
        final TypeRef tr = this.ts.type(G, utilityAccess.getTarget());
        if ((tr instanceof TypeTypeRef)) {
          final TypeArgument str = ((TypeTypeRef)tr).getTypeArg();
          boolean _xifexpression = false;
          if ((str instanceof TypeRef)) {
            _xifexpression = TypeUtils.isPromise(((TypeRef)str), tscope);
          } else {
            _xifexpression = false;
          }
          final boolean isReceiverPromise = _xifexpression;
          return isReceiverPromise;
        }
      }
    }
    return false;
  }
  
  /**
   * Constraints 51 (Name restriction in method-bodies):
   * 
   * checks, that in case the trgt refers to a plain function (not a method) and ends with "___n4",
   * it will not be contained in Method.
   */
  public static void internalCheckNameRestrictionInMethodBodies(final IdentifierRef trgt, final Procedure4<? super String, ? super EObject, ? super EStructuralFeature, ? super String> g) {
    if (((trgt.getId() instanceof TFunction) && (!(trgt.getId() instanceof TMethod)))) {
      boolean _endsWith = trgt.getId().getName().endsWith(N4JSLanguageConstants.METHOD_STACKTRACE_SUFFIX);
      if (_endsWith) {
        final N4MethodDeclaration containingMethod = EcoreUtil2.<N4MethodDeclaration>getContainerOfType(trgt, N4MethodDeclaration.class);
        if ((containingMethod != null)) {
          final String msg = IssueCodes.getMessageForCLF_METHOD_BODY_FORBIDDEN_REFERENCE_NAME();
          g.apply(msg, trgt, N4JSPackage.eINSTANCE.getIdentifierRef_Id(), 
            IssueCodes.CLF_METHOD_BODY_FORBIDDEN_REFERENCE_NAME);
        }
      }
    }
  }
  
  @Check
  public void checkNew(final NewExpression newExpression) {
    boolean _requireCheckNewExpression = this.jsVariantHelper.requireCheckNewExpression(newExpression);
    boolean _not = (!_requireCheckNewExpression);
    if (_not) {
      return;
    }
    Expression _callee = null;
    if (newExpression!=null) {
      _callee=newExpression.getCallee();
    }
    boolean _tripleEquals = (_callee == null);
    if (_tripleEquals) {
      return;
    }
    final Expression callee = newExpression.getCallee();
    final TypeRef typeRef = this.ts.tau(callee);
    if ((typeRef == null)) {
      return;
    }
    if ((typeRef instanceof UnknownTypeRef)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(newExpression);
    if ((!(typeRef instanceof TypeTypeRef))) {
      boolean _isDynamic = typeRef.isDynamic();
      if (_isDynamic) {
        final TypeTypeRef constrOfWildcard = TypeUtils.createTypeTypeRef(TypeUtils.createWildcard(), true);
        if ((this.ts.subtypeSucceeded(G, typeRef, constrOfWildcard) || this.ts.subtypeSucceeded(G, constrOfWildcard, typeRef))) {
          return;
        }
      }
      this.issueNotACtor(typeRef, newExpression);
      return;
    }
    final TypeTypeRef classifierTypeRef = ((TypeTypeRef) typeRef);
    final TypeArgument typeArg = classifierTypeRef.getTypeArg();
    final Type staticType = this.changeToCovariantUpperBoundIfTypeVar(this.tsh.getStaticType(G, classifierTypeRef));
    if (((staticType != null) && staticType.eIsProxy())) {
      return;
    }
    final boolean isCtor = classifierTypeRef.isConstructorRef();
    final boolean isDirectRef = ((callee instanceof IdentifierRef) && (((IdentifierRef) callee).getId() == staticType));
    final boolean isConcreteOrCovariant = ((!(((typeArg instanceof Wildcard) || (typeArg instanceof ExistentialTypeRef)) || (typeArg instanceof ThisTypeRef))) || ((staticType instanceof TClassifier) && N4JSLanguageUtils.hasCovariantConstructor(((TClassifier) staticType))));
    TObjectPrototype _symbolObjectType = RuleEnvironmentExtensions.symbolObjectType(G);
    boolean _tripleEquals_1 = (staticType == _symbolObjectType);
    if (_tripleEquals_1) {
      final String message = IssueCodes.getMessageForBIT_SYMBOL_NOT_A_CTOR();
      this.addIssue(message, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
        IssueCodes.BIT_SYMBOL_NOT_A_CTOR);
      return;
    } else {
      if ((((!isCtor) && (staticType instanceof TInterface)) && isDirectRef)) {
        final String message_1 = IssueCodes.getMessageForEXP_NEW_CANNOT_INSTANTIATE(this._n4JSElementKeywordProvider.keyword(staticType), staticType.getName());
        this.addIssue(message_1, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
          IssueCodes.EXP_NEW_CANNOT_INSTANTIATE);
        return;
      } else {
        if (((((!isCtor) && (staticType instanceof TClass)) && ((TClass) staticType).isAbstract()) && isDirectRef)) {
          final String message_2 = IssueCodes.getMessageForEXP_NEW_CANNOT_INSTANTIATE("abstract class", staticType.getName());
          this.addIssue(message_2, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
            IssueCodes.EXP_NEW_CANNOT_INSTANTIATE);
          return;
        } else {
          if (((isCtor && (!isConcreteOrCovariant)) && (staticType instanceof TClassifier))) {
            final String message_3 = IssueCodes.getMessageForEXP_NEW_WILDCARD_NO_COVARIANT_CTOR(typeArg.getTypeRefAsString(), staticType.getTypeAsString());
            this.addIssue(message_3, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
              IssueCodes.EXP_NEW_WILDCARD_NO_COVARIANT_CTOR);
            return;
          } else {
            if ((staticType instanceof TEnum)) {
              final String message_4 = IssueCodes.getMessageForEXP_NEW_CANNOT_INSTANTIATE("enum", ((TEnum)staticType).getName());
              this.addIssue(message_4, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
                IssueCodes.EXP_NEW_CANNOT_INSTANTIATE);
              return;
            } else {
              if ((staticType instanceof TypeVariable)) {
                final String message_5 = IssueCodes.getMessageForEXP_NEW_CANNOT_INSTANTIATE("type variable", ((TypeVariable)staticType).getName());
                this.addIssue(message_5, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
                  IssueCodes.EXP_NEW_CANNOT_INSTANTIATE);
                return;
              } else {
                if ((((staticType == null) || (!isCtor)) || (!isConcreteOrCovariant))) {
                  final String name = classifierTypeRef.getTypeRefAsString();
                  final String message_6 = IssueCodes.getMessageForEXP_NEW_WILDCARD_OR_TYPEVAR(name);
                  this.addIssue(message_6, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), 
                    IssueCodes.EXP_NEW_WILDCARD_OR_TYPEVAR);
                  return;
                }
              }
            }
          }
        }
      }
    }
    this.internalCheckTypeArguments(staticType.getTypeVars(), newExpression.getTypeArgs(), false, staticType, newExpression, 
      N4JSPackage.eINSTANCE.getNewExpression_Callee());
    if ((staticType instanceof TClassifier)) {
      this.internalCheckNewParameters(newExpression, ((TClassifier)staticType));
    }
  }
  
  private Type changeToCovariantUpperBoundIfTypeVar(final Type type) {
    if ((type instanceof TypeVariable)) {
      final TypeRef ub = ((TypeVariable)type).getDeclaredUpperBound();
      if ((ub instanceof ParameterizedTypeRef)) {
        final Type declType = ((ParameterizedTypeRef)ub).getDeclaredType();
        if ((declType instanceof TClassifier)) {
          boolean _hasCovariantConstructor = N4JSLanguageUtils.hasCovariantConstructor(((TClassifier)declType));
          if (_hasCovariantConstructor) {
            return declType;
          }
        }
      }
    }
    return type;
  }
  
  /**
   * Helper to issue the error case of having a new-expression on a non-constructor element
   */
  private void issueNotACtor(final TypeRef typeRef, final NewExpression newExpression) {
    final String message = IssueCodes.getMessageForEXP_NEW_NOT_A_CTOR(typeRef.getTypeRefAsString());
    this.addIssue(message, newExpression, N4JSPackage.eINSTANCE.getNewExpression_Callee(), IssueCodes.EXP_NEW_NOT_A_CTOR);
  }
  
  /**
   * Checks instanceof in combination with structural typing, other checks see
   * org.eclipse.n4js.xsemantics.N4JSTypeSystem.expectedTypeInRelationalExpression
   */
  @Check
  public void checkRelationalExpression(final RelationalExpression relationalExpression) {
    if (((relationalExpression.getRhs() != null) && (relationalExpression.getOp() == RelationalOperator.INSTANCEOF))) {
      final TypeRef typeRef = this.ts.tau(relationalExpression.getRhs());
      if ((typeRef instanceof TypeTypeRef)) {
        final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(relationalExpression);
        final Type staticType = this.tsh.getStaticType(G, ((TypeTypeRef)typeRef));
        if ((staticType instanceof TN4Classifier)) {
          TypingStrategy _typingStrategy = ((TN4Classifier)staticType).getTypingStrategy();
          boolean _tripleNotEquals = (_typingStrategy != TypingStrategy.DEFAULT);
          if (_tripleNotEquals) {
            final String message = IssueCodes.getMessageForTYS_INSTANCEOF_NOT_SUPPORTED_FOR_STRUCTURAL_TYPES(((TN4Classifier)staticType).getName());
            this.addIssue(message, relationalExpression, N4JSPackage.eINSTANCE.getRelationalExpression_Rhs(), 
              IssueCodes.TYS_INSTANCEOF_NOT_SUPPORTED_FOR_STRUCTURAL_TYPES);
          } else {
            if (((staticType instanceof TInterface) && 
              (EcoreUtil.getRootContainer(staticType) instanceof TypeDefs))) {
              final String message_1 = IssueCodes.getMessageForTYS_INSTANCEOF_NOT_SUPPORTED_FOR_BUILT_IN_INTERFACES(((TN4Classifier)staticType).getName());
              this.addIssue(message_1, relationalExpression, N4JSPackage.eINSTANCE.getRelationalExpression_Rhs(), 
                IssueCodes.TYS_INSTANCEOF_NOT_SUPPORTED_FOR_BUILT_IN_INTERFACES);
            }
          }
        }
      }
    }
  }
  
  /**
   * IDE-737: properties in postfixExpressions need both of getter/setter.
   *     Getter is bound to the property-field in PropertyAccessExpression, the existence of a setter needs to be validated.
   */
  @Check
  public boolean checkPostfixExpression(final PostfixExpression postfixExpression) {
    boolean _xblockexpression = false;
    {
      final Expression expression = postfixExpression.getExpression();
      _xblockexpression = ((this.holdsWritabelePropertyAccess(expression) && this.holdsWritableIdentifier(expression)) && 
        this.holdsLefthandsideNotConst(expression));
    }
    return _xblockexpression;
  }
  
  /**
   * IDE-731 / IDE-768 unary expressions of type ++ or -- need both of getter/setter.
   * Cf. Constraint 69
   */
  @Check
  public boolean checkUnaryExpressionWithWriteAccess(final UnaryExpression unaryExpression) {
    boolean _xifexpression = false;
    if (((UnaryOperator.DEC == unaryExpression.getOp()) || (UnaryOperator.INC == unaryExpression.getOp()))) {
      _xifexpression = ((this.holdsWritabelePropertyAccess(unaryExpression.getExpression()) && 
        this.holdsWritableIdentifier(unaryExpression.getExpression())) && 
        this.holdsLefthandsideNotConst(unaryExpression.getExpression()));
    }
    return _xifexpression;
  }
  
  private boolean holdsWritabelePropertyAccess(final Expression expression) {
    if ((expression instanceof ParameterizedPropertyAccessExpression)) {
      final IdentifiableElement property = ((ParameterizedPropertyAccessExpression)expression).getProperty();
      if ((property instanceof TGetter)) {
        final TypeRef propertyTargetType = this.ts.tau(((ParameterizedPropertyAccessExpression)expression).getTarget());
        Type _declaredType = null;
        if (propertyTargetType!=null) {
          _declaredType=propertyTargetType.getDeclaredType();
        }
        final Type declaredType = _declaredType;
        if ((declaredType instanceof TClassifier)) {
          final Function1<TSetter, Boolean> _function = (TSetter it) -> {
            return Boolean.valueOf(it.getName().equals(((TGetter)property).getName()));
          };
          final boolean setterExists = IterableExtensions.<TSetter>exists(Iterables.<TSetter>filter(this.containerTypesHelper.fromContext(expression).members(((ContainerType<?>)declaredType)), 
            TSetter.class), _function);
          if ((!setterExists)) {
            final String msg = IssueCodes.getMessageForTYS_PROPERTY_HAS_NO_SETTER(((TGetter)property).getName());
            this.addIssue(msg, expression, 
              N4JSPackage.Literals.PARAMETERIZED_PROPERTY_ACCESS_EXPRESSION__PROPERTY, 
              IssueCodes.TYS_PROPERTY_HAS_NO_SETTER);
            return false;
          }
        }
      }
    }
    return true;
  }
  
  /**
   * Ensures that imported elements get not reassigned any value.
   * @returns true if validation hold, false if some issue was generated.
   */
  private boolean holdsWritableIdentifier(final Expression expression) {
    if ((expression instanceof IdentifierRef)) {
      final IdentifiableElement id = ((IdentifierRef)expression).getId();
      boolean _matched = false;
      if (id instanceof TExportableElement) {
        _matched=true;
        final TModule module = EcoreUtil2.<Script>getContainerOfType(expression, Script.class).getModule();
        TModule _containingModule = ((TExportableElement)id).getContainingModule();
        boolean _notEquals = (!Objects.equal(_containingModule, module));
        if (_notEquals) {
          this.addIssue(IssueCodes.getMessageForIMP_IMPORTED_ELEMENT_READ_ONLY(((IdentifierRef)expression).getIdAsText()), expression, IssueCodes.IMP_IMPORTED_ELEMENT_READ_ONLY);
          return false;
        }
      }
    } else {
      if ((expression instanceof ParenExpression)) {
        return this.holdsWritableIdentifier(((ParenExpression)expression).getExpression());
      } else {
        if ((expression instanceof ParameterizedPropertyAccessExpression)) {
          final Expression target = ((ParameterizedPropertyAccessExpression)expression).getTarget();
          if (((((ParameterizedPropertyAccessExpression)expression).getProperty() != null) && (!((ParameterizedPropertyAccessExpression)expression).getProperty().eIsProxy()))) {
            if ((target instanceof IdentifierRef)) {
              final IdentifiableElement id_1 = ((IdentifierRef)target).getId();
              if ((id_1 instanceof ModuleNamespaceVirtualType)) {
                TModule _module = ((ModuleNamespaceVirtualType)id_1).getModule();
                TModule _module_1 = EcoreUtil2.<Script>getContainerOfType(expression, Script.class).getModule();
                boolean _notEquals = (!Objects.equal(_module, _module_1));
                if (_notEquals) {
                  final String importedElmentText = NodeModelUtils.getTokenText(
                    NodeModelUtils.findActualNodeFor(expression));
                  this.addIssue(IssueCodes.getMessageForIMP_IMPORTED_ELEMENT_READ_ONLY(importedElmentText), expression, IssueCodes.IMP_IMPORTED_ELEMENT_READ_ONLY);
                  return false;
                }
              }
            }
          }
        }
      }
    }
    return true;
  }
  
  @Check
  public void checkCallExpressionParameters(final ParameterizedCallExpression callExpression) {
    boolean _checkCallExpression = this.jsVariantHelper.checkCallExpression(callExpression);
    boolean _not = (!_checkCallExpression);
    if (_not) {
      return;
    }
    final Expression target = callExpression.getTarget();
    if ((target != null)) {
      final TypeRef targetTypeRef = this.ts.tau(target);
      if ((targetTypeRef instanceof FunctionTypeExprOrRef)) {
        EList<TFormalParameter> _fpars = ((FunctionTypeExprOrRef)targetTypeRef).getFpars();
        ArrayList<TFormalParameter> fpars = new ArrayList<TFormalParameter>(_fpars);
        final EObject parent = callExpression.eContainer();
        final boolean isPromisified = (((parent instanceof AwaitExpression) && 
          this.promisifyHelper.isAutoPromisify(((AwaitExpression) parent))) || 
          (parent instanceof PromisifyExpression));
        if (isPromisified) {
          int _size = fpars.size();
          int _minus = (_size - 1);
          fpars.remove(_minus);
        }
        this.internalCheckNumberOfArguments(fpars, callExpression.getArguments(), callExpression);
      }
    }
  }
  
  private void internalCheckNewParameters(final NewExpression newExpression, final TClassifier staticType) {
    final TMethod maybeConstructor = this.containerTypesHelper.fromContext(newExpression).findConstructor(staticType);
    if ((maybeConstructor != null)) {
      this.internalCheckNumberOfArguments(((TFunction) maybeConstructor).getFpars(), newExpression.getArguments(), newExpression);
      return;
    }
  }
  
  private void internalCheckNumberOfArguments(final List<TFormalParameter> fpars, final List<Argument> args, final Expression expr) {
    final int cmp = this.compareNumberOfArgsWithNumberOfFPars(fpars, args);
    if ((cmp < 0)) {
      this.addIssue(IssueCodes.getMessageForEXP_NUM_OF_ARGS_TOO_FEW(Integer.valueOf(fpars.size()), Integer.valueOf(args.size())), expr, 
        IssueCodes.EXP_NUM_OF_ARGS_TOO_FEW);
    } else {
      if ((cmp > 0)) {
        this.addIssue(IssueCodes.getMessageForEXP_NUM_OF_ARGS_TOO_MANY(Integer.valueOf(fpars.size()), Integer.valueOf(args.size())), expr, 
          IssueCodes.EXP_NUM_OF_ARGS_TOO_MANY);
      }
    }
  }
  
  /**
   * Compares number of arguments with number of formal parameter, taking optional and variadic parameters into consideration.
   * @return -1 if to few arguments are found, 1 if too many arguments are found, 0 if number of arguments is according to formal parameter list
   */
  private int compareNumberOfArgsWithNumberOfFPars(final List<TFormalParameter> fpars, final List<Argument> args) {
    final int argCount = args.size();
    final int fparCount = fpars.size();
    int _size = fpars.size();
    int _size_1 = args.size();
    boolean _tripleEquals = (_size == _size_1);
    if (_tripleEquals) {
      return 0;
    }
    if ((argCount > fparCount)) {
      if ((fparCount == 0)) {
        return 1;
      }
      boolean _isVariadic = IterableExtensions.<TFormalParameter>last(fpars).isVariadic();
      if (_isVariadic) {
        return 0;
      }
      return 1;
    }
    boolean _isVariadicOrOptional = fpars.get(argCount).isVariadicOrOptional();
    if (_isVariadicOrOptional) {
      return 0;
    }
    return (-1);
  }
  
  /**
   * IDE-731 / IDE-770
   *  Cf. 6.1.14. Additive Expression, Constraint 73
   */
  @Check
  public void checkAdditiveExpressionForNonADDs(final AdditiveExpression ae) {
    AdditiveOperator _op = ae.getOp();
    boolean _equals = Objects.equal(_op, AdditiveOperator.SUB);
    if (_equals) {
      this.doCheckMathOperandTypes(ae.getLhs(), ae.getRhs());
    } else {
      this.doCheckMathOperandTypeSymbol(ae.getLhs(), ae.getRhs());
    }
  }
  
  /**
   * Note: Division by 0 may lead to infinity or NaN, depending on the value of the rhs.
   * I.e. 0/0=NaN, but 1/0=Infinity. So we cannot infer from the type the result in these cases.
   */
  @Check
  public void checkMultiplicativeExpression(final MultiplicativeExpression me) {
    this.doCheckMathOperandTypes(me.getLhs(), me.getRhs());
  }
  
  @Check
  public void checkShiftExpression(final ShiftExpression se) {
    this.doCheckMathOperandTypes(se.getLhs(), se.getRhs());
  }
  
  public void doCheckMathOperandTypes(final Expression lhs, final Expression rhs) {
    if (((lhs == null) || (rhs == null))) {
      return;
    }
    final TypeRef tlhs = this.ts.tau(lhs);
    if ((tlhs == null)) {
      return;
    }
    final TypeRef trhs = this.ts.tau(rhs);
    if ((trhs == null)) {
      return;
    }
    final BuiltInTypeScope bits = BuiltInTypeScope.get(lhs.eResource().getResourceSet());
    Type _declaredType = tlhs.getDeclaredType();
    UndefinedType _undefinedType = bits.getUndefinedType();
    boolean _tripleEquals = (_declaredType == _undefinedType);
    if (_tripleEquals) {
      this.issueMathResultIsConstant("of type undefined", "NaN", lhs);
    }
    Type _declaredType_1 = trhs.getDeclaredType();
    UndefinedType _undefinedType_1 = bits.getUndefinedType();
    boolean _tripleEquals_1 = (_declaredType_1 == _undefinedType_1);
    if (_tripleEquals_1) {
      this.issueMathResultIsConstant("of type undefined", "NaN", rhs);
    }
    Type _declaredType_2 = tlhs.getDeclaredType();
    NullType _nullType = bits.getNullType();
    boolean _tripleEquals_2 = (_declaredType_2 == _nullType);
    if (_tripleEquals_2) {
      this.issueMathOperandIsConstant("null", "0", lhs);
    }
    Type _declaredType_3 = trhs.getDeclaredType();
    NullType _nullType_1 = bits.getNullType();
    boolean _tripleEquals_3 = (_declaredType_3 == _nullType_1);
    if (_tripleEquals_3) {
      this.issueMathOperandIsConstant("null", "0", rhs);
    }
    Type _declaredType_4 = tlhs.getDeclaredType();
    PrimitiveType _symbolType = bits.getSymbolType();
    boolean _equals = Objects.equal(_declaredType_4, _symbolType);
    if (_equals) {
      this.issueMathOperandTypeNotPermitted("symbol", lhs);
    }
    Type _declaredType_5 = trhs.getDeclaredType();
    PrimitiveType _symbolType_1 = bits.getSymbolType();
    boolean _equals_1 = Objects.equal(_declaredType_5, _symbolType_1);
    if (_equals_1) {
      this.issueMathOperandTypeNotPermitted("symbol", rhs);
    }
  }
  
  public void doCheckMathOperandTypeSymbol(final Expression lhs, final Expression rhs) {
    if (((lhs == null) || (rhs == null))) {
      return;
    }
    final TypeRef tlhs = this.ts.tau(lhs);
    if ((tlhs == null)) {
      return;
    }
    final TypeRef trhs = this.ts.tau(rhs);
    if ((trhs == null)) {
      return;
    }
    final BuiltInTypeScope bits = BuiltInTypeScope.get(lhs.eResource().getResourceSet());
    Type _declaredType = tlhs.getDeclaredType();
    PrimitiveType _symbolType = bits.getSymbolType();
    boolean _equals = Objects.equal(_declaredType, _symbolType);
    if (_equals) {
      this.issueMathOperandTypeNotPermitted("symbol", lhs);
    }
    Type _declaredType_1 = trhs.getDeclaredType();
    PrimitiveType _symbolType_1 = bits.getSymbolType();
    boolean _equals_1 = Objects.equal(_declaredType_1, _symbolType_1);
    if (_equals_1) {
      this.issueMathOperandTypeNotPermitted("symbol", rhs);
    }
  }
  
  public void issueMathResultIsConstant(final String operand, final String constResult, final Expression location) {
    this.addIssue(IssueCodes.getMessageForEXP_MATH_OPERATION_RESULT_IS_CONSTANT(operand, constResult), location, 
      IssueCodes.EXP_MATH_OPERATION_RESULT_IS_CONSTANT);
  }
  
  public void issueMathOperandIsConstant(final String operandType, final String constValue, final Expression location) {
    this.addIssue(IssueCodes.getMessageForEXP_MATH_OPERAND_IS_CONSTANT(operandType, constValue), location, 
      IssueCodes.EXP_MATH_OPERAND_IS_CONSTANT);
  }
  
  public void issueMathOperandTypeNotPermitted(final String operandType, final Expression location) {
    this.addIssue(IssueCodes.getMessageForEXP_MATH_TYPE_NOT_PERMITTED(operandType), location, 
      IssueCodes.EXP_MATH_TYPE_NOT_PERMITTED);
  }
  
  /**
   * IDE-731 / IDE-773
   * Cf. 6.1.17. Equality Expression
   * 
   * In N4JS mode, a warning is created, if for a given expression lhs(’===’|’!==’) rhs,
   * neither Γ |- upper(lhs) <: upper(rhs) nor Γ |- upper(rhs) <: upper(lhs), as the result is constant in these cases.
   */
  @Check
  public void checkEqualityExpressionForConstantValues(final EqualityExpression ee) {
    if (((ee.getOp() == EqualityOperator.SAME) || (ee.getOp() == EqualityOperator.NSAME))) {
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ee);
      TypeRef tlhs = this.ts.type(G, ee.getLhs());
      TypeRef trhs = this.ts.type(G, ee.getRhs());
      tlhs = this.ts.upperBound(G, tlhs);
      trhs = this.ts.upperBound(G, trhs);
      final boolean leftSubOfRight = this.ts.subtypeSucceeded(G, tlhs, trhs);
      final boolean rightSubOfLeft = this.ts.subtypeSucceeded(G, trhs, tlhs);
      final Set<Type> tdLhs = this.computeDeclaredTypeS(tlhs);
      final Set<Type> tdRhs = this.computeDeclaredTypeS(trhs);
      final boolean leftROI = this.hasInterface(tdLhs);
      final boolean rightROI = this.hasInterface(tdRhs);
      if ((leftROI && rightROI)) {
        return;
      }
      if ((leftROI || rightROI)) {
        if (leftROI) {
          boolean _isExtendable = this.isExtendable(tdRhs);
          if (_isExtendable) {
            return;
          }
        } else {
          boolean _isExtendable_1 = this.isExtendable(tdLhs);
          if (_isExtendable_1) {
            return;
          }
        }
      }
      if ((!(leftSubOfRight || rightSubOfLeft))) {
        String _warningNameOf = this.warningNameOf(tlhs);
        String _warningNameOf_1 = this.warningNameOf(trhs);
        EqualityOperator _op = ee.getOp();
        boolean _tripleEquals = (_op == EqualityOperator.NSAME);
        this.addIssue(
          IssueCodes.getMessageForEXP_WARN_CONSTANT_EQUALITY_TEST(_warningNameOf, _warningNameOf_1, Boolean.valueOf(_tripleEquals)), ee, IssueCodes.EXP_WARN_CONSTANT_EQUALITY_TEST);
      }
    }
  }
  
  private boolean isExtendable(final Set<Type> types) {
    final Function1<Type, Boolean> _function = (Type it) -> {
      return Boolean.valueOf(this.isExtendable(it));
    };
    return IterableExtensions.<Type>exists(types, _function);
  }
  
  private boolean isExtendable(final Type t) {
    boolean _isNotExtendable = this.isNotExtendable(t);
    return (!_isNotExtendable);
  }
  
  private boolean isNotExtendable(final Type t) {
    return ((((t instanceof PrimitiveType) || (t instanceof TEnum)) || (t instanceof BuiltInType)) || (t instanceof TFunction));
  }
  
  private boolean hasInterface(final Set<Type> types) {
    final Function1<Type, Boolean> _function = (Type it) -> {
      return Boolean.valueOf(this.hasInterface(it));
    };
    return IterableExtensions.<Type>exists(types, _function);
  }
  
  /**
   * true if type is a subclass of TInterface
   */
  private boolean hasInterface(final Type type) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (type instanceof TInterface) {
      _matched=true;
      _switchResult = true;
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  private String warningNameOf(final TypeRef typeRef) {
    String _xifexpression = null;
    if ((typeRef instanceof TypeTypeRef)) {
      _xifexpression = ((TypeTypeRef)typeRef).getTypeRefAsString();
    } else {
      String _xblockexpression = null;
      {
        final Set<Type> typeS = this.computeDeclaredTypeS(typeRef);
        _xblockexpression = this.warningNameOf(typeS);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  private String warningNameOf(final Set<Type> tset) {
    String _xifexpression = null;
    int _size = tset.size();
    boolean _tripleEquals = (_size == 1);
    if (_tripleEquals) {
      _xifexpression = this.warningNameOf(tset.iterator().next());
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("{");
      {
        boolean _hasElements = false;
        for(final Type s : tset) {
          if (!_hasElements) {
            _hasElements = true;
          } else {
            _builder.appendImmediate(",", "");
          }
          _builder.append(" ");
          String _warningNameOf = this.warningNameOf(s);
          _builder.append(_warningNameOf);
          _builder.append(" ");
        }
      }
      _builder.append("}");
      _xifexpression = _builder.toString();
    }
    return _xifexpression;
  }
  
  private String warningNameOf(final Type t) {
    String _xblockexpression = null;
    {
      String _elvis = null;
      String _xifexpression = null;
      if ((t == null)) {
        _xifexpression = "<type null>";
      } else {
        _xifexpression = t.getName();
      }
      if (_xifexpression != null) {
        _elvis = _xifexpression;
      } else {
        String _string = t.toString();
        _elvis = _string;
      }
      final String repr = _elvis;
      String _switchResult = null;
      boolean _matched = false;
      if (t instanceof TStructuralType) {
        _matched=true;
        _switchResult = "\'structural type\'";
      }
      if (!_matched) {
        if (t instanceof TFunction) {
          _matched=true;
          _switchResult = ("function " + repr);
        }
      }
      if (!_matched) {
        _switchResult = repr;
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private Set<Type> computeDeclaredTypeS(final TypeRef tref) {
    if ((tref instanceof ComposedTypeRef)) {
      final Comparator<Type> _function = (Type a, Type b) -> {
        int _xifexpression = (int) 0;
        if ((a == null)) {
          _xifexpression = 1;
        } else {
          int _xifexpression_1 = (int) 0;
          if ((b == null)) {
            _xifexpression_1 = (-1);
          } else {
            Comparator<String> _nullsLast = Comparator.<String>nullsLast(Comparator.<String>naturalOrder());
            String _typeAsString = null;
            if (a!=null) {
              _typeAsString=a.getTypeAsString();
            }
            String _typeAsString_1 = null;
            if (b!=null) {
              _typeAsString_1=b.getTypeAsString();
            }
            _xifexpression_1 = _nullsLast.compare(_typeAsString, _typeAsString_1);
          }
          _xifexpression = _xifexpression_1;
        }
        return _xifexpression;
      };
      final TreeSet<Type> retSet = CollectionLiterals.<Type>newTreeSet(_function);
      final Consumer<TypeRef> _function_1 = (TypeRef it) -> {
        retSet.addAll(this.computeDeclaredTypeS(it));
      };
      ((ComposedTypeRef)tref).getTypeRefs().forEach(_function_1);
      return retSet;
    }
    if ((tref instanceof BoundThisTypeRef)) {
      return Collections.<Type>singleton(((BoundThisTypeRef)tref).getActualThisTypeRef().getDeclaredType());
    } else {
      return Collections.<Type>singleton(tref.getDeclaredType());
    }
  }
  
  /**
   * Checking Constraints 79: <br>
   * Constraints 79 (Binary Logical Expression Constraints):
   * For a given binary logical expression e with e.lhs.type : L
   * and e.rhs.type : R the following constraints must hold:
   * 
   * <li> In N4JS mode L must not be undefined or null.
   * IDE-775
   */
  @Check
  public void checkBinaryLogicalExpression(final BinaryLogicalExpression e) {
    if ((((e == null) || (e.getLhs() == null)) || (e.getRhs() == null))) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(e);
    this.doCheckForbiddenType(e.getLhs(), RuleEnvironmentExtensions.nullType(G), "null");
    this.doCheckForbiddenType(e.getLhs(), RuleEnvironmentExtensions.undefinedType(G), "undefined");
  }
  
  private void doCheckForbiddenType(final Expression e, final Type forbidden, final String typeName) {
    if ((forbidden != null)) {
      TypeRef _tau = this.ts.tau(e);
      Type _declaredType = null;
      if (_tau!=null) {
        _declaredType=_tau.getDeclaredType();
      }
      final Type theType = _declaredType;
      if ((theType == forbidden)) {
        this.addIssue(
          IssueCodes.getMessageForEXP_FORBIDDEN_TYPE_IN_BINARY_LOGICAL_EXPRESSION(typeName), e, 
          IssueCodes.EXP_FORBIDDEN_TYPE_IN_BINARY_LOGICAL_EXPRESSION);
      }
    }
  }
  
  /**
   * Checking Constraint 80: <br>
   * 
   * In N4JS mode a warning will be issued if e.expression evaluates to a constant value,
   * that is e.expression in { false, true, null, undefined} or C in {void, undefined}
   * 
   * IDE-776
   */
  @Check
  public void checkConditionalExpression(final ConditionalExpression e) {
    final Expression expressionToCheck = e.getExpression();
    if ((expressionToCheck == null)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(e);
    TypeRef _tau = this.ts.tau(expressionToCheck);
    Type _declaredType = null;
    if (_tau!=null) {
      _declaredType=_tau.getDeclaredType();
    }
    final Type declaredT = _declaredType;
    ConstBoolean cboolValue = ConstBoolean.NotPrecomputable;
    if ((((declaredT == RuleEnvironmentExtensions.nullType(G)) || (declaredT == RuleEnvironmentExtensions.voidType(G))) || (declaredT == RuleEnvironmentExtensions.undefinedType(G)))) {
      cboolValue = ConstBoolean.CFalse;
    } else {
      cboolValue = this.evalConstantBooleanExpression(expressionToCheck);
    }
    if ((cboolValue == ConstBoolean.NotPrecomputable)) {
      return;
    }
    String msg1 = "?!?";
    String msg2 = "?!?";
    if ((cboolValue == ConstBoolean.CTrue)) {
      msg1 = "true";
      msg2 = "left-hand";
    } else {
      msg1 = "false";
      msg2 = "right-hand";
    }
    this.addIssue(
      IssueCodes.getMessageForEXP_WARN_DISPENSABLE_CONDITIONAL_EXPRESSION(
        NodeModelUtils.findActualNodeFor(expressionToCheck).getText().trim(), msg1, msg2), expressionToCheck, IssueCodes.EXP_WARN_DISPENSABLE_CONDITIONAL_EXPRESSION);
  }
  
  /**
   * Checks the Expression to be always constant in evaluation with
   * ECMA-Script 5.1, 2011, §9.2, p.42  ToBooleanValue(e)
   */
  private ConstBoolean evalConstantBooleanExpression(final Expression e) {
    if ((e instanceof BooleanLiteral)) {
      boolean _isTrue = ((BooleanLiteral)e).isTrue();
      if (_isTrue) {
        return ConstBoolean.CTrue;
      } else {
        return ConstBoolean.CFalse;
      }
    } else {
      if ((e instanceof NumericLiteral)) {
        final BigDecimal v = ((NumericLiteral)e).getValue();
        boolean _equals = Objects.equal(v, Integer.valueOf(0));
        if (_equals) {
          return ConstBoolean.CFalse;
        } else {
          return ConstBoolean.CTrue;
        }
      } else {
        if ((e instanceof IdentifierRef)) {
          IdentifiableElement _id = null;
          if (((IdentifierRef)e)!=null) {
            _id=((IdentifierRef)e).getId();
          }
          String _name = null;
          if (_id!=null) {
            _name=_id.getName();
          }
          boolean _equals_1 = Objects.equal(_name, "NaN");
          if (_equals_1) {
            return ConstBoolean.CFalse;
          }
        } else {
          if ((e instanceof StringLiteral)) {
            boolean _isEmpty = ((StringLiteral)e).getValue().isEmpty();
            if (_isEmpty) {
              return ConstBoolean.CFalse;
            } else {
              return ConstBoolean.CTrue;
            }
          } else {
            if ((e instanceof ObjectLiteral)) {
              return ConstBoolean.CTrue;
            }
          }
        }
      }
    }
    return ConstBoolean.NotPrecomputable;
  }
  
  /**
   * 5.5.1. Type Cast, Constraints 61
   *  updated with IDE-928 (IDEBUG-56): Casting to TypeVars
   */
  @Check
  public void checkCastExpression(final CastExpression castExpression) {
    Expression _expression = castExpression.getExpression();
    boolean _tripleEquals = (_expression == null);
    if (_tripleEquals) {
      return;
    }
    final TypeRef S = this.ts.tau(castExpression.getExpression(), castExpression);
    final TypeRef T = castExpression.getTargetTypeRef();
    if (((((S == null) || (T == null)) || (T instanceof UnknownTypeRef)) || (S instanceof UnknownTypeRef))) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(castExpression);
    boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, S, T);
    if (_subtypeSucceeded) {
      this.addIssue(IssueCodes.getMessageForEXP_CAST_UNNECESSARY(S.getTypeRefAsString(), T.getTypeRefAsString()), castExpression, IssueCodes.EXP_CAST_UNNECESSARY);
    } else {
      final boolean specialChecks = (((((((T.getDeclaredType() instanceof ContainerType<?>) || (T.getDeclaredType() instanceof TEnum)) || (T.getDeclaredType() instanceof TypeVariable)) || (T instanceof TypeTypeRef)) || (T instanceof UnionTypeExpression)) || (T instanceof FunctionTypeExpression)) || (T instanceof IntersectionTypeExpression));
      if (specialChecks) {
        this.internalCheckCastExpression(G, S, T, castExpression, true, false);
      } else {
        this.addIssue(IssueCodes.getMessageForEXP_CAST_INVALID_TARGET(), castExpression, IssueCodes.EXP_CAST_INVALID_TARGET);
      }
    }
  }
  
  /**
   * 5.5.1. Type Cast, Constraints 78 (Cast Validation At Compile-Time): 3 and 4
   */
  private boolean internalCheckCastExpression(final RuleEnvironment G, final TypeRef S, final TypeRef T, final CastExpression castExpression, final boolean addIssues, final boolean actualSourceTypeIsCPOE) {
    if ((T instanceof UnionTypeExpression)) {
      final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
        return Boolean.valueOf(this.internalCheckCastExpression(G, S, it, castExpression, false, actualSourceTypeIsCPOE));
      };
      boolean _exists = IterableExtensions.<TypeRef>exists(((UnionTypeExpression)T).getTypeRefs(), _function);
      boolean _not = (!_exists);
      if (_not) {
        if (addIssues) {
          this.addIssue(IssueCodes.getMessageForEXP_CAST_FAILED(S.getTypeRefAsString(), ((UnionTypeExpression)T).getTypeRefAsString()), castExpression, IssueCodes.EXP_CAST_FAILED);
        }
        return false;
      }
    } else {
      if ((T instanceof IntersectionTypeExpression)) {
        final Function1<TypeRef, Boolean> _function_1 = (TypeRef it) -> {
          return Boolean.valueOf(this.internalCheckCastExpression(G, S, it, castExpression, false, actualSourceTypeIsCPOE));
        };
        boolean _forall = IterableExtensions.<TypeRef>forall(((IntersectionTypeExpression)T).getTypeRefs(), _function_1);
        boolean _not_1 = (!_forall);
        if (_not_1) {
          if (addIssues) {
            this.addIssue(IssueCodes.getMessageForEXP_CAST_FAILED(S.getTypeRefAsString(), ((IntersectionTypeExpression)T).getTypeRefAsString()), castExpression, IssueCodes.EXP_CAST_FAILED);
          }
          return false;
        }
      } else {
        if ((S instanceof ComposedTypeRef)) {
          if (((!IterableExtensions.<TypeRef>exists(((ComposedTypeRef)S).getTypeRefs(), ((Function1<TypeRef, Boolean>) (TypeRef it) -> {
            return Boolean.valueOf(this.internalCheckCastExpression(G, it, T, castExpression, false, 
              (actualSourceTypeIsCPOE || ((S instanceof IntersectionTypeExpression) && IterableExtensions.<TypeRef>exists(((ComposedTypeRef)S).getTypeRefs(), ((Function1<TypeRef, Boolean>) (TypeRef it_1) -> {
                return Boolean.valueOf(this.isCPOE(G, it_1));
              }))))));
          }))) && (!IterableExtensions.<TypeRef>exists(((ComposedTypeRef)S).getTypeRefs(), ((Function1<TypeRef, Boolean>) (TypeRef it) -> {
            return Boolean.valueOf(this.ts.subtypeSucceeded(G, it, T));
          }))))) {
            if (addIssues) {
              this.addIssue(IssueCodes.getMessageForEXP_CAST_FAILED(((ComposedTypeRef)S).getTypeRefAsString(), T.getTypeRefAsString()), castExpression, IssueCodes.EXP_CAST_FAILED);
            }
            return false;
          }
        } else {
          boolean _canCheck = this.canCheck(G, S, T, actualSourceTypeIsCPOE);
          if (_canCheck) {
            boolean castOK = this.ts.subtypeSucceeded(G, T, S);
            if (((!castOK) && ((T instanceof ParameterizedTypeRef) && (S instanceof ParameterizedTypeRef)))) {
              final ParameterizedTypeRef ptrT = ((ParameterizedTypeRef) T);
              final ParameterizedTypeRef ptrS = ((ParameterizedTypeRef) S);
              Type _declaredType = ptrS.getDeclaredType();
              Type _declaredType_1 = ptrT.getDeclaredType();
              boolean _equals = Objects.equal(_declaredType, _declaredType_1);
              if (_equals) {
                final int to = ptrS.getTypeArgs().size();
                int _size = ptrT.getTypeArgs().size();
                boolean _tripleEquals = (to == _size);
                if (_tripleEquals) {
                  int i = 0;
                  while (((i < to) && 
                    this.ts.subtypeSucceeded(G, ptrT.getTypeArgs().get(i), ptrS.getTypeArgs().get(i)))) {
                    i++;
                  }
                  if ((i == to)) {
                    castOK = true;
                  } else {
                    i = 0;
                    while (((i < to) && 
                      this.ts.subtypeSucceeded(G, ptrS.getTypeArgs().get(i), ptrT.getTypeArgs().get(i)))) {
                      i++;
                    }
                    castOK = (i == to);
                  }
                }
              }
            }
            if ((!castOK)) {
              if (addIssues) {
                this.addIssue(IssueCodes.getMessageForEXP_CAST_FAILED(S.getTypeRefAsString(), T.getTypeRefAsString()), castExpression, IssueCodes.EXP_CAST_FAILED);
              }
              return false;
            }
          } else {
            if (((T.getDeclaredType() instanceof TypeVariable) && 
              (((TypeVariable) T.getDeclaredType()).getDeclaredUpperBound() != null))) {
              Type _declaredType_2 = T.getDeclaredType();
              final TypeVariable typeVariable = ((TypeVariable) _declaredType_2);
              boolean _internalCheckCastExpression = this.internalCheckCastExpression(G, S, typeVariable.getDeclaredUpperBound(), castExpression, false, actualSourceTypeIsCPOE);
              boolean _not_2 = (!_internalCheckCastExpression);
              if (_not_2) {
                if (addIssues) {
                  this.addIssue(IssueCodes.getMessageForEXP_CAST_FAILED(S.getTypeRefAsString(), T.getTypeRefAsString()), castExpression, IssueCodes.EXP_CAST_FAILED);
                }
                return false;
              }
            }
          }
        }
      }
    }
    return true;
  }
  
  /**
   * @param in case of an intersection type, S may be part of an intersection in which another element is a CPOE, i.e. concrete
   */
  private boolean canCheck(final RuleEnvironment G, final TypeRef S, final TypeRef T, final boolean actualSourceTypeIsCPOE) {
    return (((((T instanceof FunctionTypeExpression) || ((actualSourceTypeIsCPOE || this.isCPOE(G, S)) && this.isCPOE(G, T))) || ((S.getDeclaredType() instanceof TInterface) && this.isActuallyFinal(T))) || (this.isActuallyFinal(S) && (T.getDeclaredType() instanceof TInterface))) || (((S instanceof ParameterizedTypeRef) && (T instanceof ParameterizedTypeRef)) && TypeUtils.isRawSuperType(T.getDeclaredType(), S.getDeclaredType())));
  }
  
  private boolean isCPOE(final RuleEnvironment G, final TypeRef T) {
    Object d = null;
    if ((T instanceof BoundThisTypeRef)) {
      ParameterizedTypeRef _actualThisTypeRef = ((BoundThisTypeRef)T).getActualThisTypeRef();
      Type _declaredType = null;
      if (_actualThisTypeRef!=null) {
        _declaredType=_actualThisTypeRef.getDeclaredType();
      }
      d = _declaredType;
    }
    if ((T instanceof ParameterizedTypeRef)) {
      d = ((ParameterizedTypeRef)T).getDeclaredType();
    }
    if ((T instanceof TypeTypeRef)) {
      d = this.tsh.getStaticType(G, ((TypeTypeRef)T));
    }
    if ((d != null)) {
      final boolean concreteMetaType = ((((d instanceof TClass) || (d instanceof TEnum)) || (d instanceof PrimitiveType)) || (d instanceof TObjectPrototype));
      return concreteMetaType;
    }
    return false;
  }
  
  private boolean isActuallyFinal(final TypeRef typeRef) {
    return (typeRef.isFinalByType() && (!(typeRef.getDeclaredType() instanceof TypeVariable)));
  }
  
  /**
   * 7.1.7. Property Accessors, Constraints 69 (Index Access).
   */
  @Check
  public void checkIndexedAccessExpression(final IndexedAccessExpression indexedAccess) {
    boolean _requireCheckIndexedAccessExpression = this.jsVariantHelper.requireCheckIndexedAccessExpression(indexedAccess);
    boolean _not = (!_requireCheckIndexedAccessExpression);
    if (_not) {
      return;
    }
    final Expression target = indexedAccess.getTarget();
    final Expression index = indexedAccess.getIndex();
    if (((target == null) || (index == null))) {
      return;
    }
    if ((index instanceof IdentifierRef)) {
      if (((((IdentifierRef)index).getId() == null) || ((IdentifierRef)index).getId().eIsProxy())) {
        return;
      }
    }
    if ((index instanceof ParameterizedPropertyAccessExpression)) {
      if (((((ParameterizedPropertyAccessExpression)index).getProperty() == null) || ((ParameterizedPropertyAccessExpression)index).getProperty().eIsProxy())) {
        return;
      }
    }
    if ((target instanceof SuperLiteral)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(indexedAccess);
    final TypeRef targetTypeRefRaw = this.ts.type(G, target);
    if ((targetTypeRefRaw instanceof UnknownTypeRef)) {
      return;
    }
    final TypeRef targetTypeRef = this.ts.resolveType(G, targetTypeRefRaw);
    final TypeRef indexTypeRef = this.ts.type(G, index);
    if ((indexTypeRef instanceof UnknownTypeRef)) {
      return;
    }
    final Type targetDeclType = targetTypeRef.getDeclaredType();
    final boolean targetIsLiteralOfStringBasedEnum = ((targetDeclType instanceof TEnum) && AnnotationDefinition.STRING_BASED.hasAnnotation(targetDeclType));
    final TField accessedBuiltInSymbol = N4JSLanguageUtils.getAccessedBuiltInSymbol(G, index);
    Type _xifexpression = null;
    if ((targetTypeRef instanceof TypeTypeRef)) {
      _xifexpression = this.tsh.getStaticType(G, ((TypeTypeRef)targetTypeRef));
    }
    final Type accessedStaticType = _xifexpression;
    final boolean indexIsNumeric = this.ts.subtypeSucceeded(G, indexTypeRef, RuleEnvironmentExtensions.numberTypeRef(G));
    final CompileTimeValue indexValue = ASTMetaInfoUtils.getCompileTimeValue(index);
    boolean _isDynamic = targetTypeRef.isDynamic();
    if (_isDynamic) {
    } else {
      if (((RuleEnvironmentExtensions.objectType(G) == targetDeclType) && (!targetTypeRef.isUseSiteStructuralTyping()))) {
      } else {
        if ((accessedStaticType instanceof TEnum)) {
          this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_ENUM(), indexedAccess, IssueCodes.EXP_INDEXED_ACCESS_ENUM);
        } else {
          if ((indexIsNumeric && (targetTypeRef.isArrayLike() || targetIsLiteralOfStringBasedEnum))) {
          } else {
            if ((accessedBuiltInSymbol != null)) {
              this.internalCheckIndexedAccessWithSymbol(G, indexedAccess, targetTypeRef, accessedBuiltInSymbol);
            } else {
              boolean _isValid = indexValue.isValid();
              if (_isValid) {
                this.internalCheckComputedIndexedAccess(G, indexedAccess, targetTypeRef, indexValue, indexIsNumeric);
              } else {
                this.createIssuesForEvalErrors(((CompileTimeEvaluationError[])Conversions.unwrapArray(((CompileTimeValue.ValueInvalid) indexValue).getErrors(), CompileTimeEvaluationError.class)));
              }
            }
          }
        }
      }
    }
  }
  
  /**
   * In general computed-names are not allowed as index, unless it denotes a visible member by means of a string-literal.
   * 
   * @return true if allowed, false otherwise.
   */
  private void internalCheckComputedIndexedAccess(final RuleEnvironment G, final IndexedAccessExpression indexedAccess, final TypeRef receiverTypeRef, final CompileTimeValue indexValue, final boolean indexIsNumeric) {
    final String memberName = N4JSLanguageUtils.derivePropertyNameFromCompileTimeValue(indexValue);
    boolean _equals = Objects.equal(ComputedPropertyNameValueConverter.SYMBOL_ITERATOR_MANGLED, memberName);
    if (_equals) {
      this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_IMPL_RESTRICTION(), indexedAccess, 
        IssueCodes.EXP_INDEXED_ACCESS_IMPL_RESTRICTION);
      return;
    }
    boolean _isDynamic = receiverTypeRef.isDynamic();
    if (_isDynamic) {
      return;
    }
    final boolean checkVisibility = true;
    final boolean staticAccess = (receiverTypeRef instanceof TypeTypeRef);
    TypingStrategy _typingStrategy = receiverTypeRef.getTypingStrategy();
    final boolean structFieldInitMode = (_typingStrategy == TypingStrategy.STRUCTURAL_FIELD_INITIALIZER);
    final IScope scope = this.memberScopingHelper.createMemberScope(receiverTypeRef, indexedAccess, checkVisibility, staticAccess, structFieldInitMode);
    IEObjectDescription _xifexpression = null;
    if (((memberName != null) && (!memberName.isEmpty()))) {
      _xifexpression = scope.getSingleElement(this.qualifiedNameConverter.toQualifiedName(memberName));
    }
    final IEObjectDescription memberDesc = _xifexpression;
    EObject _eObjectOrProxy = null;
    if (memberDesc!=null) {
      _eObjectOrProxy=memberDesc.getEObjectOrProxy();
    }
    final EObject member = _eObjectOrProxy;
    final boolean isNonExistentMember = ((member == null) || member.eIsProxy());
    if (isNonExistentMember) {
      if (indexIsNumeric) {
        this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_FORBIDDEN(), indexedAccess, IssueCodes.EXP_INDEXED_ACCESS_FORBIDDEN);
      } else {
        this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_COMPUTED_NOTFOUND(memberName), indexedAccess, 
          IssueCodes.EXP_INDEXED_ACCESS_COMPUTED_NOTFOUND);
      }
      return;
    }
    final IEObjectDescriptionWithError errorDesc = IEObjectDescriptionWithError.getDescriptionWithError(memberDesc);
    if ((errorDesc != null)) {
      this.addIssue(errorDesc.getMessage(), indexedAccess, errorDesc.getIssueCode());
    }
  }
  
  private boolean internalCheckIndexedAccessWithSymbol(final RuleEnvironment G, final IndexedAccessExpression indexedAccess, final TypeRef receiverTypeRef, final TField accessedBuiltInSymbol) {
    final Function1<TMember, Boolean> _function = (TMember it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, "iterator"));
    };
    TMember _findFirst = IterableExtensions.<TMember>findFirst(RuleEnvironmentExtensions.symbolObjectType(G).getOwnedMembers(), _function);
    boolean _tripleNotEquals = (accessedBuiltInSymbol != _findFirst);
    if (_tripleNotEquals) {
      this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_SYMBOL_INVALID(), indexedAccess, 
        IssueCodes.EXP_INDEXED_ACCESS_SYMBOL_INVALID);
      return false;
    }
    final boolean isIterable = this.ts.subtypeSucceeded(G, receiverTypeRef, 
      RuleEnvironmentExtensions.iterableTypeRef(G, TypeRefsFactory.eINSTANCE.createWildcard()));
    final boolean isObjectImmediate = ((receiverTypeRef.getDeclaredType() == RuleEnvironmentExtensions.objectType(G)) && 
      (receiverTypeRef.getTypingStrategy() == TypingStrategy.NOMINAL));
    final boolean isDynamic = receiverTypeRef.isDynamic();
    if ((!((isIterable || isObjectImmediate) || isDynamic))) {
      this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_SYMBOL_WRONG_TYPE(), indexedAccess, 
        IssueCodes.EXP_INDEXED_ACCESS_SYMBOL_WRONG_TYPE);
      return false;
    }
    if ((!(isObjectImmediate || isDynamic))) {
      final boolean writeAccess = ExpressionExtensions.isLeftHandSide(indexedAccess);
      if (writeAccess) {
        this.addIssue(IssueCodes.getMessageForEXP_INDEXED_ACCESS_SYMBOL_READONLY(), indexedAccess, 
          IssueCodes.EXP_INDEXED_ACCESS_SYMBOL_READONLY);
        return false;
      }
    }
    return true;
  }
  
  @Check
  public Object checkAssignmentExpression(final AssignmentExpression assExpr) {
    Object _xblockexpression = null;
    {
      final Expression lhs = assExpr.getLhs();
      boolean _and = false;
      boolean _holdsWritableIdentifier = this.holdsWritableIdentifier(lhs);
      if (!_holdsWritableIdentifier) {
        _and = false;
      } else {
        boolean _holdsLefthandsideNotConst = this.holdsLefthandsideNotConst(lhs);
        _and = _holdsLefthandsideNotConst;
      }
      final Expression rhs = assExpr.getRhs();
      Object _xifexpression = null;
      if ((rhs instanceof IdentifierRef)) {
        Object _xblockexpression_1 = null;
        {
          final IdentifiableElement id = ((IdentifierRef)rhs).getId();
          Object _switchResult = null;
          boolean _matched = false;
          if (id instanceof TMethod) {
            _matched=true;
            _switchResult = null;
          }
          if (!_matched) {
            if (id instanceof TFunction) {
              _matched=true;
              final Procedure4<String, EObject, EStructuralFeature, String> _function = (String message, EObject source, EStructuralFeature feature, String issueCode) -> {
                this.addIssue(message, source, feature, issueCode);
              };
              N4JSExpressionValidator.internalCheckNameRestrictionInMethodBodies(((IdentifierRef)rhs), _function);
            }
          }
          _xblockexpression_1 = _switchResult;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * @return true if nothing was issued
   */
  private boolean holdsLefthandsideNotConst(final Expression lhs) {
    if ((lhs instanceof ParenExpression)) {
      return this.holdsLefthandsideNotConst(((ParenExpression)lhs).getExpression());
    } else {
      if ((lhs instanceof IdentifierRef)) {
        return this.holdsLefthandsideNotConst(((IdentifierRef)lhs));
      }
    }
    return true;
  }
  
  /**
   * @return true if nothing was issued
   */
  private boolean holdsLefthandsideNotConst(final IdentifierRef lhs) {
    final IdentifiableElement id = lhs.getId();
    boolean _matched = false;
    if (id instanceof VariableDeclaration) {
      boolean _isConst = ((VariableDeclaration)id).isConst();
      if (_isConst) {
        _matched=true;
        this.addIssue(IssueCodes.getMessageForEXP_ASSIGN_CONST_VARIABLE(((VariableDeclaration)id).getName()), lhs, 
          IssueCodes.EXP_ASSIGN_CONST_VARIABLE);
        return false;
      }
    }
    if (!_matched) {
      if (id instanceof TVariable) {
        boolean _isConst = ((TVariable)id).isConst();
        if (_isConst) {
          _matched=true;
          this.addIssue(IssueCodes.getMessageForEXP_ASSIGN_CONST_VARIABLE(((TVariable)id).getName()), lhs, 
            IssueCodes.EXP_ASSIGN_CONST_VARIABLE);
          return false;
        }
      }
    }
    if (!_matched) {
      if (id instanceof TField) {
        boolean _isWriteable = ((TField)id).isWriteable();
        boolean _not = (!_isWriteable);
        if (_not) {
          _matched=true;
          this.addIssue(
            IssueCodes.getMessageForVIS_WRONG_READ_WRITE_ACCESS("built-in constant", ((TField)id).getName(), "read-only"), lhs, IssueCodes.VIS_WRONG_READ_WRITE_ACCESS);
          return false;
        }
      }
    }
    return true;
  }
  
  @Check
  public void checkPromisify(final PromisifyExpression promiExpr) {
    boolean _isPromisifiableExpression = this.promisifyHelper.isPromisifiableExpression(promiExpr.getExpression());
    boolean _not = (!_isPromisifiableExpression);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForEXP_PROMISIFY_INVALID_USE(), promiExpr, IssueCodes.EXP_PROMISIFY_INVALID_USE);
    }
  }
  
  /**
   * Ensures that 'this' literals are located at a valid location.
   */
  @Check
  public void checkThisLiteral(final ThisLiteral thisLiteral) {
    ThisArgProvider context = EcoreUtil2.<ThisArgProvider>getContainerOfType(thisLiteral, ThisArgProvider.class);
    while ((context instanceof ArrowFunction)) {
      context = EcoreUtil2.<ThisArgProvider>getContainerOfType(((ArrowFunction)context).eContainer(), ThisArgProvider.class);
    }
    if ((context instanceof N4MemberDeclaration)) {
      final TMember tMember = ((N4MemberDeclaration)context).getDefinedTypeElement();
      boolean _and = false;
      ContainerType<?> _containingType = null;
      if (tMember!=null) {
        _containingType=tMember.getContainingType();
      }
      if (!(_containingType instanceof TInterface)) {
        _and = false;
      } else {
        boolean _isStatic = tMember.isStatic();
        _and = _isStatic;
      }
      if (_and) {
        final String msg = IssueCodes.getMessageForCLF_NO_THIS_IN_STATIC_MEMBER_OF_INTERFACE();
        this.addIssue(msg, thisLiteral, IssueCodes.CLF_NO_THIS_IN_STATIC_MEMBER_OF_INTERFACE);
        return;
      }
    }
    if ((context instanceof N4FieldDeclaration)) {
      final TMember tField = ((N4FieldDeclaration)context).getDefinedTypeElement();
      ContainerType<?> _containingType_1 = null;
      if (tField!=null) {
        _containingType_1=tField.getContainingType();
      }
      if ((_containingType_1 instanceof TInterface)) {
        final String msg_1 = IssueCodes.getMessageForCLF_NO_THIS_IN_FIELD_OF_INTERFACE();
        this.addIssue(msg_1, thisLiteral, IssueCodes.CLF_NO_THIS_IN_FIELD_OF_INTERFACE);
        return;
      }
      ContainerType<?> _containingType_2 = null;
      if (tField!=null) {
        _containingType_2=tField.getContainingType();
      }
      if ((_containingType_2 instanceof TClass)) {
        boolean _isStatic_1 = tField.isStatic();
        if (_isStatic_1) {
          final String msg_2 = IssueCodes.getMessageForCLF_NO_THIS_IN_STATIC_FIELD();
          this.addIssue(msg_2, thisLiteral, IssueCodes.CLF_NO_THIS_IN_STATIC_FIELD);
          return;
        }
      }
    }
  }
  
  @Check
  public void checkMandatoryCompileTimeExpression(final Expression expr) {
    EObject _eContainer = expr.eContainer();
    if ((_eContainer instanceof IndexedAccessExpression)) {
      return;
    }
    boolean _isMandatoryCompileTimeExpression = N4JSLanguageUtils.isMandatoryCompileTimeExpression(expr);
    if (_isMandatoryCompileTimeExpression) {
      final CompileTimeValue evalResult = ASTMetaInfoUtils.getCompileTimeValue(expr);
      if ((evalResult instanceof CompileTimeValue.ValueInvalid)) {
        boolean _isExpressionOfComputedPropertyNameInObjectLiteral = this.isExpressionOfComputedPropertyNameInObjectLiteral(expr);
        if (_isExpressionOfComputedPropertyNameInObjectLiteral) {
          this.addIssue(IssueCodes.getMessageForEXP_COMPUTED_PROP_NAME_DISCOURAGED(), expr, 
            IssueCodes.EXP_COMPUTED_PROP_NAME_DISCOURAGED);
          return;
        }
        this.createIssuesForEvalErrors(((CompileTimeEvaluationError[])Conversions.unwrapArray(((CompileTimeValue.ValueInvalid)evalResult).getErrors(), CompileTimeEvaluationError.class)));
      }
    }
  }
  
  private boolean isExpressionOfComputedPropertyNameInObjectLiteral(final Expression expr) {
    final EObject exprParent = expr.eContainer();
    return (((exprParent instanceof LiteralOrComputedPropertyName) && (exprParent.eContainer() instanceof PropertyAssignment)) && (exprParent.eContainer().eContainer() instanceof ObjectLiteral));
  }
  
  private void createIssuesForEvalErrors(final CompileTimeEvaluationError... errors) {
    for (final CompileTimeEvaluationError error : errors) {
      this.createIssueForEvalError(error);
    }
  }
  
  private void createIssueForEvalError(final CompileTimeEvaluationError error) {
    String _xifexpression = null;
    if ((error instanceof CompileTimeEvaluator.UnresolvedPropertyAccessError)) {
      String _xblockexpression = null;
      {
        final ParameterizedPropertyAccessExpression propAccessExpr = ((CompileTimeEvaluator.UnresolvedPropertyAccessError)error).getAstNodeCasted();
        final IdentifiableElement prop = propAccessExpr.getProperty();
        String _xifexpression_1 = null;
        if (((prop == null) || prop.eIsProxy())) {
          _xifexpression_1 = null;
        } else {
          _xifexpression_1 = "reference must point to a directly owned field (i.e. not inherited, consumed, or polyfilled) and the field must not have a computed name";
        }
        _xblockexpression = _xifexpression_1;
      }
      _xifexpression = _xblockexpression;
    } else {
      _xifexpression = error.message;
    }
    final String message = _xifexpression;
    final EObject astNode = error.astNode;
    final EStructuralFeature feature = error.feature;
    if (((message != null) && (astNode != null))) {
      final String msgFull = IssueCodes.getMessageForEXP_COMPILE_TIME_MANDATORY(message);
      this.addIssue(msgFull, astNode, feature, IssueCodes.EXP_COMPILE_TIME_MANDATORY);
    }
  }
}
