/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.AnnotationList;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.EmptyStatement;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4EnumLiteral;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.util.Triple;
import org.eclipse.xtext.util.Tuples;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class N4JSExternalValidator extends AbstractN4JSDeclarativeValidator {
  /**
   * Helper structure for 13.1. constraints validation
   */
  private final List<? extends Triple<AnnotationDefinition, String, ? extends Class<? extends N4ClassifierDeclaration>>> PROVIDES_ANNOTATION_INFO = Collections.<Triple<AnnotationDefinition, String, ? extends Class<? extends N4ClassifierDeclaration>>>unmodifiableList(CollectionLiterals.<Triple<AnnotationDefinition, String, ? extends Class<? extends N4ClassifierDeclaration>>>newArrayList(Tuples.<AnnotationDefinition, String, Class<N4InterfaceDeclaration>>create(AnnotationDefinition.PROVIDES_DEFAULT_IMPLEMENTATION, "interfaces", N4InterfaceDeclaration.class), Tuples.<AnnotationDefinition, String, Class<N4ClassifierDeclaration>>create(AnnotationDefinition.PROVIDES_INITIALZER, "classifiers", N4ClassifierDeclaration.class)));
  
  @Inject
  private IN4JSCore n4jsCore;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * @see N4Spec, 4.20. External Classes
   */
  @Check
  public void checkExternalClassesDefinedInN4JSDFile(final N4ClassDeclaration clazz) {
    boolean _holdsExternalOnlyInDefinitionFile = this.holdsExternalOnlyInDefinitionFile(clazz, "Classes");
    boolean _not = (!_holdsExternalOnlyInDefinitionFile);
    if (_not) {
      return;
    }
  }
  
  /**
   * @see N4Spec, 4.20. External Interfaces
   */
  @Check
  public void checkExternalInterfacesDefinedInN4JSDFile(final N4InterfaceDeclaration interfaceDecl) {
    boolean _holdsExternalOnlyInDefinitionFile = this.holdsExternalOnlyInDefinitionFile(interfaceDecl, "Interfaces");
    boolean _not = (!_holdsExternalOnlyInDefinitionFile);
    if (_not) {
      return;
    }
  }
  
  /**
   * @see N4Spec, 4.20. External Enums
   */
  @Check
  public void checkExternalEnumsDefinedInN4JSDFile(final N4EnumDeclaration enumDecl) {
    boolean _holdsExternalOnlyInDefinitionFile = this.holdsExternalOnlyInDefinitionFile(enumDecl, "Enumerations");
    boolean _not = (!_holdsExternalOnlyInDefinitionFile);
    if (_not) {
      return;
    }
  }
  
  private boolean holdsExternalOnlyInDefinitionFile(final N4TypeDeclaration typeDecl, final String typesName) {
    if ((typeDecl.isExternal() && (!this.jsVariantHelper.isExternalMode(typeDecl)))) {
      final String message = IssueCodes.getMessageForCLF_EXT_EXTERNAL_N4JSD(typesName);
      this.addIssue(message, typeDecl, N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, IssueCodes.CLF_EXT_EXTERNAL_N4JSD);
      return false;
    }
    return true;
  }
  
  /**
   * 13.1 ExternalDeclarations, Constraints 138.4: External class/interface members {@code @ProvidesDefaultImplementation}.
   */
  @Check
  public void checkProvidesDefaultImplementationAnnotation(final N4MemberDeclaration memberDecl) {
    String _name = memberDecl.getName();
    boolean _tripleEquals = (_name == null);
    if (_tripleEquals) {
      return;
    }
    for (final Triple<AnnotationDefinition, String, ? extends Class<? extends N4ClassifierDeclaration>> valInfo : this.PROVIDES_ANNOTATION_INFO) {
      {
        if (((AnnotationDefinition.PROVIDES_INITIALZER == valInfo.getFirst()) && 
          (memberDecl instanceof N4SetterDeclaration))) {
          return;
        }
        final AnnotationDefinition annodef = valInfo.getFirst();
        final String typeName = valInfo.getSecond();
        final Class<?> metaType = valInfo.getThird();
        boolean _hasAnnotation = annodef.hasAnnotation(memberDecl);
        if (_hasAnnotation) {
          boolean _isExternalMode = this.jsVariantHelper.isExternalMode(memberDecl);
          boolean _not = (!_isExternalMode);
          if (_not) {
            final String msg = IssueCodes.getMessageForCLF_EXT_PROVIDES_IMPL_ONLY_IN_DEFFILES(annodef.name, typeName);
            this.addIssue(msg, memberDecl, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
              IssueCodes.CLF_EXT_PROVIDES_IMPL_ONLY_IN_DEFFILES);
            return;
          }
          final N4ClassifierDefinition owner = memberDecl.getOwner();
          boolean _isInstance = metaType.isInstance(owner);
          boolean _not_1 = (!_isInstance);
          if (_not_1) {
            final String msg_1 = IssueCodes.getMessageForCLF_EXT_PROVIDES_IMPL_ONLY_IN_INTERFACE_MEMBERS(annodef.name, typeName);
            this.addIssue(msg_1, memberDecl, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
              IssueCodes.CLF_EXT_PROVIDES_IMPL_ONLY_IN_INTERFACE_MEMBERS);
            return;
          }
          boolean _hasAnnotation_1 = AnnotationDefinition.N4JS.hasAnnotation(owner);
          boolean _not_2 = (!_hasAnnotation_1);
          if (_not_2) {
            final String msg_2 = IssueCodes.getMessageForCLF_EXT_PROVIDES_IMPL_ONLY_IN_N4JS_INTERFACES(annodef.name, typeName);
            this.addIssue(msg_2, memberDecl, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
              IssueCodes.CLF_EXT_PROVIDES_IMPL_ONLY_IN_N4JS_INTERFACES);
            return;
          }
        }
      }
    }
  }
  
  /**
   * @see N4Spec, 4.20. External Classes and Roles
   */
  @Check
  public void checkExternalFunctionsDefinedInN4JSDFile(final FunctionDeclaration funDecl) {
    if ((funDecl.isExternal() && (!this.jsVariantHelper.isExternalMode(funDecl)))) {
      final String message = IssueCodes.getMessageForCLF_EXT_EXTERNAL_N4JSD("Functions");
      this.addIssue(message, funDecl, N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, IssueCodes.CLF_EXT_EXTERNAL_N4JSD);
    }
  }
  
  /**
   * No assignment in n4jsd.
   */
  @Check
  public void checkExternalVariableStatementAssigments(final ExportedVariableStatement variableStatement) {
    if ((variableStatement.isExternal() && 
      (!this.jsVariantHelper.isExternalMode(variableStatement)))) {
      final String message = IssueCodes.getMessageForCLF_EXT_EXTERNAL_N4JSD("Variables");
      this.addIssue(message, variableStatement, IssueCodes.CLF_EXT_EXTERNAL_N4JSD);
    }
    final Consumer<VariableDeclaration> _function = (VariableDeclaration evd) -> {
      if ((this.jsVariantHelper.isExternalMode(evd) && (evd.getExpression() != null))) {
        String _xifexpression = null;
        boolean _isConst = evd.isConst();
        if (_isConst) {
          _xifexpression = "constant";
        } else {
          _xifexpression = "variable";
        }
        final String mod = _xifexpression;
        final String message_1 = IssueCodes.getMessageForCLF_EXT_VAR_NO_VAL(mod);
        this.addIssue(message_1, evd, IssueCodes.CLF_EXT_VAR_NO_VAL);
      }
    };
    variableStatement.getVarDecl().forEach(_function);
  }
  
  /**
   * @see N4Spec, 4.20. External Classes and Roles
   */
  @Check
  public Object checkAllowedElementsInN4JSDFile(final EObject eo) {
    Object _xifexpression = null;
    if ((this.jsVariantHelper.isExternalMode(eo) && (eo.eContainer() instanceof Script))) {
      Object _xblockexpression = null;
      {
        final boolean found = this.findUnallowedElement(eo);
        Object _xifexpression_1 = null;
        if (found) {
          this.handleUnallowedElement(eo);
        } else {
          Object _xifexpression_2 = null;
          if ((eo instanceof ExportDeclaration)) {
            this.handleExportDeclaration(((ExportDeclaration)eo));
          } else {
            Object _xifexpression_3 = null;
            if ((!((((eo instanceof AnnotationList) || (eo instanceof Annotation)) || (eo instanceof EmptyStatement)) || 
              (eo instanceof ImportDeclaration)))) {
              _xifexpression_3 = null;
            }
            _xifexpression_2 = _xifexpression_3;
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xblockexpression = _xifexpression_1;
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  private void handleExportDeclaration(final ExportDeclaration eo) {
    final ExportableElement exported = eo.getExportedElement();
    this.holdsExternalImplementation(exported);
    boolean _matched = false;
    if (exported instanceof N4ClassDeclaration) {
      _matched=true;
      this.handleN4ClassDeclaration(eo, ((N4ClassDeclaration)exported));
    }
    if (!_matched) {
      if (exported instanceof N4InterfaceDeclaration) {
        _matched=true;
        this.handleN4InterfaceDeclaration(eo, ((N4InterfaceDeclaration)exported));
      }
    }
    if (!_matched) {
      if (exported instanceof N4EnumDeclaration) {
        _matched=true;
        this.handleN4EnumDeclaration(eo, ((N4EnumDeclaration)exported));
      }
    }
    if (!_matched) {
      if (exported instanceof FunctionDeclaration) {
        _matched=true;
        this.handleFunctionDeclaration(eo, ((FunctionDeclaration)exported));
      }
    }
  }
  
  private void handleN4ClassDeclaration(final ExportDeclaration eo, final N4ClassDeclaration exported) {
    this.validateClassifierIsExternal(exported.isExternal(), "classes", eo);
    boolean _hasAnnotation = AnnotationDefinition.N4JS.hasAnnotation(exported);
    if (_hasAnnotation) {
      this.validateExtensionOfNotAnnotatedClass(exported, eo);
      this.validateConsumptionOfNonAnnotatedInterfaces(exported.getImplementedInterfaceRefs(), eo, "classes");
    } else {
      ParameterizedTypeRef _superClassRef = exported.getSuperClassRef();
      TClass _hasExpectedTypes = null;
      if (_superClassRef!=null) {
        _hasExpectedTypes=this.<TClass>hasExpectedTypes(_superClassRef, TClass.class);
      }
      final TClass superClass = _hasExpectedTypes;
      this.validateNonAnnotatedClassDoesntExtendN4Object(exported, superClass, eo);
      this.validateConsumptionOfNonExternalInterfaces(exported.getImplementedInterfaceRefs(), eo, "classes");
    }
    this.validateNoObservableAtClassifier(eo, exported, "classes");
    this.validateMembers(exported, "classes");
  }
  
  private void handleN4InterfaceDeclaration(final ExportDeclaration eo, final N4InterfaceDeclaration exported) {
    if ((Objects.equal(exported.getTypingStrategy(), TypingStrategy.NOMINAL) || Objects.equal(exported.getTypingStrategy(), TypingStrategy.DEFAULT))) {
      this.validateClassifierIsExternal(exported.isExternal(), "interfaces", eo);
    }
    boolean _hasAnnotation = AnnotationDefinition.N4JS.hasAnnotation(exported);
    if (_hasAnnotation) {
      this.validateConsumptionOfNonAnnotatedInterfaces(exported.getSuperInterfaceRefs(), eo, "interfaces");
    } else {
      this.validateConsumptionOfNonExternalInterfaces(exported.getSuperInterfaceRefs(), eo, "interfaces");
    }
    this.validateNoObservableAtClassifier(eo, exported, "interfaces");
    this.validateMembers(exported, "interfaces");
  }
  
  private void handleN4EnumDeclaration(final ExportDeclaration eo, final N4EnumDeclaration exported) {
    boolean _isStringBased = this.isStringBased(exported);
    boolean _not = (!_isStringBased);
    if (_not) {
      final Function1<N4EnumLiteral, Boolean> _function = (N4EnumLiteral it) -> {
        String _value = it.getValue();
        return Boolean.valueOf((_value != null));
      };
      Iterable<N4EnumLiteral> _filter = IterableExtensions.<N4EnumLiteral>filter(exported.getLiterals(), _function);
      for (final N4EnumLiteral literal : _filter) {
        {
          final String message = IssueCodes.getMessageForCLF_EXT_LITERAL_NO_VALUE();
          this.addIssue(message, literal, N4JSPackage.Literals.N4_ENUM_LITERAL__VALUE, IssueCodes.CLF_EXT_LITERAL_NO_VALUE);
        }
      }
    }
  }
  
  private void handleFunctionDeclaration(final ExportDeclaration eo, final FunctionDeclaration exported) {
    Block _body = exported.getBody();
    boolean _tripleNotEquals = (_body != null);
    if (_tripleNotEquals) {
      final String message = IssueCodes.getMessageForCLF_EXT_FUN_NO_BODY();
      final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_FUN_NO_BODY);
    }
  }
  
  /**
   * Constraints 120 (Implementation of External Declarations)
   */
  private boolean holdsExternalImplementation(final ExportableElement element) {
    if ((element instanceof AnnotableElement)) {
      boolean _hasAnnotation = AnnotationDefinition.IGNORE_IMPLEMENTATION.hasAnnotation(((AnnotableElement)element));
      if (_hasAnnotation) {
        return true;
      }
      boolean _hasAnnotation_1 = AnnotationDefinition.PROVIDED_BY_RUNTIME.hasAnnotation(((AnnotableElement)element));
      if (_hasAnnotation_1) {
        return this.holdsDefinedInRuntime(element);
      }
    }
    return true;
  }
  
  /**
   * Constraints 70. 1
   */
  private boolean holdsDefinedInRuntime(final ExportableElement element) {
    Resource _eResource = null;
    if (element!=null) {
      _eResource=element.eResource();
    }
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    IN4JSProject _orNull = this.n4jsCore.findProject(_uRI).orNull();
    ProjectType _projectType = null;
    if (_orNull!=null) {
      _projectType=_orNull.getProjectType();
    }
    final ProjectType projectType = _projectType;
    if ((((projectType == null) || (projectType == ProjectType.RUNTIME_ENVIRONMENT)) || 
      (projectType == ProjectType.RUNTIME_LIBRARY))) {
      return true;
    }
    final String message = IssueCodes.getMessageForCLF_EXT_PROVIDED_BY_RUNTIME_IN_RUNTIME_TYPE();
    final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(element);
    this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), 
      IssueCodes.CLF_EXT_PROVIDED_BY_RUNTIME_IN_RUNTIME_TYPE);
    return false;
  }
  
  private void validateNoObservableAtClassifier(final ExportDeclaration ed, final N4ClassifierDeclaration declaration, final String classesOrRolesOrInterface) {
    boolean _hasAnnotation = AnnotationDefinition.OBSERVABLE.hasAnnotation(ed);
    if (_hasAnnotation) {
      final String message = IssueCodes.getMessageForCLF_EXT_NO_OBSERV_ANNO(classesOrRolesOrInterface);
      final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(declaration);
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_NO_OBSERV_ANNO);
    }
  }
  
  private void validateMembers(final N4ClassifierDeclaration declaration, final String classesOrRolesOrInterface) {
    this.validateNoBody(declaration, classesOrRolesOrInterface);
    this.validateNoFieldExpression(declaration, classesOrRolesOrInterface);
    this.validateNoObservableAtMember(declaration, classesOrRolesOrInterface);
    this.validateNoNfonAtMember(declaration, classesOrRolesOrInterface);
  }
  
  private void validateNoObservableAtMember(final N4ClassifierDeclaration declaration, final String classesOrRolesOrInterface) {
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf(AnnotationDefinition.OBSERVABLE.hasAnnotation(it));
    };
    Iterable<N4MemberDeclaration> _filter = IterableExtensions.<N4MemberDeclaration>filter(declaration.getOwnedMembers(), _function);
    for (final N4MemberDeclaration member : _filter) {
      {
        String _firstUpper = StringExtensions.toFirstUpper(this.keywordProvider.keyword(member));
        String _plus = (_firstUpper + "s");
        final String message = IssueCodes.getMessageForCLF_EXT_METHOD_NO_ANNO(_plus, classesOrRolesOrInterface, "Observable");
        this.addIssue(message, member, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_EXT_METHOD_NO_ANNO);
      }
    }
  }
  
  private void validateNoNfonAtMember(final N4ClassifierDeclaration declaration, final String classesOrRolesOrInterface) {
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf(AnnotationDefinition.NFON.hasAnnotation(it));
    };
    Iterable<N4MemberDeclaration> _filter = IterableExtensions.<N4MemberDeclaration>filter(declaration.getOwnedMembers(), _function);
    for (final N4MemberDeclaration member : _filter) {
      {
        String _firstUpper = StringExtensions.toFirstUpper(this.keywordProvider.keyword(member));
        String _plus = (_firstUpper + "s");
        final String message = IssueCodes.getMessageForCLF_EXT_METHOD_NO_ANNO(_plus, classesOrRolesOrInterface, "Nfon");
        this.addIssue(message, member, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_EXT_METHOD_NO_ANNO);
      }
    }
  }
  
  private void validateNoFieldExpression(final N4ClassifierDeclaration declaration, final String classesOrRolesOrInterface) {
    final Function1<N4FieldDeclaration, Boolean> _function = (N4FieldDeclaration it) -> {
      Expression _expression = it.getExpression();
      return Boolean.valueOf((_expression != null));
    };
    Iterable<N4FieldDeclaration> _filter = IterableExtensions.<N4FieldDeclaration>filter(declaration.getOwnedFields(), _function);
    for (final N4FieldDeclaration member : _filter) {
      {
        final String message = IssueCodes.getMessageForCLF_EXT_NO_FIELD_EXPR(classesOrRolesOrInterface);
        this.addIssue(message, member, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_EXT_NO_FIELD_EXPR);
      }
    }
  }
  
  private void validateNoBody(final N4ClassifierDeclaration declaration, final String classesOrRolesOrInterface) {
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf((!(it instanceof N4FieldDeclaration)));
    };
    final Function1<N4MemberDeclaration, Boolean> _function_1 = (N4MemberDeclaration it) -> {
      Block _body = this.getBody(it);
      return Boolean.valueOf((_body != null));
    };
    Iterable<N4MemberDeclaration> _filter = IterableExtensions.<N4MemberDeclaration>filter(IterableExtensions.<N4MemberDeclaration>filter(declaration.getOwnedMembers(), _function), _function_1);
    for (final N4MemberDeclaration member : _filter) {
      {
        String _firstUpper = StringExtensions.toFirstUpper(this.keywordProvider.keyword(member));
        String _plus = (_firstUpper + "s");
        final String message = IssueCodes.getMessageForCLF_EXT_NO_METHOD_BODY(_plus, classesOrRolesOrInterface);
        this.addIssue(message, member, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_EXT_NO_METHOD_BODY);
      }
    }
  }
  
  private void validateNonAnnotatedClassDoesntExtendN4Object(final N4ClassDeclaration exported, final TClass superType, final ExportDeclaration eo) {
    if (((superType != null) && (!superType.isExternal()))) {
      final String message = IssueCodes.getMessageForCLF_EXT_NOT_ANNOTATED_EXTEND_N4OBJECT();
      final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), 
        IssueCodes.CLF_EXT_NOT_ANNOTATED_EXTEND_N4OBJECT);
    }
  }
  
  private void validateClassifierIsExternal(final boolean external, final String classifiers, final ExportDeclaration eo) {
    if ((!external)) {
      final String message = IssueCodes.getMessageForCLF_EXT_EXTERNAL(classifiers);
      final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_EXTERNAL);
    }
  }
  
  private void validateExtensionOfNotAnnotatedClass(final N4ClassDeclaration exportedWithN4JSAnnotation, final ExportDeclaration eo) {
    final TClass superClass = this.getSuperClassType(exportedWithN4JSAnnotation);
    if ((superClass != null)) {
      boolean _isExternal = superClass.isExternal();
      if (_isExternal) {
        this.validateSuperClassAnnotatedWithN4JS(superClass, eo);
      }
    } else {
      ParameterizedTypeRef _superClassRef = exportedWithN4JSAnnotation.getSuperClassRef();
      Type _declaredType = null;
      if (_superClassRef!=null) {
        _declaredType=_superClassRef.getDeclaredType();
      }
      final Type superType = _declaredType;
      if ((((superType != null) && (!"N4Object".equals(superType.getName()))) && (!this.isSubtypeOfError(superType)))) {
        this.handleSuperClassNotAnnotatedWithN4JS(eo);
      }
    }
  }
  
  /**
   * Returns with {@code true} if the type argument is a subtype of Error. Could be direct or implicit subtype as well.
   * 13.1 ExternalDeclarations, Constraints 144/c (External allowed occurrences)
   * 
   * @see IDEBUG-512
   */
  private boolean isSubtypeOfError(final Type type) {
    final Function1<Type, TObjectPrototype> _function = (Type t) -> {
      TObjectPrototype _xifexpression = null;
      if ((t instanceof TObjectPrototype)) {
        _xifexpression = ((TObjectPrototype)t);
      } else {
        _xifexpression = null;
      }
      return _xifexpression;
    };
    Function1<Type, TObjectPrototype> toPrototype = _function;
    TObjectPrototype prototype = toPrototype.apply(type);
    while ((null != prototype)) {
      {
        final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(prototype);
        TObjectPrototype _errorType = RuleEnvironmentExtensions.errorType(G);
        boolean _tripleEquals = (_errorType == prototype);
        if (_tripleEquals) {
          return true;
        }
        ParameterizedTypeRef _superType = null;
        if (prototype!=null) {
          _superType=prototype.getSuperType();
        }
        Type _declaredType = null;
        if (_superType!=null) {
          _declaredType=_superType.getDeclaredType();
        }
        prototype = toPrototype.apply(_declaredType);
      }
    }
    return false;
  }
  
  private void validateSuperClassAnnotatedWithN4JS(final TClass classifier, final ExportDeclaration eo) {
    boolean _hasAnnotation = AnnotationDefinition.N4JS.hasAnnotation(classifier);
    boolean _not = (!_hasAnnotation);
    if (_not) {
      this.handleSuperClassNotAnnotatedWithN4JS(eo);
    }
  }
  
  private void handleSuperClassNotAnnotatedWithN4JS(final ExportDeclaration eo) {
    final String message = IssueCodes.getMessageForCLF_EXT_ANNOTATED_EXTEND();
    final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
    this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_ANNOTATED_EXTEND);
  }
  
  private TClass getSuperClassType(final N4ClassDeclaration exported) {
    ParameterizedTypeRef _superClassRef = exported.getSuperClassRef();
    TClass _hasExpectedTypes = null;
    if (_superClassRef!=null) {
      _hasExpectedTypes=this.<TClass>hasExpectedTypes(_superClassRef, TClass.class);
    }
    return _hasExpectedTypes;
  }
  
  private void validateConsumptionOfNonAnnotatedInterfaces(final Iterable<ParameterizedTypeRef> superInterfaces, final ExportDeclaration eo, final String classifiers) {
    final Function1<ParameterizedTypeRef, TInterface> _function = (ParameterizedTypeRef it) -> {
      return this.<TInterface>hasExpectedTypes(it, TInterface.class);
    };
    final Function1<TInterface, Boolean> _function_1 = (TInterface it) -> {
      return Boolean.valueOf((it != null));
    };
    Iterable<TInterface> _filter = IterableExtensions.<TInterface>filter(IterableExtensions.<ParameterizedTypeRef, TInterface>map(superInterfaces, _function), _function_1);
    for (final TInterface tinterface : _filter) {
      {
        this.validateConsumptionOfNonExternalInterface(tinterface, classifiers, eo);
        this.validateConsumptionOfNonAnnotatedRole(tinterface, classifiers, eo);
      }
    }
  }
  
  private void validateConsumptionOfNonAnnotatedRole(final TInterface consumedRole, final String classifiers, final ExportDeclaration eo) {
    if (((!AnnotationDefinition.N4JS.hasAnnotation(consumedRole)) && 
      (!Objects.equal(consumedRole.getTypingStrategy(), TypingStrategy.STRUCTURAL)))) {
      final String message = IssueCodes.getMessageForCLF_EXT_ANNOTATED_CONSUME(classifiers);
      final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_ANNOTATED_CONSUME);
    }
  }
  
  private void validateConsumptionOfNonExternalInterfaces(final Iterable<ParameterizedTypeRef> superInterfaces, final ExportDeclaration eo, final String classifiers) {
    final Function1<ParameterizedTypeRef, TInterface> _function = (ParameterizedTypeRef it) -> {
      return this.<TInterface>hasExpectedTypes(it, TInterface.class);
    };
    final Function1<TInterface, Boolean> _function_1 = (TInterface it) -> {
      return Boolean.valueOf((it != null));
    };
    Iterable<TInterface> _filter = IterableExtensions.<TInterface>filter(IterableExtensions.<ParameterizedTypeRef, TInterface>map(superInterfaces, _function), _function_1);
    for (final TInterface tinterface : _filter) {
      this.validateConsumptionOfNonExternalInterface(tinterface, classifiers, eo);
    }
  }
  
  private void validateConsumptionOfNonExternalInterface(final TInterface tinterface, final String classifiers, final ExportDeclaration eo) {
    if (((!tinterface.isExternal()) && (tinterface.getTypingStrategy() != TypingStrategy.STRUCTURAL))) {
      final String message = IssueCodes.getMessageForCLF_EXT_CONSUME_NON_EXT(classifiers);
      final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_CONSUME_NON_EXT);
    }
  }
  
  private void handleUnallowedElement(final EObject eo) {
    final String message = IssueCodes.getMessageForCLF_EXT_UNALLOWED_N4JSD();
    final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(eo);
    if ((eObjectToNameFeature == null)) {
      final Pair<Integer, Integer> offsetAndLength = this.findOffsetAndLength(eo);
      this.addIssue(message, eo, (offsetAndLength.getKey()).intValue(), (offsetAndLength.getValue()).intValue(), IssueCodes.CLF_EXT_UNALLOWED_N4JSD);
    } else {
      this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_EXT_UNALLOWED_N4JSD);
    }
  }
  
  /**
   * Returns converted declared type of type reference if it matches the expected type. Otherwise, null is returned.
   */
  private <T extends Type> T hasExpectedTypes(final ParameterizedTypeRef typeRef, final Class<T> typeClazz) {
    Type _declaredType = null;
    if (typeRef!=null) {
      _declaredType=typeRef.getDeclaredType();
    }
    final Type type = _declaredType;
    if (((type != null) && typeClazz.isAssignableFrom(type.getClass()))) {
      return ((T) type);
    }
    return null;
  }
  
  private boolean findUnallowedElement(final EObject eo) {
    if ((eo instanceof EmptyStatement)) {
      return false;
    }
    if ((eo instanceof ImportDeclaration)) {
      return false;
    }
    if ((eo instanceof Annotation)) {
      return false;
    }
    if ((eo instanceof ExportDeclaration)) {
      return this.findUnallowedElement(((ExportDeclaration)eo).getExportedElement());
    }
    if ((eo instanceof N4ClassDeclaration)) {
      boolean _isExternal = ((N4ClassDeclaration)eo).isExternal();
      if (_isExternal) {
        return false;
      }
    }
    if ((eo instanceof N4InterfaceDeclaration)) {
      if ((((N4InterfaceDeclaration)eo).isExternal() || Objects.equal(((N4InterfaceDeclaration)eo).getTypingStrategy(), TypingStrategy.STRUCTURAL))) {
        return false;
      }
    }
    if ((eo instanceof N4EnumDeclaration)) {
      boolean _isExternal_1 = ((N4EnumDeclaration)eo).isExternal();
      if (_isExternal_1) {
        return false;
      }
    }
    if ((eo instanceof FunctionDeclaration)) {
      boolean _isExternal_2 = ((FunctionDeclaration)eo).isExternal();
      if (_isExternal_2) {
        return false;
      }
    }
    if ((eo instanceof VariableStatement)) {
      return false;
    }
    return true;
  }
  
  private boolean isStringBased(final N4EnumDeclaration enumDecl) {
    return AnnotationDefinition.STRING_BASED.hasAnnotation(enumDecl);
  }
}
