/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.ArrayListMultimap;
import com.google.inject.Inject;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.AssignmentOperator;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionAnnotationList;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.GenericDeclaration;
import org.eclipse.n4js.n4JS.GetterDeclaration;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.n4JS.TypedElement;
import org.eclipse.n4js.n4JS.UnaryExpression;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.extensions.ExpressionExtensions;
import org.eclipse.n4js.scoping.N4JSScopeProvider;
import org.eclipse.n4js.scoping.members.TypingStrategyFilter;
import org.eclipse.n4js.scoping.utils.AbstractDescriptionWithError;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.AnyType;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.VoidType;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.Result;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Class for validating the N4JS types.
 */
@SuppressWarnings("all")
public class N4JSTypeValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private N4JSScopeProvider n4jsScopeProvider;
  
  @Inject
  protected ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Validates all generic type variable declarations.
   * Raises validation error for each type variable if any of their declared upper bound is a primitive type.
   * <p>
   * IDEBUG-185
   * 
   * @param declaration the generic declaration to check the upper bound declarations of its type variables.
   */
  @Check
  public void checkGenericDeclarationType(final GenericDeclaration declaration) {
    final TObjectPrototype functionType = RuleEnvironmentExtensions.functionType(RuleEnvironmentExtensions.newRuleEnvironment(declaration));
    final Consumer<TypeVariable> _function = (TypeVariable typeVar) -> {
      final TypeRef ub = typeVar.getDeclaredUpperBound();
      if ((ub != null)) {
        final Type declType = ub.getDeclaredType();
        if (((declType instanceof ContainerType<?>) && declType.isFinal())) {
          if ((declType == functionType)) {
            return;
          }
          final String message = IssueCodes.getMessageForCLF_UPPER_BOUND_FINAL(declType.getName(), typeVar.getName());
          this.addIssue(message, ub, TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE, IssueCodes.CLF_UPPER_BOUND_FINAL);
        }
      }
    };
    IterableExtensions.<TypeVariable>filterNull(declaration.getTypeVars()).forEach(_function);
  }
  
  @Check
  public void checkTModuleCreated(final Script script) {
    TModule _module = script.getModule();
    boolean _tripleEquals = (_module == null);
    if (_tripleEquals) {
      final ICompositeNode rootNode = NodeModelUtils.getNode(script);
      if ((rootNode != null)) {
        this.addIssue(IssueCodes.getMessageForTYS_MISSING(), script, rootNode.getOffset(), rootNode.getLength(), IssueCodes.TYS_MISSING);
      }
    }
  }
  
  @Check
  public void checkParameterizedTypeRef(final ParameterizedTypeRef paramTypeRef) {
    final Type declaredType = paramTypeRef.getDeclaredType();
    if (((declaredType == null) || declaredType.eIsProxy())) {
      return;
    }
    if ((declaredType instanceof TFunction)) {
      EObject _eContainer = paramTypeRef.eContainer();
      boolean _not = (!(_eContainer instanceof TypeTypeRef));
      if (_not) {
        this.addIssue(IssueCodes.getMessageForTYS_FUNCTION_DISALLOWED_AS_TYPE(), paramTypeRef, IssueCodes.TYS_FUNCTION_DISALLOWED_AS_TYPE);
        return;
      }
    }
    final boolean isInTypeTypeRef = ((paramTypeRef.eContainer() instanceof TypeTypeRef) || 
      ((paramTypeRef.eContainer() instanceof Wildcard) && (paramTypeRef.eContainer().eContainer() instanceof TypeTypeRef)));
    this.internalCheckValidLocationForVoid(paramTypeRef);
    if (isInTypeTypeRef) {
      this.internalCheckValidTypeInTypeTypeRef(paramTypeRef);
    } else {
      this.internalCheckTypeArguments(declaredType.getTypeVars(), paramTypeRef.getTypeArgs(), false, declaredType, paramTypeRef, 
        TypeRefsPackage.eINSTANCE.getParameterizedTypeRef_DeclaredType());
    }
    this.internalCheckDynamic(paramTypeRef);
    this.internalCheckStructuralPrimitiveTypeRef(paramTypeRef);
  }
  
  /**
   * Add an issue if explicit use of structural type operator with a primitive type is detected.
   */
  private void internalCheckStructuralPrimitiveTypeRef(final ParameterizedTypeRef typeRef) {
    if (((typeRef.getDeclaredType() instanceof PrimitiveType) && (!Objects.equal(typeRef.getTypingStrategy(), TypingStrategy.NOMINAL)))) {
      this.addIssue(IssueCodes.getMessageForTYS_STRUCTURAL_PRIMITIVE(), typeRef, IssueCodes.TYS_STRUCTURAL_PRIMITIVE);
    }
  }
  
  /**
   * Requirements 13, Void type.
   */
  private void internalCheckValidLocationForVoid(final ParameterizedTypeRef typeRef) {
    Type _declaredType = typeRef.getDeclaredType();
    if ((_declaredType instanceof VoidType)) {
      final boolean isValidLocationForVoid = (((((typeRef.eContainer() instanceof FunctionDefinition) && 
        (typeRef.eContainmentFeature() == N4JSPackage.eINSTANCE.getFunctionDefinition_ReturnTypeRef())) || 
        ((typeRef.eContainer() instanceof FunctionTypeExpression) && 
          (typeRef.eContainmentFeature() == TypeRefsPackage.eINSTANCE.getFunctionTypeExpression_ReturnTypeRef()))) || 
        ((typeRef.eContainer() instanceof TFunction) && (typeRef.eContainmentFeature() == TypesPackage.eINSTANCE.getTFunction_ReturnTypeRef()))) || 
        ((typeRef.eContainer() instanceof GetterDeclaration) && 
          (typeRef.eContainmentFeature() == N4JSPackage.eINSTANCE.getTypedElement_DeclaredTypeRef())));
      if ((!isValidLocationForVoid)) {
        this.addIssue(IssueCodes.getMessageForTYS_VOID_AT_WRONG_LOCATION(), typeRef, IssueCodes.TYS_VOID_AT_WRONG_LOCATION);
      }
    }
  }
  
  private void internalCheckValidTypeInTypeTypeRef(final ParameterizedTypeRef paramTypeRef) {
    boolean _isEmpty = paramTypeRef.getTypeArgs().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForAST_NO_TYPE_ARGS_IN_CLASSIFIERTYPEREF(), paramTypeRef, 
        IssueCodes.AST_NO_TYPE_ARGS_IN_CLASSIFIERTYPEREF);
    } else {
      if ((paramTypeRef instanceof FunctionTypeRef)) {
        this.addIssue(IssueCodes.getMessageForAST_NO_FUNCTIONTYPEREFS_IN_CLASSIFIERTYPEREF(), paramTypeRef, 
          IssueCodes.AST_NO_FUNCTIONTYPEREFS_IN_CLASSIFIERTYPEREF);
      } else {
        Type _declaredType = paramTypeRef.getDeclaredType();
        if ((_declaredType instanceof TFunction)) {
          this.addIssue(IssueCodes.getMessageForAST_NO_FUNCTIONTYPEREFS_IN_CLASSIFIERTYPEREF(), paramTypeRef, 
            IssueCodes.AST_NO_FUNCTIONTYPEREFS_IN_CLASSIFIERTYPEREF);
        }
      }
    }
  }
  
  @Check
  public void checkThisTypeRef(final ThisTypeRef thisTypeRef) {
    if ((thisTypeRef instanceof BoundThisTypeRef)) {
      return;
    }
    boolean _not = (!((this.isUsedStructurallyAsFormalParametersInTheConstructor(thisTypeRef) || this.isUsedAtCovariantPositionInClassifierDeclaration(thisTypeRef)) || this.isUsedInVariableWithSyntaxError(thisTypeRef)));
    if (_not) {
      this.addIssue(IssueCodes.getMessageForAST_THIS_WRONG_PLACE(), thisTypeRef, IssueCodes.AST_THIS_WRONG_PLACE);
    }
  }
  
  private boolean isUsedStructurallyAsFormalParametersInTheConstructor(final ThisTypeRef thisTypeRef) {
    boolean _isUseSiteStructuralTyping = thisTypeRef.isUseSiteStructuralTyping();
    if (_isUseSiteStructuralTyping) {
      EObject _eContainer = null;
      if (thisTypeRef!=null) {
        _eContainer=thisTypeRef.eContainer();
      }
      EObject _eContainer_1 = null;
      if (_eContainer!=null) {
        _eContainer_1=_eContainer.eContainer();
      }
      final EObject methodOrConstructor = _eContainer_1;
      if ((methodOrConstructor instanceof N4MethodDeclaration)) {
        if (((methodOrConstructor != null) && ((N4MethodDeclaration)methodOrConstructor).isConstructor())) {
          return true;
        }
      }
    }
    return false;
  }
  
  private boolean isUsedAtCovariantPositionInClassifierDeclaration(final ThisTypeRef thisTypeRef) {
    final N4ClassifierDeclaration classifierDecl = EcoreUtil2.<N4ClassifierDeclaration>getContainerOfType(thisTypeRef, N4ClassifierDeclaration.class);
    if ((classifierDecl != null)) {
      if ((classifierDecl instanceof N4InterfaceDeclaration)) {
        final N4MemberDeclaration memberDecl = EcoreUtil2.<N4MemberDeclaration>getContainerOfType(thisTypeRef, N4MemberDeclaration.class);
        if (((memberDecl != null) && memberDecl.isStatic())) {
          return false;
        }
      }
      final Variance varianceOfPos = N4JSLanguageUtils.getVarianceOfPosition(thisTypeRef);
      if ((varianceOfPos != null)) {
        return (varianceOfPos == Variance.CO);
      }
    }
    return false;
  }
  
  private boolean isUsedInVariableWithSyntaxError(final ThisTypeRef ref) {
    final EObject container = ref.eContainer();
    if ((container instanceof VariableDeclaration)) {
      String _name = ((VariableDeclaration)container).getName();
      return (_name == null);
    }
    return false;
  }
  
  @Check
  public void checkSymbolReference(final TypeRef typeRef) {
    Resource _eResource = null;
    if (typeRef!=null) {
      _eResource=typeRef.eResource();
    }
    BuiltInTypeScope bits = BuiltInTypeScope.get(_eResource.getResourceSet());
    Type _declaredType = typeRef.getDeclaredType();
    TObjectPrototype _symbolObjectType = bits.getSymbolObjectType();
    boolean _tripleEquals = (_declaredType == _symbolObjectType);
    if (_tripleEquals) {
      final EObject parent = typeRef.eContainer();
      if (((!(parent instanceof ParameterizedPropertyAccessExpression)) || 
        (((ParameterizedPropertyAccessExpression) parent).getTarget() != typeRef))) {
        this.addIssue(IssueCodes.getMessageForBIT_SYMBOL_INVALID_USE(), typeRef, IssueCodes.BIT_SYMBOL_INVALID_USE);
      }
    }
  }
  
  /**
   * Constraints 08: primitive types except any must not be declared dynamic
   */
  public void internalCheckDynamic(final ParameterizedTypeRef ref) {
    boolean _isDynamic = ref.isDynamic();
    if (_isDynamic) {
      final Type t = ref.getDeclaredType();
      if (((t instanceof PrimitiveType) && (!(t instanceof AnyType)))) {
        this.addIssue(IssueCodes.getMessageForTYS_PRIMITIVE_TYPE_DYNAMIC(t.getName()), ref, 
          IssueCodes.TYS_PRIMITIVE_TYPE_DYNAMIC);
      }
    }
  }
  
  @Check
  public void checkTypeHiddenByTypeVariable(final GenericDeclaration genDecl) {
    boolean _isEmpty = genDecl.getTypeVars().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final boolean staticAccess = ((genDecl instanceof N4MemberDeclaration) && ((N4MemberDeclaration) genDecl).isStatic());
      final IScope scope = this.n4jsScopeProvider.getTypeScope(
        genDecl.eContainer(), staticAccess);
      final Consumer<TypeVariable> _function = (TypeVariable it) -> {
        boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(it.getName());
        boolean _not_1 = (!_isNullOrEmpty);
        if (_not_1) {
          final IEObjectDescription hiddenTypeDscr = scope.getSingleElement(QualifiedName.create(it.getName()));
          EObject _eObjectOrProxy = null;
          if (hiddenTypeDscr!=null) {
            _eObjectOrProxy=hiddenTypeDscr.getEObjectOrProxy();
          }
          final EObject hiddenType = _eObjectOrProxy;
          if (((hiddenType instanceof Type) && 
            (!AbstractDescriptionWithError.isErrorDescription_XTEND_MVN_BUG_HACK(hiddenTypeDscr)))) {
            final String message = IssueCodes.getMessageForVIS_TYPE_PARAMETER_HIDES_TYPE(it.getName(), this.keywordProvider.keyword(hiddenType));
            this.addIssue(message, it, IssueCodes.VIS_TYPE_PARAMETER_HIDES_TYPE);
          }
        }
      };
      genDecl.getTypeVars().forEach(_function);
    }
  }
  
  /**
   * This handles a special case that is not checked by method {@link #checkTypeMatchesExpectedType(Expression)}.
   * In a compound assignment like += or *=, the left-hand side is both read from and written to. So we have
   * to check 1) that the type for write access is correct and 2) the type of read access is correct. Usually
   * these two types are the same, but in case of a getter/setter pair they can be different and need to be
   * checked individually. Case 1) is taken care of by method {@link #checkTypeMatchesExpectedType(Expression)}.
   * Case 2) is checked here in this method.
   */
  @Check
  public void checkCompoundAssignmentGetterSetterClashOnLhs(final AssignmentExpression assExpr) {
    if (((assExpr.getOp() == null) || (assExpr.getOp() == AssignmentOperator.ASSIGN))) {
      return;
    }
    final Expression lhs = assExpr.getLhs();
    if ((lhs instanceof ParameterizedPropertyAccessExpression)) {
      final IdentifiableElement prop = ((ParameterizedPropertyAccessExpression)lhs).getProperty();
      if ((prop instanceof TSetter)) {
        final TSetter setter = ((TSetter)prop);
        TypeRef _tau = this.ts.tau(((ParameterizedPropertyAccessExpression)lhs).getTarget());
        Type _declaredType = null;
        if (_tau!=null) {
          _declaredType=_tau.getDeclaredType();
        }
        final Type targetType = _declaredType;
        if ((targetType instanceof ContainerType<?>)) {
          final TMember m = this.containerTypesHelper.fromContext(assExpr).findMember(((ContainerType<?>)targetType), setter.getName(), 
            false, setter.isStatic());
          if ((m == null)) {
            final String message = IssueCodes.getMessageForTYS_COMPOUND_MISSING_GETTER();
            this.addIssue(message, assExpr.getLhs(), IssueCodes.TYS_COMPOUND_MISSING_GETTER);
          } else {
            if ((m instanceof TGetter)) {
              final TGetter getter = ((TGetter)m);
              RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(assExpr);
              final TypeRef expectedType = this.ts.expectedType(G, assExpr, assExpr.getRhs());
              final TypeRef typeOfGetterRAW = TypeUtils.getMemberTypeRef(getter);
              if (((expectedType != null) && (typeOfGetterRAW != null))) {
                final TypeRef typeOfGetter = this.ts.substTypeVariables(G, typeOfGetterRAW);
                if ((typeOfGetter != null)) {
                  final Result result = this.ts.subtype(G, typeOfGetter, expectedType);
                  this.createTypeError(result, assExpr.getLhs());
                }
              }
            }
          }
        }
      }
    }
  }
  
  @Check
  public void checkInconsistentInterfaceImplementationOrExtension(final N4ClassifierDeclaration classifierDecl) {
    Type _definedType = classifierDecl.getDefinedType();
    final TClassifier tClassifier = ((TClassifier) _definedType);
    if ((tClassifier == null)) {
      return;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(classifierDecl);
    RuleEnvironmentExtensions.recordInconsistentSubstitutions(G);
    final Consumer<ParameterizedTypeRef> _function = (ParameterizedTypeRef it) -> {
      this.tsh.addSubstitutions(G, it);
    };
    tClassifier.getSuperClassifierRefs().forEach(_function);
    Set<TypeVariable> _typeMappingKeys = RuleEnvironmentExtensions.getTypeMappingKeys(G);
    for (final TypeVariable tv : _typeMappingKeys) {
      if (((!tv.isDeclaredCovariant()) && (!tv.isDeclaredContravariant()))) {
        final TypeRef subst = this.ts.substTypeVariables(G, TypeUtils.createTypeRef(tv));
        if ((subst instanceof UnknownTypeRef)) {
          final List<TypeRef> badSubst = RuleEnvironmentExtensions.getInconsistentSubstitutions(G, tv);
          boolean _isEmpty = badSubst.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            boolean _allEqualType = this.tsh.allEqualType(G, ((TypeRef[])Conversions.unwrapArray(badSubst, TypeRef.class)));
            boolean _not_1 = (!_allEqualType);
            if (_not_1) {
              String _xifexpression = null;
              if ((tClassifier instanceof TClass)) {
                _xifexpression = "implement";
              } else {
                _xifexpression = "extend";
              }
              final String mode = _xifexpression;
              EObject _eContainer = tv.eContainer();
              final String ifcName = ((TInterface) _eContainer).getName();
              String _name = tv.getName();
              final String tvName = ("invariant " + _name);
              final Function1<TypeRef, String> _function_1 = (TypeRef it) -> {
                return it.getTypeRefAsString();
              };
              final String typeRefsStr = IterableExtensions.join(ListExtensions.<TypeRef, String>map(badSubst, _function_1), ", ");
              final String message = IssueCodes.getMessageForCLF_IMPLEMENT_EXTEND_SAME_INTERFACE_INCONSISTENTLY(mode, ifcName, tvName, typeRefsStr);
              this.addIssue(message, classifierDecl, N4JSPackage.eINSTANCE.getN4TypeDeclaration_Name(), 
                IssueCodes.CLF_IMPLEMENT_EXTEND_SAME_INTERFACE_INCONSISTENTLY);
            }
          }
        }
      }
    }
  }
  
  /**
   * This tests ALL expressions, including expressions used on right hand side of assignments or property definitions. That is,
   * there is no need to test field declarations or property declarations separately, as this will lead to duplicate error
   * messages.
   */
  @Check
  public void checkTypeMatchesExpectedType(final Expression expression) {
    boolean _requireCheckTypeMatchesExpectedType = this.jsVariantHelper.requireCheckTypeMatchesExpectedType(expression);
    boolean _not = (!_requireCheckTypeMatchesExpectedType);
    if (_not) {
      return;
    }
    if ((expression instanceof ExpressionAnnotationList)) {
      return;
    }
    RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(expression);
    final TypeRef inferredType = this.ts.type(G, expression);
    if ((inferredType instanceof UnknownTypeRef)) {
      return;
    }
    G = RuleEnvironmentExtensions.newRuleEnvironment(expression);
    final TypeRef expectedTypeRef = this.ts.expectedType(G, expression.eContainer(), expression);
    if ((expectedTypeRef != null)) {
      final ArrowFunction singleExprArrowFunction = N4JSASTUtils.getContainingSingleExpressionArrowFunction(expression);
      if (((singleExprArrowFunction != null) && TypeUtils.isVoid(inferredType))) {
        if ((TypeUtils.isVoid(expectedTypeRef) || singleExprArrowFunction.isReturnValueOptional())) {
          return;
        }
        TypeRef _returnTypeRef = singleExprArrowFunction.getReturnTypeRef();
        boolean _tripleEquals = (_returnTypeRef == null);
        if (_tripleEquals) {
          final String message = IssueCodes.getMessageForFUN_SINGLE_EXP_LAMBDA_IMPLICIT_RETURN_ALLOWED_UNLESS_VOID();
          this.addIssue(message, expression, 
            IssueCodes.FUN_SINGLE_EXP_LAMBDA_IMPLICIT_RETURN_ALLOWED_UNLESS_VOID);
          return;
        }
      }
      this.internalCheckUseOfUndefinedExpression(G, expression, expectedTypeRef, inferredType);
      final boolean writeAccess = ExpressionExtensions.isLeftHandSide(expression);
      if (writeAccess) {
        final Result result = this.ts.subtype(G, expectedTypeRef, inferredType);
        boolean _isFailure = result.isFailure();
        if (_isFailure) {
          final String message_1 = IssueCodes.getMessageForTYS_NO_SUPERTYPE_WRITE_ACCESS(expectedTypeRef.getTypeRefAsString(), 
            inferredType.getTypeRefAsString());
          this.addIssue(message_1, expression, IssueCodes.TYS_NO_SUPERTYPE_WRITE_ACCESS);
        }
      } else {
        final Result result_1 = this.ts.subtype(G, inferredType, expectedTypeRef);
        final boolean errorCreated = this.createTypeError(result_1, expression);
        if ((!errorCreated)) {
          this.internalCheckSuperfluousPropertiesInObjectLiteralRek(G, expectedTypeRef, expression);
        }
      }
    }
  }
  
  public void internalCheckSuperfluousPropertiesInObjectLiteralRek(final RuleEnvironment G, final TypeRef expectedTypeRef, final Expression expression) {
    if ((expression instanceof ObjectLiteral)) {
      this.internalCheckSuperfluousPropertiesInObjectLiteral(expectedTypeRef, ((ObjectLiteral)expression));
    } else {
      if ((expression instanceof ArrayLiteral)) {
        boolean _isEmpty = expectedTypeRef.getTypeArgs().isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          final TypeArgument arrayElementType = expectedTypeRef.getTypeArgs().get(0);
          final TypeRef typeArgTypeRef = this.ts.resolveType(G, arrayElementType);
          EList<ArrayElement> _elements = ((ArrayLiteral)expression).getElements();
          for (final ArrayElement arrElem : _elements) {
            {
              final Expression arrExpr = arrElem.getExpression();
              this.internalCheckSuperfluousPropertiesInObjectLiteralRek(G, typeArgTypeRef, arrExpr);
            }
          }
        }
      }
    }
  }
  
  /**
   * #225: always check for superfluous properties in object literal
   * req-id IDE-22501
   */
  public void internalCheckSuperfluousPropertiesInObjectLiteral(final TypeRef typeRef, final ObjectLiteral objectLiteral) {
    final TypingStrategy typingStrategy = typeRef.getTypingStrategy();
    if (((!Objects.equal(typingStrategy, TypingStrategy.NOMINAL)) && (!Objects.equal(typingStrategy, TypingStrategy.DEFAULT)))) {
      boolean _isDynamic = typeRef.isDynamic();
      if (_isDynamic) {
        return;
      }
      Type type = typeRef.getDeclaredType();
      if (((type == null) && (typeRef instanceof BoundThisTypeRef))) {
        ParameterizedTypeRef _actualThisTypeRef = ((BoundThisTypeRef) typeRef).getActualThisTypeRef();
        Type _declaredType = null;
        if (_actualThisTypeRef!=null) {
          _declaredType=_actualThisTypeRef.getDeclaredType();
        }
        type = _declaredType;
      }
      if ((!(type instanceof ContainerType<?>))) {
        return;
      }
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(objectLiteral);
      final EList<TStructMember> structuralMembers = typeRef.getStructuralMembers();
      if ((structuralMembers.isEmpty() && Objects.equal(type, RuleEnvironmentExtensions.objectType(G)))) {
        return;
      }
      final TMethod ctor = this.containerTypesHelper.fromContext(objectLiteral).findConstructor(((ContainerType<?>) type));
      boolean _isSpecArgumentToSpecCtor = N4JSTypeValidator.isSpecArgumentToSpecCtor(objectLiteral, ctor);
      final TypingStrategyFilter strategyFilter = new TypingStrategyFilter(typingStrategy, 
        (typingStrategy == TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS), _isSpecArgumentToSpecCtor);
      final Function1<TMember, Boolean> _function = (TMember member) -> {
        return Boolean.valueOf(strategyFilter.apply(member));
      };
      final Function1<TMember, String> _function_1 = (TMember member) -> {
        return member.getName();
      };
      final Set<String> expectedMembers = IterableExtensions.<String>toSet(IterableExtensions.<TMember, String>map(IterableExtensions.<TMember>filter(this.containerTypesHelper.fromContext(objectLiteral).allMembers(
        ((ContainerType<?>) type)), _function), _function_1));
      final Consumer<TStructMember> _function_2 = (TStructMember member) -> {
        expectedMembers.add(member.getName());
      };
      typeRef.getStructuralMembers().forEach(_function_2);
      Type _definedType = objectLiteral.getDefinedType();
      final EList<? extends TMember> inputMembers = ((ContainerType<?>) _definedType).getOwnedMembers();
      for (final TMember property : inputMembers) {
        boolean _contains = expectedMembers.contains(property.getName());
        boolean _not = (!_contains);
        if (_not) {
          EObject astElement = property.getAstElement();
          if ((astElement instanceof PropertyNameValuePair)) {
            astElement = ((PropertyNameValuePair)astElement).getDeclaredName();
          }
          final String message = IssueCodes.getMessageForCLF_SPEC_SUPERFLUOUS_PROPERTIES(property.getName(), 
            typeRef.getTypeRefAsString());
          this.addIssue(message, astElement, IssueCodes.CLF_SPEC_SUPERFLUOUS_PROPERTIES);
        }
      }
    }
  }
  
  private void internalCheckUseOfUndefinedExpression(final RuleEnvironment G, final Expression expression, final TypeRef expectedTypeRef, final TypeRef actualTypeRef) {
    if ((TypeUtils.isUndefined(actualTypeRef) && (!TypeUtils.isUndefined(expectedTypeRef)))) {
      final EObject parent = expression.eContainer();
      if (((((!(parent instanceof ExpressionStatement)) && 
        (!((parent instanceof UnaryExpression) && (((UnaryExpression) parent).getOp() == UnaryOperator.VOID)))) && 
        (!((expression instanceof UnaryExpression) && 
          (((UnaryExpression) expression).getOp() == UnaryOperator.VOID)))) && 
        (!(expression instanceof ThisLiteral)))) {
        TMember _findOwnedMember = RuleEnvironmentExtensions.globalObjectType(G).findOwnedMember("undefined", false, false);
        final TField undefinedField = ((TField) _findOwnedMember);
        boolean _xifexpression = false;
        if ((expression instanceof IdentifierRef)) {
          IdentifiableElement _id = ((IdentifierRef)expression).getId();
          _xifexpression = (_id == undefinedField);
        }
        final boolean isUndefinedLiteral = _xifexpression;
        if ((!isUndefinedLiteral)) {
          this.addIssue(IssueCodes.getMessageForEXP_USE_OF_UNDEF_EXPR(), expression, IssueCodes.EXP_USE_OF_UNDEF_EXPR);
        }
      }
    }
  }
  
  @Check
  public void checkBogusTypeReference(final TypedElement te) {
    TypeRef _bogusTypeRef = te.getBogusTypeRef();
    boolean _tripleNotEquals = (_bogusTypeRef != null);
    if (_tripleNotEquals) {
      this.addIssue(IssueCodes.getMessageForTYS_INVALID_TYPE_SYNTAX(), te.getBogusTypeRef(), IssueCodes.TYS_INVALID_TYPE_SYNTAX);
    }
  }
  
  /**
   * This validates a warning in chapter 4.10.1:<br/>
   * <i>The use of the any type in a union type produces a warning.</i>
   */
  @Check
  public void checkUnionTypeContainsNoAny(final UnionTypeExpression ute) {
    this.checkComposedTypeRefContainsNoAny(ute, IssueCodes.getMessageForUNI_ANY_USED(), IssueCodes.UNI_ANY_USED, true);
  }
  
  /**
   * This validates a warning in chapter 4.10.2:<br/>
   * <i>The use of the any type in an intersection type produces a warning.</i>
   */
  @Check
  public void checkIntersectionTypeContainsNoAny(final IntersectionTypeExpression ite) {
    this.checkComposedTypeRefContainsNoAny(ite, IssueCodes.getMessageForINTER_ANY_USED(), IssueCodes.INTER_ANY_USED, false);
  }
  
  private void checkComposedTypeRefContainsNoAny(final ComposedTypeRef ctr, final String msg, final String issueCode, final boolean soleVoidAllowesAny) {
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ctr);
    final AnyType anyType = RuleEnvironmentExtensions.anyType(G);
    final VoidType voidType = RuleEnvironmentExtensions.voidType(G);
    final EList<TypeRef> typeRefs = ctr.getTypeRefs();
    final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
      Type _declaredType = it.getDeclaredType();
      return Boolean.valueOf((_declaredType == anyType));
    };
    final Iterable<TypeRef> anyTypeRefs = IterableExtensions.<TypeRef>filter(typeRefs, _function);
    boolean dontShowWarning = false;
    if (soleVoidAllowesAny) {
      final Function1<TypeRef, Boolean> _function_1 = (TypeRef it) -> {
        Type _declaredType = it.getDeclaredType();
        return Boolean.valueOf((_declaredType == voidType));
      };
      final boolean containsVoid = IterableExtensions.<TypeRef>exists(typeRefs, _function_1);
      dontShowWarning = (containsVoid && (IterableExtensions.size(anyTypeRefs) == 1));
    }
    if ((!dontShowWarning)) {
      for (final TypeRef anyTR : anyTypeRefs) {
        this.addIssue(msg, anyTR, issueCode);
      }
    }
  }
  
  /**
   * This validates a warning in chapter 4.10.1:<br/>
   * <i>The use of unnecessary subtypes in union types produces a warning.</i>
   */
  @Check
  public void checkUnionHasUnnecessarySubtype(final UnionTypeExpression ute) {
    final List<TypeRef> tRefs = this.extractNonStructTypeRefs(ute);
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ute);
    final AnyType anyType = RuleEnvironmentExtensions.anyType(G);
    final Predicate<TypeRef> _function = (TypeRef it) -> {
      Type _declaredType = it.getDeclaredType();
      return (_declaredType == anyType);
    };
    tRefs.removeIf(_function);
    int _size = tRefs.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      final List<TypeRef> intersectionTR = this.tsh.getSuperTypesOnly(G, ((TypeRef[])Conversions.unwrapArray(tRefs, TypeRef.class)));
      tRefs.removeAll(intersectionTR);
      for (final TypeRef tClassR : tRefs) {
        {
          final String message = IssueCodes.getMessageForUNI_REDUNDANT_SUBTYPE();
          this.addIssue(message, tClassR, IssueCodes.UNI_REDUNDANT_SUBTYPE);
        }
      }
    }
  }
  
  /**
   * Entry method for validating the containing types of an intersection type.
   */
  @Check
  public void checkIntersectionType(final IntersectionTypeExpression ite) {
    final List<TypeRef> tClassRefs = this.extractNonStructTypeRefs(ite);
    int _size = tClassRefs.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ite);
      final List<TypeRef> intersectionTR = this.tsh.getSubtypesOnly(G, ((TypeRef[])Conversions.unwrapArray(tClassRefs, TypeRef.class)));
      this.checkIntersectionTypeContainsMaxOneClass(G, intersectionTR, false);
      this.checkIntersectionHasUnnecessarySupertype(tClassRefs, intersectionTR);
    }
  }
  
  /**
   * This validates constraint 25.2 ("Intersection Type") in chapter 4.10.2:<br/>
   * <i>Only one class must be contained in the intersection type.</i><br/><br/>
   * Currently, only a warning is displayed.
   */
  private void checkIntersectionTypeContainsMaxOneClass(final RuleEnvironment G, final List<TypeRef> intersectionTR, final boolean covariantTypeArgValidation) {
    int _size = intersectionTR.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      final ArrayListMultimap<Type, TypeRef> byTypes = ArrayListMultimap.<Type, TypeRef>create();
      for (final TypeRef tClassR : intersectionTR) {
        byTypes.put(tClassR.getDeclaredType(), tClassR);
      }
      int _size_1 = byTypes.keySet().size();
      boolean _greaterThan_1 = (_size_1 > 1);
      if (_greaterThan_1) {
        if (covariantTypeArgValidation) {
          final String message = IssueCodes.getMessageForINTER_TYEPARGS_ONLY_ONE_CLASS_ALLOWED();
          for (final TypeRef tClassR_1 : intersectionTR) {
            EObject _eContainer = tClassR_1.eContainer();
            boolean _not = (!(_eContainer instanceof TypeVariable));
            if (_not) {
              this.addIssue(message, tClassR_1, IssueCodes.INTER_TYEPARGS_ONLY_ONE_CLASS_ALLOWED);
            }
          }
        } else {
          final String message_1 = IssueCodes.getMessageForINTER_ONLY_ONE_CLASS_ALLOWED();
          for (final TypeRef tClassR_2 : intersectionTR) {
            this.addIssue(message_1, tClassR_2, IssueCodes.INTER_ONLY_ONE_CLASS_ALLOWED);
          }
        }
      } else {
        final Type type = IterableExtensions.<Type>head(byTypes.keys());
        boolean _isGeneric = type.isGeneric();
        if (_isGeneric) {
          final List<TypeRef> ptrs = byTypes.get(type);
          boolean _allCovariantOrWildcardWithUpperBound = this.allCovariantOrWildcardWithUpperBound(type.getTypeVars(), ptrs);
          if (_allCovariantOrWildcardWithUpperBound) {
            final int length = ((Object[])Conversions.unwrapArray(type.getTypeVars(), Object.class)).length;
            for (int v = 0; (v < length); v++) {
              {
                final int vIndex = v;
                final Function1<TypeRef, TypeRef> _function = (TypeRef ptr) -> {
                  final TypeArgument ta = ptr.getTypeArgs().get(vIndex);
                  TypeRef upper = null;
                  if ((ta instanceof TypeRef)) {
                    upper = ((TypeRef)ta);
                  }
                  if (((upper == null) && (ta instanceof Wildcard))) {
                    upper = ((Wildcard) ta).getDeclaredUpperBound();
                  }
                  if ((upper == null)) {
                    upper = type.getTypeVars().get(vIndex).getDeclaredUpperBound();
                  }
                  return upper;
                };
                final List<TypeRef> typeArgsPerVariable = this.extractNonStructTypeRefs(
                  ListExtensions.<TypeRef, TypeRef>map(ptrs, _function));
                this.checkIntersectionTypeContainsMaxOneClass(G, typeArgsPerVariable, true);
              }
            }
          } else {
            final Function1<TypeRef, Boolean> _function = (TypeRef ptr) -> {
              final Function1<TypeArgument, Boolean> _function_1 = (TypeArgument ta) -> {
                return Boolean.valueOf(((ta instanceof Wildcard) && 
                  (((Wildcard) ta).getDeclaredLowerBound() != null)));
              };
              return Boolean.valueOf(IterableExtensions.<TypeArgument>forall(ptr.getTypeArgs(), _function_1));
            };
            boolean _forall = IterableExtensions.<TypeRef>forall(ptrs, _function);
            if (_forall) {
            } else {
              final String message_2 = IssueCodes.getMessageForINTER_WITH_ONE_GENERIC();
              for (final TypeRef tClassR_3 : intersectionTR) {
                EObject _eContainer_1 = tClassR_3.eContainer();
                boolean _not_1 = (!(_eContainer_1 instanceof TypeVariable));
                if (_not_1) {
                  this.addIssue(message_2, tClassR_3, IssueCodes.INTER_WITH_ONE_GENERIC);
                }
              }
            }
          }
        }
      }
    }
  }
  
  private boolean allCovariantOrWildcardWithUpperBound(final List<TypeVariable> typeVars, final List<TypeRef> refs) throws IndexOutOfBoundsException {
    final int length = ((Object[])Conversions.unwrapArray(typeVars, Object.class)).length;
    for (int i = 0; (i < length); i++) {
      boolean _isDeclaredCovariant = typeVars.get(i).isDeclaredCovariant();
      boolean _not = (!_isDeclaredCovariant);
      if (_not) {
        for (final TypeRef ref : refs) {
          {
            final TypeArgument ta = ref.getTypeArgs().get(i);
            if ((ta instanceof Wildcard)) {
              TypeRef _declaredUpperBound = ((Wildcard)ta).getDeclaredUpperBound();
              boolean _tripleEquals = (_declaredUpperBound == null);
              if (_tripleEquals) {
                return false;
              }
            } else {
              return false;
            }
          }
        }
      }
    }
    return true;
  }
  
  /**
   * This validates a warning in chapter 4.10.2:<br/>
   * <i>The use of unnecessary supertypes in intersection types produces a warning.</i>
   */
  private void checkIntersectionHasUnnecessarySupertype(final List<TypeRef> tClassRefs, final List<TypeRef> intersectionTR) {
    tClassRefs.removeAll(intersectionTR);
    for (final TypeRef tClassR : tClassRefs) {
      {
        final String message = IssueCodes.getMessageForINTER_REDUNDANT_SUPERTYPE();
        this.addIssue(message, tClassR, IssueCodes.INTER_REDUNDANT_SUPERTYPE);
      }
    }
  }
  
  private List<TypeRef> extractNonStructTypeRefs(final ComposedTypeRef ctr) {
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(ctr);
    final List<TypeRef> tRefs = this.tsh.getSimplifiedTypeRefs(G, ctr);
    return this.extractNonStructTypeRefs(tRefs);
  }
  
  private List<TypeRef> extractNonStructTypeRefs(final List<TypeRef> simplifiedTypeRefs) {
    final List<TypeRef> tClassRefs = new LinkedList<TypeRef>();
    for (final TypeRef tR : simplifiedTypeRefs) {
      if ((tR != null)) {
        final Type type = tR.getDeclaredType();
        if ((type instanceof TClass)) {
          boolean isStructural = (tR.isDefSiteStructuralTyping() || tR.isUseSiteStructuralTyping());
          if ((!isStructural)) {
            tClassRefs.add(tR);
          }
        }
      }
    }
    return tClassRefs;
  }
  
  private static boolean isSpecArgumentToSpecCtor(final Expression expr, final TMethod ctor) {
    if ((ctor == null)) {
      return false;
    }
    EObject _eContainer = null;
    if (expr!=null) {
      _eContainer=expr.eContainer();
    }
    final EObject parent = _eContainer;
    EObject _eContainer_1 = null;
    if (parent!=null) {
      _eContainer_1=parent.eContainer();
    }
    final EObject grandParent = _eContainer_1;
    if ((parent instanceof Argument)) {
      if ((grandParent instanceof NewExpression)) {
        final int argIdx = ((NewExpression)grandParent).getArguments().indexOf(parent);
        final TFormalParameter ctorFpar = ctor.getFparForArgIdx(argIdx);
        if ((ctorFpar != null)) {
          return AnnotationDefinition.SPEC.hasAnnotation(ctorFpar);
        }
      }
    }
    return false;
  }
}
