/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.BindingPattern;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDefinition;
import org.eclipse.n4js.n4JS.N4ClassExpression;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.NewTarget;
import org.eclipse.n4js.n4JS.PropertySpread;
import org.eclipse.n4js.n4JS.TaggedTemplateString;
import org.eclipse.n4js.validation.ASTStructureValidator;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * Validations to show an error for unsupported language features, mostly ECMAScript6 features.
 * These validations will be removed over time once the corresponding features are implemented.
 */
@SuppressWarnings("all")
public class UnsupportedFeatureValidator extends AbstractN4JSDeclarativeValidator {
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkEmptyImport(final ImportDeclaration importDecl) {
    boolean _isEmpty = importDecl.getImportSpecifiers().isEmpty();
    if (_isEmpty) {
      this.unsupported("empty import", importDecl);
    }
  }
  
  @Check
  public void checkAnonymousDefaultExport(final ExportableElement decl) {
    if ((decl instanceof NamedElement)) {
      if (((((NamedElement)decl).getName() == null) && decl.isExportedAsDefault())) {
        this.unsupported("anonymous default export", decl);
      }
    }
  }
  
  @Check
  public void checkSeparateExport(final ExportDeclaration exportDecl) {
    ExportableElement _exportedElement = exportDecl.getExportedElement();
    boolean _tripleEquals = (_exportedElement == null);
    if (_tripleEquals) {
      if ((exportDecl.isDefaultExport() && (exportDecl.getDefaultExportedExpression() != null))) {
        this.unsupported(
          "exporting values (only declared classes, interfaces, enums, functions and variables can be exported)", exportDecl);
      } else {
        this.unsupported(
          "separate export statements (add keyword \'export\' directly before a class, interface, enum, function or variable declaration)", exportDecl);
      }
    }
  }
  
  @Check
  public void checkTaggedTemplateLiteral(final TaggedTemplateString tts) {
    this.unsupported("tagged template literals", tts, N4JSPackage.eINSTANCE.getTaggedTemplateString_Target());
  }
  
  @Check
  public void checkClassExpression(final N4ClassExpression classExpr) {
    if ((!UnsupportedFeatureValidator.classExpressionsAreAllowed)) {
      String _name = classExpr.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        this.unsupported("class expressions", classExpr, N4JSPackage.eINSTANCE.getN4ClassExpression_Name());
      } else {
        this.unsupported("class expressions", classExpr, NodeModelUtils.getNode(classExpr).getOffset(), 5);
      }
    }
  }
  
  private static boolean classExpressionsAreAllowed = false;
  
  @Check
  public void checkNewTarget(final NewTarget newTarget) {
    this.unsupported("new.target", newTarget);
  }
  
  @Check
  public void checkExtendsExpression(final N4ClassDefinition classDef) {
    Expression _superClassExpression = classDef.getSuperClassExpression();
    boolean _tripleNotEquals = (_superClassExpression != null);
    if (_tripleNotEquals) {
      this.unsupported("extends <expression>", classDef, N4JSPackage.eINSTANCE.getN4ClassDefinition_SuperClassExpression());
    }
  }
  
  @Check
  public void checkBindingPatternAsFpar(final BindingPattern pattern) {
    EObject _eContainer = pattern.eContainer();
    if ((_eContainer instanceof FormalParameter)) {
      this.unsupported("destructuring patterns as formal parameter", pattern);
    }
  }
  
  /**
   * NOTE: in addition to the errors produced by this and the following method, spread operator
   * in <em>array literals</em> is also unsupported; but that is checked in
   * {@link ASTStructureValidator#validateSpreadInArrayLiteral(org.eclipse.n4js.n4JS.ArrayElement,
   * org.eclipse.n4js.validation.ASTStructureDiagnosticProducer)}
   */
  @Check
  public void checkSpreadOperatorInNewAndCallExpressions(final Argument argument) {
    boolean _isSpread = argument.isSpread();
    if (_isSpread) {
      this.unsupported("spread operator in new and call expressions (only allowed in array destructuring patterns)", argument, N4JSPackage.eINSTANCE.getArgument_Spread());
    }
  }
  
  @Check
  public void checkSpreadOperatorInObjectLiteral(final PropertySpread propertySpread) {
    this.unsupported("spread operator in object literals (only allowed in array destructuring patterns)", propertySpread);
  }
  
  private void unsupported(final String msg, final EObject source) {
    this.unsupported(msg, source, null);
  }
  
  private void unsupported(final String msg, final EObject source, final EStructuralFeature feature) {
    this.addIssue(
      IssueCodes.getMessageForUNSUPPORTED(msg), source, feature, 
      IssueCodes.UNSUPPORTED);
  }
  
  private void unsupported(final String msg, final EObject source, final int offset, final int length) {
    this.addIssue(
      IssueCodes.getMessageForUNSUPPORTED(msg), source, offset, length, 
      IssueCodes.UNSUPPORTED);
  }
  
  /**
   * Turns off unsupported feature validation for class expressions, invokes given function, and turns validation
   * back on (for testing of class expressions).
   */
  public static void allowClassExpressions(final Procedure0 r) {
    try {
      UnsupportedFeatureValidator.classExpressionsAreAllowed = true;
      r.apply();
    } finally {
      UnsupportedFeatureValidator.classExpressionsAreAllowed = false;
    }
  }
}
